{
  LineEd.Pas

  LineEd line-based editor designed specifically for use with BBSkit.

  Written by Steve Madsen
}

UNIT LineEd;

{$V-}

{********************************************************************}

INTERFACE

Uses Prompts;

Const
  MaxLines = 100;

Type
  TEdData = Array[1..MaxLines] of String[80];

  TLineEditor = object(TPrompts)
    LineEdData : TEdData;
    CMark      : Byte;

    CONSTRUCTOR Init;

    FUNCTION LineEditor : Integer;
    PROCEDURE SetMarkedColor(Fore, Back : Byte);
    PROCEDURE LineEd_SetNewParagraph(Spaces : Byte);
    PROCEDURE LineEd_SetRightMargin(Col : Byte);

  private

    NewParagraph : Byte;
    NumLines     : Integer;
    RtMargin     : Byte;

    PROCEDURE CenterLine;
    PROCEDURE DeleteLines;
    PROCEDURE InsertLines;
    PROCEDURE ListEditor;
    PROCEDURE SearchAndReplace;
  end;

{********************************************************************}

IMPLEMENTATION

Uses Util;

Const
  Version = '1.3';

{--------------------------------------------------------------------}

CONSTRUCTOR TLineEditor.Init;
 begin
   TPrompts.Init;
   CMark := $1F;
   RtMargin := 74;
   NewParagraph := 4;
 end;

{--------------------------------------------------------------------}

FUNCTION TLineEditor.LineEditor : Integer;
 Var
   Wrap     : String[80];
   Quit     : Boolean;

 begin
   ComWriteLn('LineEd v' + Version + ': /S on a blank line to save, /? for help; 100 lines max');
   ComWriteLn('');
   ChangeColor(CHighlight);
   ComWriteLn('     [' + Replicate('-', RtMargin - 2) + ']');
   ChangeColor(CNorm);
   Wrap := '';
   NumLines := 0;
   Quit := False;
   Repeat
     Inc(NumLines);
     if (NumLines = 101) then
      begin
        ComWriteLn('');
        Quit := True;
        case Prompt('100 lines entered: ~S~ave or ~A~bort? ') of
          'S' : begin
                  ComWriteLn('Save');
                  NumLines := 100;
                end;
          'A' : begin
                  ComWriteLn('Abort');
                  NumLines := 255;
                end;
        end;
      end
     else
      begin
        ChangeColor(CHighlight);
        ComWrite(Right(IntToStr(NumLines), 3) + ': ');
        ChangeColor(CNorm);
        LineEdData[NumLines] := '';
        ComReadLnWrap(LineEdData[NumLines], RtMargin, Wrap);
        if (Length(LineEdData[NumLines]) = 2) and (UpCase(LineEdData[NumLines][1]) = '/') then
         begin
           case UpCase(LineEdData[NumLines][2]) of
             '?' : begin
                     ComWriteLn('');
                     ComWriteLn('               LineEd v' + Version + ' Help');
                     ComWriteLn('');
                     ComWriteLn('        /A   Abort editor; lose text');
                     ComWriteLn('        /C   Center line(s)');
                     ComWriteLn('        /D   Delete line(s)');
                     ComWriteLn('        /I   Insert blank line(s)');
                     ComWriteLn('        /L   List editor text');
                     ComWriteLn('        /N   New; lose text');
                     ComWriteLn('        /R   Search & replace');
                     ComWriteLn('        /S   Save');
                     ComWriteLn('');
                   end;
             'A' : begin
                     ComWriteLn('');
                     ComWrite('Abort: sure? ');
                     if (YNPrompt(False)) then
                      begin
                        NumLines := 0;
                        Quit := True;
                      end
                     else ComWriteLn('');
                   end;
             'C' : CenterLine;
             'D' : DeleteLines;
             'I' : InsertLines;
             'L' : ListEditor;
             'N' : begin
                     ComWriteLn('');
                     ComWrite('Clear editor text? ');
                     if (YNPrompt(False)) then
                        NumLines := 1;
                     ComWriteLn('');
                   end;
             'R' : SearchAndReplace;
             'S' : Quit := True;
           else
              Inc(NumLines);
           end;
           Dec(NumLines);
         end;
      end;
   Until (Quit);
   LineEditor := NumLines;
 end;

{--------------------------------------------------------------------}

PROCEDURE TLineEditor.LineEd_SetNewParagraph(Spaces : Byte);
 begin
   NewParagraph := Spaces;
 end;

{--------------------------------------------------------------------}

PROCEDURE TLineEditor.LineEd_SetRightMargin(Col : Byte);
 begin
   RtMargin := Col;
 end;

{--------------------------------------------------------------------}

PROCEDURE TLineEditor.SetMarkedColor(Fore, Back : Byte);
 begin
   CMark := (Back * 16) + Fore;
 end;

{********************************************************************}

PROCEDURE Switch(var A, B : Byte);
 Var
   T : Byte;

 begin
   T := B;
   B := A;
   A := T;
 end;

{--------------------------------------------------------------------}

PROCEDURE TLineEditor.CenterLine;
 Var
   Inp   : String;
   Idx   : Byte;
   First : Byte;
   Last  : Byte;

 PROCEDURE RemoveSpaces(L : Byte);
  begin
    while (Length(LineEdData[L]) > 0) and (LineEdData[L][1] = ' ') do
       LineEdData[L] := Copy(LineEdData[L], 2, Length(LineEdData[L]) - 1);
  end;

 begin
   ComWriteLn('');
   ComWrite('Center line: [L] [L1-L2] [A]ll? ');
   ComReadLn(Inp, 10);
   ComWriteLn('');
   if (Upper(Inp) = 'A') then   { all lines }
      for Idx := 1 to NumLines do
       begin
         RemoveSpaces(Idx);
         LineEdData[Idx] := Center(LineEdData[Idx]);
       end
   else
      if (Pos('-', Inp) > 0) then   { range of lines }
       begin
         First := StrToInt(Copy(Inp, 1, Pos('-', Inp) - 1));
         Last := StrToInt(Copy(Inp, Pos('-', Inp) + 1, 5));
         if (First > Last) then
            Switch(First, Last);
         if (First >= 1) and (Last <= 100) then
            for Idx := First to Last do
             begin
               RemoveSpaces(Idx);
               LineEdData[Idx] := Center(LineEdData[Idx]);
             end;
       end
      else
       begin   { one line }
         if (StrToInt(Inp) >= 1) and (StrToInt(Inp) <= 100) then
          begin
            RemoveSpaces(StrToInt(Inp));
            LineEdData[StrToInt(Inp)] := Center(LineEdData[StrToInt(Inp)]);
          end;
       end;
 end;

{--------------------------------------------------------------------}

PROCEDURE TLineEditor.DeleteLines;
 Var
   Inp   : String;
   First : Byte;
   Last  : Byte;
   Idx   : Byte;

 begin
   ComWriteLn('');
   ComWrite('Delete lines? [L] [L1-L2]: ');
   ComReadLn(Inp, 7);
   if (Inp <> '') then
    begin
      if (Pos('-', Inp) > 0) then
       begin
         First := StrToInt(Copy(Inp, 1, Pos('-', Inp) - 1));
         Last := StrToInt(Copy(Inp, Pos('-', Inp) + 1, 3));
         if (First > Last) then Switch(First, Last);
         if (First >= 1) and (Last <= NumLines) then
          begin
            for Idx := Last + 1 to NumLines do
               LineEdData[Idx - ((Last - First) + 1)] := LineEdData[Idx];
            Dec(NumLines, (Last - First) + 1);
          end;
       end
      else
       begin
         First := StrToInt(Inp);
         if (First >= 1) and (First <= NumLines) then
          begin
            for Idx := First to NumLines - 1 do
               LineEdData[Idx] := LineEdData[Idx + 1];
            Dec(NumLines);
          end;
       end;
    end;
   ComWriteLn('');
 end;

{--------------------------------------------------------------------}

PROCEDURE TLineEditor.InsertLines;
 Var
   Inp : String;
   Ln  : Byte;
   Num : Byte;
   Idx : Byte;

 begin
   ComWriteLn('');
   ComWrite('Insert lines before line #: ');
   ComReadLn(Inp, 3);
   if (Inp <> '') then
    begin
      Ln := StrToInt(Inp);
      if (Ln >= 1) and (Ln <= NumLines) then
       begin
         ComWrite('     Insert how many lines? ');
         ComReadLn(Inp, 3);
         ComWriteLn('');
         if (Inp <> '') then
          begin
            Num := StrToInt(Inp);
            if (Num + NumLines <= 100) then
             begin
               for Idx := NumLines downto Ln do
                begin
                  LineEdData[Idx + Num] := LineEdData[Idx];
                  LineEdData[Idx] := '';
                end;
               Inc(NumLines, Num);
             end
            else ComWriteLn('Not enough room in editor!');
          end;
       end
      else
       begin
         ComWriteLn('');
         ComWriteLn('That line does not exist!');
       end;
    end;
 end;

{--------------------------------------------------------------------}

PROCEDURE TLineEditor.ListEditor;
 Var
   Idx : Byte;

 begin
   if (NumLines >= 2) then
    begin
      ComWriteLn('');
      for Idx := 1 to NumLines - 1 do
         ComWriteLn(Right(IntToStr(Idx), 3) + ': ' + LineEdData[Idx]);
      ComWriteLn('');
    end
   else
    begin
      ComWriteLn('');
      ComWriteLn('No text in editor!');
      ComWriteLn('');
    end;
 end;

{--------------------------------------------------------------------}

PROCEDURE TLineEditor.SearchAndReplace;
 Type
   T25kString = Array[1..100] of String;
   P25kString = ^T25kString;

 Var
   Search        : String;
   Replace       : String;
   Inp           : String;
   PromptReplace : Boolean;
   ReplaceThis   : Boolean;
   CaseSense     : Boolean;
   WholeWords    : Boolean;
   First         : Byte;
   Last          : Byte;
   Idx           : Byte;
   Offset        : Byte;
   FormatOffset  : Byte;
   Found         : Byte;
   FoundInLine   : Byte;
   Formatted     : P25kString;

 PROCEDURE InsertOneLine(At : Byte);
  Var
    Bump : Byte;

  begin
    for Bump := 100 downto At do
       Formatted^[Bump] := Formatted^[Bump - 1];
    Inc(NumLines);
    Formatted^[At] := '';
  end;

 begin
   New(Formatted);
   ComWriteLn('');
   ComWrite('  Search for: ');
   ComReadLn(Search, RtMargin);
   if (Search <> '') then
    begin
      ComWrite('Replace with: ');
      ComReadLn(Replace, RtMargin);
      ComWriteLn('');
      ComWrite('In lines? [L] [L1-L2] [A]ll: ');
      ComReadLn(Inp, 7);
      if (Inp <> '') then
       begin
         ComWriteLn('');
         ComWrite('Prompt before replace? ');
         PromptReplace := YNPrompt(True);
         ComWriteLn('');
         ComWrite('Case sensitive? ');
         CaseSense := YNPrompt(False);
         ComWriteLn('');
         ComWrite('Whole words only? ');
         WholeWords := YNPrompt(False);
         ComWriteLn('');
         if (Upper(Inp) = 'A') then Inp := '1-' + IntToStr(NumLines - 1);
         if (Pos('-', Inp) > 0) then
          begin
            First := StrToInt(Copy(Inp, 1, Pos('-', Inp) - 1));
            Last := StrToInt(Copy(Inp, Pos('-', Inp) + 1, 3));
            if (First > Last) then Switch(First, Last);
          end
         else
          begin
            First := StrToInt(Inp);
            Last := First;
          end;
         if (First >= 1) and (Last <= NumLines - 1) then
          begin
            for Idx := 1 to 100 do
               if (Idx <= NumLines - 1) then Formatted^[Idx] := LineEdData[Idx]
               else Formatted^[Idx] := '';
            if (not CaseSense) then Search := Upper(Search);
            if (not PromptReplace) then ReplaceThis := True;
            for Idx := First to Last do
             begin
               Offset := 1;
               FormatOffset := 0;
               FoundInLine := 0;
               while (Offset > 0) do
                begin
                  if (WholeWords) then
                     if (CaseSense) then
                        Found := XPos(LineEdData[Idx], ' ' + Search + ' ', Offset)
                     else
                        Found := XPos(Upper(LineEdData[Idx]), ' ' + Search + ' ', Offset)
                  else
                     if (CaseSense) then
                        Found := XPos(LineEdData[Idx], Search, Offset)
                     else
                        Found := XPos(Upper(LineEdData[Idx]), Search, Offset);
                  if (Found > 0) then
                   begin
                     if (PromptReplace) then
                      begin
                        ChangeColor(CNorm);
                        ComWrite(Copy(LineEdData[Idx], 1, Found - 1));
                        ChangeColor(CMark);
                        ComWrite(Copy(LineEdData[Idx], Found, Length(Search)));
                        ChangeColor(CNorm);
                        ComWriteLn(Copy(LineEdData[Idx], Found + Length(Search), 80));
                        ComWriteLn('');
                        ComWrite('Replace this occurance? ');
                        ReplaceThis := YNPrompt(False);
                        ComWriteLn('');
                      end;
                     if (FormatOffset = 0) then FormatOffset := Found
                     else FormatOffset := Found + (Abs(Length(Search) - Length(Replace)) * FoundInLine);
                     if (ReplaceThis) then
                      begin
                        Formatted^[Idx] := Copy(Formatted^[Idx], 1, FormatOffset - 1) +
                                           Replace +
                                           Copy(Formatted^[Idx], FormatOffset + Length(Search), 255);
                        Inc(FormatOffset, Length(Replace));
                      end;
                     Offset := Found + Length(Search);
                     Inc(FoundInLine);
                   end
                  else Offset := 0;
                end;
             end;
            ComWrite('No further occurrances found.  Reformatting text...');
            for Idx := 1 to 100 do
             begin
               if (Length(Formatted^[Idx]) > RtMargin) then
                begin
                  Last := RtMargin;
                  while (Formatted^[Idx][Last] <> ' ') do
                     Dec(Last);
                  if (Copy(Formatted^[Idx + 1], 1, NewParagraph) = Replicate(' ', NewParagraph)) then
                     InsertOneLine(Idx + 1);
                  Formatted^[Idx + 1] := Copy(Formatted^[Idx], Last, 255);
                  Formatted^[Idx] := Copy(Formatted^[Idx], 1, Last - 1);
                end;
               LineEdData[Idx] := Formatted^[Idx];
             end;
            ComWriteLn('');
          end
         else ComWriteLn('Line(s) out of range!');
       end;
    end;
   ComWriteLn('');
   Dispose(Formatted);
 end;

{********************************************************************}

END.

