/*

 *  Fstat is (C) 1995 by Serge Pachkovsky.

 *

 *  Use of Fstat, both as an executable program and the source code

 *  is free as long as the copyright message is retained in the

 *  source as well as in the executable form of Fstat and all derived

 *  programs.

 *

 *  Author assumes no responsibility neither for the fitness of this

 *  program for any particular purpose nor for any possible damage,

 *  direct or consequential, resulted from the use the program. Fstat

 *  has NO WARRANTY, so that you agree to USE IT ON YOUR OWN RISK.

 *

 */

#include <stdio.h>

#include <stdlib.h>

#include <limits.h>

#include <dos.h>

#include <string.h>

#include <mem.h>

#include <ctype.h>

#include <setjmp.h>



typedef unsigned long   dword ;

typedef unsigned short  word ;

typedef unsigned char   byte ;



typedef struct {

    } REDIRIFS ;



typedef struct _dpb_ {

        byte                    drive ;

        byte                    unit ;

        word                    sector_bytes ;

        byte                    cluster_max_sec ;

        byte                    cluster2sec_shift ;

        word                    res_sectors ;

        byte                    fat_count ;

        word                    root_entries ;

        word                    first_data_sector ;

        word                    top_cluster ;

        word                    fat_sectors ;

        word                    root_sector ;

        struct __DV far*        driver_header ;

        byte                    media_id ;

        byte                    access_flag ;

        struct _dpb_ far *      next ;

        word                    search_from ;

        word                    free_clusters ;

    } DPB ;



typedef struct {

        word            ax ;

        word            bx ;

        word            cx ;

        word            dx ;

        word            si ;

        word            di ;

        word            ds ;

        word            es ;

        word            res1 ;

        word            computer_id ;

        word            process_id ;

    } DPL ;



typedef struct {

        byte            use_status ;

        word            size ;

        byte            pathname_csum ;

        word            first_lock ;

        union _sft_ far*sft_ptr ;

        word            seq_number ;

        byte            pathname[0] ;

    } SHARING_RECORD ;



typedef union _sft_ {

    struct {

        } _3x ;

    struct {

        word            handles_count ;

        word            open_mode ;

        byte            attrib ;

        word            device_info ;

        void far *      dpb ;           /* or char dev hdr, or REDIR data */

        word            first_cluster ;

        word            time ;

        word            date ;

        dword           size ;

        dword           offset ;

        union {

            struct {

                word            rel_last_acc_cluster ;

                dword           dir_sec ;

                byte            dir_entry ;

                } local ;

            struct {

                REDIRIFS far *  redirifs ;

                } remote ;

            } lr1 ;

        byte            filename[11] ;

        dword           shared_sft ;

        word            machine_number ;

        word            owner_psp ;

        word            sharing_record_offset ;

        word            abs_last_acc_cluster ;

        void far *      IFS_driver ;

        } _5x ;

    } SFT ;



typedef struct _share_name_info_ {

        struct _share_name_info_ *      next ;

        word                            machine_id ;

        word                            locks_count ;

        char                            pathname[0] ;

    } SHARE_NAME_INFO ;



typedef struct  __DV {

        struct  __DV    far     *next ;

        word                    attribute ;

        void            near    (*strategy)() ;

        void            near    (*intr)() ;

        char                    name[ 8 ] ;

        } DEVICE_HEADER ;



#define DS_ERROR        0x8000

#define DA_BIG          0x0002



typedef struct  {

        byte            len ;   /* Header + data        */

        byte            unt ;   /* Unit number          */

        byte            cmd ;   /* Command code         */

        word            status ;/* Device status Word   */

        byte            __1[8] ;/* Reserved             */

        byte            media ; /* Media descriptor byte        */

        byte    far     *data ; /* Data transfer buffer         */

        word            count ; /* Byte (Sector) count          */

        word            sector ;/* Starting sector ( block )    */

        unsigned far    *vol ;  /* Volume ID                    */

        dword           dos4_sector ;

        } RW_REQUEST_HEADER ;



#define RQ_INPUT        4



typedef struct _sector_chain_ {

        struct _sector_chain_ * next ;

        long                    start_sector ;

        long                    sectors_count ;

    } SECTOR_CHAIN ;



typedef struct {

        byte                    filename[11] ;

        byte                    attrib ;

        byte                    reserved[10] ;

        word                    time ;

        word                    date ;

        word                    cluster ;

        dword                   size ;

    } DIRECTORY_ENTRY ;



typedef struct _dir_list_ {

        struct _dir_list_ *     next ;

        byte                    filename[11] ;

        word                    cluster ;

    } DIRECTORY_LIST ;



#define MAX_DIRECTORY_NAME      200



enum { FAT_16, FAT_12 } ;



       unsigned                 _stklen = 16 * 1024 ;

static unsigned                 DOS_version ;

static int                      have_SHARE ;

static int                      have_Windows ;

static unsigned                 our_machine ;

static SHARE_NAME_INFO **       share_name_records ;

static unsigned                 share_name_records_count ;

static SFT **                   SFT_entries ;

static SHARE_NAME_INFO **       SFT_entries_share_records ;

static char **                  scanned_path_names ;

static unsigned                 SFT_entries_count ;

static int                      file_drives[ 'Z' - 'A' + 1 ] = { 0 } ;

static DPB far *                drive_DPB_list[ 'Z' - 'A' + 1 ] ;

static DPB                      disk_info ;

static int                      fat_type = FAT_16 ;

static unsigned                 current_first_fat_sector = -1u ;

static unsigned                 current_last_fat_sector  = -1u ;

static int                      max_fat_sector_count     = 0 ;

static byte *                   fat ;

static jmp_buf                  disk_scan_failure ;

static char                     disk_scan_error_message[ 132 ] ;

static SECTOR_CHAIN *           sector_chain_head ;

static SECTOR_CHAIN **          sector_chain_insertion_point ;

static char                     directory_name[ MAX_DIRECTORY_NAME + 13 ] ;

static int                      directory_name_length ;

static DIRECTORY_LIST *         directory_list ;

static DIRECTORY_LIST **        directory_list_insertion_point ;

static byte *                   sector_buffer ;

static int                      max_sectors_in_sector_buffer ;

static int                      use_SHARE = 1 ;

static int                      scan_disks = 1 ;

static byte far *               in_DOS_flag ;

static FILE *                   errors = stdout ;

static int                      quiet = 0 ;



static void compact_8_3_name( byte [11], char [13] ) ;



/*

 *  Device driver level absread, should work Ok on network

 *  shared drives.

 */

static byte far *

get_in_DOS_flag( void )

{

_AH = 0x34 ;

geninterrupt( 0x21 ) ;

return MK_FP( _ES, _BX ) ;

}



static RW_REQUEST_HEADER *

build_rw_request( byte unit, byte media_id, DEVICE_HEADER far * dev,

                  dword start, word len, void far *buf )

{

        static  RW_REQUEST_HEADER       req ;



req.len    = sizeof( RW_REQUEST_HEADER ) ;

req.unt    = unit ;

req.status = 0 ;

req.media  = media_id ;

req.vol    = NULL ;

req.data   = buf ;

req.count  = len ;

if( dev->attribute & DA_BIG ){

    req.sector = 0xFFFFu ;

    req.dos4_sector = start ;

    }

else {

    if( start < 0xFFFFlu ){

        req.sector = (unsigned) start ;

        }

    else {

        return NULL ;

        }

    }

return &req ;

}



static void

enter_device_critical_section( void )

{

_AX = 0x8002 ;

geninterrupt( 0x2a ) ;

}



static void

leave_device_critical_section( void )

{

_AX = 0x8102 ;

geninterrupt( 0x2a ) ;

}



static void

call_driver( DEVICE_HEADER far *dev, RW_REQUEST_HEADER far *req )

{

        void far        (*strategy)( void ) = MK_FP( FP_SEG( dev ), dev->strategy ) ;

        void far        (*intr)( void )     = MK_FP( FP_SEG( dev ), dev->intr ) ;



_SI = _DI = 0 ; /* Force TC to save 'em */

(*in_DOS_flag)++ ;

enter_device_critical_section() ;

_ES = FP_SEG( req ) ;

_BX = FP_OFF( req ) ;

strategy() ;

intr() ;

leave_device_critical_section() ;

(*in_DOS_flag)-- ;

}



static void

report_error_condition( byte drive, RW_REQUEST_HEADER *req )

{

static char *   error_codes[] = {

        "Write protect violation (?!)",

        "Unknown device unit",

        "Drive not ready",

        "Read command not supported by the driver (?!)",

        "CRC error",

        "Bad drive request structure length",

        "Seek error",

        "Unknown media",

        "Sector not found",

        "Printer out of paper (?!)",

        "Write fault (?!)",

        "Read fault",

        "General failure",

        "Reserved error code 0D",

        "Media unavailable"

        } ;

        byte    error_code = req->status & 0xFF ;



     if( error_code <  0x0F ){

        fprintf( errors, "%s reading drive %c:\n",

                        error_codes[error_code], 'A' + drive ) ;

        }

else if( error_code == 0x0F ){

        fprintf( errors, "Invalid disk change reading drive %c:\n",

                                'A' + drive ) ;

        fprintf( errors, "Disk labelled \"%s\" was expected\n", req->vol ) ;

        }

else {

        fprintf( errors, "Undefined error code %02X reading drive %c:\n",

                        error_code, 'A' + drive ) ;

        }

}



static int

abs_read( DPB far *disk_info, unsigned long start, unsigned len, void far *buf )

{

        RW_REQUEST_HEADER       *req ;



if( in_DOS_flag == NULL ){

    in_DOS_flag = get_in_DOS_flag() ;

    }

req = build_rw_request( disk_info->unit, disk_info->media_id, disk_info->driver_header, start, len, buf ) ;

if( req == NULL ){

    if( ! quiet )

        fprintf( errors, "Attempted to read sector >64K from pre-3.31 device\n" ) ;

    return -1 ;

    }

req->cmd = RQ_INPUT ;

call_driver( disk_info->driver_header, req ) ;

if( req->status & DS_ERROR ){

    if( ! quiet )

        report_error_condition( disk_info->drive, req ) ;

    return -1 ;

    }

return 0 ;

}



/*

 *  FAT allocation chains chasing routines

 */

static unsigned

read_fat( long fat_offset )

{

        unsigned        start_sector, end_sector ;

        int             sector_count ;

        unsigned        sector_offset ;



start_sector  = (unsigned)( fat_offset / disk_info.sector_bytes ) ;

sector_offset = (unsigned)( fat_offset % disk_info.sector_bytes ) ;

if( sector_offset + 2 > disk_info.sector_bytes )

     end_sector = start_sector + 1 ;

else end_sector = start_sector ;

if( start_sector < current_first_fat_sector || end_sector > current_last_fat_sector ){

    current_first_fat_sector = start_sector ;

    sector_count = min( max_fat_sector_count,

                disk_info.fat_sectors - current_first_fat_sector + 1 ) ;

    if( sector_count < max_fat_sector_count ){

        current_first_fat_sector -= max_fat_sector_count - sector_count ;

        if( current_first_fat_sector < 0 ) current_first_fat_sector = 0 ;

        sector_count = min( max_fat_sector_count,

                    disk_info.fat_sectors - current_first_fat_sector + 1 ) ;

        }

    current_last_fat_sector = current_first_fat_sector + sector_count - 1 ;

    if( end_sector > current_last_fat_sector ){

        sprintf( disk_scan_error_message, "Sector %u requested for %u-sector FAT",

                end_sector, current_last_fat_sector ) ;

        longjmp( disk_scan_failure, 1 ) ;

        }

    if( abs_read( &disk_info, disk_info.res_sectors + current_first_fat_sector,

                        sector_count, fat ) != 0 ){

        sprintf( disk_scan_error_message, "Can't read sector %u of the FAT",

                current_first_fat_sector ) ;

        longjmp( disk_scan_failure, 1 ) ;

        }

    }

return *(unsigned*) ( fat + (unsigned)( fat_offset - (long) current_first_fat_sector * disk_info.sector_bytes ) ) ;

}



static unsigned

get_next_fat_cluster( unsigned clust )

{

        unsigned        x ;



if( clust > disk_info.top_cluster ){

    sprintf( disk_scan_error_message, "Cluster number too large: %u", clust ) ;

    longjmp( disk_scan_failure, 1 ) ;

    }

switch( fat_type ){

        case FAT_12:

                x = read_fat( (long)clust + (clust>>1) ) ;

                if( ( clust & 1 ) == 0 )

                        x &= 0xfff ;

                else    x >>= 4 ;

                if( ( x & 0xfff ) > 0xff0 )

                        x |= 0xf000 ;

                break ;

        case FAT_16:

                x = read_fat( (long)clust << 1 ) ;

                break ;

        }

return x ;

}



static void

init_FAT_scanner( void )

{

if( disk_info.top_cluster > 4086 )

     fat_type = FAT_16 ;

else fat_type = FAT_12 ;

max_fat_sector_count = min( disk_info.fat_sectors, (16*1024)/disk_info.sector_bytes ) ;

if( ( fat = malloc( max_fat_sector_count * disk_info.sector_bytes ) ) == NULL ){

    fprintf( errors, "Out of memory for the FAT buffer\n" ) ;

    exit(1) ;

    }

current_first_fat_sector = -1u ;

current_last_fat_sector  = 0 ;

}



static void

reset_FAT_scanner( void )

{

free( fat ) ;

fat = NULL ;

}



static void

add_fragment( unsigned cluster, long sectors )

{

        long                    start_sector ;

        SECTOR_CHAIN *          p = calloc( 1, sizeof( SECTOR_CHAIN ) ) ;



if( p == NULL ){

    sprintf( disk_scan_error_message, "Out of memory for fragment record" ) ;

    longjmp( disk_scan_failure, 1 ) ;

    }

start_sector = ((long)(cluster-2) << disk_info.cluster2sec_shift)

                + disk_info.first_data_sector ;

p->start_sector  = start_sector ;

p->sectors_count = sectors ;

*sector_chain_insertion_point = p ;

sector_chain_insertion_point = &( p->next ) ;

}



static SECTOR_CHAIN *

trace_FAT_chain( unsigned cluster )

{

        int             cluster_size = disk_info.cluster_max_sec + 1 ;

        unsigned        start_fragment = cluster ;

        long            current_len = 0 ;

        unsigned        prev_cluster = cluster - 1 ;



sector_chain_insertion_point = &sector_chain_head ;

do {

    if( cluster == 0 || cluster > disk_info.top_cluster

                     || ( cluster >= 0xfff0 && cluster <= 0xfff7 ) ){

        sprintf( disk_scan_error_message,

                "FAT allocation chain corrupted at cluster %u", prev_cluster ) ;

        longjmp( disk_scan_failure, 1 ) ;

        }

    if( cluster == prev_cluster + 1 )

        current_len += cluster_size ;

    else {

        add_fragment( start_fragment, current_len ) ;

        start_fragment = cluster ;

        current_len = cluster_size ;

        }

    prev_cluster = cluster ;

    } while( ( cluster = get_next_fat_cluster( cluster ) ) < 0xfff8 ) ;

if( current_len != 0 )

    add_fragment( start_fragment, current_len ) ;

return sector_chain_head ;

}



static SECTOR_CHAIN *

get_root_sector_chain( void )

{

        SECTOR_CHAIN *          p = calloc( 1, sizeof( SECTOR_CHAIN ) ) ;



init_FAT_scanner() ;

if( p == NULL ){

    sprintf( disk_scan_error_message, "No memory to hold root directory sector chain" ) ;

    longjmp( disk_scan_failure, 1 ) ;

    }

p->start_sector  = disk_info.root_sector ;

p->sectors_count = ( (long) sizeof( DIRECTORY_ENTRY ) * disk_info.root_entries +

                        disk_info.sector_bytes - 1 ) / disk_info.sector_bytes ;

return p ;

}



static void

destroy_sector_chain( SECTOR_CHAIN *p )

{

        SECTOR_CHAIN *  next_p ;



for( ; p != NULL ; p = next_p ){

    next_p = p->next ;

    free( p ) ;

    }

}



/*

 *  Directory tree scanner.

 */

static void

destroy_directory_list( DIRECTORY_LIST *p )

{

        DIRECTORY_LIST *        next_p ;



for( ; p != NULL ; p = next_p ){

    next_p = p->next ;

    free( p ) ;

    }

}



static void

process_directory_entries( int count, DIRECTORY_ENTRY *p )

{

        DIRECTORY_LIST *        dir ;

        char                    filename[13] ;



for( ; count > 0 ; count--, p++ ){

    if( p->filename[0] == 0 || p->filename[0] == 229 ) continue ;

    if( !(p->attrib & 0x10) ) continue ;

    if( p->filename[0] == '.' ) continue ;

    if( p->cluster < 2 ){

        if( ! quiet ){

            compact_8_3_name( p->filename, filename ) ;

            fprintf( errors, "Subdirectory entry corrupted: %s%s\n",

                    directory_name, filename ) ;

            }

        continue ;

        }

    if( ( dir = calloc( 1, sizeof( DIRECTORY_LIST ) ) ) == NULL ){

        destroy_directory_list( directory_list ) ;

        sprintf( disk_scan_error_message, "Out of memory for subdirectory lists" ) ;

        longjmp( disk_scan_failure, 1 ) ;

        }

    dir->cluster = p->cluster ;

    memcpy( dir->filename, p->filename, 11 ) ;

    *directory_list_insertion_point = dir ;

    directory_list_insertion_point = &(dir->next) ;

    }

}



static void

match_open_file_records( long sector, int sectors, byte *data )

{

        unsigned                i ;

        SFT far *               sft ;

        long                    file_sector ;

        long                    top_sector = sector + sectors - 1 ;

        int                     offset ;

        DIRECTORY_ENTRY *       dir ;



for( i = 0 ; i < SFT_entries_count ; i++ ){

    sft = SFT_entries[i] ;

    if( sft->_5x.handles_count != 0 && (sft->_5x.device_info & 0x8080) == 0 ){

        if( ((DPB far *)(sft->_5x.dpb))->drive == disk_info.drive ){

            file_sector = sft->_5x.lr1.local.dir_sec ;

            if( file_sector >= sector && file_sector <= top_sector ){

                file_drives[ disk_info.drive ]-- ;

                offset = (int)( file_sector - sector ) * disk_info.sector_bytes +

                         sft->_5x.lr1.local.dir_entry * sizeof(DIRECTORY_ENTRY) ;

                dir = (DIRECTORY_ENTRY *)( data + offset ) ;

                compact_8_3_name( dir->filename, directory_name + directory_name_length ) ;

                scanned_path_names[ i ] = strdup( directory_name ) ;

                directory_name[ directory_name_length ] = 0 ;

                }

            }

        }

    }

}



static DIRECTORY_LIST *

collect_subdirectories( SECTOR_CHAIN *sectors )

{

        long    sector, i ;

        int     count, dir_entries ;



directory_list = NULL ;

directory_list_insertion_point = &directory_list ;

for( ; sectors != NULL ; sectors = sectors->next ){

    for( sector = sectors->start_sector, i = sectors->sectors_count ;

                        i > 0 ; i -= count, sector += count ){

        if( i > max_sectors_in_sector_buffer )

             count = max_sectors_in_sector_buffer ;

        else count = (int) i ;

        if( abs_read( &disk_info, sector, count, sector_buffer ) != 0 ){

            if( ! quiet )

                fprintf( errors, "Error reading directory: %s\n", directory_name ) ;

            goto done ;

            }

        dir_entries = count * (disk_info.sector_bytes/sizeof(DIRECTORY_ENTRY)) ;

        process_directory_entries( dir_entries, (DIRECTORY_ENTRY *)sector_buffer ) ;

        match_open_file_records( sector, count, sector_buffer ) ;

        }

    }

done:;

return directory_list ;

}



static DIRECTORY_LIST *

next_subdirectory( DIRECTORY_LIST *p )

{

        DIRECTORY_LIST *        next_p ;



if( p == NULL ) return p ;

next_p = p->next ;

free( p ) ;

return next_p ;

}



static int

append_subdir_name( int position, byte fcb_name[11] )

{

        char    name[14] ;

        int     namelen ;



compact_8_3_name( fcb_name, name ) ;

namelen = strlen( name ) ;

name[ namelen++ ] = '\\' ;

name[ namelen   ] = 0 ;

if( position + namelen >= MAX_DIRECTORY_NAME ){

    directory_name[ position ] = 0 ;

    if( ! quiet )

        fprintf( errors, "Directory name too long: %s%s\n", directory_name, name ) ;

    return -1 ;

    }

strcpy( directory_name + position, name ) ;

directory_name_length = position + namelen ;

return 0 ;

}



static void

recursively_scan_directory( unsigned dir_cluster )

{

        jmp_buf                 saved_jmp_buf ;

        SECTOR_CHAIN *          directory_sector_chain = NULL ;

        DIRECTORY_LIST *        subdirectory_list = NULL ;

        int                     dir_append_point = directory_name_length ;



memcpy( saved_jmp_buf, disk_scan_failure, sizeof( jmp_buf ) ) ;

if( setjmp( disk_scan_failure ) != 0 ){

    memcpy( disk_scan_failure, saved_jmp_buf, sizeof( jmp_buf ) ) ;

    destroy_sector_chain( directory_sector_chain ) ;

    destroy_directory_list( subdirectory_list ) ;

    directory_name[ dir_append_point ] = 0 ;

    directory_name_length = dir_append_point ;

    if( ! quiet )

        fprintf( errors, "%s: %s\n", directory_name, disk_scan_error_message ) ;

    return ;

    }

if( dir_cluster == 0 )

     directory_sector_chain = get_root_sector_chain() ;

else directory_sector_chain = trace_FAT_chain( dir_cluster ) ;

subdirectory_list = collect_subdirectories( directory_sector_chain ) ;

destroy_sector_chain( directory_sector_chain ) ;

directory_sector_chain = NULL ;

for( ; subdirectory_list != NULL ;

                subdirectory_list = next_subdirectory( subdirectory_list ) ){

    if( append_subdir_name( dir_append_point, subdirectory_list->filename ) == 0 )

        recursively_scan_directory( subdirectory_list->cluster ) ;

    }

directory_name[ dir_append_point ] = 0 ;

directory_name_length = dir_append_point ;

memcpy( disk_scan_failure, saved_jmp_buf, sizeof( jmp_buf ) ) ;

return ;

}



static void

scan_directory_tree( DPB far *disk )

{

disk_info = *disk ;

max_sectors_in_sector_buffer = max( 1, (16*1024)/disk_info.sector_bytes ) ;

if( ( sector_buffer = malloc( max_sectors_in_sector_buffer * disk_info.sector_bytes ) ) == NULL ){

    if( ! quiet )

        fprintf( errors, "No memory for sector buffer\n" ) ;

    return ;

    }

directory_name[0]     = 'A' + disk_info.drive ;

directory_name[1]     = ':' ;

directory_name[2]     = '\\' ;

directory_name[3]     = 0 ;

directory_name_length = 3 ;

recursively_scan_directory( 0 ) ;

reset_FAT_scanner() ;

free( sector_buffer ) ;

}



static void

reset_disk_buffers( void )

{

_AH = 0x0D ;

geninterrupt( 0x21 ) ;

}



/*

 *  Access routines for SFT and internal SHARE tables.

 */



static SFT far *

get_SFT( unsigned handle )

{

_ES = 0 ;

_DI = 0 ;

_BX = handle ;

_AX = 0x1216 ;

geninterrupt( 0x2f ) ;

if( _FLAGS & 1 )

     return NULL ;

else return MK_FP( _ES, _DI ) ;

}



static unsigned

count_SFT_entries( void )

{

        unsigned        i ;



for( i = 0 ; get_SFT( i ) != NULL ; i++ ) ;

return i ;

}



static SHARE_NAME_INFO *

get_sharing_record( unsigned offset, unsigned index )

{

        static  DPL             dpl ;

        char far *              pathname ;

        word                    machine_id ;

        word                    locks_count ;

        SHARE_NAME_INFO *       p ;

        int                     len ;



dpl.bx = offset ;

dpl.cx = index ;

_ES    = 0 ;

_DI    = 0 ;

_DX    = FP_OFF( &dpl ) ;

_AX    = 0x5d05 ;

geninterrupt( 0x21 ) ;

if( _FLAGS & 1 )

     return NULL ;

else {

    machine_id  = _BX ;

    locks_count = _CX ;

    pathname    = MK_FP( _ES, _DI ) ;

    if( pathname == NULL )

        return NULL ;

    len         = sizeof( SHARE_NAME_INFO ) + strlen( pathname ) + 1 ;

    if( ( p = malloc( len ) ) == NULL ){

        fprintf( errors, "Out of memory for sharing records\n" ) ;

        exit( 1 ) ;

        }

    p->next        = NULL ;

    p->machine_id  = machine_id ;

    p->locks_count = locks_count ;

    strcpy( p->pathname, pathname ) ;

    return p ;

    }

}



/*

 *  Determine DOS version, detect SHARE, Windoze and attend to

 *  other messy details.

 */

static unsigned

get_DOS_version( void )

{

        unsigned        version ;



_AX = 0x3306 ;

_BX = 0 ;

geninterrupt( 0x21 ) ;

version = _BX ;

if( (version & 0xFF) >= 5 && ((version >> 8) & 0xFF) < 100 )

        return version ;

_AX = 0x122f ;

_DX = 0 ;

geninterrupt( 0x2f ) ; /* make sure DOS 4.0 will return true values */

_AX = 0x3000 ;

geninterrupt( 0x21 ) ;

return _AX ;

}



static int

detect_SHARE( void )

{

_AX = 0x1000 ;

geninterrupt( 0x2f ) ;

if( _AL == 0xff )

     return 1 ;

else return 0 ;

}



static int

detect_Windows( void )

{

_AX = 0x1600 ;

geninterrupt( 0x2f ) ;

if( (_AL & 0x7F) != 0 )

     return 1 ;

else return 0 ;

}



static unsigned

get_VM_id( void )

{

_AX = 0x1683 ;

geninterrupt( 0x2f ) ;

return _BX ;

}



/*

 *  Processing of the data.

 */

static void

collect_share_filelist( void )

{

        int                     record ;

        SHARE_NAME_INFO *       p ;



for( record = 0 ; ( p = get_sharing_record( record, 0 ) ) != NULL ; record++ ){

    free( p ) ;

    }

share_name_records_count = record ;

if( share_name_records_count > 0 ){

    if( ( share_name_records = calloc( share_name_records_count, sizeof( SHARE_NAME_INFO * ) ) ) == NULL ){

        fprintf( errors, "Out of memory for share name records\n" ) ;

        exit( 1 ) ;

        }

    for( record = 0 ; record < share_name_records_count ; record++ ){

        share_name_records[ record ] = get_sharing_record( record, 0 ) ;

        }

    if( ! quiet )

        fprintf( errors, "Share has %d file records\n", share_name_records_count ) ;

    }

else {

    share_name_records = NULL ;

    }

}



static void

compact_8_3_name( byte fcb_name[11], char name[13] )

{

        byte    *in ;

        char    *out ;

        int     i ;



for( i = 0, in = fcb_name, out = name ; i < 8 ; i++, in++, out++ )

    *out = (char)*in ;

for( i = 8 ; i > 0 && out[-1] == ' ' ; out--, i-- ) ;

*out++ = '.' ;

for( i = 0 ; i < 3 ; i++, in++, out++ )

    *out = (char)*in ;

for( i = 3 ; i > 0 && out[-1] == ' ' ; out--, i-- ) ;

if( out[-1] == '.' ) out-- ;

*out = 0 ;

}



static int

filename_match( char *pathname, byte name[11] )

{

        char    filename[13] ;

        int     pathlen, namelen ;



compact_8_3_name( name, filename ) ;

pathlen = strlen( pathname ) ;

namelen = strlen( filename ) ;

if( namelen > pathlen || strcmp( pathname + pathlen - namelen, filename ) != 0 )

    return 0 ;

return 1 ;

}



static void

print_8_3_name( byte far name[11] )

{

        int     i ;



for( i = 0 ; i < 8 ; i++ )

    putchar( name[i] ) ;

putchar( '.' ) ;

for( i = 8 ; i < 11 ; i++ )

    putchar( name[i] ) ;

}



static void

print_owner_name( word _psp )

{

        char far *      name = MK_FP( _psp - 1, 8 ) ;

        int             i ;



for( i = 0 ; i < 8 && *name != 0 ; i++, name++ )

    putchar( *name ) ;

for( ; i < 8 ; i++ )

    putchar( ' ' ) ;

}



static void

print_open_mode( word mode )

{

switch( mode & 0x07 ){

    case 0: printf( "ro" ) ; break ;

    case 1: printf( "wo" ) ; break ;

    case 2: printf( "rw" ) ; break ;

    case 3: printf( "cs" ) ; break ;

    default:

        printf( "%02X", mode & 0x07 ) ;

        break ;

    }

switch( (mode >> 4) & 0x07 ){

    case 0: printf( "--" ) ; break ;

    case 1: printf( "da" ) ; break ; /* Deny all    */

    case 2: printf( "dw" ) ; break ; /* Deny write  */

    case 3: printf( "dr" ) ; break ; /* Deny read   */

    case 4: printf( "dn" ) ; break ; /* Deny none   */

    case 7: printf( "fc" ) ; break ; /* Network FCB */

    default:

        printf( "%02X", mode & 0x07 ) ;

        break ;

    }

if( mode & 0x80 )

     putchar( 'n' ) ; /* Noinherit */

else putchar( 'i' ) ;

putchar( ' ' ) ;

}



static void

print_attributes( byte attrib )

{

putchar( attrib & 0x01 ? 'R' : '-' ) ;

putchar( attrib & 0x02 ? 'H' : '-' ) ;

putchar( attrib & 0x04 ? 'S' : '-' ) ;

putchar( attrib & 0x08 ? 'V' : '-' ) ;

putchar( attrib & 0x10 ? 'D' : '-' ) ;

putchar( attrib & 0x20 ? 'A' : '-' ) ;

putchar( ' ' ) ;

}



static void

print_path_name( char far *p )

{

for( ; *p != 0 ; p++ ) putchar( *p ) ;

}



static void

grab_SFT( void )

{

        unsigned        i ;

        SFT far *       p ;



SFT_entries_count = count_SFT_entries() ;

if( ( SFT_entries = calloc( SFT_entries_count, sizeof( SFT far * ) ) ) == NULL ){

    fprintf( errors, "No memory for the copy of the SFT\n" ) ;

    exit( 1 ) ;

    }

for( i = 0 ; i < SFT_entries_count ; i++ ){

    p = get_SFT(i) ;

    if( ( SFT_entries[i] = malloc( sizeof( SFT ) ) ) == NULL ){

        fprintf( errors, "No memory for the copy of the SFT\n" ) ;

        exit( 1 ) ;

        }

    memcpy( SFT_entries[i], p, sizeof( SFT ) ) ;

    if( p->_5x.handles_count != 0 && (p->_5x.device_info & 0x8080) == 0 ){

        file_drives[ ((DPB far *)(p->_5x.dpb))->drive ]++ ;

        drive_DPB_list[ ((DPB far *)(p->_5x.dpb))->drive ] = (DPB far *) p->_5x.dpb ;

        }

    }

}



typedef struct {

        int             SFT_index ;

        word            sharing_record_offset ;

    } SFT_SHARING_OFFSET ;



static int

cmp_sharing_offset( const void *a, const void *b )

{

        word    offset_a = ((SFT_SHARING_OFFSET *)a)->sharing_record_offset ;

        word    offset_b = ((SFT_SHARING_OFFSET *)b)->sharing_record_offset ;



if( offset_a < offset_b ) return -1 ;

if( offset_a > offset_b ) return  1 ;

return 0 ;

}



static int

match_sharing_records( void )

{

        unsigned                i, j ;

        int                     share_index ;

        SFT *                   sft ;

        SFT_SHARING_OFFSET *    table ;

        int                     mismatch = 0 ;

        int                     result = 0 ;



if( ( table = calloc( SFT_entries_count, sizeof( SFT_SHARING_OFFSET * ) ) ) == NULL ||

    ( SFT_entries_share_records = calloc( SFT_entries_count, sizeof( SHARE_NAME_INFO * ) ) ) == NULL ){

        fprintf( errors, "No memory to sort sharing records offsets\n" ) ;

        exit( 1 ) ;

        }

for( i = 0, j = 0 ; i < SFT_entries_count ; i++ ){

    sft = SFT_entries[i] ;

    if( sft->_5x.handles_count != 0 && sft->_5x.sharing_record_offset != 0 ){

        table[j].SFT_index             = i ;

        table[j].sharing_record_offset = sft->_5x.sharing_record_offset ;

        j++ ;

        }

    }

if( j > 0 && share_name_records_count > 0 ){

    qsort( table, j, sizeof( SFT_SHARING_OFFSET ), cmp_sharing_offset ) ;

    share_index = 0 ;

    SFT_entries_share_records[ table[0].SFT_index ] = share_name_records[ share_index ] ;

    if( ! filename_match( share_name_records[ share_index ]->pathname,

                          SFT_entries[ table[0].SFT_index ]->_5x.filename ) )

        mismatch = 1 ;

    for( i = 1 ; i < j ; i++ ){

        if( table[i].sharing_record_offset != table[i-1].sharing_record_offset ){

            share_index++ ;

            if( share_index >= share_name_records_count ){

                if( ! quiet )

                    fprintf( errors, "SHARE filename table contains less records than SFT implies\n" ) ;

                result = -1 ;

                goto done ;

                }

            }

        SFT_entries_share_records[ table[i].SFT_index ] = share_name_records[ share_index ] ;

        if( ! filename_match( share_name_records[ share_index ]->pathname,

                              SFT_entries[ table[i].SFT_index ]->_5x.filename ) )

            mismatch = 1 ;

        }

    }

done:;

free( table ) ;

if( mismatch ){

    if( ! quiet )

        fprintf( errors, "Warning: SFT file names do not match name components of the SHARE records\n" ) ;

    result = -1 ;

    }

return result ;

}



typedef struct {

    int         drive ;

    byte        filename[11] ;

    void *      origin ;

    int         index ;

    } PARTIAL_NAME ;



static void

SFT_to_partial_name( SFT *sft, int index, PARTIAL_NAME *part )

{

if( (sft->_5x.device_info & 0x8080) == 0 )

     part->drive = ((DPB far *)(sft->_5x.dpb))->drive ;

else part->drive = -1 ;

memcpy( part->filename, sft->_5x.filename, 11 ) ;

part->origin = sft ;

part->index  = index ;

}



static void

share_name_to_partial_name( SHARE_NAME_INFO *sname, PARTIAL_NAME *part )

{

        char *          p ;

        int             i ;



if( isalpha( sname->pathname[0] ) && sname->pathname[1] == ':' )

     part->drive = toupper( sname->pathname[0] ) - 'A' ;

else part->drive = -1 ;

     if( ( p = strrchr( sname->pathname, '\\' ) ) != NULL ) p++ ;

else if( ( p = strrchr( sname->pathname, ':'  ) ) != NULL ) p++ ;

else p = sname->pathname ;

setmem( part->filename, 11, ' ' ) ;

for( i = 0 ; i < 8 && *p != '.' && *p != 0 ; i++, p++ )

    part->filename[i] = *p ;

for( ; *p != '.' && *p != 0 ; p++ ) ;

if( *p == '.' )

    for( p++, i = 8 ; i < 11 && *p != 0 ; i++, p++ )

        part->filename[i] = *p ;

part->origin = sname ;

}



static int

compare_partial_names( void *a, void *b )

{

        PARTIAL_NAME *  at = (PARTIAL_NAME *) a ;

        PARTIAL_NAME *  bt = (PARTIAL_NAME *) b ;



if( at->drive < bt->drive ) return -1 ;

if( at->drive > bt->drive ) return  1 ;

return memcmp( at->filename, bt->filename, 11 ) ;

}



static SHARE_NAME_INFO *

duplicate_share_name_record( SHARE_NAME_INFO *in )

{

        int                     size = sizeof( SHARE_NAME_INFO ) + strlen( in->pathname ) + 1 ;

        SHARE_NAME_INFO *       out = malloc( size ) ;



if( out == NULL ){

    fprintf( errors, "No memory for a sharing record\n" ) ;

    exit( 1 ) ;

    }

memcpy( out, in, size ) ;

return out ;

}



static void

incompletely_match_sharing_records( void )

{

        PARTIAL_NAME *          table ;

        PARTIAL_NAME            share_name ;

        PARTIAL_NAME *          match ;

        SHARE_NAME_INFO *       share_copy ;

        SFT *                   sft ;

        int                     i, shared_files ;



if( ( table = calloc( SFT_entries_count+2, sizeof( PARTIAL_NAME ) ) ) == NULL ||

    ( SFT_entries_share_records = calloc( SFT_entries_count, sizeof( SHARE_NAME_INFO * ) ) ) == NULL ){

        fprintf( errors, "No memory to sort sharing records\n" ) ;

        exit( 1 ) ;

        }

table++ ;

for( i = 0, shared_files = 0 ; i < SFT_entries_count ; i++ ){

    sft = SFT_entries[i] ;

    if( sft->_5x.handles_count != 0 && sft->_5x.sharing_record_offset != 0 ){

        SFT_to_partial_name( sft, i, table + shared_files++ ) ;

        }

    }

if( shared_files > 0 && share_name_records_count > 0 ){

    qsort( table, shared_files, sizeof( PARTIAL_NAME ), compare_partial_names ) ;

    for( i = 0 ; i < share_name_records_count ; i++ ){

        share_name_to_partial_name( share_name_records[i], &share_name ) ;

        match = bsearch( &share_name, table, shared_files,

                         sizeof( PARTIAL_NAME ), compare_partial_names ) ;

        if( match == NULL ){

            if( ! quiet )

                fprintf( errors, "There is no SFT entry for the SHARE file %s\n",

                                 share_name_records[i] ) ;

            }

        else {

            while( compare_partial_names( match-1, &share_name ) == 0 ) match-- ;

            for( ; compare_partial_names( match, &share_name ) == 0 ; match++ ){

                if( SFT_entries_share_records[ match->index ] != NULL ){

                    share_copy = duplicate_share_name_record( share_name_records[i] ) ;

                    share_copy->next = SFT_entries_share_records[ match->index ] ;

                    SFT_entries_share_records[ match->index ] = share_copy ;

                    }

                else {

                    SFT_entries_share_records[ match->index ] = share_name_records[ i ] ;

                    }

                }

            }

        }

    }

free( table - 1 ) ;

}



static void

trace_directory_entries( void )

{

        int     i ;



if( ( scanned_path_names = calloc( SFT_entries_count, sizeof( char * ) ) ) == NULL ){

    fprintf( errors, "No memory for scanned path records\n" ) ;

    exit( 1 ) ;

    }

reset_disk_buffers() ;

for( i = 0 ; i < ( 'Z' - 'A' ) ; i++ ){

    if( file_drives[i] != 0 ){

	if( ! quiet ) fprintf( errors, "Scanning drive %c:\n", 'A' + i ) ;

        scan_directory_tree( drive_DPB_list[i] ) ;

        if( file_drives[i] != 0 ){

            if( ! quiet )

                fprintf( errors, "Drive %c: directory structure differs from DOS SFT state\n", 'A' + i ) ;

            }

        }

    }

}



static void

print_SFT( void )

{

        unsigned                i ;

        SFT far *               sft ;

        SHARE_NAME_INFO *       sr ;

        int                     drive ;

        char *                  scan_name ;



printf( "  File           Refs   Owner   Mach PSP  Open  Attrib "

        "Clst    Offset    Length\n" ) ;

for( i = 0 ; i < SFT_entries_count ; i++ ){

        sft = SFT_entries[i] ;

        if( sft->_5x.handles_count != 0 ){

            if( (sft->_5x.device_info & 0x8080) == 0 ){

                drive = ((DPB far *)(sft->_5x.dpb))->drive ;

                printf( "%c:", drive + 'A' ) ;

                }

            else {

                drive = -1 ;

                printf( "  " ) ;

                }

            if( SFT_entries_share_records != NULL )

                 sr = SFT_entries_share_records[i] ;

            else sr = NULL ;

            if( scanned_path_names != NULL )

                 scan_name = scanned_path_names[i] ;

            else scan_name = NULL ;

            print_8_3_name( sft->_5x.filename ) ;

            if( sft->_5x.handles_count == 0xFFFF )

                 printf( " (unref) " ) ;

            else printf( " (%5d) ", sft->_5x.handles_count ) ;

            if( i >= 0 && i <= 2 )

                 printf( "SysInit " ) ;

            else

            if( sft->_5x.machine_number == our_machine ||

                sft->_5x.machine_number == 0 )

                 print_owner_name( sft->_5x.owner_psp ) ;

            else printf( "        " ) ;

            printf( " %04X,%04X ", sft->_5x.machine_number,

                                     sft->_5x.owner_psp ) ;

            print_open_mode( sft->_5x.open_mode ) ;

            print_attributes( sft->_5x.attrib ) ;

            switch( sft->_5x.device_info & 0x8880 ){

                case 0x0000:

                    printf( "%04X", sft->_5x.first_cluster ) ;

                    break ;

                case 0x0080:

                    printf( "ldev" ) ;

                    break ;

                case 0x0800: case 0x0880:

                case 0x8800: case 0x8880:

                    printf( "nspl" ) ;

                    break ;

                case 0x8000: case 0x8080:

                    printf( "netw" ) ;

                    break ;

                }

            putchar( ' ' ) ;

            printf( "%9ld/%9ld", sft->_5x.offset, sft->_5x.size ) ;

            putchar( '\n' ) ;

            if( sr != NULL ){

                printf( "               Share: " ) ;

                for( ; sr != NULL ; sr = sr->next ){

                    print_path_name( sr->pathname ) ;

                    if( sr->next != NULL )

                        printf( "\n                  or: " ) ;

                    }

                putchar( '\n' ) ;

                }

            if( scan_name != NULL ){

                printf( "                Scan: %s\n", scan_name ) ;

                }

            }

    }

}



static int

process_arguments( int argc, char *argv[] )

{

        int     i ;



for( i = 1 ; i < argc ; i++ ){

         if( strcmpi( argv[i], "-share"    ) == 0 ) use_SHARE  = 1 ;

    else if( strcmpi( argv[i], "-noshare"  ) == 0 ) use_SHARE  = 0 ;

    else if( strcmpi( argv[i], "-scan"     ) == 0 ) scan_disks = 1 ;

    else if( strcmpi( argv[i], "-noscan"   ) == 0 ) scan_disks = 0 ;

    else if( strcmpi( argv[i], "-quiet"    ) == 0 ) quiet      = 1 ;

    else if( strcmpi( argv[i], "-noquiet"  ) == 0 ) quiet      = 0 ;

    else if( strcmpi( argv[i], "-stderr"   ) == 0 ) errors     = stderr ;

    else if( strcmpi( argv[i], "-nostderr" ) == 0 ) errors     = stdout ;

    else if( strcmpi( argv[i], "-rights"   ) == 0 ){

        fprintf( errors,

                " Fstat is (C) 1995 by Serge Pachkovsky.\n"

                "\n"

                " Use of Fstat, both as an executable program and the source code\n"

                " is free as long as the copyright message is retained in the\n"

                " source as well as in the executable form of Fstat and all derived\n"

                " programs.\n"

                "\n"

                " Author assumes no responsibility neither for the fitness of this\n"

                " program for any particular purpose nor for any possible damage,\n"

                " direct or consequential, resulted from the use the program. Fstat\n"

                " has NO WARRANTY, so that you agree to USE IT ON YOUR OWN RISK.\n"

                "\n" ) ;

        }

    else {

        fprintf( errors,

                " Usage: Fstat [-[no]share] [-[no]scan] [-[no]quiet] [-[no]stderr] [-rights]\n"

                " Fstat reports the list of the open files on MS DOS 5.x or 6.x\n"

                "    -share  instructs Fstat to interrogate SHARE for the complete path names\n"

                "            of the open files. This is default if SHARE is installed.\n"

                "    -scan   instructs Fstat to find out the complete path names of the open\n"

                "            files located on the local block devices by scanning the\n"

                "            directory structure of the respective drives. Scanning is\n"

                "            enabled by default.\n"

                "    -quiet  suppress all non-fatal diagnostic messages. Diagnostics are\n"

                "            enabled by default.\n"

                "    -stderr emit diagnostic messages to the standard error stream, rather\n"

                "            than to the standard output. Default is -nostderr.\n"

                "    -rights displays copyright message.\n" ) ;

        return -1 ;

        }

    }

return 0 ;

}



int

main( int argc, char *argv[] )

{

if( process_arguments( argc, argv ) != 0 ){

    exit( 1 ) ;

    }

if( sizeof( SFT ) != 0x3b ){

    fprintf( errors, "Please compile with the byte alignment of data\n" ) ;

    exit( 2 ) ;

    }

DOS_version = get_DOS_version() ;

if( _version != DOS_version ){

    if( ! quiet )

        fprintf( errors, "DOS claims to be %2d.%02d, while actually it is %2d.02d\n",

            _version & 0xFF, (_version>>8) & 0xFF,

            DOS_version & 0xFF, (DOS_version>>8) & 0xFF ) ;

    }

if( (DOS_version & 0xFF) != 5 && (DOS_version & 0xFF) != 6 ){

    fprintf( errors, "Do not know how to handle DOS %2d.%02.02d\n",

        DOS_version & 0xFF, (DOS_version >> 8) ) ;

    exit( 1 ) ;

    }

our_machine = 0 ;

if( ( have_Windows = detect_Windows() ) != 0 ){

    our_machine = get_VM_id() ;

    }

if( our_machine != 0 ){

    if( ! quiet )

        fprintf( errors, "Running in VM %04X\n", our_machine ) ;

    }

if( ( have_SHARE = detect_SHARE() ) != 0 ){

    if( ! quiet )

        fprintf( errors, "SHARE is loaded\n" ) ;

    if( use_SHARE ) collect_share_filelist() ;

    }

grab_SFT() ;

if( ! quiet )

    fprintf( errors, "Maximum number of the open files is %3d\n", SFT_entries_count ) ;

if( match_sharing_records() == -1 ){

    if( ! quiet )

        fprintf( errors, "Attempting partial matching\n" ) ;

    incompletely_match_sharing_records() ;

    }

if( scan_disks ) trace_directory_entries() ;

print_SFT() ;

return 0 ;

}

