/* viewer.c - display file and process user input
 * Copyright (C) 1995-99 Andrew Pipkin (minitrue@pagesz.net)
 * MiniTrue is free software released with no warranty. See COPYING for details
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "minitrue.h"
#include "fileops.h"
#include "viewer.h"
#include "buffers.h"
#include "strings.h"
#include "lines.h"
#include "statline.h"
#include "hilites.h"
#include "attribs.h"
#include "console.h"
#include "scroll.h"
#include "tutorial.h"

static const char *Fname;      /* Filename */
static char *Rep_str;          /* Pointer to replacement text */
static int Rep_len;            /* Length of replacement text */
static String *Next_str;       /* Looked-ahead string */
static int Looked_ahead;       /* set if looked ahead for next string */
static String *Same_str;       /* Set if next viewed string must be same
                                * as previous viewed string */

static off_t Match_off;        /* offset of last match */
static int Emph_hl_i;          /* Hilight index of the last matched string */
static int Paren_hl_i;         /* Hilight index for last highlighted paren */
static int Nboolean;           /* Number of strings in boolean expression */
static FILE *Capture_file;     /* Capture file */
static char far Capture_fname[128]; /* Filename of last screen capture file */
static int Hls_shown = TRUE;   /* Set if highlights are displayed */
static int Help_displayed;     /* Set if help file being displayed */

static String *next_str(int replaced);
static String *next_valid(int replaced);
static int write_screen(off_t off, int row);
static void make_stat_line(off_t line_num, int pcnt);
static void replace_str(String *str_ptr);
static int jump(int ch, String *str);
static void write_file(int write_screen, int cursor_row);
static void write_capture(const char far *buf, size_t len);
static String *advance_screen_end(String *str_ptr, int replace);
static void toggle_help_screen(void);

static const char far subst_prompt[] = "Enter substitute replacement text: ";
static const char far no_matches[] = "No more matches found in file";

static const int TUTORIAL_NLINES = 1038;
static const size_t TUTORIAL_NBYTES = 47139u;

int Viewer(const char *fname)
{
    static int Auto_next = TRUE; /* set if next file automatically moved to
                                  * if no more matched in current file */
    static int Stat_line_cursor = FALSE; /* Set if cursor should appear
                                         * in status line*/
    int quit = FALSE, replaced = FALSE, new_auto_next = Auto_next;
    String *str_ptr = NULL;
    char subst_buf[128];         /* buffer for substitute replacement */

    Hilites_Reset();

    Fname        = fname;
    Looked_ahead = TRUE;
    Emph_hl_i    = -1;
    Match_off    = 0;

 /* View tutorial if fname NULL */
    if(!fname)
        toggle_help_screen();
    else
    {   str_ptr = Strings_Reset(fname, TRUE);
        if(!Strings_Total())
        {   Next_str  = NULL;
            Rep_str   = FALSE;
        }
        else
        {   Next_str  = str_ptr;
            str_ptr   = next_valid(FALSE);
        }
        if(!str_ptr)
            return FALSE;
        write_screen(String_off(str_ptr), DEFAULT_ROW);
    }

    while(!quit && (str_ptr || !Auto_next || Help_displayed))
    {   int ch, page_height, move_next = FALSE;
        int scroll_num = 0, move_curs_num = 0, scr, stop_at_eof = FALSE;

     /* Cache the line number of the last line on the screen */
        if(!Help_displayed)
            Buffers_Record_line(Lines_Line_num() + Lines_Num_diff(-1),
                                Lines_Off(-1), Lines_At_EOF());

     /* Do not write the status line if a message is being displayed */
        if(!StatLine_Have_msg())
        {   make_stat_line(Lines_Line_num(),
                           Buffers_Off_to_pcnt(Lines_Screen_end_off()));
            Lines_Set_unused(StatLine_Height(), TRUE);
            StatLine_Set_cursor(Stat_line_cursor);
            StatLine_Write();
        }
     /* Put the cursor in the text if it is not in the status line */
        if(!Stat_line_cursor)
            Console_Move_cursor(0, Lines_Cursor_row());

        ch = Console_Get_key();
        page_height = Lines_Height() - 1;
        if((scr = Scroll_Accel(ch)) != 0)           scroll_num = scr;
        else if(ch == ' ' || ch == PGDN_KEY) scroll_num =  page_height;
        else if(ch == 'b' || ch == PGUP_KEY) scroll_num =  -page_height;
        else if(ch == ctrl_chaR('v'))
        {   scroll_num  = page_height;
            stop_at_eof = TRUE;
        }
        else if(ch == 'f')                          scroll_num = -1;
        else if(ch == 'v')                          scroll_num = 1;
        else if(ch == 'k' || ch == ctrl_chaR('p'))  move_curs_num = -1;
        else if(ch == 'j' || ch == ctrl_chaR('n'))  move_curs_num = 1;
        else if(ch == '\t')                         Lines_Toggle();
        else if(ch == ctrl_chaR('l'))               MiniTrue_Rewrite();
        else if(ch == 'n' || ch == RIGHT_KEY)       move_next = TRUE;
        else if(ch == 'N')
        {   Same_str  = str_ptr;
            move_next = TRUE;
        }
        else if(ch == 'q')                          quit = QUIT_RESTORE_SCREEN;
        else if(ch == 'Q')                          quit = QUIT_KEEP_SCREEN;
        else if(ch == 'D')                          quit = QUIT_DIR;
        else if(ch == ctrl_chaR('d'))               quit = QUIT_SUBDIRS;
        else if(ch == '\033' || ch == 'X')          quit = EXIT_PROG;
        else if(ch == '>' || ch == '+')
        {   if(Help_displayed)
                move_next = TRUE;
            else
                break;
        }
        else if(ch == '#' && !Help_displayed)       Buffers_Count_lines();
        else if(ch == ';')                          Stat_line_cursor ^= 1;
        else if(ch == '.')
        {   new_auto_next ^= 1;
            StatLine_Message(new_auto_next
                             ? "Stop at end of file OFF"
                             : "Stop at end of file ON",
                             Attrib(STAT_LINE));
        }
        else if(ch == 'h')              write_screen(Match_off, DEFAULT_ROW);
        else if(ch == 'H')
        {   Hls_shown ^= 1;
            Lines_Toggle_hls();
            Lines_Rewrite();
        }
     /* If c pressed, find & highlight all matches to bottom of screen,
      * if C pressed, make all possible replacements to bottom of screen */
        else if((ch == 'c' || ch == 'C') && str_ptr)
            str_ptr = advance_screen_end(str_ptr, ch == 'C');
        else if(ch == '/' && !Help_displayed)
        {   char str_buf[128];
            if(StatLine_Getstr(str_buf, 127, "Enter new string: ", NULL) > 0)
            {   if(!Strings_Total())
                {   Next_str  = Strings_Add(str_buf);
                    move_next = TRUE;
                }
                else
                    Strings_Add(str_buf);
            }
        }
        else if(ch == 'W' || ch == ctrl_chaR('w'))
            write_file(ch == 'W', Stat_line_cursor ? -1 : Lines_Cursor_row());
        else if((ch == 's' || ch == 'S') && Strings_Total())
        {   if(StatLine_Getstr(subst_buf, 127, subst_prompt, NULL) != -1)
            {   size_t sub_len = str_preproc(subst_buf, FALSE);
                if(!have_input_error())
                {   Rep_len = sub_len;
                    Rep_str = subst_buf;
                    if(ch == 'S')
                        String_new_replace(str_ptr, subst_buf, Rep_len);
                }
            }
        }
        else if((ch == 'y' || ch == 'Y') && Rep_str)
        {   replaced  = TRUE;
            move_next = (ch == 'y');
        }
        else if((ch == 'A' || ch == ctrl_chaR('a')) && Rep_str)
        {   String_set_auto_rep(str_ptr,
                                 ch == 'A' ? AUTO_REP_FILE : AUTO_REP_ALL);
            move_next = replaced = TRUE;
        }
        else if(ch == '?')
        {   if(!Help_displayed)
                toggle_help_screen();
        }
        else
            jump(ch, str_ptr);

        if(scroll_num)
            Lines_Scroll(scroll_num, stop_at_eof);
        else if(move_curs_num)
            Lines_Move_cursor(move_curs_num);

        if(Help_displayed)
        {   if(move_next || quit)
            {   if(fname)
                {   toggle_help_screen();
                    move_next = quit = FALSE;
                }
                else
                    break;
            }
            continue;
        }

        if(Rep_str && replaced)
        {   replace_str(str_ptr);
            if(!move_next)
                Lines_Rewrite();
        }
        if(move_next)
        {   Auto_next = new_auto_next;
            StatLine_Show_fname(Fname);

         /* If next match is found on current screen, keep current screen
          *   screen start, otherwise jump to new position */
            if((str_ptr = next_valid(replaced)) != NULL)
            {   if(replaced)
                    Lines_Remake();
                if(   Match_off < Lines_Jump_start_off()
                   && Match_off >= Lines_Off(0))
                {   Lines_Move_cursor_off(Match_off);
                    Lines_Rewrite();
                }
                else
                    write_screen(Match_off, DEFAULT_ROW);
            }
            else if(!Auto_next)
            {   Lines_Rewrite();
                StatLine_Error(no_matches);
            }
            else if(replaced)
                Lines_Rewrite();
            move_next = replaced = FALSE;
        }
    }
    return quit;
}

/* Display tutorial */
void Viewer_Tutorial(void)
{
    Viewer(NULL);
    Help_displayed = FALSE;
    BufPtr_str_to_buf(NULL);
}

/* This function prints out the information in the tutorial file about
 * an option */
void Viewer_Option_info(char *option)
{
    int opt_ch = option[1];
    char far *search_ptr = Tutorial_text;
    char far *search_end = &Tutorial_text[TUTORIAL_NBYTES];
    char far *help_start = NULL;
    char far *help_end = NULL;

    if(*option != ':')
        invalid_param('?', *option);

    /* Find start of help text for option, which will be indicated by four
  * leading spaces and then the option */
    while(help_end == NULL)
    {   size_t nsearch = (char near *)search_end - (char near *)search_ptr;
        search_ptr = _fmemchr(search_ptr, '-', nsearch);

     /* Report error if end of file reached and help information not found */
        if(!search_ptr)
        {   char err_msg[256];
            sprintf(err_msg,
                    "-%c is not a valid option. Enter \""EXE_NAME" -?\" for help.",
                    opt_ch);
            error_msg(err_msg);
            return;
        }
     /* Once start found, look for end, which will be indicated by a blank
      * line */
        else if(   search_ptr[1] == opt_ch
                && !_fmemcmp("\n    ", &search_ptr[-5], 5))
        {   help_start = search_ptr - 4;
            for( ; ; )
            {   nsearch = (char near *)search_end - (char near *)search_ptr;
                search_ptr = _fmemchr(search_ptr, '\n', nsearch);
                if(search_ptr[1] == '\n' && search_ptr[2] == '\n')
                {   help_end = search_ptr + 2;
                    break;
                }
                ++search_ptr;
            }
        }
        ++search_ptr;
    }
 /* Print help text */
    printf("%.*"FP_FORM,
           (int)((char near *)help_end - (char near *)help_start),
           help_start);
}

/* write screen so that offset off occurs on screen row row */
static int write_screen(off_t off, int row)
{
    off_t line_num;
    BufPtr buf_ptr = BufPtr_line_num(OFF_MAX, off, &line_num);
    if(!buf_ptr)
        return FALSE;
 /* Make a dummy status line but do not print to determine how many screen
  * lines the status lines will occupy */
    make_stat_line(line_num, 50);
    Lines_Set_unused(StatLine_Height(), FALSE);
    Lines_Render(buf_ptr, line_num, row);
    return TRUE;
}

/* Find the next string - replaced is set if previous string is replaced */
static String *next_str(int replaced)
{
    String *str_ptr;
    if(!Looked_ahead)
        str_ptr = Strings_Next(replaced);
    else
    {   str_ptr = Next_str;
        if(str_ptr)
            Looked_ahead = FALSE;
    }
    if(Emph_hl_i != -1)
    {   while(Paren_hl_i > Emph_hl_i)
            Hilites_Del(Paren_hl_i--);
        Hilites_Change_attrs(Nboolean, Attrib(MATCH), 1);
    }
    if(str_ptr)
    {   int paren_len, paren_i = 1;
        off_t paren_off, off;
        String *bool_start = String_bool_start(str_ptr);
        String *bool_match = String_bool_first(bool_start);
        Match_off   = String_off(str_ptr);
        Rep_str     = String_replace(str_ptr);
        Rep_len     = String_rep_len(str_ptr);
        Nboolean    = 0;

     /* Hilight each present string in boolean match if string is in
      * a boolean expression */
        do {
            size_t len;
            if(bool_match == str_ptr)
            {   off = Match_off;
                len = String_len(str_ptr);
            }
            else
            {   off = String_prev_off(bool_match);
                len = String_prev_len(bool_match);
            }
            Paren_hl_i = Hilites_Add(off, len, Attrib(CURR_MATCH),
                                     2, bool_start);
            Emph_hl_i  = Paren_hl_i;
            ++Nboolean;
        }while((bool_match = String_bool_next(bool_match)) != NULL);

     /* If match is a regular expression with parens, do each paren in
      * a different color */
        while((paren_off = String_paren(str_ptr, paren_i, &paren_len)) != -1)
        {   if(paren_len)
                Paren_hl_i = Hilites_Add(paren_off, paren_len,
                                         Attrib(PAREN1 + ((paren_i - 1) % 3)),
                                         2 + paren_i, str_ptr);
            ++paren_i;
        }
    }
    else
        Rep_str = NULL;
    return str_ptr;
}

/* Find the next string, if string is automatically replaced, replace it
 * and if string must match last found string when 'N' pressed, keep going
 * until it is found */
static String *next_valid(int replaced)
{
    String *str_ptr;

    for( ; ; )
    {   str_ptr = next_str(replaced);

        if(str_ptr
           && (String_auto_rep(str_ptr) || (Same_str && Same_str != str_ptr)))
        {   if(String_replace(str_ptr))
            {   replace_str(str_ptr);
                replaced = TRUE;
            }
            continue;
        }
        else
            break;
    }
    if(str_ptr)
        Same_str    = NULL;

    return str_ptr;
}

static void replace_str(String *str_ptr)
{
    Buffers_Replace(String_off(str_ptr), String_len(str_ptr),
                    Rep_str, Rep_len);

    while(Paren_hl_i > Emph_hl_i)
        Hilites_Del(Paren_hl_i--);

    Hilites_Change(Emph_hl_i, Rep_len, Attrib(SUBST), 1);
    Emph_hl_i = Paren_hl_i = -1;
    Rep_str   = NULL;
}

static void make_stat_line(off_t line_num, int pcnt)
{
    StatLine_Make(Fname, line_num,
                  !Help_displayed ? Buffers_Nlines() : TUTORIAL_NLINES,
                  pcnt, Rep_str, Rep_len);
}

static const char far bad_line_num[] =
"Line number greater than number of lines in file";
static const char far bad_address[] = "Address greater than file size";
static const char far bad_pcnt[] = "Percentage must be between 0 and 100";
static const char far unknown_pcnt[] =
"Cannot go to desired percent because file size unknown";
/* These commands jump from the current position in the file to the
 * a different */
static int jump(int ch, String *str)
{
    const char far *err_msg = NULL;
    off_t line_num = OFF_MAX, off = OFF_MAX;
    int screen_row = DEFAULT_ROW, jumped = FALSE;
    int at_eof = FALSE;
    BufPtr buf_ptr = NULL;

 /* g moves to a new line */
    if(ch == 'g')
    {   line_num = StatLine_Getnum("Enter Line #: ", 10);
        err_msg  = bad_line_num;
    }

 /* ^ moves to the start of the file */
    else if((ch == '^' || ch == HOME_KEY) && Lines_Off(0))
        off = 0;

 /* $ moves to the end of the file */
    else if(ch == '$' || ch == END_KEY)
    {   if(!Help_displayed)
        {   off = Buffers_Pcnt_to_off(100);
            Buffers_Count_lines();
        }
        else
            off = OFF_MAX - 1;
        screen_row = -2;
    }
 /* @ moves to an address, treat addresses starting in 0x as hex,
  * addresses starting as 0 as octal */
    else if(ch == '@')
    {   off     = StatLine_Getnum("Enter Address: ", 0);
        err_msg = bad_address;
    }

 /* % moves a percentages into the file put the line containing the percentage
  * at the bottom so the entered percentage & the status line percent
  * are the same */
    else if(ch == '%')
    {   int pcnt = (int)StatLine_Getnum("Enter Percent: ", 10);
        if(pcnt < 0 || pcnt > 100)
        {   StatLine_Error(bad_pcnt);
            return FALSE;
        }
        if((off = Buffers_Pcnt_to_off(pcnt)) == -1)
        {   StatLine_Error(unknown_pcnt);
            return FALSE;
        }
        screen_row = -1;
    }
    else if((ch == 'p' || ch == 'P' || ch == LEFT_KEY) && !Help_displayed)
        off = Hilites_Prev(Lines_Off(0), ch != 'P' ? NULL : str);

    if((line_num !=OFF_MAX && line_num != -1) || (off != -1 && off !=OFF_MAX))
    {   buf_ptr = BufPtr_line_num(line_num, off, &line_num);
        if(!BufPtr_eof(buf_ptr) || !err_msg)
        {   write_screen(BufPtr_off(buf_ptr), screen_row);
            jumped = TRUE;
            if(ch == '%')
                Lines_Set_cursor(DEFAULT_ROW);
        }
        else
            at_eof = TRUE;
    }
 /* If end reached before desired location, report error */
    if(buf_ptr && at_eof && err_msg)
        StatLine_Error(err_msg);

    return jumped;
}

static const char far full_scrn[]      = "Write screen to file (Control-g to cancel): ";
static const char far part_scrn[]      = "Write screen (to cursor) to file (Control-g to cancel): ";
static const char far whole_file[]     = "Save to file (Control-g to cancel): ";
static const char far cant_open[]      = "Unable to open file ";
static const char far owrite_prompt[]  = " exists. Overwrite? (y/n) ";
static void write_file(int write_screen, int cursor_row)
{
    off_t start, end;
    char fname[128];
    fname[0] = '\0';
    Capture_file = NULL;
    if(write_screen)
    {   const char far *prompt;
        int fname_len;
        start = Lines_Off(0);
        if(cursor_row == -1)
        {   end    = Lines_Screen_end_off();
            prompt = full_scrn;
        }
        else
        {   end    = Lines_Off(cursor_row);
            prompt = part_scrn;
        }
        fname_len = StatLine_Getstr(fname, 127, prompt, Capture_fname);
        if(*fname)
        {   Capture_file = fopen(fname, "ab");
            if(Capture_file)
                _fmemcpy(Capture_fname, fname, fname_len + 1);
        }
    }
    else
    {   start = 0;
        end   = OFF_MAX;
        StatLine_Getstr(fname, 127, whole_file, NULL);
        if(*fname)
        {   if(x_exists(fname))
            {   char *prompt = append_strs(fname, owrite_prompt);
                if(StatLine_Getch(prompt, "nNyY") < 2)
                    *fname = 0;
                free(prompt);
            }
            if(*fname)
                Capture_file = fopen(fname, "wb");
        }
    }
    if(Capture_file)
    {   Buffers_Apply_fn(start, end, write_capture);
        fclose(Capture_file);
    }
 /* If filename entered and unable to open file, report error */
    else if(*fname)
    {   char *msg = append_strs(cant_open, fname);
        StatLine_Error(msg);
        free(msg);
    }
}

#ifndef __DOS16__
static void write_capture(const char far *buf, size_t len)
{
    fwrite(buf, 1, len, Capture_file);
}
#else
static void write_capture(const char far *buf, size_t len)
{
    x_write(fileno(Capture_file), buf, len);
}
#endif

/* Make all subsequent matches start at the bottom of the screen, if replace
 * is set make all replacements to bottom of screen automatically */
static String *advance_screen_end(String *str_ptr, int replace)
{
    int cursor_row         = Lines_Cursor_row(), need_redraw = FALSE;
    off_t screen_start_off = Lines_Off(0), prev_off = 0;
    off_t screen_end_off   = Lines_Screen_end_off();
    off_t cursor_off       = Lines_Off(cursor_row);
    String *prev_str = NULL;

    while(str_ptr && Match_off < screen_end_off)
    {   int replaced = FALSE;
        if(replace && Rep_str)
        {   need_redraw = TRUE;
            replace_str(str_ptr);
            replaced = TRUE;
        }
        else if(!Looked_ahead)
        {   if(Match_off >= screen_start_off)
                need_redraw = TRUE;
        }
        prev_str = str_ptr;
        prev_off = Match_off;
        str_ptr  = next_str(replaced);
    }
    if(prev_str)
    {   if(str_ptr)
        {   while(Paren_hl_i >= Emph_hl_i)
                Hilites_Del(Paren_hl_i--);

            Emph_hl_i = Paren_hl_i = -1;
        }
        Next_str     = str_ptr;
        str_ptr      = prev_str;
        Match_off    = prev_off;
        Rep_str      = FALSE;
        Looked_ahead = TRUE;
    }
    if(need_redraw)
        write_screen(cursor_off, cursor_row);

    return str_ptr;
}

/* Toggle between showing file and tutorial */
static void toggle_help_screen(void)
{
    static const char *File_name;
    static char *File_rep_str;
    static String *File_next_str;
    static int File_looked_ahead;
    static off_t File_match_off;
    static int Orig_hls_shown;
    static off_t Orig_top_off;

    if(!Help_displayed)
    {/* Save values corresponding to file values */
        File_name         = Fname;
        File_next_str     = Next_str;
        File_rep_str      = Rep_str;
        File_match_off    = Match_off;
        File_looked_ahead = Looked_ahead;
        Orig_hls_shown    = Hls_shown;
        Orig_top_off      = Lines_Off(0);

     /* Now set values for tutorial */
        Fname        = "<Tutorial>";
        Looked_ahead = TRUE;
        Match_off    = 0;
        Next_str     = NULL;
        Rep_str      = FALSE;
        if(Hls_shown)
            Lines_Toggle_hls();
        Lines_Render(BufPtr_str_to_buf(Tutorial_text), 1, 0);
    }
    else
    {/* Restore values for file */
        Fname        = File_name;
        Next_str     = File_next_str;
        Rep_str      = File_rep_str;
        Match_off    = File_match_off;
        Looked_ahead = File_looked_ahead;
        Hls_shown    = Orig_hls_shown;
        BufPtr_str_to_buf(NULL);
        if(Hls_shown)
            Lines_Toggle_hls();
        write_screen(Orig_top_off, 0);
    }
    Help_displayed ^= 1;
}


