/* unzip.c - decompress files before being scanned
 * Copyright (C) 1995-99 Andrew Pipkin (minitrue@pagesz.net)
 * MiniTrue is free software released with no warranty. See COPYING for details
 */

#include <stdlib.h>
#include <string.h>
#include <stdio.h>

#include "minitrue.h"
#include "unzip.h"
#include "fileops.h"

typedef struct
{   char *ext;     /* archive extension */
    int ext_len;   /* length of archive extension */
    char *argv[4];  /* array of arguments to send to execvp / spawnvp */
    int fname_i;   /* index where filename occurs in command line */
} UnzipData;

UnzipData *Unzips;           /* array start for unzip program information */
static int NUnzips;          /* Number of types of decompression */
static char Unzip_fname[L_tmpnam]; /* Name of capture file, only used in DOS*/
static int Unzip_fdesc = -1; /* File descriptor for capture file for unzipper*/
char *Params_copy;           /* Copy of parameters */

static int prog_in_path(char *prog_name);
static int open_pipe(char *unzip_argv[]);

/* enumeration describes signifance of args in argument vector */
enum unzip_args { PROG, FLAG, FNAME, ARGV_END, NARGS };
enum {  PARAM_SEP = ':' };    /* define colon as parameter separator */

/* Parse the parameters for the unzipping programs. */
/*+ Allow multiple flags? / Sort extensions so longest is first? */
void Unzip_Init(const char *params)
{
    char *param_ptr;
    if(!params)
        return;
    Params_copy = param_ptr = x_strdup(params);
    tmpnam(Unzip_fname);

    do
    {   char *prog_name;
        ++param_ptr; /* skip over colon */
        Unzips = x_realloc(Unzips, (NUnzips + 1) * sizeof(UnzipData));

     /* First parameter will be the file extension, insure that path
      * separator does not occur in extension */
        Unzips[NUnzips].ext = param_ptr;
        while(*param_ptr && *param_ptr != PARAM_SEP)
            ++param_ptr;

        Unzips[NUnzips].ext_len = param_ptr - Unzips[NUnzips].ext;

        if(*param_ptr == PARAM_SEP)
            *param_ptr++ = '\0'; /* indicate end of extension */

     /* Second parameter is program name of unzipping program */
        Unzips[NUnzips].argv[PROG] = prog_name = param_ptr;
        if((param_ptr = strchr(param_ptr, PARAM_SEP)) )
            *param_ptr++ = '\0'; /* indicate filename end */

#ifndef __MSDOS__
     /* Make sure that the decompression program exists */
        if(   *prog_name && !x_is_runnable(prog_name)
           && !path_sep_cH(*prog_name) && !prog_in_path(prog_name))
        {   char *err_text = append_3strs("Decompression program ",
                                          Unzips[NUnzips].argv[PROG],
                                          " not available");
            error_msg(err_text);
            free(err_text);
        }
#endif
     /* Third parameter is flag for decompression program to send uncompressed
      * file to standard output */
        if(param_ptr != NULL && *param_ptr != '\0' && *param_ptr != PARAM_SEP)
        {   Unzips[NUnzips].argv[FLAG] = param_ptr;
            Unzips[NUnzips].fname_i    = 2;
        }
     /* Third parameter is optional, so if not present do not use any
      * argument */
        else
            Unzips[NUnzips].fname_i    = 1;

     /* Indicate end of argument vector with NULL */
        Unzips[NUnzips++].argv[ Unzips[NUnzips].fname_i + 1] = NULL;

     /* Continue if more sets of parameters */
        if(param_ptr && (param_ptr = strchr(param_ptr, PARAM_SEP)) != NULL)
            *param_ptr = '\0';

    }while(param_ptr != NULL);
}

/* See if the decompression program can be found in the path */
#ifdef __MSDOS__
enum {DIR_CH = '\\', PATH_CH = ';'};
#else
enum {DIR_CH = '/', PATH_CH = ':'};
#endif
static int prog_in_path(char *prog_name)
{
    const char *path = getenv("PATH");

 /* Copy path to buffer. Then move backwards through path buffer, appending
  * the program name to the end of the directories */
    if(path)
    {   int prog_len = strlen(prog_name), path_len = strlen(path);
        char *path_buf = x_malloc(path_len + prog_len + 2);
        char *dir_start, *dir_end = path_buf + path_len, *dir_sep = dir_end;
        memcpy(path_buf, path, path_len + 1);
        while(dir_sep > path_buf)
        {   while(dir_sep > path_buf && *dir_sep != PATH_CH)
                --dir_sep;

         /* Append slash to end of directory if not present */
            if(!path_sep_cH(dir_end[-1]))
            {   *dir_end = DIR_CH;
                ++dir_end;
            }
         /* Now append program name to end of directory */
            memcpy(dir_end, prog_name, prog_len + 1);
            dir_end = dir_sep;

         /* Do not include leading separator in directory name */
            if(dir_sep == path_buf)
                dir_start = path_buf;
            else
            {   dir_start = dir_sep + 1;
                --dir_sep;
            }
            if(x_is_runnable(dir_start))
            {   free(path_buf);
                return TRUE;
            }
        }
        free(path_buf);
    }
    return FALSE;
}

/* See if the extension of fname matches any of the specified compressed file
 * types. If it does, start the decompression program as a subprocess and
 * return the pipe's file descriptor. Use compare_fn to compare extension &
 * file to handle possibility of case-insensitive file systems. Return -3
 * if the file should be skipped (no decompression program), -2 if the file
 * is not an archive, -1 if an error occurs during decompression and the
 * file handle if the file is decompressed successfully */
int Unzip_Open(const char *fname)
{
    int len = strlen(fname), unzip_i;
    for(unzip_i = NUnzips - 1; unzip_i >= 0; --unzip_i)
    {/* Determine where extension would start in filename */
        int ext_start_i = len - Unzips[unzip_i].ext_len;
        if(ext_start_i < 0)
            continue;

     /* If file's extension matches compression extension, put filename
      * in argument vector and stop */
        if(!fname_cmp(Unzips[unzip_i].ext, &fname[ext_start_i]))
        {/* If no program associated with extension, skip files with
          * that extension */
            if(*Unzips[unzip_i].argv[PROG] == '\0')
                return SKIP_FILE;

            Unzips[unzip_i].argv[ Unzips[unzip_i].fname_i ] = (char *)fname;
            break;
        }
    }
    return (unzip_i < 0) ? NOT_ARCHIVE : open_pipe(Unzips[unzip_i].argv);
}

void Unzip_Kill(void)
{
    if(Unzip_fdesc != -1)
    {   x_close(Unzip_fdesc);
        x_remove(Unzip_fname);
    }
    free(Params_copy);
    free(Unzips);
}

#ifndef __MSDOS__
#include <unistd.h>
#include <sys/types.h>
#include <sys/wait.h>

/* Get indication that unzipper is finished to avoid zombie process, file
 * handle still needs to be closed */
void Unzip_Close(void)
{
    int wait_status;
    wait(&wait_status);
}

/* Open pipe, use pipe->fork->exec instead of popen to avoid overhead of
 * starting shell, unzip_ptr has argument vector set to contain file
 * to unzip, return file descriptor of pipe, -1 if error */
static int open_pipe(char *unzip_argv[])
{
    pid_t pid;
    int pipe_fdescs[2];

    if(pipe(&pipe_fdescs[0]))
        return UNZIP_ERR;

    pid = fork();
 /* Send standard output of child process into pipe, then run unzip */
    if(!pid)
    {   x_close(pipe_fdescs[0]);
        x_dup2(pipe_fdescs[1], 1);
        x_close(pipe_fdescs[1]);
        execvp(unzip_argv[PROG], unzip_argv);
    }
    else if(pid < (pid_t)0) /* error condition */
        return UNZIP_ERR;
    else
    {   x_close(pipe_fdescs[1]);
        return pipe_fdescs[0];
    }
    return 0;
}

/* Need to capture output of unzipper into file in MS-DOS */
#else
#include <process.h>
static int open_pipe(char *unzip_argv[])
{
    int orig_stdout = x_dup(1); /* preserve original standard output */

 /* Create temporary file to capture the output of the decompressor */
    Unzip_fdesc = x_creat(Unzip_fname);

    x_dup2(Unzip_fdesc, 1);  /* redirect unzip output to capture file*/

 /* Use spawn instead of system to bypass beloved COMMAND.COM */
    if(spawnvp(P_WAIT, unzip_argv[0], unzip_argv))
    {   x_close(Unzip_fdesc);
        x_remove(Unzip_fname);
        Unzip_fdesc = UNZIP_ERR;
    }
    else
        x_seek(Unzip_fdesc, 0);  /* Reset to start of unzipped file */

    x_dup2(orig_stdout, 1);  /* Restore original standard output */
    x_close(orig_stdout);
    return Unzip_fdesc;
}

/* Close and remove capture file */
void Unzip_Close(void)
{
    if(Unzip_fdesc != -1)
    {   x_remove(Unzip_fname);
        Unzip_fdesc = -1;
    }
}
#endif

/* Test routine, 1st argument is unzip parameter string, rest are filenames
 * if files are compressed, determine their uncompressed size */
#ifdef UNZIP_TEST
int main(int argc, char *argv[])
{
    int arg_i;
    long total_fsize = 0;
    Unzip_Init(argv[1]);
    for(arg_i = 2; arg_i < argc; ++arg_i)
    {   char buf[4096];
        int pipe_fdesc = Unzip_Open(argv[arg_i], strcmp);
        long unzip_fsize = 0, nread;
        if(pipe_fdesc != -1)
        {   do
            {   unzip_fsize += nread = x_read(pipe_fdesc, buf, 4096);
            }while(nread);
            printf("Unzipped file size for %s: %ld\n",argv[arg_i],unzip_fsize);
            total_fsize += unzip_fsize;
            x_close(pipe_fdesc);
            Unzip_Close();
        }
    }
    printf("Total unzipped size: %ld\n", total_fsize);
    return 0;
}
#endif
