/* strings.c - parse the strings present on the command line
 * Copyright (C) 1995-99 Andrew Pipkin (minitrue@pagesz.net)
 * MiniTrue is free software released with no warranty. See COPYING for details
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <time.h>

#include "minitrue.h"
#include "strings.h"
#include "buffers.h"
#include "fixedstr.h"
#include "regexp.h"

static String *Strings, *Strings_end;
static char *Runtime_str;         /* string entered at runtime with '/' cmd */
int Nstrings, Nalloc;
static const char *Buf_start;
static char *Buf_end, *Scan_end;
static int Nargs, Vect_len;
static size_t Strlen_max = 1;
static String *Match_str;
static int Bof; /* set if buffer being scanned is at start of file */
static int Eof; /* set if buffer being scanned is at end of file */

/* # of lines in region for boolean operations */
static long Boolean_nlines = 1;
/* # of bytes in region for boolean operations */
static long Boolean_nbytes = LONG_MAX;

static int Whole_words;   /* Set if only whole words are to be found */
static int Case_preserve; /* Indicates how to preserve case in replacement
                           * text */
static int Have_boolean;  /* Set if boolean expressions are present in strings*/
static int Auto_rep_file; /* Set if string is to be replaced automatically
                           * in a single file */

enum { PRESERVE_FIRST = 1, PRESERVE_ALL = 2 };

static char * *Argv_copy;
static char * *Combined_vect;      /* Vector with containing strings from
                                    * argument vector and string input file */
static const char *Fname;          /* Name of file currently being scanned */
static char *In_file_buf;          /* Contents of string input file */
static char far *In_file_copy;     /* Original contents of string input file*/

static const char *Use_regexps;    /* Set if regular expressions to be used */
static int Record;                 /* set if statistics or log file maintained*/
static const char *Log_fname;      /* Log file name */
static FILE *Log_file;             /* Log file */
static char *Report_fname;         /* Name of file to write report to */

/* This structure records the # of finds & replacements in a file */
typedef struct filst
{   char far *fname;
    long finds;
    long reps;
}FileStat;

static FileStat far *File_stats;
static int Fstat_i  = 0;
static int Nfstats  = 0;
static long File_finds;
static long File_reps;
static long Total_files;
static long Total_bytes;
static int Any_found; /* set if any strings have been found */
static clock_t Start_time;

static String *string_add(char *src, char far *str_copy);
static int string_init(String *str, char *src);
static void string_clear(String *str);
static char * *parse_in_file(char * *arg_vect, int nargs,
                             char *buf, size_t buf_len);
static int parse_ands(char * *vect, int and_i, int term_start_i, int vect_len);
static int parse_bool_exp(char * *vect, int exp_i, int vect_len);
static int parse_bool_paren(char * *vect, int paren_i, int vect_len);
static void bool_set_nexts(int start_i, int end_i,
                           int bool_match_i, int bool_fail_i);
static void bool_set_starts(int start_i, int end_i, int vect_i);

static String *find_first(const char *scan_start, int nskip);
static const char *next_match(String *str_ptr, const char *search_start);
static int boolean_match(String *str_ptr);
static void make_replace(String *str_ptr);
static void record_match(String *match_str, int replaced);
static int is_whole_word(String *str_ptr);
static void write_str_log(const char *str, int len);
static void record_fstats(void);
static void write_report(void);
static char far *orig_src(int arg_i);
static void string_kill(String *str_ptr);

/* Set up the data structures for the strings and return the length of the
 * longest string - argv arguments will be modified */
size_t Strings_Init(char * *arg_vect, char * *argv_copy, FILE *input_file,
                    const char *case_param, const char *stats_param,
                    const char *log_fname, const char *bool_region,
                    const char *whole_words, const char *reg_exp)
{
    int vect_i = 0;
    char * *vect;

    if(!arg_vect)
        return 0;

    Argv_copy   = argv_copy;

    Use_regexps = reg_exp;

 /* See if only whole words should be found, if a char set follows the -w flag
  * use the set as the set of word chars */
    if(whole_words && *whole_words)
    {   Whole_words = TRUE;
     /* If + precedes the set, change the whole word
      *   set to the following set but do not find only whole words */
        if(*whole_words == '+')
        {   Whole_words = FALSE;
            ++whole_words;
        }
        if(*whole_words == ':')
            CharSet_Init(whole_words + 1);
    }
    else
    {   Whole_words = (whole_words != NULL);
        CharSet_Init(NULL);
    }

 /* Parse parameters for boolean region */
    if(bool_region)
    {   Boolean_nlines = LONG_MAX;
        if(isdigit(*bool_region))
            Boolean_nlines = strtol(bool_region, (char * *)&bool_region, 10);

        if(*bool_region == ':')
        {   if(isdigit(*++bool_region))
               Boolean_nbytes = strtol(bool_region,
                                       (char * *)&bool_region, 10);
        }
        if(*bool_region != '\0')
            invalid_param('u', *bool_region);
    }

 /* symbols following -c indicate how case should be preserved in replacement*/
    if(case_param)
    {   set_case_insense();
     /* + after c indicates to preserve case in entire replacement string*/
        if(*case_param == '+')
        {   Case_preserve = PRESERVE_ALL;
            ++case_param;
        }
     /* ~ indicates only preserve case in first character */
        else if(*case_param == '~')
        {   Case_preserve = PRESERVE_FIRST;
            ++case_param;
        }
        if(*case_param)
            invalid_param('c', *case_param);
    }

 /* Copy report filename because argument vector copy will be freed before
  * report is written */
    if(stats_param)
    {   Report_fname = x_strdup(stats_param);
        Start_time   = clock();
    }
    Log_fname = log_fname;

    Record = log_fname || stats_param;
 /* If string input file present, read contents of entire file into
  * In_file_buf and copy contents to In_file_copy */
 /* Determine # of arguments in vectors */
    for(vect_i = 0; arg_vect[vect_i] != 0; ++vect_i)
        ;
    Nargs = vect_i;
    if(!input_file)
    {   vect     = arg_vect;
        Vect_len = Nargs;
    }
    else
    {   size_t in_file_len;
        In_file_buf = falloc(input_file, &in_file_len);
        fclose(input_file);
        vect = Combined_vect = parse_in_file(arg_vect, Nargs,
                                             In_file_buf, in_file_len);
        In_file_copy = x_farmalloc(in_file_len);
        _fmemcpy(In_file_copy, In_file_buf, in_file_len);
    }

 /* Allocate space for string data structures */
    for(vect_i = 0; vect_i < Vect_len && vect_i != -1; )
    {   int orig_nstring = Nstrings, orig_vect_i = vect_i;
        String *str_ptr;
        if(!strcmp(vect[vect_i], "["))
        {   vect_i = parse_bool_paren(vect, vect_i, Vect_len);
            if(vect_i < Vect_len)
            {   if(!strcmp(vect[vect_i], "@"))
                   vect_i = parse_ands(vect, vect_i, orig_nstring, Vect_len);
                else if(!strcmp(vect[vect_i], "="))
                    error_msg("Cannot have replacements in boolean expressions");
            }
            bool_set_starts(orig_nstring, vect_i, orig_vect_i);
            continue;
        }
        str_ptr = string_add(vect[vect_i], orig_src(vect_i));
        str_ptr->orig_i = vect_i;

     /* If string followed by =, next string is replacement */
        if(++vect_i < Vect_len)
        {   if(!strcmp(vect[vect_i], "="))
            {   char *rep = (++vect_i < Vect_len) ? vect[vect_i] : "";
                str_ptr->replace = rep;
                set_input_str(rep);
                if(!Use_regexps)
                 /* replacement string of \z indicates zero-length replacement*/
                    str_ptr->rep_len = ((!strcmp(rep, "\\z"))
                                        ? 0 : str_preproc(rep, FALSE));
                else
                {   int nparen = RegExp_nparens(str_ptr->reg_exp);
                    str_ptr->rx_subst = RxSubst_init(rep, nparen,
                                                     &str_ptr->rep_len);
                }
                ++vect_i;
            }
         /* See if string is start of an and string */
            else if(!strcmp(vect[vect_i], "@"))
            {   vect_i = parse_ands(vect, vect_i, orig_nstring, Vect_len);
                bool_set_starts(orig_nstring, vect_i, orig_vect_i);
            }
        }
    }
 /* Chop off unused string data structures, leave an unused string
  *  for a string entered at runtime */
    Strings     = x_realloc(Strings, (Nstrings + 1) * sizeof(String));
    Strings_end = Strings + Nstrings;
    FixedStr_Set_max_skip(maX(Strlen_max, 16));

#ifdef BOOL_TEST
    {   int str_i;
        for(str_i = 0; str_i < Nstrings; ++str_i)
            printf("%d: %d %d %d\n", str_i, Strings[str_i].bool_start_i,
                   Strings[str_i].bool_match_i, Strings[str_i].bool_fail_i);
    }
#endif
    return Strlen_max;
}

/* Process and add the string *src to the string array. Store the length
 *   of the string in *len_ptr. Str_copy is a copy of the source and it appears
 *   in the error message if an error occurs in processing. The
 *   new string is returned */
static String *string_add(char *src, char far *str_copy)
{
    String *str_ptr;
    size_t len;
 /* Allocate more space in string array if neccessary */
    if(Nalloc == Nstrings)
    {   Nalloc  = Nalloc ? (Nalloc * 2) : 16;
        Strings = x_realloc(Strings, Nalloc * sizeof(String));
    }
    set_input_str(str_copy);
    str_ptr  = &Strings[Nstrings];
    len = string_init(str_ptr, src);
    ++Nstrings;
    Strlen_max = maX(Strlen_max, len);
    return str_ptr;
}

/* Set all members of string to zero or NULL */
static int string_init(String *str_ptr, char *src)
{
    size_t len;
    string_clear(str_ptr);
    if(Use_regexps)
    {   str_ptr->reg_exp = x_malloc(sizeof(RegExp));
        len = RegExp_init(str_ptr->reg_exp, src);
    }
    else
    {   len = str_preproc(src, TRUE);
        FixedStr_init(&str_ptr->fixed_str, src, len);
    }
    str_ptr->len = len;
    return len;
}

/* clear string structure */
static void string_clear(String *str)
{
    memset(str, 0, sizeof(String));
    str->bool_start_i = str->bool_fail_i = str->bool_match_i = -1;
}

/* If string input file present, make array containing starts of strings
 * in input files and store \0 at ends of strings, copy argument vector
 * to start of array */
static char * *parse_in_file(char * *arg_vect, int nargs,
                             char *buf, size_t buf_len)
{
    char * *vect, *str_ptr = buf, *buf_end = &buf[buf_len];
    int nalloc = nargs + 16, vect_i = nargs;

    vect = x_malloc(nalloc * sizeof(char *));
    memcpy(vect, arg_vect, nargs * sizeof(char *));

    while(str_ptr < buf_end)
    {   if(isspace(*str_ptr))       /* skip whitespace */
            ++str_ptr;
        else if(*str_ptr == '#')    /* skip over comments */
        {   while(*str_ptr != '\n' && str_ptr < buf_end)
                ++str_ptr;
        }
        else
        {   if(vect_i == nalloc)
                vect = x_realloc(vect, sizeof(char *) * (nalloc *= 2));
         /* If string quoted, string ends with " not preceded by \ */
            if(*str_ptr == '\"')
            {   vect[vect_i] = ++str_ptr;
                while(*str_ptr != '\"' && str_ptr < buf_end)
                {   if(*str_ptr == '\\') /* skip escape sequences */
                        ++str_ptr;
                    ++str_ptr;
                }
             /* Make sure quote is closed */
                if(str_ptr == buf_end)
                    exit(-1);
            }
         /* Otherwise, space will end string */
            else
            {   vect[vect_i] = str_ptr;
                while(!isspace(*str_ptr) && str_ptr < buf_end)
                    ++str_ptr;
            }
            *str_ptr++ = '\0';
            ++vect_i;
        }
    }
    Vect_len = vect_i;
    return x_realloc(vect, sizeof(char *) * vect_i);
}

/* Parse an and expression, and_i is the index in the string vector of the
 * first @. term_start_i is the String index of the start of the first term */
static int parse_ands(char * *vect, int and_i, int term_start_i, int vect_len)
{
    while(and_i < vect_len && !strcmp(vect[and_i], "@"))
    {
     /* Set all match indices to go to next string in previous term */
        bool_set_nexts(term_start_i, Nstrings, Nstrings, -1);
        term_start_i = Nstrings;
        if(and_i == vect_len - 1)
        {   error_msg("@ must be followed by string");
            break;
        }
        else
            and_i = parse_bool_exp(vect, ++and_i, vect_len);
    }
    return and_i;
}

static int parse_bool_exp(char * *vect, int exp_i, int vect_len)
{
    Have_boolean = TRUE;
    if(exp_i == vect_len)
        ;
else if(!strcmp(vect[exp_i], "["))
        exp_i = parse_bool_paren(vect, exp_i, vect_len);
    else if(!strcmp(vect[exp_i], "="))
        error_msg("Cannot have replacements in boolean expressions");
    else
    {   string_add(vect[exp_i], orig_src(exp_i));
        ++exp_i;
    }
    return exp_i;
}

static int parse_bool_paren(char * *vect, int paren_i, int vect_len)
{
    int term_start_i = Nstrings;
    paren_i = parse_bool_exp(vect, ++paren_i, vect_len);

    while(paren_i < vect_len)
    {   if(!strcmp(vect[paren_i], "]"))
            return paren_i + 1;
        if(!strcmp(vect[paren_i], "@"))
        {   paren_i = parse_ands(vect, paren_i, term_start_i, vect_len);
            continue;
        }
        bool_set_nexts(term_start_i, Nstrings, -1, Nstrings);
        term_start_i = Nstrings;
        paren_i = parse_bool_exp(vect, paren_i, vect_len);
    }
    error_msg("Unclosed bracket in boolean expression");
    return -1;
}

/* Set bool_match_i and bool_fail_i for the string data structures from
 * start_i to end_i. Do not change the values if they have already been set*/
static void bool_set_nexts(int start_i, int end_i,
                           int bool_match_i, int bool_fail_i)
{   String *str = &Strings[start_i];
    int str_i;
    for(str_i = start_i; str_i < end_i; ++str_i, ++str)
    {   if(str->bool_match_i == -1)
            str->bool_match_i = bool_match_i;
        if(str->bool_fail_i == -1)
            str->bool_fail_i = bool_fail_i;
    }
}

/* Set orig_i and bool_start_i for the string data structures from start_i to
 * end_i. */
static void bool_set_starts(int start_i, int end_i, int vect_i)
{   String *str = &Strings[start_i];
    int str_i;
    for(str_i = start_i ; str_i < end_i; ++str_i, ++str)
    {   str->orig_i       = vect_i;
        str->bool_start_i = start_i;
    }
}

/* If automatic replacements were desired only for a single file, clear the
 * automatic replacement flag when a new file is started */
String *Strings_Reset(const char *fname, int zero_len)
{
    String *str_ptr;
    Fname = fname;
    Eof   = FALSE;
    Bof   = BUF_BOF;
    if(Have_boolean || Auto_rep_file)
    {   for(str_ptr = Strings; str_ptr < Strings_end; ++str_ptr)
        {   str_ptr->flags &= ~AUTO_REP_FILE;
            if(str_ptr->bool_start_i != -1)
                str_ptr->prev_off = -1;
        }
        Auto_rep_file = FALSE;
    }
    ++Total_files;
    if(Strings == Strings_end)
    {   if(zero_len)
        {   string_clear(&Strings[0]);
            Buffers_Load((char * *)&Strings->loc, &Scan_end, &Buf_end, 0);
            return Strings;
        }
        else
            return NULL;
    }
    else
        return find_first(NULL, 0);
}

/* If scan_start is NULL, load the next buffer and scan it, otherwise begin
 *   scanning in the current buffer from scan_start. nskip is the number of
 *   chars to skip at the start of the next buffer. Keep loading
 *   buffers until a buffer containing a match is loaded, return string if
 *   found, NULL if file end reached */
static String *find_first(const char *scan_start, int nskip)
{
    String *str_ptr, *first_match = NULL;

 /* Keep loading a new buffer until a match is found or the end of the file
  * is reached. The end of the file will be reached if buf_end and
  * scan_end are the same (meaning there is no overlap for the next buffer)*/
    while(!Eof && !first_match)
    {   const char *first_str;
        if(!scan_start)
        {   Buffers_Load((char * *)&scan_start, &Scan_end, &Buf_end, nskip);
            Buf_start = scan_start;
        }
        Eof       = (Scan_end == Buf_end);
        if(Use_regexps)
            RegExp_Buf_init(scan_start, Buf_end, Bof | Eof);

        first_str = Scan_end + 1;
        if(Report_fname)
            Total_bytes += (Scan_end - (char *)scan_start);

     /* Find the first match in the region beginning at scan_start and ending
      * at buf_end, the start of the string must come before scan_end, but the
      * remainder of the string can come after buf_end. Set the corresponding
      * static variables appopriately */
        for(str_ptr = Strings; str_ptr < Strings_end; ++str_ptr)
        {
            if(!str_ptr->reg_exp)
            {   char *end    = Eof ? Buf_end : Scan_end + str_ptr->len;
                str_ptr->loc = FixedStr_find(&str_ptr->fixed_str,
                                             scan_start, end);
            }
            else
            {   str_ptr->loc = RegExp_find(str_ptr->reg_exp, scan_start,
                                           &str_ptr->len);
            }
            if(str_ptr->loc < first_str)
            {   first_str   = str_ptr->loc;
                first_match = str_ptr;
            }
        }
        scan_start = NULL;
    }
    if(first_match)
        make_replace(first_match);

 /* If no more matches & statistics desired, record # of matches in file */
    else if(File_finds)
    {/* Keep track if any matches have been found to determine the
      * program exit value */
        Any_found = TRUE;

        if(Report_fname)
            record_fstats();
    }
    Match_str = first_match;
    if(   first_match
       && (   (first_match->bool_start_i != -1 && !boolean_match(first_match))
           || (Whole_words && !is_whole_word(first_match))))
    {   --File_finds;
        --first_match->finds;
        Match_str = Strings_Next(FALSE);
    }
    return Match_str;
}

/* Find the next matching string in the buffer, begin searching at start
 * of last match if string not replaced, at end of replaced text if string
 * replaced. */
String *Strings_Next(int replaced)
{
    const char *match_start;
    String *str_ptr, *next_str;
    size_t orig_len;

    if(!Match_str)
        return NULL;

    match_start = Match_str->loc;
    orig_len    = Match_str->len;

 /* Record statistics if desired */
    ++File_finds;
    if(Record)
        record_match(Match_str, replaced);

 /* Find the next instance of the last found string */
    Match_str->loc = next_match(Match_str, match_start + 1);

/* If previous string replaced, matches must occur at end of replaced text */
    if(replaced)
    {   if(Match_str->len)
            match_start += orig_len;
     /* See if previous matches used in finding boolean matches are
      * overwritten by replacement, if so do not use them in boolean matches*/
        if(Have_boolean)
        {   off_t off = String_off(Match_str);
            for(str_ptr = Strings; str_ptr < Strings_end; ++str_ptr)
            {   if(   str_ptr->prev_off != -1
                   && str_ptr->prev_off + (off_t)str_ptr->prev_len > off)
                str_ptr->prev_off = -1;
            }
        }
    }

 /* Now look for the nearest match */
    for( ; ; )
    { /* If at end of buffer, set boundary one past end of buffer to allow
       * for possibility of zero-length matches at buffer end */
        const char *next_loc = (char *)Scan_end + Eof;
        next_str = NULL;

        for(str_ptr = Strings; str_ptr < Strings_end; ++str_ptr)
        {/* Need to search again if match occured in replaced text */
            if(str_ptr->loc < match_start)
                str_ptr->loc = next_match(str_ptr, match_start);

            if(str_ptr->loc < next_loc)
            {   next_loc = str_ptr->loc;
                next_str = str_ptr;
            }
        }
     /* If other strings needed for boolean match are not present,
      * keep trying */
        if(   next_str
           && (   (next_str->bool_start_i != -1 &&!boolean_match(next_str))
               || (Whole_words && !is_whole_word(next_str))))
            next_str->loc = next_match(next_str, next_str->loc + 1);
        else
            break;
    }
 /* If no more strings found in buffer, move to next buffer. If match_start
  * is greater than scan end, skip enough characters in next buffer so start
  * of next buffer will correspond to match_start */
    if(!next_str)
    {   int nskip = match_start > Scan_end ? match_start - Scan_end : 0;
        Bof      = FALSE;
        next_str = find_first(NULL, nskip);
    }
    make_replace(next_str);
    Match_str    = next_str;
    return next_str;
}

/* Find the next match for the given string, beginning at search start */
static const char *next_match(String *str_ptr, const char *search_start)
{
    if(!str_ptr->reg_exp)
    {   const char *end = Eof ? Buf_end : Scan_end + str_ptr->len;
        return FixedStr_find(&str_ptr->fixed_str, search_start, end);
    }
 /* End of next regular expression match must come after end of previous
  * regular expression match */
    else
    {   const char *orig_end = str_ptr->loc + str_ptr->len;
        for( ; ; )
        {   str_ptr->loc = RegExp_find(str_ptr->reg_exp, search_start,
                                       &str_ptr->len);

            if(str_ptr->loc + str_ptr->len <= orig_end)
                ++search_start;
            else
                break;
        }
        return str_ptr->loc;
    }
}

/* If the string is in a boolean expression, see if the other strings
 * in the expression are within the desired proximity for the boolean
 * expression to succeed */
static int boolean_match(String *start_ptr)
{
    int str_i     = start_ptr->bool_start_i;
    long off      = Buffers_Match_off(start_ptr->loc);
    long line_num = Buffers_Match_line_num(start_ptr->loc);
    String *str_ptr;
    start_ptr->prev_off      = off;
    start_ptr->prev_line_num = line_num;
    start_ptr->prev_len      = start_ptr->len;

 /* See what strings are present in the boolean match */
    for(str_ptr = &Strings[str_i];
        str_ptr < Strings_end && str_ptr->bool_start_i == str_i; ++str_ptr)
    {   int in_match = (   str_ptr->prev_off != -1
                        && off - str_ptr->prev_off < Boolean_nbytes
                        && line_num - str_ptr->prev_line_num < Boolean_nlines);
        str_ptr->bool_in_match = in_match;
    }

 /* Now see if enough strings are present for boolean match to succeed */
    for( ; ; )
    {   str_ptr = &Strings[str_i];
        if(!str_ptr->bool_in_match)
        {   str_i = str_ptr->bool_fail_i;
            if(str_i == -1)
                return FALSE;
        }
        else
        {   str_i = str_ptr->bool_match_i;
            if(str_i == -1)
                return TRUE;
        }
    }
}

/* If replacement string needs to be created, (uses parts of regular
 * expression or case is preserved, create it */
void make_replace(String *str_ptr)
{
    if(!str_ptr)
        return;

    else if(str_ptr->rx_subst)
        str_ptr->replace = RxSubst_str(str_ptr->rx_subst, str_ptr->reg_exp,
                                       &str_ptr->rep_len);

    else if(Case_preserve && str_ptr->replace)
    {   size_t len = miN(str_ptr->len, String_rep_len(str_ptr)), str_i;
        const char *match = str_ptr->loc;
        char *rep = String_replace(str_ptr);
        if(Case_preserve == PRESERVE_FIRST)
            len = miN(len, 1);
        for(str_i = 0; str_i < len; ++str_i)
        {   char match_ch = match[str_i], rep_ch = rep[str_i];
            if(up_casE(match_ch) != match_ch)
                rep[str_i] = low_casE(rep_ch);
            else if(low_casE(match_ch) != match_ch)
                rep[str_i] = up_casE(rep_ch);
        }
    }
}

/* Return TRUE if the string is a whole word, FALSE if not  */
static int is_whole_word(String *str_ptr)
{
    const char *start = str_ptr->loc;
    const char *end   = str_ptr->loc + str_ptr->len - 1;
    return (   (   !CharSet_in_worD(*start)
                || start == Buf_start
                || !CharSet_in_worD(start[-1]))
            && (   !CharSet_in_worD(*end)
                || end == Buf_end - 1
                || !CharSet_in_worD(end[1])));
}

/* Add a string at runtime, if no strings were originally present, find the
 * first instance of the new string */
String *Strings_Add(char *str)
{
    String new_str, *new_loc;
    const char *buf_start = NULL;
    char *str_copy = x_strdup(str);
    size_t len = string_init(&new_str, str);
    if(have_input_error())
        return NULL;

 /* If a string has already been added at runtime, delete previous runtime
  * string */
    if(Runtime_str)
    {   new_loc = Strings_end - 1;
        new_str.loc = ( (Match_str == new_loc)
                       ? Match_str->loc
                       : next_match(&new_str, Match_str->loc + 1));

        string_kill(new_loc);
        if(new_loc->replace)
            free(new_loc->replace);
        free(Runtime_str);
    }
 /* Otherwise use extra String allocated at end of array */
    else
    {/* If no strings originally, start search at buffer start */
        if(Strings == Strings_end)
            buf_start = Strings->loc;
        new_loc = Strings_end++;
        ++Nstrings;
    }
    memcpy(new_loc, &new_str, sizeof(String));
    Runtime_str = str_copy;
    Buffers_Add_str(len);
    return buf_start ? find_first(buf_start, 0) : NULL;
}

static void record_match(String *match_str, int replaced)
{
    if(Report_fname)
    {   ++match_str->finds;
        if(replaced)
        {   ++match_str->reps;
            ++File_reps;
        }
    }
 /* Write match information to log file, first opening file if necessary */
    if(Log_fname)
    {   off_t line_num;
        if(!Log_file)
        {   Log_file = fopen(*Log_fname ? Log_fname + 1: "minitrue.log", "ab");
            if(!Log_file)
                error_msg("Unable to open log file");
        }
        write_str_log(match_str->loc, match_str->len);
        BufPtr_line_num(OFF_MAX, Buffers_Match_off(match_str->loc),&line_num);
        fprintf(Log_file, " found in %s at line %ld", Fname, (long)line_num);
        if(replaced)
        {   fputs(", replaced/w ", Log_file);
            write_str_log(match_str->replace, match_str->rep_len);
        }
        fputc('\n', Log_file);
    }
}

/* Write a string len characters long to the log file, convert the control
 * characters in the string to their corresponding escape sequences and
 * quote the string if it contains escape sequences */
static void write_str_log(const char *str, int len)
{
 /* leave one space at start of buffer for possible quote */
    char buf[256], *buf_ptr = buf + 1, *buf_start = buf + 1;
    int str_i, quoted = FALSE;
    for(str_i = 0; str_i < len; ++str_i)
    {   if(buf_ptr > &buf[240])
        {   fwrite(buf_start, 1, buf_ptr - buf_start, Log_file);
            buf_ptr = buf_start = buf;
        }
     /* insert quote at start of buffer if string has spaces */
        if(isspace(str[str_i]) && buf_start > buf)
        {   *--buf_start = '\"';
            quoted = TRUE;
        }
        buf_ptr = ch_to_esc(buf_ptr, str[str_i]);
    }
    if(quoted)
        *buf_ptr++ = '\"';
    fwrite(buf_start, 1, buf_ptr - buf_start, Log_file);
}

/* Record the # of matches & replacements that occur in the File statistics
 * array, first allocate memory for the copied filename */
static void record_fstats(void)
{
    if(Fstat_i == Nfstats)
    {   Nfstats    = !Nfstats ? 256 : Nfstats * 2;
        File_stats = x_farrealloc(File_stats, Nfstats * sizeof(FileStat));
    }
    File_stats[Fstat_i].fname = x_fstrdup(Fname);
    File_stats[Fstat_i].finds = File_finds;
    File_stats[Fstat_i].reps  = File_reps;
    File_finds = File_reps    = 0;
    ++Fstat_i;
}

/* Accessor functions */
off_t String_off(String *str)      { return Buffers_Match_off(str->loc); }
size_t String_len(String *str)     { return str->len; }

char *String_replace(String *str)  { return str->replace; }
size_t String_rep_len(String *str) { return str->rep_len; }

off_t String_prev_off(String *str) { return str->prev_off; }
size_t String_prev_len(String *str) { return str->prev_len; }

int String_auto_rep(String *string)
{
    Auto_rep_file = TRUE;
    return string->flags & ( AUTO_REP_FILE | AUTO_REP_ALL );
}

void String_new_replace(String *string, const char *new_rep, size_t len)
{
    if(string->flags & SUBST_REP)
        free(string->replace);
    string->replace = memcpy(x_malloc(len), new_rep, len);
    string->rep_len = len;
    string->flags  |= SUBST_REP;
}

void String_set_auto_rep(String *string, int auto_rep)
{
    string->flags |= auto_rep;
}

/* If str is in a boolean expression, return the first string in the
 * boolean expression, otherwise just return the string */
String *String_bool_start(String *str)
{
    return (str->bool_start_i == -1) ? str : &Strings[str->bool_start_i];
}

/* Find the first present string in a boolean expression if str is the
 * start of a boolean expression, otherwise just return the string */
String *String_bool_first(String *str)
{
    if(str->bool_start_i != -1)
    {   while(!str->bool_in_match)
            ++str;
    }
    return str;
}

/* Find the next present string in a boolean expression, if there are no
 * more present strings or the string is not in a boolean expression, return
 * NULL */
String *String_bool_next(String *str)
{
    if(str->bool_start_i != -1)
    {   int bool_start_i = str->bool_start_i;
        while(++str < Strings_end && str->bool_start_i == bool_start_i)
        {   if(str->bool_in_match)
                return str;
        }
    }
    return NULL;
}


/* If regular expression, return the offset of the parentheses with number
 * paren_num and set *len to the length of the parentheses. If not regular
 * expression or paren_no greater than # of parens in expression, return NULL
 * and set *len to -1 */
off_t String_paren(String *string, int paren_num, int *len)
{
    if(!string->reg_exp || paren_num > RegExp_nparens(string->reg_exp))
        return -1;

    else
    {   char *paren_start = RegExp_paren(string->reg_exp, paren_num, len);
        return Buffers_Match_off(paren_start);
    }
}

int Strings_Total(void)  { return Nstrings; }
int Strings_Found(void)  { return (Any_found || !Nstrings); }

enum { FIND_COL = 53 }; /* print # of matches in column 53 */
static void write_report(void)
{
    String *str_ptr, *next_str;
    FileStat far *fstat_ptr;
    long total_finds = 0, total_reps = 0, elapsed;
    FILE *out_file = *Report_fname ? fopen(Report_fname + 1, "ab") : stdout;
    if(!out_file)
    {   error_msg("Unable to open report file");
        return;
    }
    fprintf(out_file, "String(s)%46s%17s", "Finds", "Replacements\n");
    if(Runtime_str)
        --Strings_end;
    for(str_ptr = Strings; str_ptr < Strings_end; str_ptr = next_str)
    {   int arg_i, len = 0, pad_len, end_i;
        next_str = str_ptr + 1;

     /* Add the number of finds and replacements for all strings in a
      * boolean expression to the first string in the boolean expression */
        if(str_ptr->bool_start_i != -1)
        {   while(   next_str < Strings_end
                  && next_str->bool_start_i == str_ptr->bool_start_i)
            {   str_ptr->finds += next_str->finds;
                str_ptr->reps  += next_str->reps;
                ++next_str;
            }
        }
        end_i = (next_str < Strings_end) ? next_str->orig_i : Vect_len;

     /* Print original string, enclosing in quotes if they contain spaces */
        for(arg_i = str_ptr->orig_i; arg_i < end_i; ++arg_i)
        {   char far *str = orig_src(arg_i);
            len += fprintf(out_file, have_ctype(str, isspace) ?
                           "  \"%" FP_FORM "\"" : "  %" FP_FORM, str);
        }
     /* Print enough spaces to align find/replacements column */
        pad_len = FIND_COL - len;
        fprintf(out_file,
                str_ptr->replace ? "%*ld%13ld\n" : "%*ld            -\n",
                maX(pad_len, 8), str_ptr->finds, str_ptr->reps);

        total_finds += str_ptr->finds;
        total_reps  += str_ptr->reps;
    }
 /* Print # of finds & replacements in files */
    fprintf(out_file, "\nFile\n");
    for(fstat_ptr = File_stats; fstat_ptr < &File_stats[Fstat_i];++fstat_ptr)
    {   int pad = FIND_COL - fprintf(out_file, "  %"FP_FORM, fstat_ptr->fname);
        fprintf(out_file, "%*ld%13ld\n", maX(pad, 8),
                fstat_ptr->finds, fstat_ptr->reps);
    }
    fprintf(out_file, "\nTotal:%47ld%13ld\n", total_finds, total_reps);
    fprintf(out_file, "Total Files Scanned:%33ld\n", Total_files);
    fprintf(out_file, "Total Bytes Scanned:%33ld\n", Total_bytes);
    elapsed = hsec_diff(Start_time, clock());
    fprintf(out_file, "Time Elapsed:%37ld.%02d sec\n",
           elapsed / 100, (int)(elapsed % 100));

    if(*Report_fname)
        fclose(out_file);
}

/* return the original source before processing for string at arg_i */
static char far *orig_src(int arg_i)
{
    return (  (arg_i < Nargs)
            ? Argv_copy[arg_i]
            : &In_file_copy[ Combined_vect[arg_i] - In_file_buf]);
}

/* Prevent the reporting of statistics at completion of the program */
void Strings_No_report(void) {  Report_fname = NULL; }

static void string_kill(String *str_ptr)
{
    if(str_ptr->reg_exp)
        RegExp_kill(str_ptr->reg_exp);
    else
        FixedStr_kill(&str_ptr->fixed_str);
    if(str_ptr->flags & SUBST_REP)
        free(str_ptr->replace);
    free(str_ptr->rx_subst);
}

void Strings_Kill(void)
{
    String *str_ptr;
 /* If statistics are to be printed, print them, then free memory devoted
  * to them */
    if(Report_fname)
    {
        FileStat far *fstat_ptr = File_stats;
        write_report();
        for(; fstat_ptr < &File_stats[Fstat_i]; ++fstat_ptr)
            farfree(fstat_ptr->fname);
        farfree(File_stats);
        free(Report_fname);
    }
    if(Log_file)
        fclose(Log_file);

    for(str_ptr = Strings; str_ptr < Strings_end; ++str_ptr)
        string_kill(str_ptr);

    RxSubst_kill();

    free(Runtime_str);
    free(In_file_buf);
    farfree(In_file_copy);
    free(Combined_vect);
    free(Strings);
}
