/* stdout.c - write matches and context to standard output
 * Copyright (C) 1995-99 Andrew Pipkin (minitrue@pagesz.net)
 * MiniTrue is free software released with no warranty. See COPYING for details
 *
 * Wrong again Flanders! In situations like this you just relax and let the
 * current take you back to land */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>

#include "minitrue.h"
#include "buffers.h"
#include "fileops.h"

enum stdout_flags { PRINT_MATCH = 1, PRINT_NONMATCH = 2, NUMBER_LINES =  4,
                    PRINT_FNAMES = 8, FNAMES_ONLY = 16, NO_STRINGS = 32};

static int Flags = PRINT_MATCH; /* Flags indicating what to write to stdout */
static off_t Off, Line_num;     /* Current Offset & Line number in file */
static const char *Fname;       /* Name of file being processed */
static int NLead = 1;           /* # of leading context lines to print */
static int NTrail = 1;          /* # of trailing context lines to print */
static int Have_match;          /* True if match has been found in file */
static int Prev_ch_nl;          /* True if previous char written is \n,
                                 * next text written will be preceded by
                                 * filename/line # if desired */
static off_t Prev_match_end;    /* Ending offset of previous match */

/* Function used to write to stdout */
static void (*Write_fn)(const char far *start, size_t len);

enum sep_chs { DEFAULT_SEP = ':', CONTEXT_SEP = '-' };
static char Sep_ch;             /* separator character between filename,
                                 * line number & line. Will be : if line is
                                 * match - otherwise */

static void write_trail(off_t lead_off);
static void write_region(off_t start, off_t end);
static void number_blk(const charf *start, size_t len);
static void write_stdout_info(void);
static void stdout_write(const char far *ptr, size_t len);
static void count_nl_blk(const charf *start, size_t len);

/* Parse the parameters for the -o option which determine what is written
 *  to standard output. Mult_files is true if more than one file will be
 *  scanned */
void Stdout_Init(const char *params, int mult_files, int no_strings)
{
 /* Default is to print filenames if more than one file to be scanned */
    if(mult_files)
        Flags |= PRINT_FNAMES;
 /* If no strings, dump entire contents of file to standard output */
    if(no_strings)
        Flags |= NO_STRINGS;

    for(; *params != '\0'; ++params)
    {   int ch = *params;
        if(ch == '+')           Flags |= PRINT_MATCH + PRINT_NONMATCH;
        else if(ch == '#')      Flags |= NUMBER_LINES;
        else if(ch == '$')      NLead  = NTrail = 0;
        else if(ch == '/')      Flags |= FNAMES_ONLY;
        else if(ch == '^')      Flags ^= PRINT_FNAMES;
        else if(ch == '~')
        {   Flags |=  PRINT_NONMATCH;
            Flags &= ~PRINT_MATCH;
        }
        else if(isdigit(ch))
        {   NLead = NTrail = atoi(params) + 1;
            params         = skip_num(params);
            if(params[0] == ':' && isdigit(params[1]))
            {   NTrail = atoi(++params) + 1;
                params = skip_num(params);
            }
            --params;
        }
        else
            invalid_param('o', *params);
    }
    Write_fn = (  (Flags & (NUMBER_LINES + PRINT_FNAMES))
                ? number_blk : stdout_write);
}

void Stdout_Reset(const char *fname)
{
    Off            = 0;
    Line_num       = 1;
    Prev_match_end = -1;
    Fname          = fname;
    Prev_ch_nl     = TRUE;
    Have_match     = FALSE;
}

/* Print the trailing context for the preceding match (if there was one),
 * then print the leading context for the match at match_off with length
 * len. Record the end of the match or the replacement text if a
 * replacement with length rep_len if present. Return TRUE unless
 * only file names are to be printed */
int Stdout_Print(off_t match_off, int len, char *replace, int rep_len)
{
    off_t match_line_off, lead_start_off, nl_found;
    BufPtr buf_ptr = BufPtr_goto(match_off);

    if(Flags & FNAMES_ONLY)
    {   Have_match = TRUE;
        if(!(Flags & PRINT_NONMATCH))
            puts(Fname);
        return FALSE;
    }
 /* First determine start of line containing match */
    if(NLead)
    {   buf_ptr        = BufPtr_find_nl(buf_ptr, Off, -1, &nl_found);
        match_line_off = BufPtr_off(buf_ptr);
    }
    else
        match_line_off = match_off;

    write_trail(match_line_off);
    if(NLead > 1 && nl_found && match_line_off > Off)
    {   buf_ptr = BufPtr_find_nl(BufPtr_goto(match_line_off - 1), Off,
                                 -NLead + 1, &nl_found);
        lead_start_off = BufPtr_off(buf_ptr);
    }
    else
        lead_start_off = match_line_off;

 /* Process region between end of last context and start of current context
  * If lead_lines is -1, we are at end of file so it is not neccessary to
  * number the last nonmatching lines if they are printed */
    if(Flags & PRINT_NONMATCH)
    {   Sep_ch = DEFAULT_SEP;
        write_region(Off, lead_start_off);
    }
 /* Process region between start of leading context and current position */
    if(Flags & PRINT_MATCH)
    {   Sep_ch = CONTEXT_SEP;
        write_region(lead_start_off, match_line_off);
        Sep_ch = DEFAULT_SEP;
        write_region(match_line_off, match_off);
    }
    Off = match_off;
    if(!replace)
        Prev_match_end = maX(Prev_match_end, match_off + len);
    else
    {   Sep_ch     = DEFAULT_SEP;
        if(Flags & PRINT_MATCH)
            Write_fn(replace, rep_len);

        Prev_match_end = (Off += len);
    }
/*  printf("%ld %ld %ld\n", Off, Prev_match_end, Line_num); */
    Have_match = TRUE;
    return TRUE;
}

/* If a preceding match was found, write the match and its following context,
 *  stopping if the location of the current match is reached */
static void write_trail(off_t stop_off)
{
    Sep_ch = DEFAULT_SEP;
    if(!Have_match)
        return;

 /* If preceding match does not end before current location, just write
  * to current location if desired */
    if(Prev_match_end >= stop_off)
    {   if(Flags & PRINT_MATCH)
            write_region(Off, stop_off);
        else if(Flags & NUMBER_LINES)
            Buffers_Apply_fn(Off, stop_off, count_nl_blk);
        Off = stop_off;
    }
    else
    {   off_t trail_end_off, nl_found;
        BufPtr match_end, trail_end;
     /* Find end of trailing context for preceding match */
        if(NTrail)
        {   match_end = BufPtr_goto(  Prev_match_end != 0
                                    ? Prev_match_end - 1
                                    : Prev_match_end);
            trail_end = BufPtr_find_nl(match_end, stop_off, NTrail, &nl_found);
        }
        else
            match_end = trail_end = BufPtr_goto(Prev_match_end);

        trail_end_off = BufPtr_off(trail_end);

     /* Print up to end of context if match printing desired */
        if(Flags & PRINT_MATCH)
        {   write_region(Off, Prev_match_end);
            Sep_ch = CONTEXT_SEP;
            write_region(Prev_match_end, trail_end_off);
        }
        else if(Flags & NUMBER_LINES)
            Buffers_Apply_fn(Off, trail_end_off, count_nl_blk);
        Off = trail_end_off;
    }
}

/* Write the region bounded by the offsets start & end to standard output */
static void write_region(off_t start, off_t end)
{
 /* If line numbers are to be printed, first count all uncounted \ns up to
  *   start of write */
    if(start > Off)
    {   if(Flags & NUMBER_LINES)
            Buffers_Apply_fn(Off, start, count_nl_blk);
        if(Have_match && (NLead > 1 || NTrail > 1))
            stdout_write("--\n", 3);
    }
 /* Otherwise can just dump region to standard output */
    Buffers_Apply_fn(start, end, Write_fn);
    Off = end;
}

/* Print the block of text between start and end on standard output,
 *  with the appropriate line numbers */
static void number_blk(const charf *start, size_t len)
{
    /* Make sure number is not printed if region has zero length */
    if(!len)
        ;
 /* If numbering on, need to scan through block to determine line endings */
    else
    {   const charf *endln, *end = start + len;
     /* If previous character written to stdout \n, start with line no. */
        if(Prev_ch_nl)
            write_stdout_info();

        for( ; ; )
        {   if((endln = _fmemchr(start, NL, (size_t)(end - start))) == NULL)
                endln = end;
            else
            {   ++Line_num;
                ++endln;
            }
            stdout_write(start, (size_t)(endln - start));
            if((start = endln) < end)
                write_stdout_info();
            else
                break;
        }
     /* If last char \n, next write to stdout should begin with line no.*/
        Prev_ch_nl = (end[-1] == NL);
    }
}

/* Print the filename and/or line number corresponding to the current line
 * being written to stdout */
static void write_stdout_info(void)
{
    if(Flags & PRINT_FNAMES)
    {   stdout_write(Fname, strlen(Fname));
        stdout_write(&Sep_ch, 1);
    }
    if(Flags & NUMBER_LINES)
    {   char num_text[40];
        int num_len = sprintf(num_text, "%ld", (long)Line_num);
        num_text[num_len] = Sep_ch;
        stdout_write(num_text, num_len + 1);
    }
}

/* Write len bytes starting at ptr to standard output */
static void stdout_write(const char far *start, size_t len)
{
    x_write(STD_OUT, start, len);
}

/* Count the number of \n in the block bounded by the pointers start and end*/
static void count_nl_blk(const charf *start, size_t len)
{
    Line_num += count_nl(start, len);
}

void Stdout_Flush(void)
{
    Sep_ch = DEFAULT_SEP;
 /* If no strings to search for, write entire file unless only filenames
  * are to be printed */
    if(Flags & NO_STRINGS)
    {   if(!(Flags & FNAMES_ONLY))
            write_region(0, OFF_MAX);
        else if(!(Flags & PRINT_NONMATCH))
            puts(Fname);
    }
 /* Print filenames if Non-matching filenames to be printed & no match found*/
    else if(Flags & FNAMES_ONLY)
    {   if((Flags & PRINT_NONMATCH) && !Have_match)
            puts(Fname);
    }
    else
    {   write_trail(OFF_MAX);
        if(Flags & PRINT_NONMATCH)
            write_region(Off, OFF_MAX);
    }
}
