/* statline.c - generate status line at bottom of screen
 * Copyright (C) 1995-99 Andrew Pipkin (minitrue@pagesz.net)
 * MiniTrue is free software released with no warranty. See COPYING for details
 */

#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <errno.h>

#include "minitrue.h"
#include "statline.h"
#include "console.h"
#include "attribs.h"
#include "lines.h"
#include "winbuf.h"

static Window *Win_ptr, *Win_copy;
static int Len = -1;
static int Cursor_pos;
static int Have_cursor;
static int Have_message;
static int Getting_str = FALSE;

static int nlines(int len);
static int append(const char far *str, attr_t attr);
static int append_nchars(const char far *str, attr_t, int nchars);
static void append_command_str(const char far *str, attr_t attr);
static void truncate(int ndel);
static void write_part(int start_i, int end_i, int clear_eol);

void StatLine_Init(void)
{
    Win_ptr  = WinBuf_Init(8);
    Win_copy = WinBuf_Init(8);
}

int StatLine_Used(void) { return Win_ptr != NULL; }

static char far rep_cmds[] = " ^yes/^no/^All/^quit/^?help";

void StatLine_Make(const char *fname, off_t line_num, off_t file_lines,
                   int pcnt, const char *far replace, int rep_len)
{
    char str_buf[40], *buf_ptr = str_buf;
    attr_t attr = Attrib(STAT_LINE);
    Len = 0;

 /* Start by adding filename */
    append(fname, attr);

 /* now write line number/ percentage */
    buf_ptr += sprintf(buf_ptr, " %ld", (long)line_num);
    if(file_lines != -1)
        buf_ptr += sprintf(buf_ptr, "/%ld", (long)file_lines);
    buf_ptr += sprintf(buf_ptr, pcnt == -1 ? " ??%%" : " %02d%%", pcnt);
    append(str_buf, attr);
 /* Make sure percentage takes up 4 spaces */
    if(pcnt < 100)
        append(" ", attr);

 /* If replacement string present, append it to status line, putting
  * backslashed escape sequences in a different color */
    if(replace)
    {   append(" Replace/w ", attr);
        while(rep_len)
        {   char buf[10], *esc_end = ch_to_esc(buf, *replace);
            if(!append(buf, Attrib((esc_end == buf + 1)
                                   ? REP_TEXT : REP_CTRL)))
            {   truncate(_fstrlen(rep_cmds) - 1);
                append("...", Attrib(STAT_LINE));
                break;
            }
            ++replace;
            --rep_len;
        }
        append_command_str(rep_cmds, attr);
    }
    else
        append_command_str(" ^next/^quit/^?help", attr);

    Cursor_pos   = Len;
    Have_message = FALSE;
}

/* indicate if a message is being displayed */
int StatLine_Have_msg(void)
{
    int have_msg = Have_message;
    Have_message = FALSE;
    return have_msg;
}

/* Read in up to max_len chars into str_buf, it is assumed that buf has at
 * least max_len characters allocated. prompt_str contains the prompt
 * and the string will originally be set to default_str if it is not NULL
 * return the length of the string */
int StatLine_Getstr(char *str_buf, int max_len,
                    const char far *prompt_str, const char far *default_str)
{
    int curs_i = 0, str_len = 0, prompt_len = _fstrlen(prompt_str);
    int prev_lines = 0, prev_len = 0, prev_curs_i = 0;
    max_len = miN(max_len, max_len - 2);

 /* If default input present, copy it into the string buffer */
    if(default_str)
    {   int default_len = _fstrlen(default_str);
        default_len     = miN(default_len, max_len);
        _fmemcpy(str_buf, default_str, default_len);
        curs_i = str_len = default_len;
    }
    Getting_str = TRUE;
    for( ; ; )
    {   int ch;

     /* Copy the current contents of the string and prompt to the status line*/
        Len        = 0;
        Cursor_pos = prompt_len + curs_i;
        append(prompt_str, Attrib(PROMPT_TEXT));
        str_buf[str_len] = '\0';
        append(str_buf, Attrib(ENTERED_TEXT));

     /* If number of lines occupied by status lines has changed, write
      * entire status line to screen */
        if(prev_lines != nlines(Len))
        {   StatLine_Write();
            prev_lines = nlines(Len);
        }
     /* Otherwise determine changed portion and write that to the screen */
        else
        {   int start_i = prompt_len + miN(curs_i, prev_curs_i);
            int end_i   = maX(str_len, prev_len) + prompt_len;
            write_part(start_i, end_i, FALSE);
        }
        prev_curs_i = curs_i;
        prev_len    = str_len;
        ch = Console_Get_key();
        if(ch == '\r' || ch == '\n')
            break;
     /* backspace or DEL moves the cursor back, deleting the previous char */
        else if((ch == '\b' || ch == 0x7f) && curs_i > 0)
        {   memmove(&str_buf[curs_i - 1], &str_buf[curs_i], str_len - curs_i);
            --str_len;
            --curs_i;
        }
     /* control-D or the delete key deletes the character at the cursor */
        else if((ch == ctrl_chaR('d') || ch == DEL_KEY) && curs_i < str_len)
        {   memmove(&str_buf[curs_i], &str_buf[curs_i + 1], str_len - curs_i);
            --str_len;
        }
     /* control-G or escape exits and returns -1 */
        else if(ch == ctrl_chaR('g') || ch == '\033')
        {   str_len    = -1;
            str_buf[0] = '\0';
            break;
        }
     /* control-B or left arrow moves cursor to the right */
        else if((ch == LEFT_KEY || ch == ctrl_chaR('b')) && curs_i)
            --curs_i;
     /* control-F or right arrow moves cursor to the right */
        else if((ch == RIGHT_KEY || ch ==ctrl_chaR('f')) && curs_i < str_len)
            ++curs_i;
     /* control-A moves cursor to start of string */
        else if(ch == ctrl_chaR('a') || ch == HOME_KEY)
            curs_i = 0;
     /* control-E moves cursor to end of string */
        else if(ch == ctrl_chaR('e') || ch == END_KEY)
            curs_i = str_len;
     /* control-U erases string */
        else if(ch == ctrl_chaR('u'))
            curs_i = str_len = 0;
     /* control-K deletes from cursor to end of string */
        else if(ch == ctrl_chaR('k'))
            str_len  = curs_i;

     /* insert character if printing char */
        else if(isprint(ch) && str_len < max_len)
        {   memmove(&str_buf[curs_i + 1], &str_buf[curs_i], str_len - curs_i);
            str_buf[curs_i++] = ch;
            ++str_len;
        }
    }
    Getting_str = FALSE;
    return str_len;
}
/* Prompt for a single character, valid_chars is a string containing all
 * the valid chars. return the index of the entered char in valid_chars */
int StatLine_Getch(const char far *prompt_str, const char *valid_chars)
{
    StatLine_Message(prompt_str, Attrib(STAT_LINE));
    Have_message = FALSE;
    for( ; ; )
    {   int ch = Console_Get_key();
        const char *ch_loc = strchr(valid_chars, ch);

        if(ch_loc)
            return ch_loc - valid_chars;
        else if(ch == ctrl_chaR('g'))
            return -1;
    }
}

/* read in a positive long integer, base has same meaning as in strtol,
 * return -1 string not a valid positive number */
long StatLine_Getnum(const char far *prompt_str, int base)
{
    char num_buf[33], *num_end;
    long num;
    StatLine_Getstr(num_buf, 32, prompt_str, NULL);
    num = strtol(num_buf, &num_end, base);
    if(num < 0 || !*num_buf || *num_end || errno == ERANGE)
    {   if(*num_buf)
            StatLine_Error("Invalid Number");
        errno = 0;
        num   = -1;
    }
    return num;
}

/* Write a message to the status line */
void StatLine_Message(const char far *msg_text, attr_t attr)
{
    Len = 0;
    append(msg_text, attr);
    Cursor_pos = Len;
    StatLine_Write();
    Have_message = TRUE;
}

void StatLine_Error(const char far *err_msg)
{
    StatLine_Message(err_msg, Attrib(ERR_MSG));
}

/* Report an error reading a file fname, Display the list of options
 * with its corresponding command chars for dealing with the error */
int StatLine_File_error(const char far *err_msg, const char *fname,
                        const char far *options, const char *valid_chars)
{
    int orig_len = StatLine_Save();
    Len = 0;
    append(err_msg, Attrib(ERR_MSG));
    append(fname, Attrib(ERR_MSG));
    append_command_str(options, Attrib(STAT_LINE));
    Cursor_pos = Len;
    StatLine_Write();

    for( ; ; )
    {   int ch = Console_Get_key();
        const char *ch_loc = strchr(valid_chars, tolower(ch));

        if(ch_loc)
        {   StatLine_Restore(orig_len);
            return ch_loc - valid_chars;
        }
    }
}

/* Display the file name which is currently being scanning in the status
 * line */
void StatLine_Show_fname(const char *fname)
{
    Len = 0;
    append("Scanning ", Attrib(STAT_LINE));
    append(fname, Attrib(STAT_LINE));
    Cursor_pos = Len;
    StatLine_Write();
}

/* append the string str to the status line, giving it the attribute attr
 *   stop when '\0' is encountered. Return number of chars written */
static int append(const char far *str, attr_t attr)
{
    return append_nchars(str, attr, INT_MAX);
}

/* append the string str to the status line, giving it the attribute attr
 *   stop when '\0' is encountered or nchars have been written. Return
 *   1 if enough room to write string, 0 if not enough */
static int append_nchars(const char far *str, attr_t attr, int nchars)
{
    int ncols = Console_Cols(), nwrote, finished = TRUE;
    do
    {   int row = Len / ncols, col = Len % ncols;
        ScreenPtr line_buf = WinBuf_row(Win_ptr, row);
        if(!line_buf)
        {   finished = FALSE;
            break;
        }
        nwrote = LineBuf_put_nchars(line_buf, col, str, attr, nchars);
        Len += nwrote;
        str += nwrote;
        nchars -= nwrote;
    }while(*str && nchars);

    return finished;
}

/* append the string str to the status line, highlight letters preceded
 * by a ^ */
static void append_command_str(const char far *str, attr_t attr)
{
    int caret_i = 0;

    while(str[caret_i] != '\0' && str[caret_i] != '^')
        ++caret_i;
    append_nchars(str, attr, caret_i);

    while(str[caret_i] == '^')
    {   int orig_i = caret_i + 2;
        append_nchars(&str[++caret_i], Attrib(COMMAND_CH), 1);
        ++caret_i;
        while(str[caret_i] != '\0' && str[caret_i] != '^')
            ++caret_i;

        append_nchars(&str[orig_i], attr, caret_i - orig_i);
    }
}

/* Truncate ndel characters from the end of the command line */
static void truncate(int ndel)
{
    Len -= ndel;
    if(Len < 0)
        Len = 0;
}

/* Return the number of lines occupied by len characters */
int nlines(int len)
{
    return (len - 1) / Console_Cols() + 1;
}

int StatLine_Height(void) { return nlines(Len); }

/* These functions are needed to determine the where to put the cursor
 * after a screen rewrite */
void StatLine_Set_cursor(int stat_line_curs) { Have_cursor = stat_line_curs;}
int StatLine_Cursor(void) { return Have_cursor || Getting_str; }

/* write the entire status line */
void StatLine_Write(void) { write_part(0, Len, TRUE); }

/* copy the current status line into a buffer. Return the original
 *  length of the status line */
int StatLine_Save(void)
{
    WinBuf_copy(Win_copy, Win_ptr);
    return Len;
}

/* Restore the previous status line */
void StatLine_Restore(int orig_len)
{
    WinBuf_copy(Win_ptr, Win_copy);
    Len = orig_len;
    StatLine_Write();
}

/* write a part of the status line beginning at start_i and ending with
 * end_i, if clear_eol set clear to the end of the last line */
static void write_part(int start_i, int end_i, int clear_eol)
{
    int stat_lines = nlines(Len);
    Lines_Set_unused(stat_lines, TRUE);
    if(Len != -1)
    {   int nrows  = Console_Rows(), ncols = Console_Cols(), line;
        int nwrote = start_i;
        if(start_i != end_i)
        {   /* Clear the remainder of the status line */
            LineBuf_clear(WinBuf_row(Win_ptr, stat_lines - 1),
                          Len % ncols, Attrib(STAT_LINE));

            for(line = start_i / ncols; line < stat_lines; ++line)
            {   int screen_row = nrows - stat_lines + line;
                int line_chars = miN(end_i - nwrote, ncols);
                if(!line_chars)
                    ++line_chars;
                Console_Write(WinBuf_row(Win_ptr, line), screen_row,
                              nwrote % ncols, line_chars, clear_eol);
                nwrote += line_chars;
            }
        }
     /* If cursor should appear in the status line, put the cursor at
      * the Cursor_pos position */
        if(StatLine_Cursor())
        {   int curs_row = nrows - stat_lines + Cursor_pos/ncols;
            Console_Move_cursor(Cursor_pos % ncols, curs_row);
        }
    }
}
