/* signals.c - signal handling routines
 * Copyright (C) 1995-99 Andrew Pipkin (minitrue@pagesz.net)
 * MiniTrue is free software released with no warranty. See COPYING for details
 */

#include <stdlib.h>
#include <signal.h>
#include "minitrue.h"
#include "signals.h"
#include "console.h"
#include "strings.h"

static void abort_prog(int signum);
static void susp_handler(int sig_num);
static void cont_handler(int sig_num);
static void resize_handler(int sig_num);

void Signals_Init(int need_console)
{
#ifdef SIGHUP
    int sigs[] = { SIGINT, SIGTERM, SIGHUP }, nsigs = 3, sig_i;
#else
    int sigs[] = { SIGINT, SIGTERM }, nsigs = 2, sig_i;
#endif
    for(sig_i = 0; sig_i < nsigs; ++sig_i)
    {   if (signal(sigs[sig_i], abort_prog) == SIG_IGN)
            signal (sig_i, SIG_IGN);
    }
/* Initialize suspension & continuation handlers */
#ifdef SIGTSTP
    if(need_console)
    {   if(signal(SIGTSTP, susp_handler) == SIG_IGN)
            signal(SIGTSTP, SIG_IGN);
        if(signal(SIGCONT, cont_handler) == SIG_IGN)
            signal(SIGCONT, SIG_IGN);
    }
#endif
#ifdef SIGWINCH
    if(need_console)
    {   if(signal(SIGWINCH, resize_handler) == SIG_IGN)
            signal(SIGWINCH, SIG_IGN);
    }
#endif
}

/* reset keyboard before temporarily supsending program */
#ifdef  SIGTSTP
#include <unistd.h>
static void susp_handler(int sig_num)
{
    sig_num = sig_num; /* bypass unused variable warning */
    Console_Restore(TRUE);
    signal(SIGTSTP, SIG_DFL);
    raise(SIGTSTP);
}

/* Reset keyboard and reprint screen after resumption following suspension */
static void cont_handler(int sig_num)
{
    sig_num = sig_num; /* bypass unused variable warning */
    signal(SIGCONT, cont_handler);
    signal(SIGTSTP, susp_handler);
    Console_Init(NULL);
    MiniTrue_Rewrite();
}
#endif /* SIGTSTP */
#ifdef SIGWINCH
static void resize_handler(int sig_num)
{
    sig_num += 0;
    MiniTrue_Rewrite();
}
#endif

static void abort_prog(int sig_num)
{
    Strings_No_report();
    exit(sig_num);
}
