/* rxprint.c - convert regular expression data structure to text
 * Copyright (C) 1995-99 Andrew Pipkin (minitrue@pagesz.net)
 * MiniTrue is free software released with no warranty. See COPYING for details
 */

#include <stdlib.h>
#include "regexp.h"
#include "charset.h"
static int print_atom_data(RxAtom *rx);
static void Rx_print_all(RegExp *rx);
#ifdef RX_DUMP
#include "rxopname.h"

int main(int argc, char *argv[])
{
    int arg_i = 1;
    char line_buf[256], *rx_src;
    RegExp rx;

    CharSet_Init(NULL);

    while((rx_src = (argc > 1) ? argv[arg_i++] : fgets(line_buf, 256, stdin))
          != NULL)
    {   RegExp_init(&rx, rx_src);
        Rx_print_all(&rx);
        RegExp_kill(&rx);
    }
    return EXIT_SUCCESS;
}

static void Rx_print_all(RegExp *rx)
{
    int rx_i;
    for(rx_i = 0; rx_i < rx->len; ++rx_i)
    {   RegExp_print_atom(rx_i, rx);
        putchar('\n');
    }
}

void RegExp_print_atom(int rx_i, RegExp *reg_exp)
{
    RxAtom *rx_ptr = &reg_exp->start[rx_i];

    printf("%3d ", (int)(rx_ptr - reg_exp->start));
    printf((rx_ptr->flags & RX_REV) ? "<-" : "->");

    if(!Rx_is_data(rx_ptr) || rx_ptr->min)
    {   int fail_i = rx_ptr->fail.ptr - reg_exp->start;
        printf((fail_i >= reg_exp->len || fail_i < 0)
               ? " END "
               : " %3d ", fail_i);
    }
    else
        printf("     ");

    if(Rx_atom_len(rx_ptr))
        printf((rx_ptr->max < INT_MAX / 2)
               ? " %3d "
               : " INF ", rx_ptr->max / Rx_atom_len(rx_ptr));

    if(!print_atom_data(rx_ptr))
    {   if(rx_ptr->opcode == GROUP_END || rx_ptr->opcode == GROUP_END_REV)
            printf("%d, %d ", rx_ptr->min, rx_ptr->max);
        if(rx_ptr->flags & RX_BREF)
            printf("{%d} ", (int)(rx_ptr->data.paren_loc - reg_exp->start));
    }
    fputs(Op_names[rx_ptr->opcode], stdout);
}

/* Print atom if atom data has textual representation, return TRUE if anything
 * printed */
static int print_atom_data(RxAtom *rx)
{
    int type = Rx_type(rx);

    if(type == RX_CH || type == RX_STR)
    {   const char *ptr, *end;
        if(type == RX_STR)
        {   ptr = rx->data.str.start;
            end = rx->data.str.end;
        }
        else
        {   ptr = &rx->data.ch;
            end = ptr + 1;
        }
        for( ; ptr < end; ++ptr)
        {   char esc_buf[5];
            ch_to_esc(esc_buf, *ptr);
            fputs(esc_buf, stdout);
        }
    }
    else if(type == RX_SET)
        CharSet_dump(&rx->data.set);
    else
        return FALSE;

    return TRUE;
}

#endif /* RX_DUMP */


