/* rxfind.c - runtime routine for regular expression searching
 * Copyright (C) 1995-99 Andrew Pipkin (minitrue@pagesz.net)
 * MiniTrue is free software released with no warranty. See COPYING for details
 *
 * See all that stuff in there [Robo-Itchy's circuitry]. That's why your
 * robot didn't work
 */

#include <stdlib.h>
#include <string.h>
#include <assert.h>

#include "minitrue.h"
#include "regexp.h"
#include "fixedstr.h"
#include "charset.h"

static const char *Buf_start;
static const char *Buf_end;
static int Buf_loc, Buf_no;

static char *have_str(const char *str, const char *buf, int len);
static char const * *paren_buf_write(RxAtom *buf_atom);
static const char *paren_buf_read(RxAtom *buf_atom, int i);
static void restore_rparens(RxAtom *rmost_buf);
static SubRX *nearest_sub_rx(RxAtom *top_alt_atom);

#ifdef RX_TRACE
#include "rxopname.h"
static void print_trace(int rx_i, const char *op_name,
                        const char *match_start, const char *match_end);
#endif
#ifdef PRINT_PARS
static void print_pars(RegExp *rx, const char *buf_start, const char *buf_end);
#endif

/* Assign buffer boundary variables */
void RegExp_Buf_init(const char *buf_start, const char *buf_end, int buf_loc)
{
    Buf_start = buf_start;
    Buf_end   = buf_end;
    Buf_loc   = buf_loc;
    ++Buf_no;
}

const char *RegExp_find(RegExp *rx, const char *search_start, size_t *len)
{
    const char *match_start = search_start, *match_end = search_start;
    const char *anchor = search_start;
    RxAtom *rx_ptr = rx->start;

    for( ; ; )
    {
next_op:
        assert(   (rx->start <= rx_ptr && rx_ptr < &rx->start[rx->len])
               || rx_ptr->opcode == RX_FAIL);

#ifdef RX_TRACE
        print_trace(rx_ptr - rx->start, Op_names[rx_ptr->opcode],
                    match_start, match_end);
#ifdef PRINT_PARS
        print_pars(rx, Buf_start, Buf_end);
#endif
#endif
        switch(rx_ptr->opcode)
        { case RX_MATCH:
            *len = match_end - match_start;
            return match_start;

          case RX_FAIL:
            return Buf_end + 1;

          case RX_ADVANCE:
            if((match_start = anchor) > Buf_end)
                rx_ptr = rx_ptr->fail.ptr;
            else
            {   match_end = match_start;
                ++anchor;
                ++rx_ptr;
            }
            break;

          case RX_JUMP:
            rx_ptr = rx_ptr->fail.ptr;
            break;

          case JUMP_OFF:
            rx_ptr = rx_ptr->fail.ptr - rx_ptr->min;
            break;

       /* If the jump is NULL, jump to the end of the parentheseses */
          case JUMP_NULL:
            if(rx_ptr->fail.ptr != NULL)
                rx_ptr = rx_ptr->fail.ptr;
            else
                rx_ptr = rx_ptr->data.paren_loc;
            break;

          case RX_RESET:
            match_end = match_start;
            ++rx_ptr;
            break;

          case SKIP_NEXT:
            rx_ptr += 2;
            break;

          case SET_JUMP:
            (rx_ptr->data.paren_loc)->fail.ptr = rx_ptr->fail.ptr;
            ++rx_ptr;
            break;

          case CLEAR_PAREN:
            {   RxAtom *par_ptr           = rx_ptr->fail.ptr;
                par_ptr->data.paren.start = par_ptr->data.paren.end = NULL;
            }
            ++rx_ptr;
            break;

          case CLEAR_PARENS:
            {   int clear_i;
                AtomLoc *paren_list = rx_ptr->data.clear_list;
                for(clear_i = 0; clear_i < rx_ptr->max; ++clear_i)
                {   RxAtom *par_ptr           = paren_list[clear_i].ptr;
                    par_ptr->data.paren.start = par_ptr->data.paren.end =NULL;
                }
            }
            ++rx_ptr;
            break;

          case INIT_BACKTRACK:
            rx_ptr->data.backtrack.start = rx_ptr[-1].fail.opt;
            rx_ptr->data.backtrack.end   = match_end;
            rx_ptr->data.backtrack.orig  = match_start;
            rx_ptr += 2;
            break;

          case BACKTRACK:
            match_start = rx_ptr[-1].data.backtrack.orig;
            match_end   = --rx_ptr[-1].data.backtrack.end;
            rx_ptr = ((match_end >= rx_ptr[-1].data.backtrack.start)
                      ? rx_ptr + 1
                      : rx_ptr->fail.ptr);
            break;

          case INIT_FORTRACK:
            rx_ptr->data.backtrack.start = match_start;
            rx_ptr->data.backtrack.end   = rx_ptr[-1].fail.opt;
            rx_ptr->data.backtrack.orig  = match_end;
            rx_ptr += 2;
            break;

          case FORTRACK:
            match_end   = rx_ptr[-1].data.backtrack.orig;
            match_start = ++rx_ptr[-1].data.backtrack.start;
            rx_ptr = ((match_start <= rx_ptr[-1].data.backtrack.end)
                      ? rx_ptr + 1
                      : rx_ptr->fail.ptr);
            break;

          case SING_CH:   /* test for single instance of a character */
            if(match_end == Buf_end || low_casE(*match_end) != rx_ptr->data.ch)
                rx_ptr = rx_ptr->fail.ptr;
            else
            {   ++match_end;
                ++rx_ptr;
            }
            break;

          case MULT_CHS:  /* test for multiple instances of a character */
            if(Buf_end - match_end < rx_ptr->max)
                rx_ptr = rx_ptr->fail.ptr;

            else
            {   const char *buf_ptr = match_end;
                const char *search_end = buf_ptr + rx_ptr->max;
                char ch = rx_ptr->data.ch;
                do
                {   if(low_casE(*buf_ptr) != ch)
                    {   rx_ptr = rx_ptr->fail.ptr;
                        goto next_op;
                    }
                }while(++buf_ptr < search_end);
                match_end = search_end;
                ++rx_ptr;
            }
            break;

          case OPT_CHS:  /* test for optional instances of a character */
            {   const char *search_end =((Buf_end - match_end < rx_ptr->max)
                                         ? Buf_end : match_end + rx_ptr->max);
                char ch                 = rx_ptr->data.ch;

                rx_ptr->fail.opt = match_end;
                while(low_casE(*match_end) == ch && match_end < search_end)
                    ++match_end;

                ++rx_ptr;
            }
            break;

          case FIND_CH:
            anchor = memchr(anchor, rx_ptr->data.ch, Buf_end - anchor);
            if(!anchor)
                rx_ptr = rx_ptr->fail.ptr;
            else
            {   match_start = anchor;
                match_end   = anchor + 1;
                ++anchor;
                ++rx_ptr;
            }
            break;

          case SING_CH_REV:  /* test for single char at start of match */
            if(   match_start == search_start
               || low_casE(match_start[-1]) != rx_ptr->data.ch)
                rx_ptr = rx_ptr->fail.ptr;
            else
            {   --match_start;
                ++rx_ptr;
            }
            break;

          case MULT_CHS_REV:  /* test for multiple instances of a character */
            if(match_start - search_start < rx_ptr->max)
                rx_ptr = rx_ptr->fail.ptr;

            else
            {   const char *buf_ptr    = match_start;
                const char *search_end = match_start - rx_ptr->max;
                char ch = rx_ptr->data.ch;
                while(buf_ptr-- > search_end)
                {   if(low_casE(*buf_ptr) != ch)
                    {   rx_ptr = rx_ptr->fail.ptr;
                        goto next_op;
                    }
                }
                match_start = search_end;
                ++rx_ptr;
            }
            break;

          case OPT_CHS_REV:  /* test for optional instances of a character */
            {   const char *search_end;
                char ch          = rx_ptr->data.ch;
                rx_ptr->fail.opt = match_start;
                search_end = ( (match_start - search_start < rx_ptr->max)
                              ? search_start
                              : match_start - rx_ptr->max);

                while(   match_start > search_end
                      && low_casE(match_start[-1]) == ch)
                    --match_start;

                ++rx_ptr;
            }
            break;

          case SING_SET: /* test for single instance of a character set */
            {   if(   match_end == Buf_end
                   || !CharSet_iN(rx_ptr->data.set, *match_end))
                    rx_ptr = rx_ptr->fail.ptr;
                else
                {   ++match_end;
                    ++rx_ptr;
                }
            }
            break;

          case MULT_SETS:
            if(Buf_end - match_end < rx_ptr->max)
                rx_ptr = rx_ptr->fail.ptr;

            else
            {   const char *buf_ptr    = match_end;
                const char *search_end = buf_ptr + rx_ptr->max;
                CharSet ch_set         = rx_ptr->data.set;

                do
                {   if(!CharSet_iN(ch_set, *buf_ptr))
                    {   rx_ptr = rx_ptr->fail.ptr;
                        goto next_op;
                    }
                }while(++buf_ptr < search_end);
                match_end = search_end;
                ++rx_ptr;
            }
            break;

          case OPT_SETS:
            {   CharSet ch_set         = rx_ptr->data.set;
                const char *search_end = ((Buf_end - match_end < rx_ptr->max)
                                          ? Buf_end : match_end + rx_ptr->max);

                rx_ptr->fail.opt = match_end;
                while(CharSet_iN(ch_set, *match_end) && match_end < search_end)
                    ++match_end;

                ++rx_ptr;
            }
            break;

       /* Use Boyer-Moore type of algorithm to find character set(s) */
          case FIND_SET:
            {   int far *set_skips = rx_ptr->data.skips, len = rx_ptr->min;
                int backtrack_i, skip;
                anchor += len - 1;
find_set:
                do
                {   anchor += set_skips[(unsigned char)*anchor];
                    anchor += set_skips[(unsigned char)*anchor];
                    anchor += set_skips[(unsigned char)*anchor];
                    skip    = set_skips[(unsigned char)*anchor];
                    anchor += skip;
                    if(anchor > Buf_end)
                    {   rx_ptr = rx_ptr->fail.ptr;
                        goto next_op;
                    }
                }while(skip);

                for(backtrack_i = -1; backtrack_i > -len; --backtrack_i)
                {   if(set_skips[ (unsigned char)anchor[ backtrack_i] ] )
                    {   ++anchor;
                        goto find_set;
                    }
                }
                match_end   = anchor + 1;
                match_start = match_end - len;
                anchor      = match_start + 1;
                ++rx_ptr;
            }
            break;

          case FIND_SING_SET:
            {   int far *set_skips = rx_ptr->data.skips;
                while(anchor < Buf_end)
                {   if(!set_skips[(unsigned char)*anchor])
                    {   match_start = anchor;
                        match_end   = ++anchor;
                        ++rx_ptr;
                        goto next_op;
                    }
                    ++anchor;
                }
                rx_ptr = rx_ptr->fail.ptr;
            }
            break;

        /* Look for multiple instances of a commonly occurring character set
         * Do not use a Boyer-Moore algorithm because that would entail a
         * lot of unneeded backtracking */
          case FIND_FIRST_SETS:
            {   int len = rx_ptr->min, far *set_skips = rx_ptr->data.skips;
                --anchor;
                while((anchor += len) < Buf_end)
                {   int backtrack_i = len;
                    while(!set_skips[(unsigned char)*anchor])
                    {   if(!--backtrack_i)
                        {   match_start = anchor;
                            match_end   = anchor = anchor + len;
                            ++rx_ptr;
                            goto next_op;
                        }
                        --anchor;
                    }
                }
                rx_ptr = rx_ptr->fail.ptr;
            }
            break;

         /* This is similiar to FIND_FIRST_SETS, but since anchor is preceded
          * by the the desired # of sets, we will have enough sets if *anchor
          * is a member of the character set */
          case FIND_NEXT_SETS:
            {   int far *set_skips = rx_ptr->data.skips;
                if(anchor == Buf_end)
                    rx_ptr = rx_ptr->fail.ptr;
                else if(!set_skips[(unsigned char)*anchor])
                {   match_end = ++anchor;
                    match_start = anchor - rx_ptr->min;
                    ++rx_ptr;
                }
                else
                {   int len = rx_ptr->min;
                    while((anchor += len) < Buf_end)
                    {   int backtrack_i = len;
                        while(!set_skips[(unsigned char)*anchor])
                        {   if(!--backtrack_i)
                            {   match_start = anchor;
                                match_end   = anchor = anchor + rx_ptr->min;
                                ++rx_ptr;
                                goto next_op;
                            }
                            --anchor;
                        }
                    }
                    rx_ptr = rx_ptr->fail.ptr;
                }
                break;
            }

          case SING_SET_REV: /* test for single instance of a character set */
            {   if(match_start == search_start
                   || !CharSet_iN(rx_ptr->data.set, match_start[-1]))
                    rx_ptr = rx_ptr->fail.ptr;
                else
                {   --match_start;
                    ++rx_ptr;
                }
            }
            break;

          case MULT_SETS_REV:  /* test for multiple instances of a character */
            if(match_start - search_start < rx_ptr->max)
                rx_ptr = rx_ptr->fail.ptr;

            else
            {   const char *buf_ptr    = match_start;
                const char *search_end = match_start - rx_ptr->max;
                CharSet ch_set         = rx_ptr->data.set;
                while(buf_ptr-- > search_end)
                {   if(!CharSet_iN(ch_set, *buf_ptr))
                    {   rx_ptr = rx_ptr->fail.ptr;
                        goto next_op;
                    }
                }
                match_start = search_end;
                ++rx_ptr;
            }
            break;

          case OPT_SETS_REV:
            {   CharSet ch_set         = rx_ptr->data.set;
                const char *search_end;
                rx_ptr->fail.opt = match_start;
                search_end = ( (match_start - search_start < rx_ptr->max)
                              ? search_start
                              : match_start - rx_ptr->max);

                while(   match_start > search_end
                      && CharSet_iN(ch_set, match_start[-1]))
                    --match_start;

                ++rx_ptr;
            }
            break;

          case UNIV_SET:
            {   size_t bytes_after = Buf_end - match_end;
                if(bytes_after >= (size_t)rx_ptr->min)
                {   rx_ptr[1].fail.opt = match_end + rx_ptr->min;
                    match_end = (  (bytes_after >= (size_t)rx_ptr->max)
                                 ?  match_end + rx_ptr->max
                                 :  Buf_end);
                    rx_ptr += 2;
                }
                else
                    rx_ptr = rx_ptr->fail.ptr;
            }
            break;

          case NEG_CHAR_SET:
            {   size_t bytes_after = Buf_end - match_end;
                size_t n = ((bytes_after >= (size_t)rx_ptr->max)
                            ? rx_ptr->max
                            : bytes_after);
                const char *set_end = memchr(match_end, rx_ptr->data.ch, n);
                if(!set_end)
                    set_end = match_end + n;
                if(set_end - match_end >= rx_ptr->min)
                {   rx_ptr[1].fail.opt = match_end + rx_ptr->min;
                    match_end = set_end;
                    rx_ptr += 2;
                }
                else
                    rx_ptr = rx_ptr->fail.ptr;
            }
            break;

          case UNIV_SET_REV:
            {   size_t bytes_before = match_start - search_start;
                if(bytes_before >= (size_t)rx_ptr->min)
                {   rx_ptr[1].fail.opt = match_start - rx_ptr->min;
                    match_start = (  (bytes_before >= (size_t)rx_ptr->max)
                                   ?  match_start - rx_ptr->max
                                   :  search_start);
                    rx_ptr += 2;
                }
                else
                    rx_ptr = rx_ptr->fail.ptr;
            }
            break;

          case NEG_CHAR_SET_REV:
            {   size_t bytes_before = match_start - search_start;
                size_t n = ((bytes_before >= (size_t)rx_ptr->max)
                            ? rx_ptr->max
                            : bytes_before);
                const char *set_start = memrchr(match_start - n,
                                                rx_ptr->data.ch, n);
                set_start = (!set_start) ? match_start - n : set_start + 1;

                if(match_start - set_start >= rx_ptr->min)
                {   rx_ptr[1].fail.opt = match_start - rx_ptr->min;
                    match_start = set_start;
                    rx_ptr += 2;
                }
                else
                    rx_ptr = rx_ptr->fail.ptr;
            }
            break;

          case SING_STR:
            {   int len = rx_ptr->data.str.len;
                if(   Buf_end - match_end < len
                   || !have_str(rx_ptr->data.str.start, match_end, len))
                    rx_ptr = rx_ptr->fail.ptr;
                else
                {   match_end += len;
                    ++rx_ptr;
                }
            }
            break;

          case MULT_STRS:
            if(Buf_end - match_end < rx_ptr->max)
                rx_ptr = rx_ptr->fail.ptr;
            else
            {   const char *search_end = match_end + rx_ptr->max;
                const char *buf_ptr    = match_end;
                while(   buf_ptr < search_end
                      && (buf_ptr = have_str(rx_ptr->data.str.start, buf_ptr,
                                             rx_ptr->data.str.len)) != NULL)
                    ;
                if(!buf_ptr)
                    rx_ptr = rx_ptr->fail.ptr;
                else
                {   match_end = buf_ptr;
                    ++rx_ptr;
                }
            }
            break;

          case OPT_STRS:
            {   const char *buf_ptr = match_end, *str = rx_ptr->data.str.start;
                int len = rx_ptr->data.str.len;
                const char *search_end;

                rx_ptr->fail.opt = match_end;
                search_end = (  (Buf_end - buf_ptr < rx_ptr->max)
                              ? match_end + rx_ptr->max
                              : (Buf_end - (Buf_end - match_end) % len));

                while(buf_ptr < search_end
                      && !(buf_ptr = have_str(str, buf_ptr, len)))
                    match_end = buf_ptr;
                ++rx_ptr;
            }
            break;

          case FIND_STR:
            anchor = FixedStr_find(rx_ptr->data.bmoore.table, anchor, Buf_end);
            if(anchor >= Buf_end)
                rx_ptr = rx_ptr->fail.ptr;
            else
            {   match_start = anchor;
                match_end   = anchor + rx_ptr->max;
                ++anchor;
                ++rx_ptr;
            }
            break;

          case SING_STR_REV:
            {   int len = rx_ptr->data.str.len;
                if(    match_start - search_start < len
                   || !(have_str(rx_ptr->data.str.start,
                                 match_start - len, len)))
                    rx_ptr = rx_ptr->fail.ptr;
                else
                {   match_start -= len;
                    ++rx_ptr;
                }
            }
            break;

          case MULT_STRS_REV:
            if(match_start - search_start < rx_ptr->max)
                rx_ptr = rx_ptr->fail.ptr;
            else
            {   const char *buf_ptr = match_start - rx_ptr->max;
                while(   buf_ptr < match_start
                      && (buf_ptr = have_str(rx_ptr->data.str.start, buf_ptr,
                                             rx_ptr->data.str.len)) != NULL)
                    ;
                if(!buf_ptr)
                    rx_ptr = rx_ptr->fail.ptr;
                else
                {   match_end -= rx_ptr->max;
                    ++rx_ptr;
                }
            }

          case OPT_STRS_REV:
            {   const char *str = rx_ptr->data.str.start, *search_end;
                int len = rx_ptr->data.str.len;
                int bytes_before = match_start - search_start;

                rx_ptr->fail.opt = match_start;
                search_end  = ((bytes_before < rx_ptr->max)
                               ? match_start - rx_ptr->max
                               : search_start + (bytes_before % len));
                while(   match_start > search_end
                      && have_str(str, match_start - len, len))
                    match_start -= len;

            }
            ++rx_ptr;
            break;

          case SING_BREF:
            {   RxAtom *bref_loc = rx_ptr->data.paren_loc;
                const char *start = bref_loc->data.paren.start;
                int len           = bref_loc->data.paren.end - start;

                if(Buf_end - match_end >= len)
                {   int str_i;
                    for(str_i = 0; str_i < len; ++str_i)
                    {   if(low_casE(match_end[str_i]) !=low_casE(start[str_i]))
                        {   rx_ptr = rx_ptr->fail.ptr;
                            goto next_op;
                        }
                    }
                    match_end += len;
                    ++rx_ptr;
                }
                else
                    rx_ptr = rx_ptr->fail.ptr;

                break;
            }

          case SING_BREF_CS:
            {   RxAtom *bref_loc = rx_ptr->data.paren_loc;
                const char *start = bref_loc->data.paren.start;
                int len           = bref_loc->data.paren.end - start;

                if(Buf_end - match_end >= len && !memcmp(match_end, start,len))
                {   match_end += len;
                    ++rx_ptr;
                }
                else
                    rx_ptr = rx_ptr->fail.ptr;

                break;
            }

          case SING_BREF_CH:
            {   char ch = *((rx_ptr->data.paren_loc)->data.paren.start);
                if(low_casE(*match_end) !=low_casE(ch) || Buf_end == match_end)
                    rx_ptr = rx_ptr->fail.ptr;
                else
                {   ++match_end;
                    ++rx_ptr;
                }
                break;
            }

          case MULT_BREFS:
            {   RxAtom *bref_loc    = rx_ptr->data.paren_loc;
                const char *start   = bref_loc->data.paren.start;
                const char *end     = bref_loc->data.paren.end;
                int len             = end - start;
                const char *buf_ptr = match_end;
                if(Buf_end - match_end < len * rx_ptr->max)
                    rx_ptr = rx_ptr->fail.ptr;
                else
                {   int bref;
                    for(bref = 0; bref < rx_ptr->max; ++bref)
                    {   if(!have_str(start, buf_ptr, len))
                        {   rx_ptr = rx_ptr->fail.ptr;
                            goto next_op;
                        }
                        buf_ptr += len;
                    }
                    ++rx_ptr;
                    match_end = buf_ptr; 
                }
                break;
            }

          case OPT_BREFS:
            {   RxAtom *bref_loc = rx_ptr->data.paren_loc;
                const char *start = bref_loc->data.paren.start;
                int len           = bref_loc->data.paren.end - start;
                int bref;

                rx_ptr->fail.opt = match_end;
                for(bref = 0; bref < rx_ptr->max; ++bref)
                {   if(   Buf_end - match_end < len
                       || !have_str(start, match_end, len))
                        break;
                    match_end += len;
                }
                ++rx_ptr;
                break;
            }

       /* If the distance between a backreference and its corresponding paren
        * is fixed, that backreference can be found by first finding two
        * similar chars that distance apart */
          case FIND_BREF:
            {   int dist = rx_ptr->data.bref.dist, len = rx_ptr->data.bref.len;
find_first_chars:
                while(anchor + dist + len < Buf_end)
                {   if(low_casE(anchor[0]) == low_casE(anchor[dist]))
                        break;
                    if(low_casE(anchor[1]) == low_casE(anchor[dist + 1]))
                    {   ++anchor;
                        break;
                    }
                    if(low_casE(anchor[2]) == low_casE(anchor[dist + 2]))
                    {   anchor += 2;
                        break;
                    }
                    if(low_casE(anchor[3]) == low_casE(anchor[dist + 3]))
                    {   anchor += 3;
                        break;
                    }
                    anchor += 4;
                }
             /* See if characters after first char match */
                if(anchor + len + dist <= Buf_end)
                {   int str_i = 0;
                    do
                    {   if(++str_i == len)
                        {   match_start = anchor++ + dist;
                            match_end   = match_start + len;
                            ++rx_ptr;
                            goto next_op;
                        }
                    }while(   low_casE(anchor[str_i])
                           == low_casE(anchor[dist + str_i]));

                    ++anchor;
                    goto find_first_chars;
                }
                rx_ptr = rx_ptr->fail.ptr;
                break;
            }

       /* If case-sensitive, do not need to conver chars to lower case
        * before comparing */
          case FIND_BREF_CS:
            {   int dist = rx_ptr->data.bref.dist, len = rx_ptr->data.bref.len;
find_first_chars_cs:
                while(anchor + dist + len < Buf_end)
                {   if(anchor[0] == anchor[dist])
                        break;
                    if(anchor[1] == anchor[dist + 1])
                    {   ++anchor;
                        break;
                    }
                    if(anchor[2] == anchor[dist + 2])
                    {   anchor += 2;
                        break;
                    }
                    if(anchor[3] == anchor[dist + 3])
                    {   anchor += 3;
                        break;
                    }
                    anchor += 4;
                }
             /* Compare remaining characters in backref */
                if(   anchor + len + dist <= Buf_end)
                {   if(!memcmp(anchor, anchor + dist, len))
                    {   match_start = anchor++ + dist;
                        match_end   = match_start + len;
                        ++rx_ptr;
                        goto next_op;
                    }
                    ++anchor;
                    goto find_first_chars_cs;
                }
            }
            rx_ptr = rx_ptr->fail.ptr;
            break;

          case FIND_FILE_START:
            if(anchor == Buf_start && Buf_loc & BUF_BOF)
            {   match_start = match_end = anchor++;
                ++rx_ptr;
            }
            else
                rx_ptr = rx_ptr->fail.ptr;
            break;

          case FIND_FILE_END:
            if(anchor == Buf_start && Buf_loc & BUF_EOF)
            {   match_start = match_end = Buf_end;
                ++anchor;
                ++rx_ptr;
            }
            else
                rx_ptr = rx_ptr->fail.ptr;
            break;

          case FIND_LINE_START:
         /* Consider start of file a line start */
            if(anchor == Buf_start && Buf_loc & BUF_BOF)
            {   ++rx_ptr;
                match_start = match_end = anchor++;
            }
            else if(anchor >= Buf_end)
                rx_ptr = rx_ptr->fail.ptr;

         /* See if anchor is already at line start */
            else if(anchor > Buf_start && anchor[-1] == NL)
            {   ++rx_ptr;
                match_start = match_end = anchor++;
            }
         /* Otherwise search for the next \n */
            else
            {   anchor = memchr(anchor, NL, Buf_end - anchor);
                if(!anchor)
                    rx_ptr = rx_ptr->fail.ptr;
                else
                    ++rx_ptr;
                match_start = match_end = ++anchor;
            }
            break;

          case FIND_LINE_END:
            if(anchor > Buf_end)
                rx_ptr = rx_ptr->fail.ptr;
            else
            {   anchor = memchr(anchor, NL, Buf_end - anchor);
                if(!anchor)
                {   if(Buf_loc & BUF_EOF)
                    {   anchor = Buf_end;
                        ++rx_ptr;
                    }
                    else
                        rx_ptr = rx_ptr->fail.ptr;
                }
                else
                    ++rx_ptr;
            }
            match_start = match_end = anchor;
            ++anchor;
            break;

          case LINE_START:
            if(match_end != Buf_start)
                rx_ptr = (match_end[-1] == NL) ? ++rx_ptr : rx_ptr->fail.ptr;
            else if(Buf_loc & BUF_BOF)
                ++rx_ptr;
            else
                rx_ptr = rx_ptr->fail.ptr;
            break;

          case LINE_END:
            if(match_end != Buf_end)
                rx_ptr = (*match_end == NL) ? ++rx_ptr : rx_ptr->fail.ptr;
            else if(Buf_loc & BUF_EOF)
                ++rx_ptr;
            else
                rx_ptr = rx_ptr->fail.ptr;
            break;

          case LINE_START_REV:
            if(match_start != Buf_start)
                rx_ptr = (match_start[-1] == NL) ? ++rx_ptr : rx_ptr->fail.ptr;
            else if(Buf_loc & BUF_BOF)
                ++rx_ptr;
            else
                rx_ptr = rx_ptr->fail.ptr;
            break;

          case LINE_END_REV:
            if(match_start != Buf_end)
                rx_ptr = (*match_start == NL) ? ++rx_ptr : rx_ptr->fail.ptr;
            else if(Buf_loc & BUF_EOF)
                ++rx_ptr;
            else
                rx_ptr = rx_ptr->fail.ptr;
            break;

          case FIND_WORD_BREAK:
            if(anchor > Buf_end)
                rx_ptr = rx_ptr->fail.ptr;
            else if(anchor == Buf_start || !CharSet_in_worD(anchor[-1]))
            {   while(!CharSet_in_worD(*anchor))
                {   if(anchor >= Buf_end - 1)
                    {   if((Buf_loc & BUF_EOF) || anchor == Buf_end)
                        {   rx_ptr = rx_ptr->fail.ptr;
                            goto next_op;
                        }
                    }
                    ++anchor;
                }
            }
            else
            {   while(CharSet_in_worD(*anchor))
                {   if(anchor >= Buf_end - 1)
                    {   if(Buf_loc & BUF_EOF)
                            break;
                        else if(anchor == Buf_end)
                        {   rx_ptr = rx_ptr->fail.ptr;
                            goto next_op;
                        }
                    }
                    ++anchor;
                }
            }
            ++rx_ptr;
            match_start = match_end = anchor++;
            break;

          case FIND_NON_BREAK:
            {   int lead_word, trail_word;
                if(anchor == Buf_start)
                    lead_word = FALSE;
                else
                    lead_word = (CharSet_in_worD(anchor[-1]) != 0);

                for( ; ; )
                {   if(anchor > Buf_end)
                    {   rx_ptr = rx_ptr->fail.ptr;
                        goto next_op;
                    }
                    trail_word = ((anchor == Buf_end && (Buf_loc & BUF_EOF))
                                  ? FALSE
                                  : (CharSet_in_worD(*anchor) != 0));

                    if((lead_word ^ trail_word) == 0)
                        break;
                    lead_word = trail_word;
                    ++anchor;
                }
                match_start = match_end = anchor++;
                ++rx_ptr;
            }
            break;

          case WORD_BREAK:
            rx_ptr = (  (CharSet_in_worD(*match_end)
                       ^ CharSet_in_worD(match_end[-1]))
                      ? rx_ptr + 1
                      : rx_ptr->fail.ptr);
            break;

          case NON_BREAK:
            rx_ptr = ((  CharSet_in_worD(*match_end)
                       ^ CharSet_in_worD(match_end[-1]))
                      ? rx_ptr->fail.ptr
                      : rx_ptr + 1);
            break;

          case WORD_BREAK_REV:
            rx_ptr = ((  CharSet_in_worD(*match_start)
                       ^ CharSet_in_worD(match_start[-1]))
                      ? rx_ptr + 1
                      : rx_ptr->fail.ptr);
            break;

          case NON_BREAK_REV:
            rx_ptr = ((  CharSet_in_worD(*match_start)
                       ^ CharSet_in_worD(match_start[-1]))
                      ? rx_ptr->fail.ptr
                      : rx_ptr + 1);
            break;

          case PAREN_START:
            {   RxAtom *paren_loc = rx_ptr->fail.ptr;
                paren_loc->data.paren.start = match_end;
                ++rx_ptr;
                break;
            }

          case PAREN_END:
            {   rx_ptr->data.paren.end = match_end;
                ++rx_ptr;
                break;
            }

          case FIXED_LEN_PAR:
            {   rx_ptr->data.paren.start = match_end - rx_ptr->max;
                rx_ptr->data.paren.end   = match_end;
                ++rx_ptr;
                break;
            }

          case RESET_STACK:
            {   RxAtom *paren_loc         = rx_ptr->fail.ptr;
                RxAtom *buf_loc           = paren_loc[-1].fail.ptr - 1;
                buf_loc->data.paren_buf.i = 0;

                ++rx_ptr;
            }
            break;


       /* Indicate the start of a stack by placing NULL in it.
        * Cannot assume stack start is when i = 0 because of possibility
        * of nested parentheses
        * Assumptions: fail points to end of parentheses, from which
        *   start will be obtained */
          case START_STACK:
            {   RxAtom *paren_loc = rx_ptr->fail.ptr;
                RxAtom *stack_loc = paren_loc[-1].fail.ptr - 1;
                int i             = stack_loc->data.paren_buf.i;
                char const * *buf = stack_loc->data.paren_buf.start;

                buf[i] = NULL;
                stack_loc->data.paren_buf.i += stack_loc->min;

             /* Make sure there is enough free space on the stack */
                if(  stack_loc->data.paren_buf.i + stack_loc->min
                   > stack_loc->max)
                {   char const * *buf = stack_loc->data.paren_buf.start;
                    int new_size = stack_loc->max = stack_loc->max * 2;
                    buf = x_realloc(buf, sizeof(char *) * new_size);
                    stack_loc->data.paren_buf.start = buf;
                }
            }
            ++rx_ptr;
            break;

          case GROUP_START:
            {   char const * *buffer;
                int i                = rx_ptr->data.paren_buf.i;
                RxAtom *paren_loc    = rx_ptr->fail.ptr;

                paren_loc->data.paren.start = match_end;
                paren_loc->data.paren.end   = match_end;
                paren_loc[-1].data.count    = 0;

                buffer    = &rx_ptr->data.paren_buf.start[i];
                buffer[0] = buffer[1] = match_end;
                memset(buffer + 2, 0, (rx_ptr->min - 2) * sizeof(char *));

                ++rx_ptr;
            }
            break;

       /* If end of quantified paren reached, exit paren if maximum attained
        * try for another if count less than maximum.
        * Assumptions: GROUP_END is followed by GROUP_FAIL which stores the
        *   paren start & end. The fail points to the atom after the parens
        *   GROUP_START atom */
          case GROUP_END:
            {   if(rx_ptr->max < 0)
                    rx_ptr->max = abs(rx_ptr->max);

                rx_ptr[1].data.paren.end = match_end;
                paren_buf_write(rx_ptr->fail.ptr - 1);

             /* Jump to atom following GROUP_FAIL if max reached */
                if(++(rx_ptr->data.count) == rx_ptr->max)
                    rx_ptr += 2;
             /* Otherwise jump back to start */
                else
                {   rx_ptr[1].data.paren.start = match_end;
                    rx_ptr = rx_ptr->fail.ptr;
                }
            }
            break;

          case GROUP_FAIL:

            if(rx_ptr[-1].max <= 0)
            {   rx_ptr[-1].max = abs(rx_ptr[-1].max);
                rx_ptr = rx_ptr->fail.ptr;
            }
            else if(rx_ptr[-1].data.count >= rx_ptr[-1].min)
            {   match_end = paren_buf_read(rx_ptr[-1].fail.ptr - 1, 1);
                ++rx_ptr;
            }
            else
                rx_ptr = rx_ptr->fail.ptr;
            break;

          case REENTER_PAREN:
            {   RxAtom *count_ptr = rx_ptr->fail.ptr - 1;
                count_ptr->max = -count_ptr->max;
                --count_ptr->data.count;
                ++rx_ptr;
            }
            break;

       /* Pop the paren data off the stack and set match_end to the end
        *   of the previous parentheses
        * Assumptions: data.count contains the relative offset of atom which
        *   contains the stack */
          case POP_PAREN:
            {   RxAtom *buf_loc = rx_ptr + rx_ptr->data.count;
                char const * *buf_ptr = (  buf_loc->data.paren_buf.start
                                         + buf_loc->data.paren_buf.i
                                         - buf_loc->min);

             /* If stack is empty, first value will be NULL */
                if(buf_loc->data.paren_buf.i < buf_loc->min || *buf_ptr ==NULL)
                    rx_ptr = rx_ptr->fail.ptr;

             /* If stack not empty, copy parentheses data to the appropriate
              * locations */
                else
                {   char const * * *srcs  = buf_loc->data.paren_buf.locs;
                    int i;
                    match_end = buf_ptr[1];
                    for(i = 0; i < buf_loc->min; ++i)
                        *srcs[i] = buf_ptr[i];

                    ++rx_ptr;
                }
                buf_loc->data.paren_buf.i -= buf_loc->min;
            }
            break;

          case PAREN_START_REV:
            {   RxAtom *paren_loc = rx_ptr->fail.ptr;
                paren_loc->data.paren.end   = match_start;
                ++rx_ptr;
                break;
            }

          case PAREN_END_REV:
            {   rx_ptr->data.paren.start = match_start;
                ++rx_ptr;
                break;
            }

          case FIXED_LEN_PAR_REV:
            {   rx_ptr->data.paren.start = match_start;
                rx_ptr->data.paren.end   = match_start + rx_ptr->max;
                ++rx_ptr;
                break;
            }

          case GROUP_START_SPLIT:
            {   RxAtom *paren_loc           = rx_ptr->fail.ptr;
                paren_loc->data.paren.start = match_start;
                paren_loc->data.paren.end   = match_start;
                paren_loc[-1].data.count    = 0;

                ++rx_ptr;
            }
            break;

          case GROUP_START_REV:
            {   char const * *buffer;
                int i             = rx_ptr->data.paren_buf.i;
                RxAtom *paren_loc = rx_ptr->fail.ptr;

                paren_loc->data.paren.start = match_start;
                paren_loc->data.paren.end   = match_start;

                memset(paren_loc->data.paren.rmost, 0,
                       paren_loc->min * sizeof(char * *));
                paren_loc[-1].data.count    = 0;

                buffer    = &rx_ptr->data.paren_buf.start[i];
                buffer[0] = buffer[1] = match_start;

                memset(buffer + 2, 0, (rx_ptr->min - 2) * sizeof(char * *));
                ++rx_ptr;
            }
            break;

       /* If end of reverse quantified paren reached, exit paren if maximum
        * attained, try for another if count less than maximum.
        * Assumptions: GROUP_END_REV is followed by GROUP_FAIL_REV which
        *   stores the paren start & end. The fail points to the atom
        *   after the parens GROUP_START_REV atom */
          case GROUP_END_REV:
            {   RxAtom *par_start      = rx_ptr->fail.ptr - 1;
                char const * *buf_ptr;

                rx_ptr[1].data.paren.start = match_start;
                buf_ptr = paren_buf_write(rx_ptr->fail.ptr - 1);
                rx_ptr[1].data.paren.end = match_end;

                if(rx_ptr->max < 0)
                    rx_ptr->max = abs(rx_ptr->max);

             /* Store subparens for first reverse parentheses into rmost
              * buffer */
                if(!rx_ptr->data.count)
                    memcpy(rx_ptr[1].data.paren.rmost, buf_ptr,
                           sizeof(char * *) * rx_ptr[1].min);

                if(++(rx_ptr->data.count) == rx_ptr->max)
                {   restore_rparens(rx_ptr + 1);
                    rx_ptr += 2;
                }
                else
                    rx_ptr  = par_start + 1;
            }
            break;

          case GROUP_FAIL_REV:
            if(rx_ptr[-1].max <= 0)
            {   rx_ptr[-1].max = abs(rx_ptr[-1].max);
                rx_ptr = rx_ptr->fail.ptr;
            }
            else if(rx_ptr[-1].data.count >= rx_ptr[-1].min)
            {   match_start = paren_buf_read(rx_ptr[-1].fail.ptr - 1, 0);

                if(rx_ptr[-1].data.count > 1)
                    restore_rparens(rx_ptr);

                ++rx_ptr;
            }
            else
                rx_ptr = rx_ptr->fail.ptr;
            break;

       /* Pop the paren data off the stack and set match_start to the start
        * of the previous parentheses
        * Assumptions: data.count contains relative offset of the atom
        * containing the stack */
          case POP_PAREN_REV:
            {   RxAtom *buf_loc = rx_ptr + rx_ptr->data.count;
                char const * *buf_ptr = (  buf_loc->data.paren_buf.start
                                         + buf_loc->data.paren_buf.i
                                         - buf_loc->min);

             /* If stack is empty, first value will be NULL */
                if(buf_loc->data.paren_buf.i < buf_loc->min || *buf_ptr ==NULL)
                    rx_ptr = rx_ptr->fail.ptr;

             /* If stack not empty, copy parentheses data to the appropriate
              * locations */
                else
                {   char const * * *srcs  = buf_loc->data.paren_buf.locs;
                    int i;
                    match_start = buf_ptr[0];
                    if(match_start == rx_ptr[-2].data.paren.end)
                        rx_ptr[-2].data.paren.start = match_start;

                    for(i = 2; i < buf_loc->min; ++i)
                        *srcs[i] = buf_ptr[i];

                    ++rx_ptr;
                }
                buf_loc->data.paren_buf.i -= buf_loc->min;
            }
            break;

          case PAREN_START_SPLIT:
            {   rx_ptr->fail.ptr->data.paren.start = match_start;
                ++rx_ptr;
            }
            break;

          case ADVANCE_STACK:
            if(rx_ptr[1].data.count >= rx_ptr[1].min && rx_ptr[1].max >= 0)
            {   RxAtom *stack_loc = rx_ptr->fail.ptr;
                int new_i = stack_loc->data.paren_buf.i + stack_loc->min;
                if(new_i + (stack_loc->min * 2) > stack_loc->max)
                {   char const * *buf = stack_loc->data.paren_buf.start;
                    int new_size = stack_loc->max = stack_loc->max * 2;
                    buf = x_realloc(buf, sizeof(char *) * new_size);
                    stack_loc->data.paren_buf.start = buf;
                }
                stack_loc->data.paren_buf.i = new_i;
            }
            ++rx_ptr;
            break;

       /* See if an ambiguous alternative has more untried alternatives,
        * if it does then attempt the alternative, if not then advance
        * Assumptions: The GROUP_FAIL atom for the alternative is 2 atoms
        * before the atom and the link for the atom points to the atom where
        * the next alternative to try is stored. If there are no more
        * alternatives to try, the next alternative will be set to the
        * GROUP_FAIL atom */
          case TEST_UNTRIED:
            {   RxAtom *next_alt = (rx_ptr->fail.ptr)->fail.ptr;
                paren_buf_read(rx_ptr[-3].fail.ptr - 1, 0);
                rx_ptr = (next_alt != rx_ptr - 2) ? next_alt : rx_ptr + 1;
                break;
            }

        /* Before starting with a quantified parentheses, clear the
         * sub-paren buffer */
          case CLEAR_SUB_PARENS:
            {   RxAtom *sub_paren_atom = rx_ptr->fail.ptr + 1;
                char * *buf_ptr        = sub_paren_atom->data.sub_parens.buf;
                memset(buf_ptr, 0, 2 * sizeof(char *) * sub_paren_atom->max);
            }
            ++rx_ptr;
            break;

          case COPY_PAREN:
            {   RxAtom *src  = rx_ptr->fail.ptr;
                RxAtom *dest = rx_ptr->data.paren_loc;
                dest->data.paren.start = src->data.paren.start;
                dest->data.paren.end   = src->data.paren.end;
            }
            ++rx_ptr;
            break;

        /* If quantified paren contents can have a length for zero,
         * exit quantified paren if contents zero length to avoid
         * infinite loop */
          case TEST_ZERO_PAREN:
            {   RxAtom *paren_loc = rx_ptr->fail.ptr;
                if(paren_loc->data.paren.end == match_end)
                    rx_ptr = paren_loc + 1;
                else
                    ++rx_ptr;
            }
            break;

          case TEST_ZERO_PAREN_REV:
            {   RxAtom *paren_loc = rx_ptr->fail.ptr;
                if(paren_loc->data.paren.start == match_start)
                    rx_ptr = paren_loc + 1;
                else
                    ++rx_ptr;
            }
            break;

       /* If alternative matched in non-quantified paren, jump to atom
        * following paren after setting paren end */
          case HAVE_ALT:
            {   RxAtom *jump = rx_ptr->fail.ptr;
                jump->data.paren.end = match_end;
                rx_ptr = jump + 1;
            }
            break;

       /* Before attempting next alternative after a failed alternative,
        * need to reset paren */
          case ALT_FAIL:
            match_end = (rx_ptr->fail.ptr)->data.paren.start;
            ++rx_ptr;
            break;

          case HAVE_ALT_REV:
            {   RxAtom *jump = rx_ptr->fail.ptr;
                jump->data.paren.start = match_start;
                rx_ptr = jump + 1;
            }
            break;

          case ALT_FAIL_REV:
            match_start = (rx_ptr->fail.ptr)->data.paren.end;
            ++rx_ptr;
            break;

       /* Initialize the top-level alternatives structure */
          case TOP_ALT_INIT:
            {   SubRX *sub_rxs  = rx_ptr->data.top_alt.sub_rxs, *curr_alt;
                TopAlt *top_alt = &rx_ptr->data.top_alt;
                int len = abs(top_alt->len);

             /* If buffer is new, find the anchors for all the alternatives
              * otherwise, just find the next instance of the anchor used
              * in the last match */
                if(rx_ptr->min != Buf_no)
                {   int alt_i;
                    SubRX *srx = sub_rxs;
                    top_alt->i    = 0;
                    top_alt->len  = len;
                    for(alt_i = 0; alt_i < len; ++alt_i, ++srx)
                        srx->anchor = srx->start = srx->end = search_start;

                    rx_ptr->min = Buf_no;
                }
             /* If top_alt->len negative, find the next instance of the
              * alternative that was used in the previous match */
                else
                    top_alt->len = -len;

                curr_alt        = &sub_rxs[ top_alt->i ];
                anchor          = curr_alt->anchor;
                rx_ptr          = curr_alt->code.ptr;
            }
            break;

       /* Handle the non-existence of a top-level alternative */
          case TOP_ALT_FAIL:
            {   TopAlt *top_alt = &rx_ptr[-1].data.top_alt;
                SubRX *sub_rx = &top_alt->sub_rxs[top_alt->i], *near_sub_rx;

                sub_rx->start = Buf_end + 2;

             /* If all the alternatives have not been searched for, continue
              * searching */
                while(++top_alt->i < top_alt->len)
                {   ++sub_rx;
                    if(sub_rx->start <= Buf_end)
                    {   anchor        = sub_rx->anchor;
                        rx_ptr        = sub_rx->code.ptr;
                        goto next_op;
                    }
                }
                top_alt->len = -abs(top_alt->len);

             /* If all the alternatives have been searched for, use the
              * leftmost/longest alternative if present */
                if(!(near_sub_rx = nearest_sub_rx(rx_ptr - 1)))
                    rx_ptr = rx_ptr->fail.ptr;
                else
                {   match_start        = near_sub_rx->start;
                    match_end          = near_sub_rx->end;
                    rx_ptr             = rx_ptr[1].fail.ptr;
                }
            }
            break;

       /* Handle match of top level alternative found */
          case TOP_ALT_MATCH:
            {   TopAlt *top_alt = &rx_ptr[-2].data.top_alt;
                SubRX *sub_rx   = &top_alt->sub_rxs[top_alt->i], *near_sub_rx;

                sub_rx->start  = match_start;
                sub_rx->end    = match_end;
                sub_rx->anchor = anchor;
             /* Store subparen locations in buffer */
                if(sub_rx->nsub_paren)
                {   int i     = sub_rx->sub_paren_i;
                    int end_i = sub_rx->sub_paren_i + sub_rx->nsub_paren;
                    char * *par_buf_ptr = rx_ptr->data.sub_parens.buf;
                    par_buf_ptr += 2 * (i - rx_ptr->data.sub_parens.start_i);

                    for(; i < end_i; ++i, par_buf_ptr += 2)
                    {   RxAtom *src = rx->par_locs[i];
                        par_buf_ptr[0] = (char *)src->data.paren.start;
                        par_buf_ptr[1] = (char *)src->data.paren.end;
                    }
                }
             /* Search for alternatives which have not been searched for */
                while(++top_alt->i < top_alt->len)
                {   ++sub_rx;
                    if(sub_rx->start <= Buf_end)
                    {   anchor        = sub_rx->anchor;
                        rx_ptr        = sub_rx->code.ptr;
                        goto next_op;
                    }
                }
             /* Use leftmost/longest of the found alternatives */
                top_alt->len       = -abs(top_alt->len);
                near_sub_rx        = nearest_sub_rx(rx_ptr - 2);
                match_start        = near_sub_rx->start;
                match_end          = near_sub_rx->end;
                rx_ptr             = rx_ptr->fail.ptr;
            }
            break;

       /* If regular expression fails after top-level alternative found,
        * try to find next instance of used alternative */
          case TOP_ALT_NEXT:
            {   TopAlt *top_alt = &rx_ptr[-3].data.top_alt;
                SubRX *sub_rx   = &top_alt->sub_rxs[ top_alt->i ];
                anchor          = sub_rx->anchor;
                rx_ptr          = sub_rx->code.ptr;
            }
            break;

          default: fatal_error("Invalid regular expression opcode");
        }
    }
}

/* Return end of found string if the string starting at str_start and ending
 * at str_end is found at *loc, false otherwise */
static char *have_str(const char *str, const char *buf, int len)
{
    int str_i = 0;
    while(str_i < len)
    {   if(low_casE(buf[str_i]) != low_casE(str[str_i]))
            return NULL;
        ++str_i;
    }
    return (char *)buf + len;
}

/* Load the data corresponding to a parentheses from the buffer contained
 *  in the atom buf_atom. Return the pointer at index i in the current
 *  buffer  */
static const char *paren_buf_read(RxAtom *buf_atom, int i)
{
    char const * * *srcs  = buf_atom->data.paren_buf.locs;
    char const * *buf_ptr = (  buf_atom->data.paren_buf.start
                             + buf_atom->data.paren_buf.i);
    int buf_i;
    for(buf_i = 0; buf_i < buf_atom->min; ++buf_i)
        *srcs[buf_i] = buf_ptr[buf_i];

    return buf_ptr[i];
}

/* Write the data corresponding to a parentheses into the buffer contained
 * in the atom buf_atom. Return the start of the first atom written */
static char const * *paren_buf_write(RxAtom *buf_atom)
{
    char const * * *srcs  = buf_atom->data.paren_buf.locs;
    char const * *buf_ptr = (buf_atom->data.paren_buf.start
                             + buf_atom->data.paren_buf.i);
    int i;

    for(i = 0; i < buf_atom->min; ++i)
        buf_ptr[i] = *srcs[i];

    return buf_ptr;
}


/* Copy the contents of the buffer containing the locations the rightmost
 *  parentheses to the */
static void restore_rparens(RxAtom *rmost_buf)
{
    RxAtom *buf_atom      = rmost_buf[-1].fail.ptr - 1;
    char const * * *srcs  = buf_atom->data.paren_buf.locs;
    int i;

    for(i = 0; i < rmost_buf->min; ++i)
        *srcs[i] = rmost_buf->data.paren.rmost[i];
}



static SubRX *nearest_sub_rx(RxAtom *top_alt_atom)
{
    TopAlt *top_alt = &top_alt_atom->data.top_alt;
    SubRX *sub_rx   = top_alt->sub_rxs, *nearest = NULL;
    int i;
    const char *start = Buf_end + 1, *end = Buf_end + 1;
    SubParens *sub_parens = &top_alt_atom[2].data.sub_parens;

 /* Find alternative closest to search start */
    for(i = 0; i < abs(top_alt->len); ++i, ++sub_rx)
    {   if(   sub_rx->start < start
           || (sub_rx->start == start && sub_rx->end > end))
        {   nearest    = sub_rx;
            start      = sub_rx->start;
            end        = sub_rx->end;
            top_alt->i = i;
        }
    }
    if(nearest && sub_parens->buf)
    {   int clear_i, restore_i, end_i;
        char * *paren_buf_ptr;
        RxAtom * *par_locs = sub_parens->par_locs;

     /* Clear all the subparens in the alternation */
        for(clear_i = 0; clear_i < top_alt_atom[2].max; ++clear_i)
        {   RxAtom *paren_atom = par_locs[clear_i];
            paren_atom->data.paren.start = paren_atom->data.paren.end = NULL;
        }

     /* Now restore the subparens for the used alternative */
        restore_i     = nearest->sub_paren_i - sub_parens->start_i;
        end_i         = restore_i + nearest->nsub_paren;
        paren_buf_ptr = sub_parens->buf + (restore_i * 2);

        for(; restore_i < end_i; ++restore_i)
        {   RxAtom *paren_atom           = par_locs[restore_i];
            paren_atom->data.paren.start = paren_buf_ptr[0];
            paren_atom->data.paren.end   = paren_buf_ptr[1];
            paren_buf_ptr += 2;
        }
    }
    return nearest;
}

char *RegExp_paren(RegExp *rx, int paren_no, int *paren_len)
{
    if(paren_no > rx->nparen || paren_no <= 0)
    {   *paren_len = -1;
        return NULL;
    }
    else
    {   RxAtom *paren_atom = rx->par_locs[paren_no];
        char *paren_start  = (char *)paren_atom->data.paren.start;
        *paren_len = paren_atom->data.paren.end - paren_start;
        return paren_start;
    }
}

int RegExp_nparens(RegExp *rx) { return rx->nparen; }

#ifdef RX_TRACE
enum { MAX_CONTEXT = 30 };
static void print_trace(int rx_i, const char *op_name,
                        const char *match_start, const char *match_end)
{
    printf("%d %s  ", rx_i, op_name);
    if(match_start != match_end)
    {   int nscanl = miN(MAX_CONTEXT, match_start - Buf_start);
        int nscanr = miN(MAX_CONTEXT, Buf_end - match_end);

        const char *line_start = memrchr(match_start - nscanl, '\n', nscanl);
        const char *line_end = memchr(match_end, '\n', nscanr);
        line_start = (!line_start) ? match_start - nscanl : line_start + 1;
        if(!line_end)
            line_end = match_start + nscanr;

        fwrite(line_start, match_start - line_start, 1, stdout);
        fputs("<|", stdout);
        fwrite(match_start, match_end - match_start, 1, stdout);
        fputs("|>", stdout);
        fwrite(match_end, line_end - match_end, 1, stdout);
    }
    putchar('\n');
}
#endif
#ifdef PRINT_PARS
static void print_pars(RegExp *rx, const char *buf_start, const char *buf_end)
{
    int par_i, nprint = 0;
    for(par_i = 1; par_i <= rx->nparen; ++par_i)
    {   RxAtom *paren_atom = rx->par_locs[par_i];
        const char *start = paren_atom->data.paren.start;
        const char *end   = paren_atom->data.paren.end;
        if(start >= buf_start && end < buf_end)
        {   printf("(%d):%p:%d:", par_i, start, end - start);
            if(start <= end)
                fwrite(start, 1, miN(24, end - start), stdout);
            putchar(' ');
            ++nprint;
        }
    }
    if(nprint)
        putchar('\n');
}
#endif

