/* resizer.c - adjustable-size buffers
 * Copyright (C) 1995-99 Andrew Pipkin (minitrue@pagesz.net)
 * MiniTrue is free software released with no warranty. See COPYING for details
 */

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include "minitrue.h"
#include "resizer.h"

/* Initialize the resizeable buffer, set the length of the item and
 * start by allocating enough memory for start_size items */
void Resizer_init(Resizer *resizer, size_t item_len, size_t chunk_len)
{
    resizer->item_len  = item_len;
    resizer->free_i    = resizer->nalloc = 0;
    resizer->chunk_len = chunk_len * item_len;
    resizer->start     = NULL;
}

/* Return a pointer to the the start of the item at the specified index */
void far *Resizer_ptr(Resizer *resizer, size_t index)
{
 /* Make sure that memory has been allocated for the item at the index */
    if(index * resizer->item_len == resizer->nalloc)
    {   resizer->nalloc += resizer->chunk_len;
        resizer->start   = x_farrealloc(resizer->start, resizer->nalloc);
    }
    return (char far *)resizer->start + (index * resizer->item_len);
}

/* Copy nunits at src to the buffer starting at the byte at dest_i, allocate
 *  more memory if necessary and set the end of the buffer to the end of the
 *  copy */
static size_t copy(Resizer *resizer, size_t dest_i,
                   void far *src, size_t nunits)
{
    size_t end_i = dest_i + nunits * resizer->item_len;
    if(end_i > resizer->nalloc)
    {   resizer->nalloc += resizer->chunk_len;
        resizer->start   = x_farrealloc(resizer->start, resizer->nalloc);
    }
    resizer->free_i = end_i;
    _fmemcpy((char far *)resizer->start + dest_i, src, end_i - dest_i);
    return dest_i / resizer->item_len;
}

/* Write nunits starting at src to the buffer starting at write_i. */
size_t Resizer_write(Resizer *resizer, void far *src,
                  size_t nunits, size_t write_i)
{
    return copy(resizer, write_i * resizer->item_len, src, nunits);
}

/* Append nunits starting at src to the buffer */
size_t Resizer_append(Resizer *resizer, void far *src, size_t nunits)
{
    return copy(resizer, resizer->free_i, src, nunits);
}

/* Return the number of items in the resizeable buffer */
size_t Resizer_size(Resizer *resizer)
{
    return resizer->free_i / resizer->item_len;
}

/* Make the next item appended overwrite the truncate_i th item */
void Resizer_truncate(Resizer *resizer, size_t truncate_i)
{
    resizer->free_i = truncate_i * resizer->item_len;
}

/* Free unsused memory and returning start of array */
void far *Resizer_trim(Resizer *resizer)
{
    if (resizer->free_i < resizer->nalloc)
    {   resizer->start  = x_farrealloc(resizer->start, resizer->free_i);
        resizer->nalloc = resizer->free_i;
    }
    return resizer->start;
}

/* Free the resizeable buffer */
void Resizer_kill(Resizer *resizer) { farfree(resizer->start); }

#ifdef RESIZER_TEST
int main(void)
{
    int i;
    Resizer int_buf;
    Resizer_init(&int_buf, sizeof(int), 16);
    for(i = 0; i < 32; ++i)
    {   Resizer_append(&int_buf, &i, 1);
        printf("%p:%i\n", Resizer_item(&int_buf, i), i);
    }
    Resizer_truncate(&int_buf, 16);
    for(i = 16; i < 32; ++i)
    {   Resizer_append(&int_buf, &i, 1);
        printf("%p:%i\n", Resizer_item(&int_buf, i), i);
    }
    Resizer_kill(&int_buf);
    return 0;
}
#endif /* RESIZER_TEST */
