/* misc.c - Miscellaneous routines
 * Copyright (C) 1995-99 Andrew Pipkin (minitrue@pagesz.net)
 * MiniTrue is free software released with no warranty. See COPYING for details
 */

#include <stdio.h>
#include <stdlib.h>
#include <ctype.h>
#include <string.h>
#include <limits.h>
#include <time.h>
#include <errno.h>

#include "minitrue.h"
#include "casetran.h" /* Case translation tables */
#include "console.h"  /* need these for error messages */
#include "statline.h"

/* If case sensitivity desired, set all values in translation tables to
 * translate to the original value */
void set_case_insense(void)
{
    int ch_i;
    for(ch_i = 0; ch_i < NCHAR; ++ch_i)
    {   Lower_table[ch_i] = tolower(ch_i);
        Upper_table[ch_i] = toupper(ch_i);
    }
}

/* This function converts a C backslash escape sequence starting at src
 *   stores the converted value in *dest, and returns the end of the
 *   escape sequence - src is assumed to point to '\' */
char *esc_to_ch(char *dest, const char *src)
{
    const char *start = src;
    int val = -1, ch = *src++, chars_read = 1;

    if(ch != '\\') {
        *dest = ch;
        return (char *)src;
    }
 /* if backslash followed by nonalphanumeric character, treat as literal */
    if(ispunct((ch = *src)))
        val = ch;

 /* \x precedes a 2-digit hexadecimal escape sequence */
    else if (ch == 'x')
        sscanf(src, "x%2x%n", &val, &chars_read);

 /* Use array to map lower-case letters to the control characters they
  *   represent if followd by a backslash */
    else if (islower(ch)) {
        char esc_map[] = {'\a','\b',0,0,  '\033','\f',0,0,  0,0,0,0,
                           0,'\n',0,0,    0,'\r',0,'\t',    0,0,0,0, 0,0};

        val = esc_map[ch - 'a'] ? esc_map[ch - 'a'] : -1;
    }
 /* If nothing has matched, try to read octal number */
    else
        sscanf(src, "%3o%n", &val, &chars_read);

    if(val < 0 || val >= NCHAR)
    {   char err_msg[128];
        sprintf(err_msg, (val < 0
                          ? "%.2s is not a valid escape sequence"
                          : "%.4s is an octal number greater than 255"),
                start);
        input_error(err_msg);
    }
    *dest = (char)val;
    return (char *)src + chars_read;
}

/* ch_to_esc - Convert character to backslash escape sequence if it is not
 *   not a ASCII printing character. Write results to dest which must have
 *   have at least 6 free characters */
char *ch_to_esc(char *dest, int ch)
{
    if(isprint(ch))
        *dest++ = ch;
    /* If not printing character, write escape sequence - escape sequence
     *   will be 2-digit hex value unless char is a control character */
    else {
        static char Ctrl_esc[32] = {
            '0',0,0,0,0,0,0,'a', 'b','t','n','v','f','r',0,0,
            0,0,0,0,0,0,0,0,     0,0,0,'e',0,0,0,0
        };
        *dest++ = '\\';
        if(ch < 32 && Ctrl_esc[ch])
            *dest++ = Ctrl_esc[ch];
        else
            dest += sprintf(dest, "x%02x", ch);
    }
    *dest = '\0';
    return dest;
}

/* Compare n chars of str1 and str2 with case insensitivity if case
 * insensitivity desired, return TRUE if match, FALSE if not */
int strn_cmp_ci(const char *str1, const char *str2, int n)
{
    int str_i;
    for(str_i = 0; str_i < n; ++str_i, ++str1, ++str2)
    {   if(low_casE(*str1) != low_casE(*str2))
            return FALSE;
    }
    return TRUE;
}

/* Parse an integer, store the integer value in int_ptr, return a pointer
 * to the end of the number */
char *read_int(const char *src, int *int_ptr)
{
    if(!isdigit(*src))
        return NULL;

    *int_ptr = atoi(src);
    while(isdigit(*src))
        ++src;
    return (char *)src;
}

/* return first non-whitespace character after src */
char *skip_ws(const char *ptr) {
    while(isspace(*ptr))
        ++ptr;
    return (char *)ptr;
}

/* return first non-numeric character after ptr */
char *skip_num(const char *ptr) {
    while(isdigit(*ptr))
        ++ptr;
    return (char *)ptr;
}

/* return true if the nul-terminated string has a character for which
 *  char_fn returns true, false otherwise */
int have_ctype(const char far *str, int (*char_fn)(int ch))
{
    while(*str && !char_fn(*str))
        ++str;
    return *str;
}

/* preprocess a string - convert all backslash escape sequences and set
 *  characters to lower case if case_conv set and case insensitivity desired
 *  return the length of the string */
size_t str_preproc(char *src, int conv_case)
{
    char *dest = (char *)src, *orig_dest = dest;
    while(*src)
    {/* As in C, a backslash must precede an escape charater */
        if(*src == '\\')
            src = (char *)esc_to_ch(dest, src);
        else
        {   *dest = (conv_case) ? (char)low_casE(*src) : *src;
            ++src;
        }
        ++dest;
    }
    *dest = '\0';
    return (dest - orig_dest);
}

/* Read in entire file, allocating memory as needed, set len to
 length of file */
char *falloc(FILE *file, size_t *len)
{
    size_t  nalloc = 2048, nread = 0;
    char *buf = NULL;
    do
    {   if((nalloc * 2) <= 0 || (buf = realloc(buf, nalloc *= 2)) == NULL)
        {   farfree(buf);
            return NULL;
        }
        nread += fread(&buf[nread], 1, nalloc - nread, file);
    }while(nread == nalloc);

 /* Trim allocated memory to file size, setting \0 at end of buffer */
    *len = nread;
    if((buf = realloc(buf, nread + 1)) != NULL)
        buf[nread] = '\0';

    return buf;
}

/* Do a deep copy of an array of nstrs strings, put NULL at the
 *  end of arrays */
char * *copy_str_vect(char *str_vect[], int nstrs)
{
    int str_i;
    char * *copy = x_malloc(sizeof(char *) * (nstrs + 1));
    for(str_i = 0; str_i < nstrs; ++str_i)
        copy[str_i] = x_strdup(str_vect[str_i]);

    copy[str_i] = NULL;
    return copy;
}

/* Free a string vector consisting of nstrs */
void kill_str_vect(char *str_vect[], int nstrs)
{
    int str_i;
    for(str_i = 0; str_i < nstrs; ++str_i)
        free(str_vect[str_i]);
    free(str_vect);
}

/* if the sum of positive integers a1 and a2 overflows, return INT_MAX,
 * otherwise return the sum */
int add_truncate(int a1, int a2)
{
    int sum = a1 + a2;
    return ( sum >= 0 ) ? sum : INT_MAX;
}

/* if the product of m1 and m2 does not overflow, return the product,
 *  otherwise return INT_MAX */
int mult_truncate(int m1, int m2)
{
    int prod = m1 * m2;
    if(!prod)
        return 0;
    else
        return (prod / m1) == m2 ? prod : INT_MAX;
}

/* Return a value proportional to the possiblitity of character ch appearing
 * in a text file. The values have been normalized so that the most common
 * character (space) returns a value of 255. All characters return at least
 * one */
int char_freq(int ch)
{
    static unsigned char Char_freqs[] = {
        10, 1, 1, 1, 1, 1, 1, 1, 1, 4, 26, 1, 1, 13, 1, 1,
        1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1,
        255, 1, 7, 1, 1, 1, 1, 5, 1, 1, 1, 1, 14, 5, 19, 1,
        3, 2, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1,
        1, 3, 1, 1, 1, 1, 1, 1, 2, 7, 1, 1, 1, 2, 1, 1,
        1, 1, 1, 4, 3, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 2,
        1, 77, 16, 22, 44, 124, 19, 24, 60, 59, 1, 10, 42, 24, 65, 72,
        16, 1, 56, 62, 83, 30, 7, 22, 1, 19, 1, 1, 1, 1, 1, 1,
        1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1,
        1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1,
        1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1,
        1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1,
        1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1,
        1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1,
        1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1,
        1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 4,
    };
    return Char_freqs[(unsigned char)ch];
}

/* return the # of hundredths of seconds between the end time and the start
 * time */
long hsec_diff(clock_t start, clock_t end)
{
/* determine the # of clock ticks per ten seconds, need to avoid floating
 * point multiplication to avoid linking in emulator in DOS 16 bit version*/
#ifdef __DOS16__
    return (end - start) * 1000L / 182; /* 18.2 DOS CLOCKS PER SEC */
#else
    return (long)(end - start) * 100 / CLOCKS_PER_SEC;
#endif
}

/* This function is equivalent to strcpy, but it returns the end of the
 * destination string instead of the start */
char *copy_str(char *dest, const char *src)
{
    while((*dest = *src) != '\0')
    {   ++dest;
        ++src;
    }
    return dest;
}

/* Copy two strings, together, first allocating memory for the combined
 * string */
char *append_strs(const char far *str1, const char far *str2)
{
    size_t len1 = _fstrlen(str1), len2 = _fstrlen(str2) + 1;
    char *dest = x_malloc(len1 + len2); /* include trailing NUL */
    _fmemcpy(dest, str1, len1);
    _fmemcpy(&dest[len1], str2, len2);
    return dest;
}

/* Copy three strings together, first allocating memory for the combined
 * string */
char *append_3strs(const char far *str1, const char far *str2,
                   const char far *str3)
{
    char *part1 = append_strs(str1, str2), *str = append_strs(part1, str3);
    free(part1);
    return str;
}

/* Allocate & create a strings containg n instances of str with length n */
char *str_mult(const char *str, size_t len, size_t n)
{
    size_t str_i;
    char *mem = x_malloc(len * n + 1), *dest = mem;
    for(str_i = 0; str_i < n; ++str_i)
    {   memcpy(dest, str, len);
        dest += len;
    }
    *dest = '\0';
    return mem;
}

/* memrchr - find the last instance of character c in the n bytes beginning
 * with start, return NULL if character not found */
void *memrchr(const void *start, int c, size_t n)
{
    char *ptr = (char *)start + n;

 /* Get n to a multiple of 4 before the loop is unrolled */
    while(n & 3)
    {   if(*(--ptr) == c)
            return ptr;
        --n;
    }
    while(n)
    {   if(ptr[-1] == c)
            return ptr - 1;
        else if(ptr[-2] == c)
            return ptr - 2;
        else if(ptr[-3] == c)
            return ptr - 3;
        else if(ptr[-4] == c)
            return ptr - 4;

        ptr -= 4;
        n   -= 4;
    }
    return NULL;
}

/* Return the number of \n found in the len bytes beginning at start */
size_t count_nl(const char far *start, size_t len)
{
    size_t nl_num = 0;
    const charf *end = start + len;
    while((start = _fmemchr(start, NL, end - start)) != NULL)
    {   ++nl_num;
        ++start; /* skip over \n */
    }
    return nl_num;
}

/* This function parses a pathname, it returns the start of the filename,
 *   and sets *ext to the dot preceding an extension if present, or NULL
 *   if no extension present and sets fname_len to the filename length */
char *parse_path(char *pathname, char * *ext_pptr, int *path_len)
{
    char *fname = pathname + (*path_len = strlen(pathname)) - 1, *ext = fname;
 /* Filename will start after last path separator in path if present */
    while(fname > pathname && !path_sep_cH(*fname))
        --fname;
 /* Do not include path separator in filename */
    if(path_sep_cH(*fname))
        ++fname;

 /* Extension will start at last period in filename, ignore period if
  * it is first character in filename */
    while(ext > fname + 1 && *ext != '.')
        --ext;

    *ext_pptr   = (*ext == '.') ? ext : NULL;

    return fname;
}
static const char far out_of_mem[] = "Out of Memory error";

/* If these memory allocation functions fail, cleanup is performed and the
 *   program exits */
void *x_malloc(size_t size)
{
    void *new_mem = malloc(size);
    if(!new_mem && size)
        fatal_error(out_of_mem);

    return new_mem;
}

void *x_realloc(void *p, size_t size)
{
    void *new_mem = realloc(p, size);
    if(!new_mem && size)
        fatal_error(out_of_mem);

    return new_mem;
}

void *x_calloc(size_t nobj, size_t size)
{
    void *new_mem = calloc(nobj, size);
    if(!new_mem && size)
        fatal_error(out_of_mem);

    return new_mem;
}

void far *x_farmalloc(size_t size)
{
    void far *new_mem = farmalloc(size);
    if(!new_mem && size)
        fatal_error(out_of_mem);

    return new_mem;
}

void far *x_farrealloc(void far *p, size_t size)
{
    void far *new_mem = farrealloc(p, size);
    if(!new_mem && size)
        fatal_error(out_of_mem);

    return new_mem;
}

char *x_strdup(const char *str)
{
    size_t len = strlen(str) + 1;
    char *copy = x_malloc(len);
    return memcpy(copy, str, len);
}

char far *x_fstrdup(const char far *str)
{
    size_t len     = _fstrlen(str) + 1;
    char far *dest = x_farmalloc(len);
    return _fmemcpy(dest, str, len);
}

static int Initializing     = TRUE;
static int Have_error       = FALSE;
static int Input_error      = FALSE;
static int Console_err_msgs = FALSE;
static const char far *Error_string; /* String currently being parsed */

/* When error encountered, reset the display before printing mtr: to start
 * the error message */
void error_msg(const char far *error)
{
    Have_error = TRUE;
 /* write error messages to standard error messages if Console_err_msgs
  * not set */
    if(!Console_err_msgs)
        fprintf(stderr, EXE_NAME ": %" FP_FORM"\r\n", error);
#ifndef TEST_PROG
    else
        StatLine_Error(error);
#endif
}

/* kill the program, first resetting the console and writing an error message
 *  to standard error */
void fatal_error(const char far *error)
{
#ifndef TEST_PROG
    Console_Restore(TRUE);
#endif
    Console_err_msgs = FALSE;
    error_msg(error);
    exit(2);
}

/* End the initialization phase of the program, if console_used is set,
 *  write subsequent non-fatal errors to the status line return true if errors
 *  have occured during the initialization phase */
int end_init(int console_used)
{
    Initializing     = FALSE;
    Console_err_msgs = console_used;
    Error_string     = NULL;
    return Have_error;
}

void invalid_param(int option_ch, int param_ch)
{
    char msg_buf[256];
    int msg_i = sprintf(msg_buf, (param_ch
                                  ? "%c is not a valid paramter"
                                  : "Invalid parameter"),
                        param_ch);
    sprintf(&msg_buf[msg_i],
            " for option -%c. Enter \"" EXE_NAME " -?%c\" for help.",
            option_ch, option_ch);
    error_msg(msg_buf);
}

void set_input_str(const char far *err_str) {Error_string = err_str;}

void input_error(const char far *error)
{
    Input_error = TRUE;
    if(Error_string)
    {   char *msg   = append_3strs(error, " in ", Error_string);
        error_msg(msg);
        free(msg);
    }
    else
        error_msg(error);
}

int have_input_error(void)
{
    int orig_error = Input_error;
    Input_error    = FALSE;
    return orig_error;
}

#ifndef __MSDOS__
#include <unistd.h>
#include <sys/time.h>
#include <sys/resource.h>
long maj_faults(void)
{
    struct rusage ru;
    getrusage(RUSAGE_SELF, &ru);
    return ru.ru_majflt;
}
#endif
