/* lines.c - generate a screenful of lines corresponding to a buffer location
 * Copyright (C) 1995-99 Andrew Pipkin (minitrue@pagesz.net)
 * MiniTrue is free software released with no warranty. See COPYING for details
 */

#include <stdio.h>
#include <stdlib.h>
#include <ctype.h>
#include <string.h>

#include "lines.h"
#include "minitrue.h"
#include "buffers.h"
#include "charset.h"
#include "console.h"
#include "winbuf.h"
#include "hilites.h"
#include "attribs.h"

enum {HEX_LINE_LEN = 16, MAX_LINE_LEN = 4096, MAX_SCROLL = 4};
enum {EOF_CH = '~', WRAP_CH = '\\', TRUNC_CH = '$' };

typedef struct
{   off_t  off;       /* # of bytes between start of file and start of line */
    int    len;       /* number of bytes between start & end offset of line */
    int    nl_num;    /* # of \n in line, at most 1 unless in hex mode */
    int    cols;      /* number of columns occupied by the line */
    int    type;      /* type of text in line, argument for Attrib function */
}line;

static far line *Lines;
static int Nlines;         /* total # of lines both on & off-screen */

static int Rows, Cols;     /* number of rows and columns visible on screen */
static int Height;         /* height of lines window */
static int Unused_lines;   /* number of unused lines at bottom of screen
                            * used by status line */
static int Top_i;          /* index of top line in buffer */
static int Top_vis_i;      /* index of top line appearing on screen */
static int Bottom_i;       /* index of bottom line on screen */
static int Cursor_i;       /* index of line with cursor */
static off_t Line_num;     /* line number of line with cursor */

enum fold { WORD_WRAP = 1, TRUNCATE };
static int Fold_mode;      /* tells how lines longer than screen width will
                            * be handled */
static int Tab_len = 8;    /* number of spaces in tab */
static int Default_row;    /* default line # for match */
static int Jump_nlines = -1;/* If previous match less than Jump_nlines from
                             *  current match and current match found on
                             *  previous screen, do not jump screen start */
static int Hex_mode;       /* boolean variable for hex screen */
static int Show_hls = TRUE;/* set if matches are to be highlighted */

static CharSet Print_set;
static Window *Win_ptr;

static void make_screen(BufPtr buf_ptr, off_t line_num, int lines_above);
static void text_line(int line_i, BufPtr buf_pptr, off_t stop_off, int type);
static void hex_line(int line_i, BufPtr buf_pptr, off_t stop_off);
static char convert_ctrl_char(unsigned char ch);
static BufPtr find_discont(BufPtr buf_ptr, charf * *ch_pptr,
                           attr_t *attr, off_t stop_off);

static int top_append(void);
static void reverse_wraps(int wrap_start, int wrap_last);
static int bottom_append(int type);
static int bottom_type(void);

static void screen_resize(void);
static void line_set(int line_i, off_t off, int type);
static void indicate_eof(off_t eof_off);
static int line_below(int row_i, int pos_change);
static int line_above(int row_i, int pos_change);
static int row_num(int row_i);
static int lines_between(int upper_i, int lower_i);
static int nls_between(int line_i);
static int on_screen(int row_i);
static int bottom_vis_line(void);

static char far set_err[] = "Cannot have any text following set for -p option";

void Lines_Init(const char *fold_mode, const char *hex_mode,
                const char *lines_above, const char *print_set,
                const char *tab_len)
{
    Win_ptr = WinBuf_Init(Console_Rows() + 64);
    screen_resize();
    if(fold_mode != NULL)
    {   if(!fold_mode[0])
            Fold_mode = WORD_WRAP;
        else if(fold_mode[0] == '+')
            Fold_mode = TRUNCATE;
    }
    if(tab_len)
    {   if(!isdigit(*tab_len))
           error_msg("-t must be followed by tab size");
        else
        {   Tab_len = atoi(tab_len);
            if(Tab_len > 40)
                Tab_len = 40;
        }
    }
    Hex_mode    = hex_mode != NULL;
 /* -m parameter indicates how many lines to display above match, if preceded
  * by +, indicates how many lines below match */
    Default_row = VCENTER;
    if(lines_above)
    {   int have_plus = (*lines_above == '+');
        if(have_plus)
            ++lines_above;
        if(isdigit(*lines_above))
        {   Default_row = atoi(lines_above);
            if(have_plus)
                Default_row = -Default_row;
            lines_above = skip_num(lines_above);
        }
        if(*lines_above == ':')
        {   if(isdigit(*(++lines_above)))
            {   Jump_nlines = atoi(lines_above);
                lines_above = skip_num(lines_above);
            }
        }
        if(*lines_above != '\0')
            invalid_param('m', *lines_above);
    }
    if(!print_set || !*print_set)
        print_set = ":[\x20-\\x7e\\xa0-\\xff]";
    print_set = CharSet_init(&Print_set, print_set + 1);
    if(*print_set != '\0')
        error_msg(set_err);
}

static void screen_resize(void)
{
    Console_Resize();
    WinBuf_Resize_all();
    Rows   = Console_Rows();
    Cols   = Console_Cols();
    Height = Rows - Unused_lines;
    Nlines = WinBuf_nrows(Win_ptr);
    Lines  = x_farrealloc(Lines, sizeof(line) * Nlines);
}

/* Generate the screen, then write it */
void Lines_Render(BufPtr buf_ptr, off_t line_num, int lines_above)
{
    make_screen(buf_ptr, line_num, lines_above);
    Lines_Write();
}

/* Generate the screen so that the text pointed to by buf_ptr
 *  has lines_above lines between its line and the top of the screen, use the
 *  default screen position if lines_above is -1, line_num is the line number
 *  corresponding to buf_ptr */
static void make_screen(BufPtr buf_ptr, off_t line_num, int lines_above)
{
    off_t offset, nl_found;
    int type = NORMAL;
    Line_num = line_num;
    Top_i    = Top_vis_i = -1;
    Bottom_i = 0;

    offset      = BufPtr_off(buf_ptr);
    lines_above = row_num(lines_above);

 /* First determine start of line containing text of line pointer
  * byte_diff will have length of line start */
    if(Hex_mode)
    {   buf_ptr   = BufPtr_find_nl(buf_ptr, offset & ~(HEX_LINE_LEN -1),
                                 -100, &nl_found);
        Line_num += nl_found;
    }
 /* If leading nl not found, put text in green */
    else
    {   buf_ptr = BufPtr_find_nl(buf_ptr, offset - MAX_LINE_LEN, -1,
                                 &nl_found);
        if(BufPtr_off(buf_ptr) && !nl_found)
            type = UNKNOWN_START;
    }
    line_set(Bottom_i, BufPtr_off(buf_ptr), type);
    while(   bottom_append(type)
          && Lines[Bottom_i].off + Lines[Bottom_i].len <= offset)
        type = bottom_type();
    Cursor_i = Bottom_i;
    Top_i    = 1;

 /* Write top part of screen */
    while(lines_between(Top_i, Cursor_i) < lines_above && top_append())
        ;

 /* Set top visible line so that desired # of lines are above line
  * with found string */
    Top_vis_i = Top_i;
    if(lines_between(Top_i, Cursor_i) > lines_above)
        Top_vis_i = line_above(Cursor_i, lines_above);

 /* To finish screen, keep writing lines until bottom line is off the screen*/
    while(on_screen(line_below(Bottom_i, 1)) && bottom_append(bottom_type()))
        ;
}

/* Set the number of lines which will be used for the status line, if rewrite
 * is set & the number of unused lines has decreased, write lines which were
 * formally occupied by the status line */
void Lines_Set_unused(int status_lines, int rewrite)
{
    Height = Rows - status_lines;
    if(Unused_lines > status_lines && rewrite)
    {   int row;
        for(row = Rows - Unused_lines; row < Rows - status_lines; ++row)
        {   int line_i = (lines_between(Top_vis_i, Bottom_i) > row
                          ? line_below(Top_vis_i, row)
                          : Bottom_i);
            if(on_screen(Bottom_i))
                bottom_append(bottom_type());
            Console_Writeln(WinBuf_row(Win_ptr, line_i), row,
                            Lines[line_i].cols + 1);
        }
    }
    Unused_lines = status_lines;
}

int Lines_Height(void) { return Height; }

/* Indicate the end of the file by making the bottom line black */
static void indicate_eof(off_t eof_off)
{
    line_set(Bottom_i, eof_off, EOFILE);
    LineBuf_clear(WinBuf_row(Win_ptr, Bottom_i), 0, Attrib(EOFILE));
    if(!Console_Is_color())
        LineBuf_putc(WinBuf_row(Win_ptr, Bottom_i), 0, EOF_CH, Attrib(SYMB));

    Lines[Bottom_i].cols = 1;
}

/* Write the lines to the screen */
void Lines_Write(void)
{
    int line_i = Top_vis_i, row_num = 0;
    int last_row = row_num + Height;

    for(; row_num < last_row; ++row_num)
    {   Console_Writeln(WinBuf_row(Win_ptr, line_i), row_num,
                        Lines[line_i].cols + 1);
        if(line_i != Bottom_i)
            line_i = line_below(line_i, 1);
    }
    Console_Move_cursor(0, Rows);
}

/* If the screen size has changed, regenerate the screen, just copy back
 * the screen if the size has remained constant since the last rewrite */
void Lines_Rewrite(void)
{
    Lines_Remake();
    Lines_Write();
}

/* Regenerate the screen with the top of the screen unchanged */
void Lines_Remake(void)
{
    screen_resize();
 /* if Bottom_i == Top_i, no screen has been written so do not rewrite */
    if(Bottom_i != Top_i)
    {   int orig_cursor_row = lines_between(Top_vis_i, Cursor_i);
        Lines_Set_cursor(0);
        Lines_Render(BufPtr_goto(Lines[Top_vis_i].off), Line_num, 0);
        Lines_Set_cursor(orig_cursor_row);
    }
}

/* If scroll_lines positive scroll down scroll_lines lines, if negative scroll
 * up. If stop_at_eof set, stop scrolling if scrolling down when end of file
 * becomes visibile, otherwise keep scrolling, filling the bottom of the
 * screen with visible lines */
int Lines_Scroll(int scroll_lines, int stop_at_eof)
{
    int line_num_change = 0, orig_top_vis = Top_vis_i, scrolled = 0;
 /* If scrolling upwards, write new lines at top of screen */
    if(scroll_lines < 0)
    {   for( ;scroll_lines < 0 && Lines[Top_vis_i].off != 0; ++scroll_lines)
        {   if(Top_i == Top_vis_i)
                top_append();

            if(Top_vis_i != Top_i)
            {   Top_vis_i        = line_above(Top_vis_i, 1);
                line_num_change += Lines_Move_cursor(-1);
            }
        }
        scrolled = -lines_between(Top_vis_i, orig_top_vis);
    }
 /* Do not scroll down if end of file already visible */
    if(scroll_lines > 0 && Lines[bottom_vis_line()].type != EOFILE)
    {   int orig_cursor_row = lines_between(Top_vis_i, Cursor_i);
        while(scroll_lines-- > 0 && Lines[bottom_vis_line()].type != EOFILE)
        {   Top_vis_i = line_below(Top_vis_i, 1);
            if(on_screen(Bottom_i))
                bottom_append(bottom_type());
            line_num_change += Lines_Move_cursor(1);
        }
     /* If end of file reached, alter Top_vis_i so bottom line of last
      *  screen is top line of new screen, put cursor on same row as
      *  it was originally if there are enough lines, but on bottom if not */
        if(scroll_lines > 0 && !stop_at_eof)
        {   int scrn_height, new_cursor_i;
            Top_vis_i        = line_below(Top_vis_i, scroll_lines + 1);
            scrn_height      = lines_between(Top_vis_i, Bottom_i);
            orig_cursor_row  = miN(scrn_height, orig_cursor_row);
            new_cursor_i     = line_below(Top_vis_i, orig_cursor_row);
            line_num_change += nls_between(new_cursor_i);
            Line_num        += nls_between(new_cursor_i);
            Cursor_i         = new_cursor_i;
        }
        scrolled = lines_between(orig_top_vis, Top_vis_i);
    }
 /* If number of lines scrolled less than MAX_SCROLL, and it is possible to
  * use escape sequences to scroll the console, write new lines at the top
  * or bottom of the screen */
    if(abs(scrolled) <= MAX_SCROLL && Console_Scroll(scrolled, 0, Height))
    {   int i, screen_row = (scrolled < 0) ? 0 : Height - scrolled;
        for(i = 0; i < abs(scrolled); ++i, ++screen_row)
        {   int line_i = line_below(Top_vis_i, screen_row);
            Console_Writeln(WinBuf_row(Win_ptr, line_i), screen_row,
                            Lines[line_i].cols + 1);
        }
    }
 /* Otherwise rewrite entire screen */
    else if(scrolled)
        Lines_Write();

    return line_num_change;
}
void Lines_Kill(void)
{
    if(Lines)
    {   farfree(Lines);
        CharSet_kill(&Print_set);
    }
}

/* append a line to the top of the screen */
int top_append(void)
{
    off_t stop_off = Lines[Top_i].off;
    int type       = NORMAL;
    BufPtr buf_ptr;
    int wrap_arounds = 0, top_i_orig = Top_i;
    int bottom_vis_orig = bottom_vis_line();

 /* Find start of previous line - go back 16 bytes if in hex mode otherwise
  *  try to find preceding newline, first make sure that not at file start */
    if(   !stop_off
       || !(buf_ptr = BufPtr_goto(stop_off - (Hex_mode ? HEX_LINE_LEN : 0))))
        return FALSE;

    if(!Hex_mode)
    {   off_t nl_found;

     /* If line immediately preceded by \n, need to move back one byte
      *    so that \n will not be counted */
        if(Lines[Top_i].type == NORMAL)
            buf_ptr = BufPtr_dec(buf_ptr);
        buf_ptr = BufPtr_find_nl(buf_ptr, stop_off - MAX_LINE_LEN,
                                 -1, &nl_found);
        if(!nl_found && BufPtr_off(buf_ptr))
            type = UNKNOWN_START;

     /* If start of top line has come in range when it was previously out of
      *   range, move top line down do first line with a known newline
      *   causing all of the wrapped around top lines to be rewritten */
        else if(Lines[Top_i].type == UNKNOWN_START)
        {   int orig_cursor_row = lines_between(Top_i, Cursor_i);
            Lines_Move_cursor(-orig_cursor_row);
            Lines_Render(BufPtr_goto(stop_off), Line_num, 0);
            Lines_Move_cursor(orig_cursor_row);
            return TRUE;
        }
    }
 /* Determine where line starts, then write line */
    for( ; ; )
    {   off_t end_off;
        Top_i = line_above(Top_i, 1);

     /* Do not overwrite lower visible lines, overwrite previous wraparounds*/
        if(Top_i == bottom_vis_orig || Top_i == top_i_orig)
            Top_i = line_above(top_i_orig, 1);
        else if (Top_i == Bottom_i)
            Bottom_i = line_above(Bottom_i, 1);

     /* write line, quit if end of line offset is start of next line */
        if(Hex_mode)
            hex_line(Top_i, buf_ptr, stop_off);
        else
            text_line(Top_i, buf_ptr, stop_off, type);

        if((end_off = Lines[Top_i].off + (off_t)Lines[Top_i].len) == stop_off)
            break;

        buf_ptr = BufPtr_goto(end_off);

     /* If line wraps around, make wrapped around text in gray */
        if(type != UNKNOWN_START)
            type = WRAPPED;

        ++wrap_arounds;
    }
 /* Wrapped around lines will be in reverse order, so reverse them */
    if(wrap_arounds)
    {   int end_wrap = line_above(top_i_orig, 1);
        reverse_wraps(Top_i, end_wrap);
        if(wrap_arounds > lines_between(bottom_vis_orig, end_wrap))
            reverse_wraps(line_below(bottom_vis_orig,1), line_above(Top_i,1));
    }
    return TRUE;
}

/* reverse wraps - reverses the order of the wrapped around line segments
 *   written to the top part of the screen */
static void reverse_wraps(int wrap_start, int wrap_last)
{
    char swap_buf[2 * 132];
    line line_temp;
    while(wrap_start != wrap_last && wrap_start != line_below(wrap_last, 1))
    {/* Copy top screen line & line info to swap buffer, copy bottom to top,
      *   then copy temp to bottom line */
        ScreenPtr top_wrap_ptr    = WinBuf_row(Win_ptr, wrap_start);
        ScreenPtr bottom_wrap_ptr = WinBuf_row(Win_ptr, wrap_last);
        _fmemcpy(swap_buf, top_wrap_ptr, 2 * Cols);
        _fmemcpy(top_wrap_ptr, bottom_wrap_ptr, 2 * Cols);
        _fmemcpy(bottom_wrap_ptr, swap_buf, 2 * Cols);

        line_temp           = Lines[wrap_start];
        Lines[wrap_start]   = Lines[wrap_last];
        Lines[wrap_last]    = line_temp;

        wrap_start  = line_below(wrap_start, 1);
        wrap_last   = line_above(wrap_last, 1);
    }
}

/* The new bottom line will have the normal color if the preceding line has
 *  an \n, otherwise the new bottom line will have the the WRAPPED attribute
 *  unless the line start is not known */
int bottom_type(void)
{
    if(Lines[Bottom_i].nl_num)                      return NORMAL;
    else if(Lines[Bottom_i].type == UNKNOWN_START)  return UNKNOWN_START;
    else                                            return WRAPPED;
}

/* This function adds a line to the bottom of the screen, it returns
 *   FALSE if at end of file, TRUE otherwise */
int bottom_append(int type)
{
    BufPtr buf_ptr;

 /* Line will be truncated when trunc_off occurs, trunc_off will be either
  *   2048 bytes from line start or at end of file */
    off_t start_off = Lines[Bottom_i].off + Lines[Bottom_i].len;
    off_t trunc_off = start_off + MAX_LINE_LEN;

 /* If attribute of current line indicates end of file, quit */
    if(Lines[Bottom_i].type == EOFILE)
        return FALSE;

 /* Shift the bottom line index and change top line if overwritten */
    if((Bottom_i = line_below(Bottom_i, 1)) == Top_i)
        Top_i = line_below(Top_i, 1);

    buf_ptr = BufPtr_goto(start_off);
    if(BufPtr_eof(buf_ptr))
    {   indicate_eof(start_off);
        return FALSE;
    }
    if(Hex_mode)
        hex_line(Bottom_i, buf_ptr, trunc_off);
    else
        text_line(Bottom_i, buf_ptr, trunc_off, type);

    return TRUE;
}

/* this function writes a line of text, starting at start_off with attribute
 *   attr. The line ends when the line is full or stop_off is reached */
static void text_line(int line_i, BufPtr buf_ptr, off_t stop_off, int type)
{
    charf *ch_ptr      = BufPtr_curr(buf_ptr), *discont_ptr = ch_ptr;
    ScreenPtr line_buf = WinBuf_row(Win_ptr, line_i);
    char ch, prev_ch = '\0';
    attr_t line_attr = Attrib(type), prev_attr = 0;
    int screen_col = 0, len = 0, end_col = Cols;
    int wrap_col = 0, wrap_len = 0, prev_bksp = FALSE;
 /* Leave space for \ indicating wrap at end of line if in monochrome mode */
    if(!Console_Is_color())
        --end_col;

    line_set(line_i, BufPtr_off(buf_ptr), type);

    for( ; ; )
    {   attr_t attr;
     /* See if discontinuity encountered, discontinuities defined as change
      * in text attributes or buffer boundaries */
        if(ch_ptr == discont_ptr)
        {   BufPtr_set(buf_ptr, ch_ptr);
            attr = line_attr;
         /* If at end of file (not at stop offset), make rest of line black */
            if(!(buf_ptr = find_discont(buf_ptr, &ch_ptr, &attr, stop_off)))
            {   if(Lines[line_i].off + len != stop_off)
                    line_attr = Attrib(EOFILE);

                break;
            }
            discont_ptr = BufPtr_curr(buf_ptr);
            continue;
        }
     /* If new line found, clear remainder of line and record presence of \n*/
        if((ch = *ch_ptr) == NL)
        {   Lines[line_i].nl_num = 1;
            ++len;
            break;
        }
      /* If char in set of printing characters, then just put on screen */
        else if(CharSet_iN(Print_set, ch))
        {   attr_t ch_attr = attr;
         /* If previous character backspace, see if that signifies emphasis
          * or underlining */
            if(prev_bksp && screen_col > 0)
            {   if(prev_attr == line_attr)
                {   if(prev_ch == ch)       ch_attr = Attrib(EMPH);
                    else if(prev_ch == '_') ch_attr = Attrib(UNDERLINED);
                }
                prev_bksp = FALSE;
                --screen_col;
            }
            if(screen_col == end_col)
                break;
            LineBuf_putc(line_buf, screen_col++, ch, ch_attr);
         /* If word wrapping desired & character not part of a word, keep
          * track of location in case a line break is needed */
            if(Fold_mode == WORD_WRAP && !CharSet_in_worD(ch))
            {   wrap_col = screen_col;
                wrap_len = len + 1;
            }
            prev_ch   = ch;
            prev_attr = attr;
        }
     /* If tab, print spaces until tab stop reached */
        else if(ch == '\t')
        {   int tab_spaces = Tab_len - (screen_col % Tab_len);
            for( ; tab_spaces && screen_col < end_col; --tab_spaces)
                LineBuf_putc(line_buf, screen_col++, ' ', attr);
        }
        else if(ch == '\b')
            prev_bksp = TRUE;
        ++ch_ptr;
        ++len;
    }
    Lines[line_i].len  = len;
    Lines[line_i].cols = screen_col;

 /* Assume end of screen line reached if line lacks \n and not at eof */
    if(!Lines[line_i].nl_num && buf_ptr)
    {/* If word wrapping desired, set end of line to after last non-word char
      * and clear line after that, do not wrap more than half a line */
        if(Fold_mode == WORD_WRAP && Cols/2 < wrap_col && wrap_col < Cols)
        {   Lines[line_i].len = wrap_len;
            Lines[line_i].cols = screen_col = wrap_col;
        }
     /* If lines longer than screen width are to be truncated, scan for next
      *   \n so next line will start after \n */
        if(Fold_mode == TRUNCATE)
        {   off_t nl_num;
            BufPtr_set(buf_ptr, ch_ptr);
            buf_ptr = BufPtr_find_nl(buf_ptr, stop_off, 1, &nl_num);
            if(nl_num)
                Lines[line_i].nl_num = 1;
            Lines[line_i].len = (int)(BufPtr_off(buf_ptr) - Lines[line_i].off);
         /* Put $ at end of line to indicate truncation */
            LineBuf_putc(line_buf, Cols - 1, TRUNC_CH, Attrib(SYMB));
        }
      /* If in monochrome, put \ at end of wrapped line */
        else if(!Console_Is_color())
        {   LineBuf_putc(line_buf, Cols - 1, WRAP_CH, Attrib(SYMB));
            Lines[line_i].cols = Cols;
        }
    }
 /* If line not full, clear remainder of line */
    if(screen_col < Cols)
        LineBuf_clear(line_buf, screen_col, line_attr);
}
enum { HEX_START  = 10,  /* begin hexadecimal nos in column 10 */
       TEXT_START = 63}; /* begin text in col 63 */
static void hex_line(int line_i, BufPtr buf_ptr, off_t stop_off)
{
 /* Set up pointers at start of hex numbers & text region */
    ScreenPtr line_buf = WinBuf_row(Win_ptr, line_i);
    off_t off          = BufPtr_off(buf_ptr);
    charf *ch_ptr      = BufPtr_curr(buf_ptr), *discont_ptr;
    int byte_i, addr_col, hex_col = HEX_START, text_col = TEXT_START;
    char addr_text[16];
    attr_t attr, norm_attr = Attrib(NORMAL);

    line_set(line_i, off, NORMAL);
    LineBuf_clear(line_buf, 0, norm_attr);

 /* Convert address to text, address must have at least 6 digits */
    sprintf(addr_text, (off < 0x100000) ? "  %06lX:" : "%8lX:" , (long)off);
    for(addr_col = 0; addr_col < 9; ++addr_col)
        LineBuf_putc(line_buf, addr_col, addr_text[addr_col], norm_attr);

 /* Write HEX_LINE_LEN characters */
    for(byte_i = 0, discont_ptr = ch_ptr; byte_i < HEX_LINE_LEN; ++byte_i)
    {   unsigned char ch;
        char *hex_digits = "0123456789abcdef";
        if(ch_ptr == discont_ptr)
        {   BufPtr_set(buf_ptr, ch_ptr);
            attr        = norm_attr;
            if((buf_ptr = find_discont(buf_ptr, &ch_ptr,
                                       &attr, stop_off  )) == NULL)
                break;
            discont_ptr = BufPtr_curr(buf_ptr);
            --byte_i;
            continue;
        }
     /* Need to # of \n in line to know row num for text display */
        if((ch = (unsigned char)*ch_ptr++) == NL)
            ++Lines[line_i].nl_num;

     /* Write hex digits corresponding to current character */
        LineBuf_putc(line_buf, hex_col, hex_digits[ch / 16], attr);
        LineBuf_putc(line_buf, hex_col + 1, hex_digits[ ch & 0xf ], attr);

     /* Move 3 columns to write next hex num, put extra space every 4 bytes */
        hex_col += 3 + !((byte_i + 1) % 4);

        if(!CharSet_iN(Print_set, ch))
        {   attr_t at = attr == norm_attr ? Attrib(CTRL) : Attrib(CTRL_MATCH);
            LineBuf_putc(line_buf, text_col, convert_ctrl_char(ch), at);
        }
        else
            LineBuf_putc(line_buf, text_col, ch, attr);

        ++text_col;
    }
    Lines[line_i].len  = byte_i;
    Lines[line_i].cols = 79;
}

/* Return character which best represents a control character, if char
 * has a C escape sequence, return the escape sequence letter, otherwise
 * return upper case letter for control char. Return d and e for delete
 * and escape, * if no char available */
enum {CONTROL_CHARS = 28};
static char convert_ctrl_char(unsigned char ch)
{
    static char Control_chars[CONTROL_CHARS] =
    {   '0','A','B','C',   'D','E','F','a',  'b','t','n','v', 'f','r',
        'N','O',  'P','Q','R','S',   'T','U','V','W',  'X','Y','Z','e'
    };
    char ctrl_char;
    if(ch < CONTROL_CHARS)
        ctrl_char = Control_chars[(int)ch];
    else
        ctrl_char = (ch == '\x7f') ? 'd' : '*';
    return ctrl_char;
}

/* Find the next discontinuity, discontinuities are defined to be the
 * end of the buffer, a change in the text attribute, or reaching the
 * stop offset. ch_pptr is set to point to the next discontinuity. *Attr_ptr
 * is originally set to the default attribute of the text and it is
 * set to the text color by the routine. buf_ptr points to the current
 * position */
BufPtr find_discont(BufPtr buf_ptr, charf * *ch_pptr,
                    char *attr_ptr, off_t stop_off)
{
    off_t off    = BufPtr_off(buf_ptr), discont_off, end_buf_off;
    off_t hl_off = Show_hls ? Hilites_Off(off, attr_ptr) : OFF_MAX;
    charf *curr  = BufPtr_curr(buf_ptr);
    if(off == stop_off)
        return NULL;
    else if(curr == BufPtr_end(buf_ptr))
    {   buf_ptr = BufPtr_next(buf_ptr);
        if(BufPtr_eof(buf_ptr))
            return NULL;
        curr = *ch_pptr = BufPtr_curr(buf_ptr);
    }
    end_buf_off = off + (BufPtr_end(buf_ptr) - curr);
    discont_off = miN(end_buf_off, hl_off);
    discont_off = miN(discont_off, stop_off);
    BufPtr_set(buf_ptr, curr + (size_t)(discont_off - off));
    return buf_ptr;
}

/* Toggle between ASCII and hexadecimal display modes
 *   The new screen will be drawn around the current highlighted line */
int Lines_Toggle(void)
{
    int lines_above;  /* lines above current highlighted line */
    int line_num_diff = 0;
    off_t cursor_off = Lines[Cursor_i].off;
    BufPtr buf_ptr;
    Hex_mode        ^= 1;

 /* If cursor at end of file, move cursor up a line unless file has zero
  * length, in which case just return */
    if(Lines[Cursor_i].type == EOFILE)
    {   if(!cursor_off)
            return 0;
        else
        {   Lines_Move_cursor(-1);
            cursor_off = Lines[Cursor_i].off;
        }
    }
    buf_ptr = BufPtr_goto(cursor_off);

 /* Now determine where to start first line on screen
  *  If toggling into hex, move to next address which is power of 16 */
    if(Hex_mode)
    {   off_t new_off = (cursor_off + HEX_LINE_LEN - 1) & ~(HEX_LINE_LEN - 1);
        off_t nl_found;
     /* If at end of file, move back to first address which is power of 16 */
        buf_ptr = BufPtr_find_nl(buf_ptr, new_off, HEX_LINE_LEN, &nl_found);
        if(BufPtr_eof(buf_ptr))
        {   new_off -= HEX_LINE_LEN;
            buf_ptr  = BufPtr_find_nl(BufPtr_goto(cursor_off), new_off,
                                      -HEX_LINE_LEN, &nl_found);
        }
        line_num_diff = (int)nl_found;
        Line_num     += nl_found;
    }
    lines_above = lines_between(Top_vis_i, Cursor_i);
    Lines_Render(buf_ptr, Line_num, lines_above);
    return line_num_diff;
}

/* Return the offset of the start of the screen row */
off_t Lines_Off(int row)
{
    return Lines[ line_below(Top_vis_i, row_num(row)) ].off;
}

/* Return the offset of the end of the screen */
off_t Lines_Screen_end_off(void)
{
    int bvl = bottom_vis_line();
    return Lines[bvl].off + Lines[bvl].len;
}

/* Return the line number the cursor is on */
off_t Lines_Line_num(void) { return Line_num; }

/* return the row number of cursor */
int Lines_Cursor_row(void)
{
    return lines_between(Top_vis_i, Cursor_i);
}

/* If next match is found before the value this function returns, do not
 *  jump to a new screen position */
off_t Lines_Jump_start_off(void)
{
    if(Jump_nlines == -1)
        return Lines_Screen_end_off();
    else
    {   int line_i = Cursor_i, row_num, bvl = bottom_vis_line();
        for(row_num = 0; row_num < Jump_nlines; ++row_num)
        {   line_i = line_below(line_i, 1);
            if(line_i == bvl)
                break;
        }
        return Lines[line_i].off + Lines[line_i].len;
    }
}

void Lines_Toggle_hls(void) { Show_hls ^= 1; }

static void line_set(int line_i, off_t off, int type)
{
    Lines[line_i].off    = off;
    Lines[line_i].type   = type;
    Lines[line_i].nl_num = Lines[line_i].len = 0;
}

/* Return the index of the line pos_change rows below row_i */
static int line_below(int row_i, int pos_change)
{
    return (Nlines + row_i + pos_change) % Nlines;
}

/* return the index of the line pos_change rows above row_i */
static int line_above(int row_i, int pos_change)
{
    return (Nlines + row_i - pos_change) % Nlines;
}

/* Determine # of lines between 2 row indices include upper & lower in count*/
static int lines_between(int upper_i, int lower_i)
{
    return (Nlines + lower_i - upper_i) % Nlines;
}

/* Return the difference in line number between the line the cursor is on
 * and the line at row number row. */
int Lines_Num_diff(int row)
{   int line_i = line_below(Top_vis_i, row_num(row));
    return nls_between(line_i);
}

/* Return the number of \n between the cursor line and the line at line_i */
static int nls_between(int line_i)
{
    int nl_cnt = 0, nl_i = Cursor_i;
    if(lines_between(Cursor_i, line_i) <= 0)
    {   while(nl_i != line_i)
        {   nl_i    = line_above(nl_i, 1);
            nl_cnt -= Lines[nl_i].nl_num;
        }
    }
    else
    {   while(nl_i != line_i)
        {   nl_cnt += Lines[nl_i].nl_num;
            nl_i    = line_below(nl_i, 1);
        }
    }
    return nl_cnt;
}

/* Return TRUE if the line at row_i appears on screen, FALSE if not */
static int on_screen(int row_i)
{
    return (lines_between(Top_vis_i, row_i) < Height);
}

/* Return TRUE if last screen line is at end of file, FALSE otherwise
 * end of file will be true if line length is zero */
int Lines_At_EOF(void)
{
    return Lines[ bottom_vis_line() ].type == EOFILE;
}

/* Return the index of the bottom line appearing on the screen */
static int bottom_vis_line(void)
{
    if(lines_between(Top_vis_i, Bottom_i) >= Height)
        return line_below(Top_vis_i, Height - 1);
    else
        return Bottom_i;
}

/* Move the cursor, positive pos_change moves cursor down
 *  negative moves it up, if boundaries of screen reached, scroll, return
 *  number of lines scrolled */
int Lines_Move_cursor(int pos_change)
{
    int line_num_change = 0;
  /*If hl line is going to go off screen, scroll in appropriate direction*/
    while(pos_change < 0 && Cursor_i != Top_vis_i)
    {   Cursor_i  = line_above(Cursor_i, 1);
        Line_num -= Lines[Cursor_i].nl_num;
        ++pos_change;
    }
    while(pos_change > 0 && Cursor_i != bottom_vis_line())
    {   Line_num  += Lines[Cursor_i].nl_num;
        Cursor_i   = line_below(Cursor_i, 1);
        --pos_change;
    }
    /* If line at top or bottom of screen and pos_change not zero, scroll */
    if(pos_change)
        line_num_change += Lines_Scroll(pos_change, TRUE);

    return line_num_change;
}

/* Set the cursor on row number row */
int Lines_Set_cursor(int row)
{
    int cursor_row = lines_between(Top_vis_i, Cursor_i);
    return Lines_Move_cursor(row_num(row) - cursor_row);
}

/* Move the cursor to the line containing the offset off */
void Lines_Move_cursor_off(off_t off)
{
    int row_i = Top_vis_i, bvl = bottom_vis_line();
    while(row_i != bvl)
    {   if(off < Lines[row_i].off + Lines[row_i].len)
            break;

        row_i = line_below(row_i, 1);
    }
    Lines_Set_cursor(lines_between(Top_vis_i, row_i));
}

/* Determine the row number correspoding to row. If row is VCENTER, return
 * the number of the vertically centered row. If row is negative -1, is
 * bottom line, -2 is 2nd from bottom line, etc. Make sure row is on screen */
static int row_num(int row)
{
    if(row == DEFAULT_ROW)
        row = Default_row;
    if(row == VCENTER)
        row = Height/2 - 1;
    else if(row >= Rows)
        row = Rows - 1;
    else if(row < 0)
    {   int vis_rows = lines_between(Top_vis_i, Bottom_i);
        row = maX(Height + row, 0);
        if(row > vis_rows && Top_vis_i != -1)
            row = vis_rows;
    }
    return row;
}

#ifdef LINES_TEST
#include "files.h"
int main(int argc, char *argv[])
{
    Buffer *buf_ptr;
    char *scan_start, *scan_end, *buf_end;
    argc = argc;
    Files_Init(&argv[1]);
    Buffers_Init(16, "");
    Lines_Init(24, 80, 0, NULL);
    Buffers_Bind(Files_open());
    buf_ptr = Buffers_Load(&scan_start, &scan_end, &buf_end);
    BufPtr_set(buf_ptr, scan_start);
    Lines_Render(buf_ptr);
    Lines_Write();
    return EXIT_SUCCESS;
}
#endif /* LINES_TEST */
