/* linenums.c - container for line number locations
 * Copyright (C) 1995-99 Andrew Pipkin (minitrue@pagesz.net)
 * MiniTrue is free software released with no warranty. See COPYING for details
 */

#include <stdlib.h>
#include "minitrue.h"
#include "linenums.h"

/* This module implements the cache for line numbers */

/* Allocate memory for line numbers cache */
void LineNums_init(LineNums *ln)
{
    LineLoc far *first = ln->line_locs = x_farmalloc(1024 * sizeof(LineLoc));
    ln->nalloc         = 1024;

 /* Initialize first line */
    first->line_num    = 1;
    first->off         = 0;
}

void LineNums_reset(LineNums *ln)
{
    ln->last_i     = 0;
    ln->know_total = FALSE;
}

/* Add a line number to the line number cache, if last_line is set line
 * number is total number of lines in the file */
void LineNums_add(LineNums *ln, off_t num, off_t loc, int last_line)
{
    LineLoc far *locs = ln->line_locs;
    int ln_i          = ln->last_i;

 /* Do not add line location if before last location */
    if(loc <= locs[ln_i].off)
        return;

 /* erase previous line number if difference in line numbers is less than
  * LINENUM_GAP */
    if(!ln_i || locs[ln_i].line_num  <= num - NL_CACHE_GAP)
        ++ln_i;

 /* Double array size if array full */
    if(ln_i == ln->nalloc)
    {   ln->nalloc   *= 2;
        locs = x_farrealloc(locs, ln->nalloc * sizeof(LineLoc));
        ln->line_locs = locs;
    }
 /* Store line_number and location */
    locs[ln_i].off      = loc;
    locs[ln_i].line_num = num;
    ln->last_i          = ln_i;

    if(last_line)
        ln->know_total = TRUE;
}

/* Use binary search to find first cached line number before the offset
 * off or line number line_num.  */
LineLoc far *LineNums_nearest(LineNums *ln, off_t num, off_t off)
{
    LineLoc far *locs       = ln->line_locs;
    int left_i = 0, right_i = ln->last_i;
    while(right_i != left_i)
    {   int mid_i = (right_i + left_i + 1) / 2;
        if(num >= locs[mid_i].line_num && off >= locs[mid_i].off)
            left_i  = mid_i;
        else
            right_i = mid_i - 1;
    }
    return &locs[left_i];
}

/* return last line number added */
LineLoc far *LineNums_last(LineNums *ln)
{
    return &ln->line_locs[ ln->last_i ];
}

/* After a replacement is made, the line numbers after the replaced text
 * need to be adjusted by the difference in size and line numbers
 * between the replaced and replacement text. If cache entries occur in the
 * replaced text, set all the replaced entries to the first entry before
 * the replaced text */
void LineNums_adjust(LineNums *ln, off_t replace_off, off_t replaced_len,
                      off_t len_diff, off_t line_diff)
{
    int src_i, adjust_i;
    LineLoc far *locs = ln->line_locs;
    off_t rep_end_off = replace_off + replaced_len;

 /* Adjust all LineLocs after the end of the replaced text */
    for(adjust_i = ln->last_i; locs[adjust_i].off > rep_end_off; --adjust_i)
    {   locs[adjust_i].off      += len_diff;
        locs[adjust_i].line_num += line_diff;
    }
 /* Move to src_i to first LineLoc before replaced text */
    for(src_i = adjust_i; locs[adjust_i].off > replace_off; --adjust_i)
        ;

 /* Set all LineLocs in replaced text to first LineLoc before replaced text */
    for( ; adjust_i > src_i ; --adjust_i )
    {   locs[adjust_i].off      = locs[src_i].off;
        locs[adjust_i].line_num = locs[src_i].line_num;
    }
}
/* return total # of lines if known, -1 if unknown */
off_t LineNums_total(LineNums *ln)
{
    return ln->know_total ? ln->line_locs[ ln->last_i ].line_num : -1;
}

void LineNums_kill(LineNums *ln)  { farfree(ln->line_locs); }

