/* hilites.c - container for location of highlighted regions on screen
 * Copyright (C) 1995-99 Andrew Pipkin (minitrue@pagesz.net)
 * MiniTrue is free software released with no warranty. See COPYING for details
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "minitrue.h"
#include "console.h"

typedef struct
{   off_t off;      /* offset of highlight */
    int len;        /* length of highlight */
    attr_t attr;    /* text attribute of hilight */
    int level;      /* if highlights overlap, highlight with higher level
                     * will be in the foreground */
    int overlap_i;  /* index of first highlight which overlaps current
                     *  highlight, set to highlight index if no overlap */
    void *str;      /* pointer to string structure need to test for equality*/
    int nhilites;   /* number of hilights present when hilight was added */
}Hilite;

static Hilite far *Hilites;
static int Nhilites;  /* index of highlights used */
static int Nalloc;    /* number of highlights allocated */

static off_t end_off(Hilite far *hl_ptr);
static int hl_bin_search(off_t off);

int Hilites_Add(off_t off, int len, attr_t attr, int level, void *str)
{
    Hilite far *hl_ptr;
    int hl_i, ol_i;
 /* If no more spaces in highlight buffer, try to double its size if
  *   possible. If new size overflows, just drop last highlight and reuse */
    if(Nhilites == Nalloc)
    {   long new_alloc = sizeof(Hilite) * Nalloc * 2L;
        if(new_alloc <= 0 || (unsigned long)new_alloc >= UINT_MAX)
            --Nhilites;
        else
        {   Nalloc += Nalloc;
            Hilites = x_farrealloc(Hilites, (size_t)new_alloc);
        }
    }
    hl_i             = hl_bin_search(off) + 1;
    hl_ptr           = &Hilites[ hl_i ];
    _fmemmove(hl_ptr + 1, hl_ptr, sizeof(Hilite) * (Nhilites - hl_i));
    hl_ptr->off      = off;
    hl_ptr->len      = len;
    hl_ptr->attr     = attr;
    hl_ptr->level    = level;
    hl_ptr->str      = str;
    hl_ptr->nhilites = Nhilites;

    for(ol_i = hl_ptr[-1].overlap_i; ol_i < Nhilites; ++ol_i)
    {   if(end_off(&Hilites[ ol_i ]) > off)
            break;
    }
    hl_ptr->overlap_i = ol_i;

    Nhilites++;
    return hl_i;
}

static off_t end_off(Hilite far *hl_ptr) { return hl_ptr->off + hl_ptr->len;}

/* remove the highlight at hl_i */
void Hilites_Del(int hl_i)
{
    size_t move_len = sizeof(Hilite) * (Nhilites - hl_i - 1);
    _fmemmove(&Hilites[hl_i], &Hilites[hl_i + 1], move_len);
    --Nhilites;
}

/* Change the length, attribute and level of the highlight at hl_i */
void Hilites_Change(int hl_i, int new_len, attr_t new_attr, int new_level)
{
    Hilite far *hl_ptr = &Hilites[hl_i];
    if(new_len != -1)
        hl_ptr->len = new_len;
    hl_ptr->attr    = new_attr;
    hl_ptr->level   = new_level;
}

/* Change the attributes of the last nchange hilights added to new_attr */
void Hilites_Change_attrs(int nchange, attr_t new_attr, int new_level)
{
    int hl_i = Nhilites - 1, nchanged = 0;
    Hilite far *hl_ptr = &Hilites[hl_i];
    for( ; hl_i >= 0 && nchanged < nchange; --hl_i, --hl_ptr)
    {   if(hl_ptr->nhilites >= Nhilites - nchange)
        {   hl_ptr->attr  = new_attr;
            hl_ptr->level = new_level;
            ++nchanged;
        }
    }
}

/* If the location at offset off is highlighted, set *attr_ptr to the
 * appropriate attribute, return the offset of where the attribute next
 * changes after off */
off_t Hilites_Off(off_t off, attr_t *attr_ptr)
{
    off_t hl_off, hl_end_off = 0;
    int level          = -1;
    Hilite far *hl_ptr = &Hilites[ hl_bin_search(off) ];
    hl_ptr             = &Hilites[ hl_ptr->overlap_i ];

    for( ; (hl_off = hl_ptr->off) <= off ; ++hl_ptr)
    {   off_t end = hl_off + hl_ptr->len;
        if(end > off && hl_ptr->level > level)
        {   hl_end_off = end;
            level      = hl_ptr->level;
            *attr_ptr  = hl_ptr->attr;
        }
    }
 /* if off not in a highlight, return the offset of the next highlight */
    if(level == -1)
        return hl_off;
 /* Truncate highlight if following highlights overlap and have a higher
  * level */
    else
    {   for(; (hl_off = hl_ptr->off) < hl_end_off; ++hl_ptr)
        {   if(hl_ptr->level > level)
                hl_end_off = hl_off;
        }
        return hl_end_off;
    }
}

/* Find the highlight before off, if str is not NULL, the string must
 *  match str */
off_t Hilites_Prev(off_t off, void *str)
{
    int hl_i = -1;

    if(Nhilites)
    {   hl_i = hl_bin_search(off);
        if(Hilites[hl_i].off >= off)
            --hl_i;
        while(hl_i >= 0 && str != NULL && Hilites[hl_i].str != str)
            --hl_i;
    }
    return hl_i != -1 ? Hilites[hl_i].off : -1;
}

/* Do a binary search for the first highlight at or before off */
static int hl_bin_search(off_t off)
{
    int left_i = 0, right_i = Nhilites;
    Nhilites = right_i;

    while(left_i + 1 < right_i)
    {   int mid_i = (right_i + left_i) / 2;
        if(off >= Hilites[mid_i].off)
            left_i  = mid_i;
        else
            right_i = mid_i;
    }
    return left_i;
}

void Hilites_Reset(void)
{
 /* Boundary values at start & end, needed for binary search */
    static Hilite Bounds[2] =
    {   {0,       0, 0, 0, 0, NULL},
        {OFF_MAX, 0, 0, 0, 1, NULL}
    };
 /* Allocate memory if not allocated, start with 1024 highlights */
    if(!Nalloc)
    {   Nalloc  = 1024;
        Hilites = x_farmalloc(Nalloc * sizeof(Hilite));
    }
 /* Copy boundary values to highlights array */
    _fmemcpy(Hilites, Bounds, 2 * sizeof(Hilite));
    Nhilites = 2;
}
void Hilites_Kill(void)  {  farfree(Hilites); }

#ifdef HILITES_TEST
int main(void)
{
    off_t off;
    char buf[256];
    Hilites_Reset();
    while(fgets(buf, 256, stdin))
    {   int len, level, attr;
        sscanf(buf, "%ld %d %d %d", &off, &len, &attr, &level);
        Hilites_Add(off, len, attr, level, NULL);
    }
    off = 0;
    while(off < OFF_MAX)
    {   attr_t attr = 0;
        off         = Hilites_Off(off, &attr);
        printf("%ld %d\n", (long)off, (int)attr);
    }
    Hilites_Kill();
    return EXIT_SUCCESS;
}
#endif
