/* flags.c - parsing routines for command-line options
 * Copyright (C) 1995-99 Andrew Pipkin (minitrue@pagesz.net)
 * MiniTrue is free software released with no warranty. See COPYING for details
 */

#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <stddef.h>

#include "minitrue.h"
#include "flags.h"

enum {NON_FLAG = 28, NUM_FLAGS };
static char *Params[NUM_FLAGS]; /* Initialize all pointers to params to NULL */
static char *Env_copy;   /* Points to copy of MINITRUE environment variable */

static int All_files;  /* If true, treat all non-flags as files */
static char Default[] = DEFAULT_FLAGS; /* defined in minitrue.h */

static void parse_str(char *str);
static void parse_arg(char *arg_ptr);
static int param_idx(int ch);

int Flags_Init(char * *arg_vect)
{
    int arg_i, param_i;
    char *arg, *env_var = getenv("MINITRUE");
    parse_str(Default);

 /* If environment variable present, copy it because it will be modified,
  *   then parse it. */
    if(env_var)
    {   Env_copy = x_malloc(strlen(env_var) + 1);
        strcpy(Env_copy, env_var);
        parse_str(Env_copy);
    }
 /* Now parse command line flags. Flags will end with the first argument
  *   not starting with a - or a - or -- with no chars afterwards */
    for(arg_i = 0; (arg = arg_vect[arg_i]) != NULL; ++arg_i)
    {   int str_file_i;
        if(*arg != '-' || !arg[1] || (arg[1] == '-' && !arg[2]))
            break;
        parse_arg(arg);
        str_file_i = param_idx('i');
     /* If paramter of -i is +, next argument will be name of string file
      * and treat all non-flags on command line as files */
        if(Params[str_file_i] && !strcmp(Params[str_file_i], "+"))
        {   All_files          = TRUE;
            if((Params[str_file_i] = arg_vect[++arg_i]) == NULL)
                break;
        }
    }
 /* Clear out all parameters beginning with - */
    for(param_i = 0; param_i < NUM_FLAGS; ++param_i)
    {   if(Params[param_i] && *Params[param_i] == '-')
            Params[param_i] = NULL;
    }
 /** If question mark is a flag, or no arguments and -@ and -i not present
  *   just print help
    if(   (!*arg_vect && !Params[param_idx('i') ] && !Params[param_idx('@') ])
       || Params[ param_idx('?') ])
        exit_program(0, NULL); */

    return arg_i;
}
int Flags_All_files(void) { return All_files; }

/* Return index into parameter pointer array corresponding to flag char ch*/
static int param_idx(int ch)
{
    if(islower(ch))        return ch - 'a';
    else if(ch == '@')     return 26;
    else if(ch == '?')     return 27;
    else                   return NON_FLAG;
}

/* Return pointer to parameter corresponding to */
char *Flags_Param(int ch) {
    return Params[ param_idx(ch) ];
}

/* Only thing destructor will do is free copy of environment variable */
void Flags_Kill(void) { free(Env_copy); }

/* This function chops a string containing flags and then parses the flags.
 *   The string is chopped at whitespace unless it is quoted */
static char far Unclosed_env[] =
"Unclosed \" in environment variable MINITRUE";
static void parse_str(char *str)
{
    char *arg_start = str;
 /* Ignore quotes enclosing entire string */
    if(*str == '\"')
    {   char *str_end = str + strlen(str) - 1;
        if(*str_end == '\"')
        {   ++arg_start;
            *str_end = '\0';
        }
    }
    while(*(arg_start = skip_ws(arg_start)) != '\0')
    {   char *arg_end = arg_start;
        if(*arg_start != '\"')
        {   while(!isspace(*arg_end) && *arg_end)
                ++arg_end;
        }
     /* If string quoted, locate closing quote */
        else
        {   arg_end = ++arg_start;
            while(*arg_end != '\"' && *arg_end)
            {   if(!*++arg_end)
                    error_msg(Unclosed_env);
            }
        }
     /* Insert NUL to terminate string unless NUL already at end, in which
      *   case end of string has been reached so quit after parsing. */
        if(*arg_end)
        {   *arg_end = '\0';
            parse_arg(arg_start);
            arg_start = ++arg_end;
        }
        else
        {   parse_arg(arg_start);
            break;
        }
    }
}

static char far Bad_env_var[] =
"Environment variable MINITRUE can only contain command-line options\n.";

static void parse_arg(char *arg_ptr)
{
 /* This enumerations represents the types of parameters */
    enum param_types
    {   INV = 0,       /* flag corresponding to index invalid */
        NONE,          /* flag valid but no parameters allowed */
        PLUS,          /* parameter can only be plus sign */
        NUM = 4,       /* numeric parameter allowed */
        STR = 8,       /* paramter nul-terminated string beg*/
        SYMB = 16      /* string of non-flag characters */
    };
    static char valid_params[NUM_FLAGS + 1] =
    {   NONE /* a */, STR + PLUS     /* b */, SYMB /* c */, NONE    /* d */,
        PLUS /* e */, PLUS           /* f */, INV  /* g */, NONE    /* h */,
        STR  +PLUS /* i */, INV      /* j */, STR /* k */, STR     /* l */,
        SYMB /* m */, NONE           /* n */, SYMB /* o */, STR     /* p */,
        NONE /* q */, SYMB           /* r */, SYMB /* s */, NUM     /* t */,
        SYMB /* u */, STR + PLUS     /* v */, STR + PLUS  /* w */, PLUS /* x */,
        PLUS /* y */, STR            /* z */,
        STR + PLUS /* @ */, STR, /* ? */ INV /* all other chars */
    };
 /* Flags must begin with dash */
    if(*arg_ptr++ != '-')
    {   error_msg(Bad_env_var);
        return;
    }
    while(*arg_ptr)
    {   int option_ch = *arg_ptr++, flag_i = param_idx(option_ch), param_type;
        char *param_start = arg_ptr;

     /* Replace dashes with NULs to chop up flags into parameters */
        arg_ptr[-1] = '\0';

        if((param_type = valid_params[flag_i]) == INV)
        {   char err_msg[256];
            sprintf(err_msg,
                    "-%c is not a valid option. Enter \""EXE_NAME" -?\" for help.",
                    option_ch);
            error_msg(err_msg);
        }
     /* Dash following flag means to use defaults */
        if(*arg_ptr == '-')
            ++arg_ptr;

        if(param_type & PLUS && *arg_ptr == '+')
            ++arg_ptr;

     /* Skip over argument parameters */
        if(param_type & STR && *arg_ptr == ':')
        {   while(*arg_ptr)
                ++arg_ptr;
        }
        else if(param_type & NUM)
        {   while(isdigit(*arg_ptr))
                ++arg_ptr;
        }
     /* flags with symbol parameters cannot have flag characters */
        else if(param_type & SYMB)
        {   while(*arg_ptr && param_idx(*arg_ptr) == NON_FLAG)
                ++arg_ptr;
        }
     /* If argument has no parameters */
        if(param_start == arg_ptr && Params[flag_i] && *Params[flag_i] == '-')
            ++Params[flag_i];
        else
            Params[flag_i] = param_start;
    }
}

#ifdef FLAGS_TEST
#include <stdio.h>
int main(int argc, char * *argv)
{
    int ch_i;
    Flags_Init(&argv[1]);
    for(ch_i = 0; ch_i < 127; ++ch_i)
    {   char *param = Flags_Param(ch_i);
        if(param)
            printf("-%c = %s\n", ch_i, param);
    }
    Flags_Kill();
    return EXIT_SUCCESS;
}
#endif /* FLAGS_TEST */
