/* files.c - handle file I/O for files to be scanned
 * Copyright (C) 1995-99 Andrew Pipkin (minitrue@pagesz.net)
 * MiniTrue is free software released with no warranty. See COPYING for details
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include "minitrue.h"
#include "files.h"
#include "fileops.h"
#include "dirwalk.h"
#include "wildcard.h"
#include "unzip.h"
#include "backup.h"

enum flags { STDIN_PIPE = 1, UNZIP_PIPE = 2 };
enum backup { BACKUP_ORIG = 1, BACKUP_MOD  = 2 };

static char * *Fname_vect; /* Vector of command line filenames */
static int Cmdln_num;      /* Number of filenames found in command line */
static int Scan_stdin;     /* Tells if standard input is to be scanned */
static int Single;         /* Set if only one file is to be scanned */
static int Make_backup;    /* Set if Backups to be made for changed files */
static int Recurse;        /* Set if recursing through directories */
static int Keep_fdates;    /* Set if files' datestamps are not to be modified*/
static DirWalk Dir_walk;   /* Set if currently recursing through directory */
static WildCard *Wild_card;/* Pointer to bytecodes for shell pattern matching*/
static int Unzip;          /* Set if unzip parameter present */
static file_t Stdin_dup;   /* Duplicate of standard input */
static Files In_file = { UNOPENED };
static Files Out_file = { UNOPENED } ;

static FILE *Fname_file;         /* File containing filenames to scan */
static char *Fname_buf;          /* Buffer for filenames read from file */
static int (*Is_fn_sep)(int ch); /* Function for determining if char is
                                  * filename separator, default is spaces,
                                  * -@+ forces nuls */
static FILE *String_file;        /* File containing strings */

static char *next_fname(fdata_t *fdata_ptr, int *got_fdata);
static char *read_fname_file(void);
static void process_fdata(Files *files, int have_fdata);
static void file_reset(Files *files);
static char *find_unused(const char *fname);
static int same_file_sys(Files *files, const char *output_fname);
int isnul(int ch) { return (ch == '\0'); }

/* Determine number of filenames found on command line, return pointer to
 * pointer to first string on command line */
char * *Files_Init(char * *arg_vect, int nargs, int all_fnames,
                   const char *all_files, const char *backup_ext,
                   const char *keep_fdates, const char *string_fname,
                   const char *recurse_params, const char *unzip_params,
                   const char *fn_file)
{
    int arg_i;
    static char Temp_name[L_tmpnam];
    char * *str_vect = NULL;

    Out_file.name = tmpnam(Temp_name);
    DirWalk_init(&Dir_walk, recurse_params ? recurse_params : "0:0",
                 all_files != NULL);
    Recurse = (recurse_params != NULL);
    Keep_fdates = (keep_fdates != NULL);
    Unzip_Init(unzip_params);
    Unzip = (unzip_params != NULL);

    if(backup_ext)
    {   if(*backup_ext == '+')
        {   ++backup_ext;
            Make_backup = BACKUP_MOD;
        }
        else
            Make_backup = BACKUP_ORIG;
        if(*backup_ext != ':' || backup_ext[1] == '\0')
            error_msg("-b must be followed by backup extension");
        Backup_Init(backup_ext + 1);
    }
 /* If stdin piped in, duplicate stdin in case keyboard reopened */
    if(!x_isatty(0))
    {   Stdin_dup  = x_dup(0);
        Scan_stdin = TRUE;
    }
 /* Open filename containing strings */
    if(string_fname)
    {   if(*string_fname != '\0')
        {   if(*string_fname == ':' || all_fnames)
            {   if(!all_fnames)
                    ++string_fname;
                String_file = fopen(string_fname, "rb");
                if(!String_file)
                    error_msg("Unable to open string input file");
            }
            else
                invalid_param('i', *string_fname);
        }
        else
        {   Scan_stdin  = FALSE;
            String_file = stdin;
        }
    }
 /* Parse filename file argument, if preceded by + nuls will be filename
  * separators instead of whitespace, use standard input if no filename */
    if(fn_file)
    {   if(*fn_file != '+')
            Is_fn_sep = isspace;
        else
        {   Is_fn_sep = isnul;
            ++fn_file;
        }
        if(*fn_file == '\0')
        {   if(Scan_stdin)
            {   Scan_stdin = FALSE;
                Fname_file = fdopen(Stdin_dup, "rb");
            }
        }
        else if(*fn_file == ':')
        {   Fname_file = fopen(fn_file + 1, "rb");
            if(!Fname_file)
                error_msg("Unable to open filename file");
        }
        else
            invalid_param('@', *fn_file);
    }
 /* Now determine which items in argument vector are filenames */
    Fname_vect = arg_vect;
    Cmdln_num  = -1;
    for(arg_i = 0; arg_i < nargs && Cmdln_num == -1 && !all_fnames; ++arg_i)
    { /* Single dash implies end of filenames - remainder strings */
        if(arg_vect[arg_i][0] == '-')
        {   if(!arg_vect[arg_i][1])
            {   Cmdln_num = arg_i;
                str_vect  = arg_vect + arg_i + 1;
            }
         /* Two dashes imply start of filenames - preceding nonflags strings*/
            else if(arg_vect[arg_i][1] == '-' && !arg_vect[arg_i][2])
            {   Fname_vect      = &arg_vect[arg_i + 1];
                Cmdln_num       = nargs - arg_i - 1;
             /* replace "--" with NULL to indicate end of strings */
                arg_vect[arg_i] = NULL;
                str_vect        = arg_vect;
            }
        }
    }
    if(all_fnames)
    {   Cmdln_num = nargs;
        str_vect  = arg_vect + nargs;
    }
 /* If no dash(es) present only one filename will appear on command line
  *   unless stdin is to be scanned, then no arguments are files */
    else if(Cmdln_num == -1)
    {   Cmdln_num = (arg_vect[0] && !Scan_stdin && !Fname_file) ? 1 : 0;
        str_vect  = arg_vect + Cmdln_num;
    }
    if(!Cmdln_num && !Scan_stdin && !Fname_file)
    {   if(nargs)
            error_msg("No filenames present");
        else
            return NULL;
    }

 /* If only one file will be scanned, filename will be ommitted from
  *   grep output, treat wildcards and filename file as multiple files
  *   even if they only result in only one file */
    Single = !(   Cmdln_num + Scan_stdin > 1 || Fname_file || Recurse
               || (Cmdln_num >= 1 && WildCard_Is(Fname_vect[0])));

    return str_vect;
}
FILE *Files_String_file(void) { return String_file; }

/* Open the next file and set the parameters of the files structure */
Files *Files_Next(void)
{
    int got_fdata = FALSE;
    file_reset(&In_file);
    file_reset(&Out_file);

    /* First scan standard input if present */
    if(Scan_stdin == TRUE)
    {   set_error_fname(In_file.name = "<STDIN>");
        Scan_stdin     = FALSE;
        In_file.flags |= STDIN_PIPE;
        In_file.desc   = Stdin_dup;
        process_fdata(&In_file, got_fdata);
    }
 /* Then go through files listed on the command line */
    else
    {   while((In_file.name = next_fname(&In_file.fdata, &got_fdata)) != NULL)
        {   int unzip_err;
            set_error_fname(In_file.name);
         /* See if file can be decompressed */
            if(Unzip && (unzip_err = Unzip_Open(In_file.name)) != NOT_ARCHIVE)
            {   if(unzip_err == SKIP_FILE)
                    continue;
                else if(unzip_err == UNZIP_ERR)
                    read_error("Unable to decompress ", In_file.name, FALSE);
                else
                {   In_file.desc   = unzip_err;
                    In_file.flags |= UNZIP_PIPE;
                }
            }
            else
            {/* If only regular files desired, test and skip if not
              * regular */
                if(DirWalk_only_reg(&Dir_walk))
                {   x_get_fattr(In_file.name, &In_file.fdata);
                    if(!fdata_isreg(&In_file.fdata))
                        continue;
                }
                In_file.desc = x_open(In_file.name);
            }

         /* Report error if file a directory */
            if(In_file.desc != UNOPENED)
            {   process_fdata(&In_file, got_fdata);
             /* skip file if directory files not desired */
                if(!fdata_isdir(&In_file.fdata))
                    break;
                else
                    read_error("is a directory", In_file.name, FALSE);
            }
        }
    }
    return In_file.name ? &In_file : NULL;
}

static char *next_fname(fdata_t *fdata_ptr, int *got_fdata)
{
    static int Fname_i = 0;
    char *fname, *basename, *ext;
    do
    {   fname = NULL;
        *got_fdata = FALSE;
     /* If scanning directory, see if wildcard pattern matches next filename */
        if(Wild_card)
        {   do
            {   if(!(fname = DirWalk_next(&Dir_walk, fdata_ptr, got_fdata)))
                {   WildCard_kill(Wild_card);
                    Wild_card = NULL;
                }
            } while(fname && !(WildCard_match(Wild_card, fname)));
        }
     /* Otherwise get filename from command line or filename file */
        else if(Fname_i < Cmdln_num || (fname = read_fname_file()) != NULL)
        {   int fname_len;
            if(Fname_i < Cmdln_num)
                fname = Fname_vect[Fname_i++];

            basename = parse_path(fname, &ext, &fname_len);
         /* If filename is directory, and recursion desired, recurse into it*/
            if(   Recurse && !WildCard_Is(basename)
               && !x_get_fattr(fname, fdata_ptr))
            {   if(fdata_isdir(fdata_ptr))
                    basename = fname + fname_len;
                *got_fdata = GOT_FATTR;
            }
            if((Wild_card = WildCard_init(basename, FN_CASE_INS, Recurse)))
            {/* Copy leading dirs into dir_copy */
                int dir_len       = basename - fname;
                char *dir_copy    = x_strdup(fname);
                dir_copy[dir_len] = '\0';

                if(!(DirWalk_start(&Dir_walk, dir_copy,
                                   have_ctype(basename, isupper))))
                {   WildCard_kill(Wild_card);
                    Wild_card = NULL;
                }
                free(dir_copy);
                fname = NULL;
            }
        }
        else
            return NULL;
    }while(fname == NULL);
    return fname;
}

/* Read filename from Fname_file, the function Is_fn_sep points to determines
 *   where the filenames are separated */
static char *read_fname_file(void)
{
    static int Fname_buf_len;
    int ch, ch_i = 0;
    if(!Fname_file)
        return NULL;

 /* Skip over leading non-filename characters */
    while(Is_fn_sep((ch = fgetc(Fname_file))) && !feof(Fname_file))
        ;

    if(feof(Fname_file))
    {   fclose(Fname_file);
        Fname_file = NULL;

        return NULL;
    }
 /* Copy filename chars to buffer, resizing buffer if necessary */
    else
    {   do
        {   if(ch_i + 1 >= Fname_buf_len)
                Fname_buf = x_realloc(Fname_buf, Fname_buf_len += 128);
            Fname_buf[ch_i++] = ch;
        } while(!Is_fn_sep(ch = fgetc(Fname_file)) && !feof(Fname_file));
    }
    Fname_buf[ch_i] = '\0';
    return Fname_buf;
}

/* Get the file data if it has not already been obtained, and set the file
 * size to -1 if the file size is not known */
static void process_fdata(Files *files, int got_fdata)
{
    if(got_fdata != GOT_FDATA)
#ifndef __MSDOS__
        if(files->desc != UNOPENED)
            x_fstat(files->desc, &files->fdata);

#else
    {   files->fdata.size = x_seekend(files->desc);
        x_seek(files->desc, 0);
        if(got_fdata != GOT_FATTR)
        {   if(files->flags & (UNZIP_PIPE | STDIN_PIPE))
                files->fdata.attrib = _A_NORMAL;
            else
                x_get_fattr(files->name, &files->fdata);
        }
        if(Keep_fdates)
            x_get_ftime(files->desc, &files->fdata);
    }
#endif
    files->size = (  fdata_seekable(&files->fdata, (files->flags & UNZIP_PIPE))
                   ? fdata_size(&files->fdata) : -1);
}

static void file_reset(Files *files)
{
    files->nread = files->size = 0;
    files->mod   = NULL;
    files->desc  = UNOPENED;
    files->flags = 0;
}

/* Access routines */
const char *Files_name(Files *files) { return files->name; }
off_t Files_size(Files *files)       { return files->size; }
Files *Files_mod(Files *files)       { return files->mod; }
int Files_Single(void)               { return Single; }
int Files_eof(Files *files)
{
    return files->nread >= files->size && files->size != -1;
}
int Files_seekable(Files *files)
{
    return fdata_seekable(&files->fdata, files->flags & UNZIP_PIPE);
}

/* Read the file into the buffer bounded by *start & *end and return the
 * number of characters read, if memory mapping used, set *start & *end
 * the the boundaries of the memory mapped region */
size_t Files_read(Files *files, char *buf, size_t len)
{
    size_t nread  = x_read(files->desc, buf, len);
    files->nread += nread;

 /* If read does not fill buffer, must be at EOF so set file size */
    if(nread < len)
        files->size = files->nread;
    return nread;
}

/* Load file using memory mapping, keep reading until a non-contiguous
 * page mapped or the end of the file reached. Return the number of bytes
 * read and set *start and *end to the start and end of the loaded region.
 * Pad with an extra block at the start and end of the file. Set *end to
 * NULL on mmap failure if no pages read */
#if defined(unix) && !defined(__DJGPP__)
#include <unistd.h>
#include <sys/mman.h>
#endif /* DJGPP or UNIX */
size_t Files_mmap(Files *files, char * *start, char * *end)
{
#ifdef _POSIX_MAPPED_FILES
 /* If memory mapping available and file a regular file, use mmap */
    int page_len = getpagesize();
    char *blk_start = NULL, *blk_end = NULL, *pad_page = NULL;
    off_t orig_nread = files->nread;

 /* Add padding page at start of file */
    if(!orig_nread)
        pad_page = mmap(NULL, page_len, PROT_READ, MAP_SHARED, files->desc, 0);

 /* Keep reading blocks as long as start of next block is end of last block */
    do
    {   char *next_blk = mmap(NULL, page_len, PROT_READ,
                              MAP_SHARED, files->desc, files->nread);
     /* If error occurs, set the file pointer to the # of bytes memmapped so
        the remainder of the file can be read conventionally */
        if((int)next_blk == -1)
        {   x_seek(files->desc, files->nread);
            break;
        }
        if(!blk_end)
            blk_start = next_blk;
     /* If page not contiguous to previously mapped pages, unmap it and exit*/
        else if(next_blk != blk_end)
        {   munmap(next_blk, page_len);
            break;
        }
        blk_end  = next_blk + page_len;

    } while((files->nread += page_len) < (files->size + page_len ));

    *start = blk_start;
    *end   = blk_end;
    if(pad_page && (int)pad_page != -1)
        munmap(pad_page, page_len);
    return miN(files->size, files->nread) - orig_nread;
#endif /* mmap */
}

int Files_Have_mmap(void)
{
#ifdef _POSIX_MAPPED_FILES
    return 1;
#else
    return 0;
#endif
}

size_t Files_write(Files *files, const charf *buf, size_t len)
{
    size_t nwrote     = x_write(files->mod->desc, buf, len);
    files->mod->size += len;
    if(nwrote != len)
        fatal_error("Disk full");

    return nwrote;
}

/* Initialize modified file before first replacement is written,
 * write up to offset off of the original file into the modified file,
 * using buf with buf_len to transfer */
void Files_start_mod(Files *files, off_t copy_end_off,
                     charf *buf, size_t buf_len)
{
    files->mod = &Out_file;
    Out_file.desc = x_creat(Out_file.name);
    if(copy_end_off)
    {   off_t fpos = 0;
        x_seek(files->desc, 0);
        while(fpos < copy_end_off) {
            x_read(files->desc, buf, buf_len);
            x_write(Out_file.desc, buf, buf_len);
            fpos += buf_len;
        }
        x_seek(files->desc, files->nread);
    }
    Out_file.size = copy_end_off;
}

size_t Files_read_off(Files *files, off_t off, charf *buf, int len,
                      off_t *off_ptr, int reset_eof)
{
    size_t nread;
    x_seek(files->desc, (*off_ptr = off & ~(len - 1)));
    nread = x_read(files->desc, buf, len);
    x_seek(files->desc, reset_eof ? files->size : files->nread);

    return nread;
}

void Files_finish_dir(int skip_sub_dirs)
{
    DirWalk_finish(&Dir_walk, skip_sub_dirs);
}

/* Rename modified temporary file and original files to create backups
 * in desired manner */
void Files_close(Files *files)
{
    int fn_max_len = fname_max(files->desc);
    x_close(files->desc);
    files->desc = UNOPENED;

    if(files->mod)
    {   const char *backup_fname = (  Make_backup
                                    ? Backup_Fname(files->name, fn_max_len)
                                    : NULL);
        char fname_buf[128], backup_buf[128];
        const char *output_fname = (  (Make_backup != BACKUP_MOD)
                                    ? files->name
                                    : backup_fname);
        int attempt_write = !(files->flags & (STDIN_PIPE | UNZIP_PIPE));

        if(Make_backup && !backup_fname)
        {   backup_fname = FileOps_Get_writeable(files->name, -1,
                                                 backup_buf, 128);
            if(Make_backup == BACKUP_MOD)
                output_fname = backup_fname;
        }

     /* Make sure that file is writeable before attempting to modify
      * file. */
        output_fname = FileOps_Get_writeable(output_fname, attempt_write,
                                             fname_buf, 128);
        if(!output_fname || (Make_backup && !backup_fname))
        {   x_close(files->mod->desc);
            x_remove(files->mod->name);
            if(files->flags & UNZIP_PIPE)
                Unzip_Close();
            return;
        }

        if(Make_backup == BACKUP_ORIG && x_can_write(files->name))
            x_rename(files->name, backup_fname);

     /* If file and the temporary file containing a modified version of the
      * original file are on different file systems, must copy modified version
      * back to original directory byte by byte, otherwise just need to
      * rename */
        if(same_file_sys(files, output_fname))
        {   x_close(files->mod->desc);
            if(output_fname == files->name)
                x_clone(files->mod->name, &files->fdata, Keep_fdates);
            else
                x_norm_perms(files->mod->name);
            x_rename(files->mod->name, output_fname);
        }
        else
        {   size_t nread, nwrote;
            char far *buf = x_farmalloc(16384);
            const char *copy_dest_fname = output_fname;
            int copy_fdesc;

         /* If backups not wanted, change original file name to prevent
          * overwriting in case copying goes wrong */
            if(output_fname == files->name && !Make_backup)
               copy_dest_fname = find_unused(files->name);

            copy_fdesc = x_creat(copy_dest_fname);

            x_seek(files->mod->desc, 0);
            do
            {   nread  = x_read(files->mod->desc, buf, 16384);
                nwrote = x_write(copy_fdesc, buf, nread);

            }while(nwrote == 16384);
            x_close(copy_fdesc);
            x_close(files->mod->desc);

            if(nread != nwrote)
            {   x_remove(copy_dest_fname);
                if(Make_backup == BACKUP_ORIG)
                    x_rename(backup_fname, files->name);

                x_remove(files->mod->name);
                files->mod->desc = UNOPENED;
                fatal_error("Disk full");
            }

            if(output_fname == files->name)
                x_clone(copy_dest_fname, &files->fdata, Keep_fdates);
            else
                x_norm_perms(copy_dest_fname);
            x_remove(files->mod->name);

            if(!Make_backup && output_fname == files->name)
            {   x_rename(copy_dest_fname, files->name);
                free((char *)backup_fname);
            }
            farfree(buf);
        }
        files->mod->desc = UNOPENED;
    }

    if(files->flags & UNZIP_PIPE)
        Unzip_Close();
}
/* Find an unused filename in the directory where file is located */
static char *find_unused(const char *path)
{
    int len = strlen(path);
    char *buf = x_malloc(len + 13), *ext;
    unsigned long num;
    char *fname = parse_path(memcpy(buf, path, len + 1), &ext, &len);
    for(num = 0; num < 0xffffffff; ++num)
    {   sprintf(fname, "%lx.mtr", num);
        if(!x_exists(buf))
            break;
    }
    return buf;
}

/* Return TRUE if the original file and it temporary file containing
 * modifications are found on the same file system */
#ifndef __MSDOS__
static int same_file_sys(Files *files, const char *output_fname)
{   struct stat temp_stat;
    fstat(files->mod->desc, &temp_stat);
    if(output_fname == files->name)
        return files->fdata.st_dev == temp_stat.st_dev;
    else
    {   struct stat output_stat;
        char *output_dir = x_get_dir(output_fname);
        stat(output_dir, &output_stat);
        free(output_dir);
        return output_stat.st_dev == temp_stat.st_dev;
    }
}

#else
static int same_file_sys(Files *files, const char *output_fname)
{
    int default_drive = x_get_drive();
    int mod_drive     = (  files->mod->name[1] == ':'
                         ? tolower(files->mod->name[0]) - 'a' + 1
                         : default_drive);
    int output_drive  = (  output_fname[1] == ':'
                         ? tolower(output_fname[0]) - 'a' + 1
                         : default_drive);
    return mod_drive == output_drive;
}
#endif

/* Shut files and delete temporary file if open, should already be closed
 *   unless program interrupted */
void Files_Kill(void)
{
    free(Fname_buf);
    DirWalk_kill(&Dir_walk);
    Unzip_Kill();
    Backup_Kill();
    if(Wild_card)
        WildCard_kill(Wild_card);
    if(In_file.desc != UNOPENED)
    {   x_close(In_file.desc);
        if(In_file.flags & UNZIP_PIPE)
            Unzip_Close();
    }
    if(Out_file.desc != UNOPENED)
    {   x_close(Out_file.desc);
        x_remove(Out_file.name);
    }
}

#ifdef _POSIX_MAPPED_FILES
void Files_Unmap(void *start, size_t len)
{
    munmap(start, len);
}
#endif /* _POSIX_MAPPED_FILES */
