/* fileops.c - low level file operations
 * Copyright (C) 1995-99 Andrew Pipkin (minitrue@pagesz.net)
 * MiniTrue is free software released with no warranty. See COPYING for details
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <fcntl.h>
#include <errno.h>
#include "fileops.h"
#include "minitrue.h"
#include "statline.h"

static void file_error(const char *fname);
static int path_max(const char *fname);
static const char *Fname;  /* Default file name for writing error messages */
#ifndef __DOS16__
#include <unistd.h>
#else
enum { F_OK = 0, R_OK = 4, W_OK = 2, X_OK = 1};
#endif

#ifndef __MSDOS__
#include <sys/stat.h>
#include <dirent.h>
#else
#include <io.h>
#include <dos.h>
#define open  _open
#define close _close
#define read  _read
#define write _write
#endif

enum { NEXT_FILE = 0, RETRY = 1 };

static char *Ignore_errors;

void FileOps_Init(char *ignore_errors)
{
    Ignore_errors = ignore_errors;
}

/* Set the default filename in error message to fname */
void set_error_fname(const char *fname)
{
    Fname = fname;
}

file_t x_open(const char *fname)
{
    int fdesc;
    for( ; ; )
    {   fdesc = open(fname, O_RDONLY);
        if(fdesc == UNOPENED)
        {   if(read_error("Unable to open ", fname, TRUE) < 2)
                break;
        }
        else
            break;
    }
    return fdesc;
}

file_t x_creat(const char *fname)
{
#ifndef __MSDOS__
    int fdesc = open(fname, O_RDWR | O_CREAT, S_IRUSR);
#else
    int fdesc = _creat(fname, _A_NORMAL);
#endif
    if(fdesc == UNOPENED)
        file_error(fname);
    return fdesc;
}
#ifndef __DOS16__

int x_read(file_t fdesc, charf *buf, int len)
{
    int nread_buf = 0, nread = 0;
    for( ; ; )
    {   while(   (nread = read(fdesc, buf + nread_buf, len - nread_buf)) > 0
              && (nread_buf += nread) < (ssize_t)len)
            ;
        if(nread == -1)
        {   if(read_error("Unable to read ", NULL, TRUE) < 2)
                break;
        }
        else
            break;
    }
    return nread_buf;
}

int x_write(file_t fdesc, const charf *buf, int len)
{
    int nread = write(fdesc, buf, len);
    if(nread == -1)
        file_error(NULL);
    return nread;
}

/* Return the directory for the file fname. If fname has no directory, use
 * the current directory. The directory will need to be freed later */
char *x_get_dir(const char *fname)
{
    int len, max_len = path_max(fname);
    char *ext_start, *dir_end = parse_path((char *)fname, &ext_start, &len);
    char *dir_buf = x_malloc(max_len);
 /* Use current working directory if path contains no directories */
    if(dir_end == fname)
    {   if(getcwd(dir_buf, max_len) == NULL)
        {   free(dir_buf);
            return FALSE;
        }
    }
    else
    {   int dir_len = dir_end - fname;
        memcpy(dir_buf, fname, dir_end - fname);
        dir_buf[dir_len] = '\0';
    }
    return dir_buf;
}

/* return the maximum filename length for the where filesystem fdesc is found*/
int fname_max(file_t fdesc)
{
    return fpathconf(fdesc, _PC_NAME_MAX);
}

/* return the maximum path length of the file system where fname is found */
static int path_max(const char *fname)
{
    return pathconf(fname, _PC_PATH_MAX);
}

/* Need to use _dos_read/write because they accept far pointers */
#else
int x_read(file_t fdesc, charf *buf, int len)
{
    unsigned nread;
    for( ; ; )
    {   if(_dos_read(fdesc, buf, len, &nread))
        {   if(read_error("Unable to read ", NULL, TRUE) < 2)
                return -1;
        }
        else
            break;
    }
    return (int)nread;
}

int x_write( file_t fdesc, const charf *buf, int len)
{
    unsigned nwrote;
    if(_dos_write(fdesc, buf, len, &nwrote))
    {   file_error(NULL);
        return (size_t)-1;
    }
    return nwrote;
}

int fname_max(file_t fdesc) { return 12; }
static int path_max(const char *fname) { return 80; }
#endif

int x_close(file_t fdesc)
{
    if(close(fdesc))
    {   file_error(NULL);
        return -1;
    }
    return 0;
}

int x_rename(const char *oldname, const char *newname)
{
    if(x_exists(newname))
        x_remove(newname);

    if(rename(oldname, newname))
    {   file_error(oldname);
        return -1;
    }
    return 0;
}

int x_remove(const char *fname)
{
    if(remove(fname))
    {   file_error(fname);
        return -1;
    }
    return 0;
}

off_t x_seek(file_t fdesc, off_t off)
{
    off_t new_off = lseek(fdesc, off, SEEK_SET);
    if(new_off == -1)
        file_error(NULL);
    return new_off;
}

off_t x_seekend(file_t fdesc)
{
    off_t fsize = lseek(fdesc, 0, SEEK_END);
    if(fsize == -1)
        file_error(NULL);
    return fsize;
}

int x_dup(int fdesc)
{
    int new_fdesc = dup(fdesc);
    if(new_fdesc == -1)
        file_error(NULL);
    return new_fdesc;
}

int x_dup2(int old_fdesc, int new_fdesc)
{
    if(dup2(old_fdesc, new_fdesc) == -1)
    {   file_error(NULL);
        return -1;
    }
    return 0;
}
int x_isatty(int fdesc) { return isatty(fdesc); }

int x_exists(const char *fname)      { return !access(fname, F_OK); }
int x_can_write(const char *fname)
{
    return access(fname, F_OK) == -1 || !access(fname, W_OK);
}
int x_is_runnable(const char *fname) { return !access(fname, X_OK); }

#ifndef __MSDOS__
DIR *x_opendir(const char *dir_name)
{
    DIR *dir;
    for( ; ; )
    {   dir = opendir(dir_name);
        if(!dir)
        {   if(   read_error("Unable to open directory ", dir_name, TRUE)
               == NEXT_FILE)
                break;
        }
        else
            break;
    }
    return dir;
}

int x_get_fattr(const char *fname, fdata_t *stat_ptr)
{
    return x_stat(fname, stat_ptr);
}

int x_get_ftime(int fdesc, fdata_t *stat_ptr)
{
    return x_fstat(fdesc, stat_ptr);
}

int x_stat(const char *fname, fdata_t *stat_ptr)
{
    if(stat(fname, stat_ptr))
    {   file_error(fname);
        return -1;
    }
    else
        return 0;
}

int x_fstat(int fdesc, fdata_t *stat_ptr)
{
    if(fstat(fdesc, stat_ptr))
    {   file_error(NULL);
        return -1;
    }
    else
        return 0;
}

int x_lstat(const char *fname, fdata_t *stat_ptr)
{
    if(lstat(fname, stat_ptr))
    {   file_error(fname);
        return -1;
    }
    else
        return 0;
}

/* Set the file attributes of dest to those of src. If clone_ftime is set,
 *  copy the file time as well */
#include <utime.h>
int x_clone(const char *fname, fdata_t *st, int clone_ftime)
{
    if(chmod(fname, st->st_mode) || chown(fname, st->st_uid, st->st_gid))
    {   file_error(fname);
        return -1;
    }
    else
    {   if(clone_ftime)
        {   struct utimbuf utime_buf;
            utime_buf.actime  = st->st_atime;
            utime_buf.modtime = st->st_mtime;
            if(utime(fname, &utime_buf))
                return -1;
        }
        return 0;
    }
}

/* Make the file permissions of the file fname normal */
int x_norm_perms(const char *fname)
{
    if(chmod(fname, 0664))
    {   file_error(fname);
        return -1;
    }
    else
        return 0;
}

/* Return TRUE if it is possible to write into the directory of path,
 * FALSE otherwise */
int x_can_write_dir(const char *path)
{
    char *dir_buf = x_get_dir(path);
    int can_write = x_can_write(dir_buf);
    free(dir_buf);
    return can_write;
}

int x_get_drive(void) { return 0; }

/* If not MSDOS, compare filenames case sensitively */
int fname_cmp(const char far *fname1, const char far *fname2)
{
    return strcmp(fname1, fname2);
}
off_t fdata_size(struct stat *st)       { return st->st_size; }
int   fdata_isdir(struct stat *st)      { return S_ISDIR(st->st_mode); }
int   fdata_isreg(struct stat *st)      { return S_ISREG(st->st_mode); }
int fdata_seekable(struct stat *st, int unzipped)
{
    return unzipped ? FALSE : S_ISREG(st->st_mode);
}
#else

int x_findfirst(const char *fname, int attr, fdata_t *fdata)
{
    if(_dos_findfirst((char *)fname, attr, fdata))
    {   file_error(fname);
        return -1;
    }
    else
        return 0;
}

/* Get only the file attribute, store in attribute field of fdata */
int x_get_fattr(const char *fname, fdata_t *fdata)
{
    unsigned fattr;
    int err_code = _dos_getfileattr(fname, &fattr);
    if(err_code)
        file_error(fname);
    fdata->attrib = fattr;
    return err_code;
}

/* Get the time stamp of the file */
int x_get_ftime(int fdesc, fdata_t *fdata)
{
    if(_dos_getftime(fdesc, &fdata->wr_date, &fdata->wr_time))
    {   file_error(NULL);
        return -1;
    }
    return 0;
}

/* Set the file attributes of dest to those of src. If clone_ftime is set,
 *   copy the file time as well */
int x_clone(const char *fname, fdata_t *fdata, int clone_ftime)
{
    if(_dos_setfileattr(fname, fdata->attrib))
    {   file_error(fname);
        return -1;
    }
    else
    {   if(clone_ftime)
        {   int fdesc = x_open(fname);
            if(_dos_setftime(fdesc, fdata->wr_date, fdata->wr_time))
            {   file_error(fname);
                return -1;
            }
            x_close(fdesc);
        }
        return 0;
    }
}

/* Make the file permissions of the file fname normal */
int x_norm_perms(const char *fname) { return 0; }

/* Get the current drive, A = 1, B = 2, .. */
int x_get_drive(void)
{
    unsigned int drive;
    _dos_getdrive(&drive);
    return drive;
}

/* In MSDOS, compare filenames case insensitively */
int fname_cmp(const char far *fname1, const char far *fname2)
{
    return _fstricmp(fname1, fname2);
}

int x_can_write_dir(const char *dir) { return TRUE; }

int fdata_seekable(fdata_t *fdata, int unzipped) { return TRUE; }
off_t fdata_size(fdata_t *fdata){ return fdata->size; }
int fdata_isdir(fdata_t *fdata) { return fdata->attrib & _A_SUBDIR; }
int fdata_isreg(fdata_t *fdata) { return fdata->attrib & (_A_NORMAL |_A_ARCH);}

/* Including this function will suppress automatic wildcard expansion for
 *   the DJGPP version */
char **__crt0_glob_function(char *_argument) { return 0;}

#endif

static void file_error(const char *fname)
{
    fprintf(stderr, EXE_NAME ": %s: %s\r\n", fname ? fname : Fname,
            sys_errlist[errno]);
    if(errno != EACCES && errno != ENOENT && errno != EROFS)
        exit(errno);
    errno = 0;
}

int read_error(const char far *err_msg, const char *fname, int retry)
{
    int response = 0;
    if(!Ignore_errors)
    {   if(!StatLine_Used())
            fprintf(stderr, EXE_NAME ": %s %s\r\n", fname, err_msg);

        else
        {   char *options = retry ? " ^Next file/^Retry " : " ^Next file ";
            response = StatLine_File_error(err_msg, fname ? fname : Fname,
                                           options, retry ? "nNrR" : "nN");
        }
    }
    return response;
}

/* If the file fname can be written to return it. Otherwise prompt for
 * an alternate filename. If no alternate is given, return NULL */
const char Write_err[]  = "Unable to write to ";
const char Backup_err[] = "Unable to make backup for ";

const char *FileOps_Get_writeable(const char *fname, int test_writeable,
                                  char *fname_buf, int fname_max)
{
    int orig_len = 0;

    if(!fname)
        return NULL;

    if(StatLine_Used())
        orig_len = StatLine_Save();

    else
    {   if(test_writeable == -1 || !x_can_write(fname))
        {   fprintf(stderr, EXE_NAME ": %s%s\r\n",
                    test_writeable == -1 ? Backup_err: Write_err, fname);
            return NULL;
        }
        else
            return fname;
    }
    if(Ignore_errors && *Ignore_errors == '+')
        return NULL;

    for( ; ; )
    {   if(   fname && test_writeable && test_writeable != -1
           && x_can_write(fname))
            return fname;
        else
        {   int response;
            if(test_writeable == -1)
                response = StatLine_File_error(Backup_err,
                                               fname, " ^Next file/^Change name ",
                                               "nNnncC");
            else if(test_writeable)
                response = StatLine_File_error(Write_err, fname,
                                               " ^Next file/^Retry/^Change name ",
                                               "nNrRcC");
            else
                response = StatLine_File_error(Write_err, fname,
                                               " ^Next file/^Change name",
                                               "nNnncC");
            if(response < 2)
                return NULL;
            else if(response >= 4)
            {   int fname_len = StatLine_Getstr(fname_buf, fname_max,
                                                "Write changes to file: ",
                                                NULL);
                if(fname_len <= 0)
                    continue;
                fname          = fname_buf;
                test_writeable = TRUE;
                if(x_exists(fname_buf))
                {   response = StatLine_Getch("File exists. Overwrite? (Y/N)",
                                              "yYnN");
                    if(response >= 2)
                        continue;
                }
                else if(x_can_write_dir(fname_buf))
                    break;
            }
        }
    }
    StatLine_Restore(orig_len);
    return fname;
}

