/* dirwalk.c - recurse through a directory tree
 * Copyright (C) 1995-99 Andrew Pipkin (minitrue@pagesz.net)
 * MiniTrue is free software released with no warranty. See COPYING for details
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>

#include "minitrue.h"
#include "dirwalk.h"
#include "fileops.h"
#include "resizer.h"

enum dw_flags {
    DW_HIDDEN = 1, DW_DEPTH = 2, DW_FOLLOW = 4, DW_REGFILE = 8, DW_SORT = 16,
    DW_LC_CONV = 32
};
enum { SKIP_SUBDIRS = 2};

static void path_append(DirWalk *dw, char far *str, int str_len, int append_i);
static int add_fname(DirWalk *dw, char *fname);
static int read_subdir(DirWalk *dw, fdata_t *fdata_ptr, int *got_fdata);
static int open_subdir(DirWalk *dw, DirLevel far *lev_ptr);
static void qsort_dir(FInfo far *first, FInfo far *last, char far *fnames);
static void finfo_swap(FInfo far *f1, FInfo far *f2);
static DirLevel far *descend_subdir(DirWalk *dw, int finfo_i, int new_level);

void DirWalk_init(DirWalk *dw, const char *params, int get_hidden)
{
    dw->path       = NULL;
    dw->path_alloc = 0;
    Resizer_init(&dw->levels, sizeof(DirLevel), 8);
    Resizer_init(&dw->fnames, 1, 8192);
    Resizer_init(&dw->finfo, sizeof(FInfo), 1024);
    Resizer_init(&dw->subdirs, sizeof(int), 512);
#ifndef __MSDOS__
    Resizer_init(&dw->dir_inodes, sizeof(ino_t), 256);
    Resizer_init(&dw->dir_devs,   sizeof(dev_t), 256);
#endif
    dw->flags     = dw->level_min = 0;
    dw->level_max = 0x4000;

    if(get_hidden)
        dw->flags |= DW_HIDDEN;
    if(params != NULL)
    {   for( ; *params != '\0'; ++params)
        {   if(*params == '^')
                dw->flags |= DW_DEPTH;
            else if(*params == '+')
                dw->flags |= DW_FOLLOW;
            else if(*params == '~')
                dw->flags |= DW_REGFILE;
            else if(*params == '#')
                dw->flags |= DW_SORT;
            else if(isdigit(*params))
            {   params = read_int(params, &dw->level_max);
                if(*params == ':')
                {   dw->level_min = dw->level_max;
                    if(isdigit(*++params))
                        params = read_int(params, &dw->level_max);
                    else
                        dw->level_max = 0x4000;
                }
                --params;
            }
            else
                invalid_param('r', *params);
        }
    }
}

/* start walking through the subdirectories, beginning with start_dir,
 * use current directory if start_dir is NULL or \0. basename_upcase
 * will be true if basename has uppercase chars, false otherwise
 * Return 1 on success 0 on failure */
int DirWalk_start(DirWalk *dw, char *start_dir, int basename_upcase)
{
#ifndef __MSDOS__
    dw->path_sep_ch = '/';
    Resizer_truncate(&dw->dir_inodes, 0);
    Resizer_truncate(&dw->dir_devs, 0);
    basename_upcase += 0; /* dummy statement to avoid warning */
#else
    int dir_i;
 /* If long file names not used, set filenames to lower case unless upper
  *   case letters in base directory */
    char *lfn_env = getenv("LFN");
    dw->flags &= ~DW_LC_CONV;
    if((!lfn_env || (*lfn_env != 'y' && *lfn_env != 'Y')) && !basename_upcase)
    {   if(start_dir)
        {   dir_i = 0;
            for(; start_dir[dir_i] && !isupper(start_dir[dir_i]); ++dir_i)
                ;
        }
        if(!start_dir || !start_dir[dir_i])
            dw->flags |= DW_LC_CONV;
    }
 /* Use / as path separator in DOS filenames if used in start_dir,
  *   otherwise use \ */
    dw->path_sep_ch = '\\';
    if(start_dir)
    {   for(dir_i = 0; start_dir[dir_i] && start_dir[dir_i] != '/'; ++dir_i)
            ;
        if(start_dir[dir_i] == '/')
            dw->path_sep_ch = '/';
    }
#endif
    dw->level     = 0;
    Resizer_truncate(&dw->fnames, 0);
    Resizer_truncate(&dw->finfo, 0);
    dw->path_skip = (!start_dir || !start_dir[0]) ? 2 : 0;
    add_fname(dw, (start_dir && start_dir[0]) ? start_dir : ".");
    return (int)(descend_subdir(dw, 0, 0));
}

/* Append the name whose index in the file info array is finfo_i to the path
 * name, then open the subdirectory
 * Return a pointer to the new directory level if successful, on failure
 * return a pointer to the previous level, NULL if failure and level is 0*/
static DirLevel far *descend_subdir(DirWalk *dw, int finfo_i, int new_level)
{
    DirLevel far *lev = Resizer_ptr(&dw->levels, new_level);
    FInfo far *finfo  = Resizer_ptr(&dw->finfo, finfo_i);
 /* Start of subdirectory will be at previous levels basename */
    int subdir_i      = new_level ? lev[-1].base_name_i : 0;
    char far *fname   = Resizer_ptr(&dw->fnames, finfo->name_i);
    path_append(dw, fname, finfo->name_len, subdir_i);

 /* Append path separator char to end if one not already present */
    if(!(path_sep_cH(dw->path[dw->path_len - 1])))
        path_append(dw, &dw->path_sep_ch, 1, dw->path_len);

 /* Add NUL to end set basename start to location of NUL */
    lev->base_name_i  = dw->path_len;
    lev->finfo_i      = lev->finfo_start  = Resizer_size(&dw->finfo);
    lev->subdir_i     = lev->subdir_start = Resizer_size(&dw->subdirs);
    lev->fnames_start = Resizer_size(&dw->fnames);
    lev->done         = FALSE;
    dw->prev_subdir_i = finfo_i;

 /* On failure, return pointer to previous level, NULL if new_level = 0 */
    if(new_level > dw->level_max || !open_subdir(dw, lev))
        lev = new_level ? Resizer_ptr(&dw->levels, dw->level) : NULL;
 /* On success, set the current directory level to the new level */
    else
    {   dw->level = new_level;
        if(dw->flags & DW_SORT)
            qsort_dir(Resizer_ptr(&dw->finfo, lev->finfo_i),
                      Resizer_ptr(&dw->finfo, Resizer_size(&dw->finfo) - 1),
                      Resizer_ptr(&dw->fnames, 0));
    }
    return lev;
}

/* Append the string str to the path at the index append_i, allocate more
 *  memory for the path if neccessary and adjust the path length to the end
 *  of the appended text */
static void path_append(DirWalk *dw, char far *str, int str_len, int append_i)
{
    dw->path_len = append_i + str_len;
    if(dw->path_alloc < dw->path_len + 4)
        dw->path = x_realloc(dw->path, dw->path_alloc += 128);
    _fmemcpy(dw->path + append_i, str, str_len);
    dw->path[dw->path_len] = '\0';
}

/* Sort the filenames belonging to a read directory using the qsort
 *  algorithm. Cannot use the library function because the filename array
 *  is required in addition to the array boundaries */
static void qsort_dir(FInfo far *first, FInfo far *last, char far *fnames)
{
    FInfo far *ptr, far *partition = first;
    finfo_swap(first, first + ((FInfo near *)last - (FInfo near *)first)/2);

    for(ptr = first + 1; ptr <= last; ++ptr)
    {   if(fname_cmp(&fnames[first->name_i], &fnames[ptr->name_i]) > 0)
            finfo_swap(++partition, ptr);
    }
    if(first < partition)
    {   finfo_swap(first, partition);
        qsort_dir(first, partition - 1, fnames);
    }
    if(partition + 1 < last)
        qsort_dir(partition + 1, last, fnames);
}

/* Swap file info structures */
static void finfo_swap(FInfo far *f1, FInfo far *f2)
{
    FInfo temp;
    _fmemcpy(&temp, f1, sizeof(FInfo));
    _fmemcpy(f1, f2, sizeof(FInfo));
    _fmemcpy(f2, &temp, sizeof(FInfo));
}

/* Return next file in directory if available, NULL if at end. If the file
 * data was obtained, set fdata_pptr to point to it. Set fdata_pptr to NULL
 * if file data was not obtained */
char *DirWalk_next(DirWalk *dw, fdata_t *fdata_ptr, int *got_fdata)
{
 /* If at end of current directory, go to lower directory if depth-first
  * searching, otherwise see if there are subdirectories in the subdir
  * buffer before moving to a lower level */
    while(dw->level >= 0 && !read_subdir(dw, fdata_ptr, got_fdata))
    {   DirLevel far *lev = Resizer_ptr(&dw->levels, dw->level);
        if(!(dw->flags & DW_DEPTH))
        {   if(lev->subdir_i < Resizer_size(&dw->subdirs))
            {   int far *fi_ptr = Resizer_ptr(&dw->subdirs, lev->subdir_i++);
                if(   lev->done != SKIP_SUBDIRS
                   && descend_subdir(dw, *fi_ptr, dw->level + 1))
                    continue;
            }
        }
     /* If at end of directory, reuse memory allocated for level */
        Resizer_truncate(&dw->subdirs, lev->subdir_start);
        Resizer_truncate(&dw->fnames, lev->fnames_start);
        Resizer_truncate(&dw->finfo, lev->finfo_start);
        --dw->level;
    }
    return dw->level >= 0 ? dw->path + dw->path_skip : NULL;
}

/* Copy the filename info the filename buffer and add the corresponding
 * file info structure to the file info array. return the file info index */
static int add_fname(DirWalk *dw, char *fname)
{
    FInfo finfo;
    int fname_len = finfo.name_len = strlen(fname), finfo_i, fname_i;
 /* Convert filename to lower case if desired */
    if(dw->flags & DW_LC_CONV)
    {   for(fname_i = 0; fname_i < fname_len; ++fname_i)
            fname[fname_i] = tolower(fname[fname_i]);
    }
    finfo.name_i  = Resizer_append(&dw->fnames, fname, fname_len + 1);
    finfo_i       = Resizer_append(&dw->finfo, &finfo, 1);
    return finfo_i;
}

#ifdef __MSDOS__
#include <dos.h>
/* Open the subdirectory corresponding to the level pointed to by lev_ptr */
static int open_subdir(DirWalk *dw, DirLevel far *lev_ptr)
{
    int attr       = _A_NORMAL | _A_ARCH | _A_RDONLY;
    fdata_t fdata;

    if(!(dw->flags & DW_REGFILE))
        attr |= _A_SYSTEM;
    if(dw->level <= dw->level_max)
        attr |= _A_SUBDIR;
    if(dw->flags & DW_HIDDEN)
        attr |= _A_HIDDEN; /* _A_HIDDEN; */

    path_append(dw, "*.*", 3, dw->path_len);

    if(x_findfirst(dw->path, attr, &fdata))
        return FALSE;

 /* Skip over . and .. , will be present in all directories except root */
    do {
        FInfo far *finfo_ptr;
        if(   fdata.name[0] == '.'
           && (!fdata.name[1] || (fdata.name[1] == '.' && !fdata.name[2])))
            continue;

     /* Write file size and attribute to finfo array in addition to filename*/
        finfo_ptr = Resizer_ptr(&dw->finfo, add_fname(dw, fdata.name));
        finfo_ptr->attrib = fdata.attrib;
        finfo_ptr->size   = fdata.size;
        finfo_ptr->date   = fdata.wr_date;
        finfo_ptr->time   = fdata.wr_time;

    }while(!_dos_findnext(&fdata));

    return TRUE;
}

static int read_subdir(DirWalk *dw, fdata_t *fdata_ptr, int *got_fdata)
{
    DirLevel far *lev_ptr  = Resizer_ptr(&dw->levels, dw->level);

    while(lev_ptr->finfo_i < Resizer_size(&dw->finfo))
    {   FInfo far *finfo   = Resizer_ptr(&dw->finfo, lev_ptr->finfo_i);
        char far *fname    = Resizer_ptr(&dw->fnames, finfo->name_i);
        ++lev_ptr->finfo_i;
     /* Append basename to path */
        path_append(dw, fname, finfo->name_len, lev_ptr->base_name_i);
        if((finfo->attrib & _A_SUBDIR) && lev_ptr->done != SKIP_SUBDIRS)
        {   int finfo_i = lev_ptr->finfo_i - 1;
            if(dw->flags & DW_DEPTH)
                lev_ptr = descend_subdir(dw, finfo_i, dw->level + 1);
            else
                Resizer_append(&dw->subdirs, &finfo_i, 1);
            continue;
        }
        else if(dw->level >= dw->level_min && !lev_ptr->done)
        {   *got_fdata = GOT_FDATA;
            fdata_ptr->attrib  = finfo->attrib;
            fdata_ptr->size    = finfo->size;
            fdata_ptr->wr_date = finfo->date;
            fdata_ptr->wr_time = finfo->time;
            return TRUE;
        }
    }
    return FALSE;
}

#else
#include <unistd.h>
#include <dirent.h>
#include <sys/stat.h>
/* copy the pertinent portions of the stat structure to the file info
 * structure at index finfo_i */
static void copy_stat_parts(FInfo far *finfo_ptr, struct stat *st)
{
    finfo_ptr->nlinks = st->st_nlink;
    finfo_ptr->inode  = st->st_ino;
    finfo_ptr->dev    = st->st_dev;
}

/* Open directory, store all filenames in directory in the levels fnames
 * array, and then close the directory. Return TRUE if directory successfully
 * read, FALSE on failure */
static int open_subdir(DirWalk *dw, DirLevel far *lev_ptr)
{
    struct dirent *dir_ent;
    DIR *dir             = x_opendir(dw->path);
    FInfo far *finfo_ptr = Resizer_ptr(&dw->finfo, dw->prev_subdir_i);
    if(!dir)
        return FALSE;

 /* If directory starting directory, need to stat to get pertinent info */
    if(!dw->prev_subdir_i)
    {   struct stat st;
        if(x_stat(dw->path, &st))
            return FALSE;
        copy_stat_parts(finfo_ptr, &st);
    }
 /* If not following symlinks, # of subdirs in directory will be number of
  *  links to it, subtract 2 to skip . and .. */
    if(!(dw->flags & DW_FOLLOW))
        lev_ptr->subdirs_left = finfo_ptr->nlinks - 2;
 /* If following symlinks, check inodes of traversed directories to insure
  * that new directory has not been traversed */
    else
    {   ino_t far *inodes = Resizer_ptr(&dw->dir_inodes, 0);
        dev_t far *devs   = Resizer_ptr(&dw->dir_devs, 0);
        int i, ndirs      = Resizer_size(&dw->dir_inodes);
        for(i = 0; i < ndirs; ++i)
        {   if(inodes[i] == finfo_ptr->inode && devs[i] == finfo_ptr->dev)
                return FALSE;
        }
     /* Add inode & device numbers of current dir to arrays */
        Resizer_append(&dw->dir_inodes, &finfo_ptr->inode, 1);
        Resizer_append(&dw->dir_devs, &finfo_ptr->dev, 1);
        lev_ptr->subdirs_left = -1;
    }
 /* read in all entries in dir, (except . and ..) and store in fnames
  * resizeable array */
    while((dir_ent = readdir(dir)) != NULL)
    {   char *fname = dir_ent->d_name;
     /* Skip over . and .. */
        if(*fname == '.')
        {   if(!fname[1] || (fname[1] == '.' || !fname[2]))
                continue;
        }
        add_fname(dw, fname);
    }
    closedir(dir);
    return TRUE;
}

/* If more names in current directory, append filename to path and return 1
 * return 0 at end of dir. If got data is set to a non-zero number, fdata_ptr
 *  will contain the results of a stat call */
static int read_subdir(DirWalk *dw, fdata_t *fdata_ptr, int *got_fdata)
{
    DirLevel far *lev_ptr  = Resizer_ptr(&dw->levels, dw->level);

    while(lev_ptr->finfo_i < Resizer_size(&dw->finfo))
    {   FInfo far *finfo   = Resizer_ptr(&dw->finfo, lev_ptr->finfo_i);
        char far *fname    = Resizer_ptr(&dw->fnames, finfo->name_i);
        int skip_file      = (*fname == '.' && !(dw->flags & DW_HIDDEN));
        ++lev_ptr->finfo_i;

     /* Append base name to path before statting */
        path_append(dw, fname, finfo->name_len, lev_ptr->base_name_i);

     /* Need to stat if file might possibly be subdirectory */
        if(lev_ptr->subdirs_left && lev_ptr->done != SKIP_SUBDIRS)
        {/* Ignore symbolic links if link following not desired */
            if(!(dw->flags & DW_FOLLOW))
            {   if(x_lstat(dw->path, fdata_ptr) || S_ISLNK(fdata_ptr->st_mode))
                    continue;
            }
            else if(x_stat(dw->path, fdata_ptr))
                continue;
            *got_fdata = GOT_FDATA;

         /* If directory, copy relevant file information into file
          *   information buffer, then store file index in subdir buffer
          *   if breadth first, descend into subdir if search depth first */
            if(S_ISDIR(fdata_ptr->st_mode))
            {   --lev_ptr->subdirs_left;
                if(!skip_file)
                {   int finfo_i = lev_ptr->finfo_i - 1;
                    copy_stat_parts(finfo, fdata_ptr);
                    if(dw->flags & DW_DEPTH)
                        lev_ptr = descend_subdir(dw, finfo_i, dw->level + 1);
                    else
                        Resizer_append(&dw->subdirs, &finfo_i, 1);
                }
                continue;
            }
        }
     /* No need to stat if file not subdir, indicate that file not statted */
        else
            *got_fdata = FALSE;

     /* Return if file valid non-directory */
        if(dw->level >= dw->level_min && !skip_file && !lev_ptr->done)
            return TRUE;
    }
    return FALSE;
}
#endif

/* Do not read any more files in the current directory. If skip_sub_dirs
 * is set, skip all subdirectories in the current subdirectory as well */
void DirWalk_finish(DirWalk *dw, int skip_sub_dirs)
{
    DirLevel far *lev_ptr  = Resizer_ptr(&dw->levels, dw->level);
    lev_ptr->done = (skip_sub_dirs) ? SKIP_SUBDIRS : TRUE;
}

/* Return non-zero if only regular files should be read */
int DirWalk_only_reg(DirWalk *dw) { return (dw->flags & DW_REGFILE); }
void DirWalk_kill(DirWalk *dw)
{
    free(dw->path);
    Resizer_kill(&dw->subdirs);
    Resizer_kill(&dw->levels);
    Resizer_kill(&dw->fnames);
    Resizer_kill(&dw->finfo);
#ifndef __MSDOS__
    Resizer_kill(&dw->dir_inodes);
    Resizer_kill(&dw->dir_devs);
#endif

}
#ifdef DIRWALK_TEST
int main(int argc, char * *argv)
{
    DirWalk dw;
    char *fname, *params = NULL;
    int arg_i = 1;
    if(argc > 1 && argv[1][0] == '-')
    {   params = argv[1] + 1;
        arg_i = 2;
    }
    DirWalk_init(&dw, params, FALSE);
    do
    {   fdata_t fdata;
        int statted;
        if(DirWalk_start(&dw, argv[arg_i]))
        {   while((fname = DirWalk_next(&dw, &fdata, &statted)) != NULL)
                puts(fname);
            DirWalk_kill(&dw);
        }
    } while(++arg_i < argc);
    return 0;
}
#endif
