/* console.c - routines for console I/O
 * Copyright (C) 1995-99 Andrew Pipkin (minitrue@pagesz.net)
 * MiniTrue is free software released with no warranty. See COPYING for details
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "console.h"
#include "attribs.h"
#include "minitrue.h"
#include "ansiesc.h"

typedef struct
{   int mode;    /* video mode */
    int cols;    /* number of columns on screen */
    int rows;    /* number or rows */
    int curx;    /* horizontal cursor position */
    int cury;    /* vertical cursor position */
    attr_t attr; /* current text attribute */
} Scrn;

Scrn Screen       = {-1, -1, -1, -1, -1, -1};
Scrn Orig_screen  = {-1, -1, -1, -1, -1, -1};
static ScreenPtr Capture_buf; /* Buffer containing contents of captured
                               * screen */
static int Raw_mode;
static int Written;           /* Set if screen has been written to */
static char *Term_buf;        /* Buffer for termcap data */

enum term_caps
{   AUTO_WRAP, MOVE_CURS, CLEAR_EOL, SCROLL_UP, SCROLL_DOWN, SCROLL_REGION,
    TERM_INIT, TERM_RESET, NORM_START, REV_START, BOLD_START, UL_START,
    KEYPAD_INIT, KEYPAD_RESET, KEYPAD_ESCS, UP_ESC = KEYPAD_ESCS, DOWN_ESC,
    LEFT_ESC, RIGHT_ESC, PGUP_ESC, PGDN_ESC,
    HOME_ESC, END_ESC, INS_ESC, DEL_ESC,
    NTERMCAPS, TERM_BUF_LEN = 2048, NKEYPAD = 10
};

/* Termcap for ANSI.SYS */
static char *Term_caps[NTERMCAPS] =
{   (char *)TRUE, "\033[%i%d;%dH", "\033[K", NULL, "\n", NULL,
    NULL, NULL, "\033[0m", "\033[7m", "\033[1m", "\033[4m" };

static char *Cap_names[] =
{   "am", "cm", "ce", "sr", "sf", "cs",
    "ti", "te", "me", "mr", "md", "us",
    "ks", "ke", "ku", "kd",
    "kl", "kr", "kP", "kN",
    "kh", "@7", "kI", "kD",
};

#ifdef __MSDOS__
static int Tty = FALSE;       /* default in MSDOS is to write directly to
                               * video memory */
#else
static int Tty = TRUE;        /* in UNIX, must write to terminal driver */
#endif

static int Mono = FALSE;      /* Set if monochrome monitor is to be used */
static int Pad_eol = FALSE;  /* if set, pad to end of screen line with
                               * spaces if line shorter than screen line */
static int Dumb_tty = FALSE;  /* Set if dumb terminal is used */

static void get_screen_info(Scrn *screen_ptr);
static void set_term(void);
static void reset_term(void);
static void vmem_writeln(ScreenPtr line_buf, int line);
static void write_esc_seq(int cap);
static void reset_cursor(void);

void Console_Init(const char *tty_params)
{
    const char *colors = NULL;
    if(tty_params != NULL)
    {   if(*tty_params == '+')
        {   if(!Tty)
                Tty = TRUE;
            else
                Pad_eol = TRUE;
            ++tty_params;
        }
        if(!strcmp(tty_params, ":mono"))
            Mono = TRUE;

        else if(*tty_params == ':')
            colors = tty_params;
        else if(*tty_params != '\0')
        {   invalid_param('v', '\0');
            return;
        }
    }
    set_term();
    Attribs_Init(Console_Is_color(), Tty, colors);
    memcpy(&Orig_screen, &Screen, sizeof(Scrn));

    setvbuf(stdout, NULL, _IONBF, 0);
}

/* Determine the new size of the screen, return TRUE if the screen size
 * has changed, FALSE otherwise */
void Console_Resize(void)
{
    get_screen_info(&Screen);
}

int Console_Rows(void)   {  return Screen.rows; }
int Console_Cols(void)   {  return Screen.cols; }
/* Return type of tty */
int Console_Tty(void) { return Tty; }
int Console_Is_color(void) { return !Mono; }

/* Write nchars of line_buf starting at start_col to row row of the
 * screen. If clear_eol set, clear the remainder of the line */
void Console_Write(ScreenPtr line_buf, int row, int start_col,
                   int nchars, int clear_eol)
{
    char buf[256], *buf_ptr = buf;
    attr_t attr;
    int col = start_col, end_col = start_col + nchars, next_row;
    Written = TRUE;
 /* Write directly to video memory if not tty */
    if(!Tty)
    {   vmem_writeln(line_buf, row);
        return;
    }
    if(   clear_eol
       && (end_col + 6 > Screen.cols || Pad_eol || !Term_caps[CLEAR_EOL]))
    {   end_col   = Screen.cols;
        clear_eol = FALSE;
    }
    Console_Move_cursor(col, row);

    for(; col < end_col ; ++col)
    {   if(buf_ptr > &buf[224])
        {   fwrite(buf, 1, buf_ptr - buf, stdout);
            buf_ptr = buf;
        }
     /* Print escape sequence corresponding to attribute if attr changed*/
        if((attr = line_buf[col].attr) != Screen.attr)
        {   char far *esc_seq;

            if(attr >= 0)
                esc_seq = Ansi_esc[(int)attr];
            else
                esc_seq = Term_caps[NORM_START + (attr - NORM_ATTR) ];

            if(esc_seq && !Dumb_tty)
            {   if(attr < 0)
                    buf_ptr = copy_str(buf_ptr, Term_caps[NORM_START]);

                _fstrcpy(buf_ptr, esc_seq);
                buf_ptr    += _fstrlen(esc_seq);
                Screen.attr = attr;
            }
        }
        *buf_ptr++ = line_buf[col].ch;
    }
 /* If line does not extend to screen end, clear remainder of line */
    if(clear_eol)
        buf_ptr = copy_str(buf_ptr, Term_caps[CLEAR_EOL]);

 /* Set next row if cursor should move to start of next row after write */
    next_row = (clear_eol || end_col == Screen.cols) && row < Screen.rows - 1;

 /* Need to move to next line if terminal does not wrap to next line
  * automatically or line has fewer chars than screen columns */
    if(next_row && (end_col < Screen.cols || !Term_caps[AUTO_WRAP]))
        buf_ptr = copy_str(buf_ptr, "\r\n");

 /* If only one char is to be written, use putchar to write it because
  * a single char outputed with fwrite on an xterm or telnet sometimes
  * is not displayed immediately */
    if(buf_ptr == buf + 1)
        putchar(*buf);
    else
        fwrite(buf, 1, buf_ptr - buf, stdout);

 /* Set cursor to position after last character written */
    Screen.curx = next_row ? 0 : end_col;
    Screen.cury = row + next_row;
}

/* Write an entire line to the screen */
void Console_Writeln(ScreenPtr line_buf, int row, int line_len)
{
    Console_Write(line_buf, row, 0, line_len, TRUE);
}

/* Scroll nlines between top_row and bottom_row if terminal has escape
 * sequence for scrolling, return number of lines scrolled */
int Console_Scroll(int nlines, int top_row, int bottom_row)
{
    int nscrolled = 0;
 /* Only scroll if escape sequence is ANSI */
    if(   !Term_caps[SCROLL_REGION]
       || strcmp("\033[%i%d;%dr", Term_caps[SCROLL_REGION]))
        return 0;

 /* Set scroll region */
    printf("\033[%d;%dr", top_row + 1, bottom_row);
    reset_cursor();
    if(nlines < 0 && Term_caps[SCROLL_UP])
    {   Console_Move_cursor(0, top_row);
        for( ; nscrolled > nlines; --nscrolled)
            write_esc_seq(SCROLL_UP);
    }
    else if(nlines > 0 && Term_caps[SCROLL_DOWN])
    {   Console_Move_cursor(0, bottom_row - 1);
        for( ; nscrolled < nlines; ++nscrolled)
            write_esc_seq(SCROLL_DOWN);
    }
 /* Reset scroll region to be entire screen */
    printf("\033[%d;%dr", 1, Screen.rows);
    reset_cursor();
    Screen.attr = UNKNOWN_ATTR;
    return nscrolled;
}
/* Force next cursor movement command to do move cursor even if the
 * cursor is believed to be in desired position */
static void reset_cursor(void) { Screen.curx = Screen.cury = -1; }

/* Write the escape sequence corresponding to the terminal capability cap_name*/
static void write_esc_seq(int cap_name)
{
    if(Term_caps[cap_name] && !Dumb_tty)
        fputs(Term_caps[cap_name], stdout);
}

/* Restore the original contents of the screen if restore screen is set
 * If console has been initialized with TERM_INIT, TERM_RESET must be
 * called, if screen restoration not desired, return FALSE to indicate
 * that the program screen should be redrawn before exiting */
int Console_Restore(int restore_screen)
{
    if(!Written)
        return TRUE;

    reset_term();

    if(restore_screen && Capture_buf && !Tty)
    {   int line;
        for(line = 0; line < Orig_screen.rows; ++line)
            vmem_writeln(&Capture_buf[line * Orig_screen.cols], line);
        Console_Move_cursor(Orig_screen.curx ? Orig_screen.curx - 1: 0,
                            Orig_screen.cury);
    }
    else if(Term_caps[TERM_RESET])
    {   write_esc_seq(TERM_RESET);
        if(!restore_screen)
            return FALSE;
    }
    else
    {   Console_Move_cursor(0, Orig_screen.rows - 1);
        if(Tty)
        {   if(Mono && Term_caps[NORM_START])
                write_esc_seq(NORM_START);
            else
                printf("\033[;;m\r\n");
        }
    }
    Written = FALSE;
    return TRUE;
}

/* Reset terminal to original state & free up console resources */
void Console_Kill(void)
{
 /* If rows -1, assume that console not initialized so nothing to restore */
    if(Screen.rows == -1)
        return;

    if(Written && Tty)
    {   if(Mono && Term_caps[NORM_START])
            write_esc_seq(NORM_START);
        else
            printf("\033[;;m\r\n");
    }

    reset_term();

    farfree(Capture_buf);
    free(Term_buf);
}

/* ========================== MS-DOS Specific Functions ================ */
#ifdef __MSDOS__
static void bios_move_cursor(int x, int y);
void Console_Move_cursor(int x, int y)
{
    if(Dumb_tty)
        return;

 /* If already at desired position, return */
    if(Screen.curx == x && Screen.cury == y)
        return;

    if(x >= Screen.cols)
        x = Screen.cols - 1;
    if(y >= Screen.rows)
        y = Screen.rows - 1;

    if(!Tty)
        bios_move_cursor(x, y);
    else
        printf("\033[%d;%dH", y + 1, x + 1);

    Screen.curx = x;
    Screen.cury = y;
    Screen.attr = UNKNOWN_ATTR;
}
enum {KPD_START = 71, KPD_END = 83};
static int get_keypad(char ch)
{
    static int keypad[] = {HOME_KEY, UP_KEY, PGUP_KEY, 0, LEFT_KEY, 0,
                            RIGHT_KEY, 0, END_KEY, DOWN_KEY, PGDN_KEY,
                            INS_KEY, DEL_KEY };

    return (KPD_START <= ch && ch <= KPD_END) ? keypad[ch - KPD_START] : 0;
}

/* =================== Borland C Specific Functions ==================== */
/* Must use BIOS calls instead of conio routines to get console information
 *  because conio routines do not handle non-standard screen sizes */

#ifdef __DOS16__
#include <dos.h>
/* This function returns a non-zero # if a vga or ega card is detected 0
 *  otherwise */
static unsigned char have_vga_ega(void)
{
    asm mov bl, 0x10
    asm mov ah, 0x12
    asm int 0x10
    asm sub bl, 0x10
    return _BL;
}

/* This returns the text mode and sets the number of rows and colums */
static void get_screen_info(Scrn *screen)
{
    char mode, cols, curx, cury;
    asm mov ah, 0xf;
    asm int 0x10;
 /* Need to store in intermediate values because converting AH/AL to integer
  *  clobbers other register half */
    mode = _AL;
    cols = _AH;
    screen->mode = mode;
    screen->cols = cols;
    screen->rows = have_vga_ega() ? *(char far *)MK_FP(0x40, 0x84) + 1 : 25;

 /* Get cursor position */
    asm mov ah, 3
    asm xor bh, bh
    asm int 0x10;

    curx = _DL;
    cury = _DH;
    screen->curx = curx;
    screen->cury = cury;
}

/* Return a pointer to the start of the video memory for a line */
static ScreenPtr vmem_line(int line)
{
/* Monochrome screens (mode 7) start at segments 0xb000, video memory for
 *  all other modes start at 0xb800 */
    ScreenPtr vmem = Screen.mode != 7 ? MK_FP(0xb800, 0) : MK_FP(0xb000, 0);
    return &vmem[line * Screen.cols];
}

/* Write line directly to video memory */
static void vmem_writeln(ScreenPtr line_buf, int line)
{
    _fmemcpy(vmem_line(line), line_buf, Screen.cols * sizeof(CharAttr));
}

/* move cursor using bios routines */
static void bios_move_cursor(int x, int y)
{
    unsigned char col = (unsigned char)x;
    unsigned char row = (unsigned char)y;
    asm mov dh, row;
    asm mov dl, col;
    asm xor bh, bh;
    asm mov ah, 2;
    asm int 0x10;
}

#include <signal.h>
int Console_Get_key(void)
{
    int key;
asm mov ax, 0x0700
asm int 0x21;
    if((key = (int)_AL) == '\003')
        raise(SIGINT);
 /* if key '\0' keypad key pressed, read next key to determine keypad char */
    return key ? key : get_keypad(Console_Get_key());
}

static void set_term(void)
{
    int screen_size;
    Console_Resize();
    screen_size = Screen.rows * Screen.cols * sizeof(CharAttr);

    freopen("CON", "rb", stdin);
    Capture_buf     = x_farrealloc(Capture_buf, screen_size);
    _fmemcpy(Capture_buf, vmem_line(0), screen_size);
}
static void reset_term(void) {}

/* ====================== DJGPP specific functions ======================= */

#elif defined (__DJGPP__)
#include <conio.h>
#include <pc.h>

static void get_screen_info(Scrn *screen)
{
    struct text_info ti;
    gettextinfo(&ti);
    screen->mode = ti.currmode;
    screen->rows = ti.screenheight;
    screen->cols = ti.screenwidth;
    screen->curx = ti.curx - 1;
    screen->cury = ti.cury - 1;
    screen->attr = ti.normattr;
}

static void vmem_writeln(ScreenPtr line_buf, int line)
{
    ScreenUpdateLine(line_buf, line);
}

static void bios_move_cursor(int x, int y) { ScreenSetCursor(y, x); }
#include <signal.h>
static void set_term()
{
    int screen_size;
    Console_Resize();
    screen_size = Screen.rows * Screen.cols * sizeof(CharAttr);
    freopen("CON", "rb", stdin);
    Capture_buf     = x_farrealloc(Capture_buf, screen_size);
    ScreenRetrieve(Capture_buf);
}
static void reset_term() {}

int Console_Get_key()
{
    int ch = getch();
    if(ch == '\003')
        raise(SIGINT);

 /* if key '\0' keypad key pressed, read next key to determine keypad char */
    return ch ? ch : get_keypad(Console_Get_key());
}
#endif /* __MSDOS__ */

/* ====================== Unix-specific functions ======================== */

#else
#include <termcap.h>
#include <sys/ioctl.h>
#include <unistd.h>
#include <termios.h>
#include <fcntl.h>

/* put terminal in single-character, raw output mode */
static struct termios Term_orig;
static int TTY_fd;

void set_term(void)
{
    struct termios term_mod;
    int cap_i, got_term;
    char *term_name = getenv("TERM"), *term_buf_ptr;

/* get termcap information - first half of buffer will store tgetent
 *  result, 2nd half will store specific capabilities */
    if(!Term_buf)
        Term_buf = x_malloc(2 * TERM_BUF_LEN);
    if(   !term_name || !(got_term = tgetent(Term_buf, term_name))
       || got_term == -1)
    {   free(Term_buf);
        Term_buf = NULL;
    }
    else
    {   term_buf_ptr = &Term_buf[TERM_BUF_LEN];
        Term_caps[AUTO_WRAP] = tgetflag("am") ? (char *)1 : NULL;
        for(cap_i = 1; cap_i < NTERMCAPS; ++cap_i)
            Term_caps[cap_i] = tgetstr(Cap_names[cap_i], &term_buf_ptr);
    }
    TTY_fd = open("/dev/tty", O_RDONLY);
    if(TTY_fd == -1)
        TTY_fd = 2;

    tcgetattr(TTY_fd, &Term_orig);
    memcpy(&term_mod, &Term_orig, sizeof(struct termios));

 /* Disable canonical mode, and set buffer size to 1 byte */
    term_mod.c_oflag &= ~OPOST;
    term_mod.c_lflag &= ~(ECHO|ECHONL|ICANON|IEXTEN);

    term_mod.c_cc[VTIME] = 0;
    term_mod.c_cc[VMIN]  = 1;

    tcsetattr(TTY_fd, TCSANOW, &term_mod);
    Raw_mode = TRUE;

 /* If terminal lacks cursor-moving capability, treat terminal as raw */
    if(!Term_caps[MOVE_CURS])
        Dumb_tty = TRUE;

 /* If cursor movement commands not those of an ANSI terminal, set terminal
  * type to mono because ANSI color capabilities cannot be used */
    else if(!Mono && strcmp(Term_caps[MOVE_CURS], "\033[%i%d;%dH")
            && strcmp(Term_caps[MOVE_CURS], "\033[%i%2;%2H"))
        Mono = TRUE;

    Console_Resize();
    write_esc_seq(TERM_INIT);
    write_esc_seq(KEYPAD_INIT);
}

static void get_screen_info(Scrn *screen)
{
    screen->rows = 0;
    screen->cols = 0;

/* First try ioctl to determine window size */
#ifdef TIOCGWINSZ
    {   struct winsize ws;
        if(ioctl(1, TIOCGWINSZ, &ws) == 0)
        {   screen->cols = ws.ws_col;
            screen->rows = ws.ws_row;
        }
    }
#elif defined TIOCGSIZE
    {   struct ttysize ts;
        if(ioctl(1, TIOCGSIZE, &ts) == 0)
        {   screen->cols = ts.cols;
            screen->rows = ts.lines;
        }
    }
#endif /* TIOCGWINSZ */
 /* Now try reading screen parameters from environment */
    if(!screen->cols || !screen->rows )
    {   char *env_ptr;
        if((env_ptr = getenv("LINES")) != NULL)
            screen->rows = atoi(env_ptr);
        if((env_ptr = getenv("COLUMNS")) != NULL)
            screen->cols = atoi(env_ptr);
    }
 /* Look in termcap */
    if((!screen->cols || !screen->rows) && Term_buf)
    {   screen->cols = tgetnum("co");
        screen->rows = tgetnum("li");
    }
 /* Use defaults if everything fails */
    if(!screen->cols || !screen->rows)
    {   screen->rows = 24;
        screen->cols = 80;
    }
}

int Console_Get_key(void)
{
    char ch, buf[32];  /* buffer used to compare escape sequences */
    int buf_i = 0, esc_i, esc_seq_match;

    do
    {   esc_seq_match = FALSE;
        read(TTY_fd, &ch, sizeof(char));
        buf[buf_i++] = ch;

     /* Compare chars read so far with keypad escape sequences,
      * if key not part of sequence, return key, otherwise examine more chars*/
        for(esc_i = 0; esc_i < NKEYPAD; ++esc_i)
        {   char *esc_seq = Term_caps[KEYPAD_ESCS + esc_i];
            if(esc_seq != NULL && !memcmp(buf, esc_seq, buf_i))
            {/* If entire escape sequence matched, return the keys value */
                if(esc_seq[buf_i] == '\0')
                    return KEYPAD_OFF + esc_i;
                else
                    esc_seq_match = TRUE;
                break;
            }
        }
    }while(esc_seq_match);
    return ch;
}

/* Move the cursor after tgoto handles the parameters in the cursor
 * movement termcap string */
void Console_Move_cursor(int x, int y)
{
 /* Dumb terminals cannot move cursor */
    if(Dumb_tty)
        return;

 /* No need to move cursor if already at desired position */
    if(Screen.curx == x && Screen.cury == y)
        return;

/* Make sure cursor position does not exceed screen boundaries */
    if(x >= Screen.cols)
        x = Screen.cols - 1;
    if(y >= Screen.rows)
        y = Screen.rows - 1;

    if(Term_caps[MOVE_CURS])
    {   char *esc_seq = tgoto(Term_caps[MOVE_CURS], x, y);
        if(esc_seq)
        {   fputs(esc_seq, stdout);
            Screen.curx = x;
            Screen.cury = y;
            Screen.attr = UNKNOWN_ATTR;
        }
    }
}

void reset_term(void)
{
    write_esc_seq(KEYPAD_RESET);
    if(Raw_mode)
    {   tcsetattr(TTY_fd, TCSANOW, &Term_orig);
        Raw_mode = FALSE;
    }
    return;
}

/* This function should never be invoked, body is just to prevent compiler
 warning for unused variables */
static void vmem_writeln(ScreenPtr line_buf, int line) { line_buf += line; }

#endif /* unix */

    /* Write the character/attribute pair to line at column col */
void LineBuf_putc(ScreenPtr line, int col, char ch, attr_t attr)
{
    if(col < Screen.cols)
    {   ScreenPtr ptr = line + col;
        ptr->ch       = ch;
        ptr->attr     = attr;
    }
}

/* Copy a null-terminated string with attribute attr to a line buffer
 * beginning at column start_col, return the number of chars written */
int LineBuf_puts(ScreenPtr line, int col, const char far *str, attr_t attr)
{
    return LineBuf_put_nchars(line, col, str, attr, INT_MAX);
}

/* Copy at most nchars of a  null-terminated string with attribute attr
 *   to a line buffer beginning at column start_col, return the number
 *   of chars written */
int LineBuf_put_nchars(ScreenPtr line, int col, const char far *str,
                       attr_t attr, int nchars)
{
    const char far *ptr = str;
    if(!nchars)
        return 0;

    for(ptr = str; *ptr && nchars > 0; ++ptr, ++col, --nchars)
    {   if(col == Screen.cols)
            break;
        line[col].ch   = *ptr;
        line[col].attr = attr;
    }
    return (char near *)ptr - (char near *)str;
}

/* Clear the line from column col to the end of the line with attribute attr*/
void LineBuf_clear(ScreenPtr line, int col, attr_t attr)
{
    ScreenPtr ptr = line + col;
    for( ; col < Screen.cols; ++col)
    {   ptr->ch   = ' ';
        ptr->attr = attr;
        ++ptr;
    }
}
