/* buffers.c - interface between the file I/O and searching routines
 * Copyright (C) 1995-99 Andrew Pipkin (minitrue@pagesz.net)
 * MiniTrue is free software released with no warranty. See COPYING for details
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <limits.h>
#include <assert.h>

#include "buffers.h"
#include "minitrue.h"
#include "files.h"
#include "fileops.h"
#include "linenums.h"
#include "charset.h"

static Buffer *In_bufs;     /* Initial input buffer */
enum { OVERWRITE_BUF, OUT_BUF, SCROLL_BUF1, SCROLL_BUF2, NUM_BUFS,
    MMAP_MOD_START = SCROLL_BUF1};
static Buffer *Misc_bufs;
static Buffer Overlap_buf;
static Buffer String_buf;   /* This buffer is used to handle a string as
                             * if it were a file */
static int Scan_buf_i;      /* index of input buffer currently being scanned */
static int Wrap_buf_i;      /* if set, next buf will wrap around to In_bufs */
static int Load_buf_i;      /* Index of buffer currently being loaded */
static int Whole_files;     /* if set, read in whole files at a time
                             * instead of in chunks */
static int In_bufs_alloc;   /* # of input buffers allocated*/
static int In_overlap;      /* true if overlap buffer currently being scanned*/
static int Wrapped;         /* set if bufs wrapped around */
static int Misc_bufs_alloc;

int Buf_loc;                /* Position in buffer currently being scanned */
enum { BUF_START, BUF_MID, BUF_END };

#ifdef __DOS16__ /* In 16-bit DOS, only 32K can be used for input buffers */
static size_t In_buf_mem_left  = 0x8000;
static size_t In_buf_len       = 0x8000;
enum { PAD_LEN = 0 };
#else
static size_t In_buf_mem_left  = INT_MAX;
static size_t In_buf_len       = 0x40000;  /* 256K buffer length */
enum { PAD_LEN = 4096 };
#endif

static int Load_len = 0x4000; /* size of input reads */
static int Scan_len;          /* Length of block to scan */
static int Use_mmap;          /* Set if memmap used to read file */

static LineNums Line_nums;

static off_t Fsize_diff; /* Size of modified file - original file size */

static off_t Nwrote;   /* Number of bytes written to output file/buffer */
static int Written;    /* Set if data written to output files or buffers */
static Files *Files_ptr;

static int Strlen_max; /* maximum length of string or regexp */
static int Overlap_len;

static int File_types; /* Tells whether to get text files, binary files
                          * or both */
enum  { ALL_FILES = 0, TEXT_ONLY, BINARY_ONLY };

static int is_binary(BufPtr buf_ptr);
static int next_in_buf(int buf_i);
static void init_load_buf(int buf_i, off_t off);
static void load_buffer(void);
static void load_whole_file(void);
static size_t alloc_in_buf(BufPtr buf_ptr, size_t len);
static void write_blk(const charf *start, size_t len);
static off_t buf_end_off(Buffer *buf_ptr);
static off_t ptr_to_off(Buffer *buf_ptr, const char far *ptr);
static void shift_start(Buffer *buf_ptr, int n);
static Buffer *load_scroll_buf(off_t off);
static Buffer *load_ahead(off_t off);
static off_t end_off(int find_end);
static BufPtr off_in_buf(BufPtr buf, off_t off);

/* Allocate memory for buffers */
void Buffers_Init(size_t strlen_max, const char *binary_params)
{
    In_bufs          = x_malloc(sizeof(Buffer));
    In_bufs_alloc    = 1;
    Misc_bufs        = x_malloc(sizeof(Buffer) * NUM_BUFS);
    Misc_bufs_alloc  = NUM_BUFS;

 /* If not unable to allocate memory for buffer, cut buffer size in half
  *   (should only happen in 16-bit DOS) also set Wrap_buf_ptr to initial
  *   input buffer */
    for( ; ; )
    {/* Must have two blocks in input buf*/
        if((size_t)Load_len > In_buf_len / 2)
            Load_len = In_buf_len / 2;
        if(alloc_in_buf(In_bufs, In_buf_len))
            break;
        In_buf_mem_left = In_buf_len = In_buf_len / 2;
        Wrap_buf_i      = 1;
    }
 /* Overlap buffer must be at least 400 bytes to allow for strings entered
  * at runtime */
    if(sizeof(int) == 2 && strlen_max > 1024)
        strlen_max = 1024;

    if(strlen_max <= 1024)
    {   int overrun_len = miN(1024, 4 * strlen_max);
        Overlap_len = strlen_max + overrun_len;
        if(Overlap_len < 128)
            Overlap_len = 128 + strlen_max;

        Overlap_buf.start = x_malloc(Overlap_len + strlen_max + overrun_len);
        Overlap_buf.alloc = Overlap_buf.start;
        Strlen_max        = strlen_max;
        Scan_len          = Load_len;
    }
    else
    {   Whole_files = TRUE;
        Scan_len    = INT_MAX;
    }

    Misc_bufs[OUT_BUF].start     = x_farmalloc(3 * Load_len);
    Misc_bufs[SCROLL_BUF1].start = Misc_bufs[OUT_BUF].start + Load_len;
    Misc_bufs[SCROLL_BUF2].start = Misc_bufs[SCROLL_BUF1].start + Load_len;

    LineNums_init(&Line_nums);

    if(!binary_params)
        File_types = ALL_FILES;
    else if(!binary_params[0])
        File_types = TEXT_ONLY;
    else if(binary_params[0] == '+' && !binary_params[1])
        File_types = BINARY_ONLY;
}

/* Bind an open file to the buffers structures, then reset buffers  */
BufPtr Buffers_Reset(Files *files)
{
 /* Initialize other buffers */
    Misc_bufs[SCROLL_BUF1].len = Misc_bufs[SCROLL_BUF2].len = 0;
    Misc_bufs[OUT_BUF].len     = Misc_bufs[OVERWRITE_BUF].len = 0;
    Overlap_buf.len            = 0;
    Misc_bufs[OUT_BUF].off     = 0;

    Files_ptr    = files;
 /* Reset, then load input buffer */
    In_bufs->off = 0;
    In_bufs->len = 0;
    Scan_buf_i   = Load_buf_i = Wrapped = 0;
    Use_mmap     = Files_seekable(files) && Files_Have_mmap();
    init_load_buf(0, 0);
    load_buffer();
    if(Whole_files)
        load_whole_file();
    Buf_loc      = BUF_START;

    Nwrote       = 0;
    Fsize_diff   = 0;
    Written      = FALSE;
    LineNums_reset(&Line_nums);

 /* Check and make sure the file */
    if(File_types != ALL_FILES)
    {   int bin_file = is_binary(&In_bufs[0]);
        if(   (bin_file  && File_types == TEXT_ONLY)
           || (!bin_file && File_types == BINARY_ONLY))
           return NULL;
    }
    return In_bufs;
}

/* Return true if more than BIN_PCNT of the tested characters are binary
 * characters, FALSE otherwise */
enum { TEST_START_I = 1024, NTEST_BYTES = 256, BIN_PCNT = 10 };
static int is_binary(BufPtr buf_ptr)
{
    int binary_bytes = 0, total_bytes, test_i, end_i, ch;

 /* determine where to start test & number of bytes to test, want to test
  * NTEST_BYTES bytes beginning at TEST_START_I index to avoid possibility
  * of text header, but make sure file is large enough */
    if(!buf_ptr->len)
        return FALSE; /* avoid possibility of division by zero */
    else if(buf_ptr->len < NTEST_BYTES)
    {   test_i = 0;
        end_i  = buf_ptr->len;
    }
    else
    {   if(buf_ptr->len < TEST_START_I + NTEST_BYTES)
            test_i = buf_ptr->len - NTEST_BYTES;
        else
            test_i = TEST_START_I;
        end_i = test_i + NTEST_BYTES;
    }
    total_bytes = end_i - test_i;

 /* Consider chars binary if neither printing ASCII chars, \r, \n or \t*/
    for( ; test_i < end_i; ++test_i)
    {   ch = buf_ptr->start[test_i];
        if(!isprint(ch) && ch != '\n' && ch != '\t' && ch != '\r')
            ++binary_bytes;
    }
    return ( (100 * binary_bytes) / total_bytes > BIN_PCNT);
}

/* Load in a buffer, return the start of the buffer and set buf_end to the
 *  end of the buffer and scan_end to the end of the region to search
 *  if a replacement occured at the end of the previous buffer, nrep will
 *  be set to the # of replaced characters at the start of the next buffer */
void Buffers_Load(char * *start, char * *overlap, char * *end, int nrep)
{
    static char *Load_end;
    char *end_ch;  /* points to unused spaced in buffer */
    Buffer *buf_ptr = &In_bufs[Scan_buf_i];

 /* If current buffer not full, read in some more chars into it */
    if(Buf_loc != BUF_END)
    {   if(Buf_loc == BUF_START)
        {   *start  = Load_end = (charn *)buf_ptr->start;
            Buf_loc = BUF_MID;
        }
        else
        {   *start = Load_end - Overlap_len;
            if(!buf_ptr->mmap && Load_end == buf_ptr->start + buf_ptr->len)
                load_buffer();
        }
     /* If reading file in conventional manner, only scan the bytes which have
      * just been read */
        if(!buf_ptr->mmap && !Whole_files)
            Load_end += Load_len;
     /* Otherwise, scan Scan_len bytes, provided that there are enough chars*/
        else if(buf_ptr->end - Load_end < Scan_len)
            Load_end  = (char near *)buf_ptr->end;
        else
            Load_end += Scan_len;

     /* If buffer full, set Buf_loc appropriately */
        if(Load_end == buf_ptr->end)
            Buf_loc = BUF_END;

        *overlap   = Load_end - Overlap_len;
        In_overlap = FALSE;
    }
 /* If current buffer full, set up overlap buffer. First copy last Overlap_len
  * bytes of full buffer into overlap buffer */
    else
    {   char *copy_start = (charn *)buf_ptr->end - Overlap_len;
        Overlap_buf.start = Overlap_buf.alloc;
        *start = memcpy((charn *)Overlap_buf.start, copy_start, Overlap_len);
        Overlap_buf.off = copy_start - (charn *)buf_ptr->start + buf_ptr->off;
        Overlap_buf.len = buf_ptr->len -(copy_start - (charn *)buf_ptr->start);
        *overlap        = (charn *)Overlap_buf.start + Overlap_buf.len;

     /* If not at eof, then look ahead into next buffer,
      * and copy start of next buffer into the gap buffer */
        if(Overlap_buf.len == (size_t)Overlap_len)
        {   int next_buf_i = next_in_buf(Scan_buf_i);
            if(Scan_buf_i == Load_buf_i)
            {   init_load_buf(next_buf_i, buf_end_off(buf_ptr));
                load_buffer();
            }
            buf_ptr          = &In_bufs[Scan_buf_i = next_buf_i];
            Overlap_buf.len += miN((size_t)Strlen_max, buf_ptr->len);
            memcpy(*overlap, (charn *)buf_ptr->start, Strlen_max);
        }
        In_overlap = TRUE;
        Buf_loc    = BUF_START;
        buf_ptr    = &Overlap_buf;
    }
 /* Block out replaced characters */
    if(nrep)
    {   shift_start(buf_ptr, (charn *)buf_ptr->start - *start - nrep);
        *start = (charn *)buf_ptr->start;
    }
    end_ch = (char near *)buf_ptr->start + buf_ptr->len;
    if(end_ch - *overlap >= Strlen_max)
        *end = *overlap + Strlen_max;
    else
        *overlap = *end = end_ch;
}

/* Move to the next input buffer, first allocate more input buffer structures
 * if no free buffers left, then allocate space for buffer */
static int next_in_buf(int buf_i)
{
 /* Allocate 8 more buffers if no free buffers left */
    if(++buf_i == In_bufs_alloc && (Use_mmap || !Wrap_buf_i))
    {   int new_len = sizeof(Buffer) * (In_bufs_alloc + 8), new_buf_i;
        In_bufs = x_realloc(In_bufs, new_len);
        for(new_buf_i = 0; new_buf_i < 8; ++new_buf_i)
            In_bufs[In_bufs_alloc++].alloc = NULL;
    }
 /* Allocate buffer if file not memory mapped and buf not already alloced */
    if(!Use_mmap && !In_bufs[buf_i].alloc && !Wrap_buf_i)
    {   if(!alloc_in_buf(&In_bufs[buf_i], In_buf_len))
            Wrap_buf_i = buf_i;
    }
    if(buf_i == Wrap_buf_i && !Use_mmap)
    {   buf_i = 0;
        Wrapped = 1;
    }
    return buf_i;
}

/* Allocate the space for an input buf. Return the amount of memory alloced */
static size_t alloc_in_buf(BufPtr buf_ptr, size_t len)
{
    if(len > In_buf_mem_left)
        len = In_buf_mem_left & ~Load_len;

    if(len)
       buf_ptr->start = buf_ptr->alloc = malloc(len + PAD_LEN);

    if(!len || !(char near *)buf_ptr->start)
        return 0;
    buf_ptr->alloc_end = buf_ptr->start + len;
    In_buf_mem_left   -= len;
    return len;
}

static void init_load_buf(int buf_i, off_t off)
{
    Buffer *buf_ptr = &In_bufs[buf_i];
    if(Wrapped)
        Misc_bufs[OVERWRITE_BUF] = *buf_ptr;
    Load_buf_i     = buf_i;
    buf_ptr->off   = off;
    buf_ptr->len   = 0;
    buf_ptr->start = buf_ptr->alloc;
    buf_ptr->end   = buf_ptr->alloc_end;
    buf_ptr->mmap  = NULL;
}

/* Load more chars into the current load buffer. Return TRUE if desired
 *   number of characters read, FALSE otherwise */
static void load_buffer(void)
{
    BufPtr buf_ptr = &In_bufs[Load_buf_i];
    char *load_start = (charn *)buf_ptr->start + buf_ptr->len;

 /* If overwritten buffer present, first flush part which will be overwritten
  *   then adjust parameters of overwritten buffer */
    if(Misc_bufs[OVERWRITE_BUF].len)
    {   BufPtr ow_ptr = &Misc_bufs[OVERWRITE_BUF];
        int nflush    = (load_start + Load_len) - (charn *)ow_ptr->start;
        if(nflush > 0)
        {   if(Written)
            {   Buffers_Apply_fn(Nwrote, Nwrote + nflush, write_blk);
                Nwrote += nflush;
            }
            shift_start(ow_ptr, -nflush);
        }
    }
    if(!Use_mmap)
        buf_ptr->len += Files_read(Files_ptr, load_start, Load_len);

 /* If buf_ptr->start is set to NULL, mmap has failed and it is necessary
  * to read the remainder of the file using conventional means */
    else
    {   buf_ptr->len  = Files_mmap(Files_ptr, &buf_ptr->start, &buf_ptr->end);
        if(!buf_ptr->end)
        {   off_t off;
            Use_mmap       = FALSE;
            buf_ptr->start = buf_ptr->alloc;
            buf_ptr->end   = buf_ptr->alloc_end;
            Files_read_off(Files_ptr, 0, NULL, 0, &off, FALSE);
            load_buffer();
        }
        else
            buf_ptr->mmap  = (charn *)buf_ptr->start;
    }
}

/* Load the remainder of the entire  file into the buffer, resize the
 *  buffer so the entire file will fit, return the difference between the
 *  start of the original buffer and the buffer containing the whole buffer */
static void load_whole_file(void)
{
    off_t fsize        = Files_size(Files_ptr);
    BufPtr buf_ptr     = &In_bufs[Scan_buf_i];
    size_t nread;

    if(fsize != -1)
        fsize += Fsize_diff;

 /* Nothing to read if entire file already loaded */
    if(   buf_end_off(buf_ptr) == fsize
       && buf_ptr->end - buf_ptr->start >= fsize + (Use_mmap ? PAD_LEN : 0))
        return;

#ifndef __MSDOS__
    if(buf_ptr->mmap)
    {   off_t off;
        if((int)buf_ptr->len >= buf_ptr->alloc_end - buf_ptr->alloc)
        {   buf_ptr->alloc     = x_realloc(buf_ptr->alloc, buf_ptr->len);
            buf_ptr->alloc_end = buf_ptr->alloc + buf_ptr->len;
        }
        memcpy(buf_ptr->alloc, buf_ptr->mmap, buf_ptr->len);
        buf_ptr->start = buf_ptr->alloc;
        Files_Unmap(buf_ptr->mmap, buf_ptr->len);
        buf_ptr->mmap = NULL;
        buf_ptr->end  = buf_ptr->alloc_end;

     /* This will set the file offset to the appropriate location */
        Files_read_off(Files_ptr, buf_ptr->len, NULL, 0, &off, FALSE);
    }
#endif

 /* Now load remainder of file, double buffer size if buffer is full */
    do
    {   char *load_start = (charn *)buf_ptr->start + buf_ptr->len;
        if(load_start + Load_len + PAD_LEN > buf_ptr->end)
        {   size_t new_len = 2 * (buf_ptr->alloc_end - buf_ptr->alloc);
            buf_ptr->start = (charn *)x_realloc((charn *)buf_ptr->alloc,
                                                new_len);
            buf_ptr->alloc     = buf_ptr->start;
            buf_ptr->alloc_end = buf_ptr->end = buf_ptr->alloc + new_len;
        }
        nread = Files_read(Files_ptr, load_start, Load_len);
        buf_ptr->len += nread;
    }while(nread == (size_t)Load_len);
}

/* Write replacement text to output buffer and adjust input buffers */
void Buffers_Replace(off_t off, int len, char *replace, int rep_len)
{
    BufPtr buf_ptr = In_overlap ? &Overlap_buf : &In_bufs[Scan_buf_i];
    char *match    = (int)(off - buf_ptr->off) + (charn *)buf_ptr->start;
    int nbefore    = match - (charn *)buf_ptr->start; /* chars before match*/
    int len_diff   = rep_len - len, buf_i;
    int nl_diff    = count_nl(replace, rep_len) - count_nl(match, len);

 /* Start modified file if nothing has been written */
    if(!Written)
    {   off_t end_fwrite_off = (  Misc_bufs[OVERWRITE_BUF].len
                                ? Misc_bufs[OVERWRITE_BUF].off : 0);
        Files_start_mod(Files_ptr, end_fwrite_off,
                        Misc_bufs[OUT_BUF].start, Load_len);
        Misc_bufs[OUT_BUF].off = end_fwrite_off;
        Misc_bufs[OUT_BUF].len = 0;
        Written = TRUE;
        Nwrote  = end_fwrite_off;
    }
 /* Write text between last write and start of replaced string */
    if(buf_ptr->off == Nwrote)
        write_blk(buf_ptr->start, nbefore);
    else
        Buffers_Apply_fn(Nwrote, off, write_blk);

 /* Write up to where replacement starts, then write replacement */
    write_blk(replace, rep_len);
    Fsize_diff   += len_diff;
    Nwrote        = off + rep_len;

    LineNums_adjust(&Line_nums, off, rep_len, len_diff, nl_diff);

 /* Adjust input buffer paramemters to prevent access to written portion */
    shift_start(buf_ptr, -nbefore - len);

 /* Now adjust the offset all buffers by the difference between the string
  *  lengths in the buffers following the current one */
    buf_i = Wrapped ? (Wrap_buf_i - 1) : Load_buf_i;
    for(; buf_i >= 0 ; --buf_i)
        In_bufs[buf_i].off += len_diff;

    Overlap_buf.off += len_diff;

    Misc_bufs[OVERWRITE_BUF].len = 0;
}

/* if the length of a string added at runtime is greater than the length
 *  previous strings, it is necessary to change the overlap length */
void Buffers_Add_str(int new_str_len)
{
    if(new_str_len > Strlen_max)
    {/* If in overlap buffer, copy enough characters so there are enough
      * chars at the end of the buffer */
        if(In_overlap)
        {   char *dest = (charn *)In_bufs[Scan_buf_i].start + Strlen_max;
            char *src  = (charn *)Overlap_buf.start + Overlap_len;
            memcpy(dest, src, new_str_len);
        }
        Strlen_max  = new_str_len;
    }
}

/* return the offset of the match occuring at loc */
off_t Buffers_Match_off(const char *loc)
{
    return ptr_to_off(In_overlap ? &Overlap_buf : &In_bufs[Scan_buf_i], loc);
}

/* return the line number of the match occuring at loc */
off_t Buffers_Match_line_num(const char *loc)
{
    off_t line_num;
    BufPtr_line_num(OFF_MAX, Buffers_Match_off(loc), &line_num);

    return line_num;
}

/* Flush all buffers when file done */
void Buffers_Close(void)
{
    Buffer *buf_ptr = &In_bufs[Load_buf_i];
    if(Misc_bufs[OUT_BUF].len)
    {   Buffers_Apply_fn(Nwrote, buf_ptr->off + buf_ptr->len, write_blk);
     /* Flush output buffer */
        Files_write(Files_ptr, Misc_bufs[OUT_BUF].start,
                    Misc_bufs[OUT_BUF].len);
    }

 /* Unmap all memory mapped buffers */
#ifndef __MSDOS__
    for(buf_ptr = In_bufs; buf_ptr <= &In_bufs[Load_buf_i] ; ++buf_ptr)
    {   if(buf_ptr->mmap)
            Files_Unmap(buf_ptr->mmap, buf_ptr->end - buf_ptr->mmap);
    }
#endif
}

void Buffers_Kill(void)
{
    int free_i;
    for(free_i = 0; free_i < In_bufs_alloc; ++free_i)
    {   if(In_bufs[free_i].alloc)
            free((charn *)In_bufs[free_i].alloc);
    }
    farfree(Misc_bufs[OUT_BUF].start);
    free(In_bufs);
    free(Misc_bufs);
    free((charn *)Overlap_buf.alloc);
    LineNums_kill(&Line_nums);
}

/* Return the offset corresponding to the end of a buffer */
static off_t buf_end_off(Buffer *buf_ptr)
{
    return buf_ptr->off + buf_ptr->len;
}

/* Return the pointer corresponding to a pointer in a buffer */
static off_t ptr_to_off(Buffer *buf_ptr, const char far *ptr)
{
    return buf_ptr->off + (ptr - buf_ptr->start);
}

/* Shift the start of the buffer by n bytes of a buffer */
static void shift_start(Buffer *buf_ptr, int n)
{
    buf_ptr->start -= n;
    buf_ptr->off   -= n;
    buf_ptr->len   += n;
}

/* write a block of text to the output buffer, if the block is too large
 * for the buffer, fill the buffer, flush it and the reset */
static void write_blk(const charf *src, size_t len)
{
    size_t nwrote = 0;

 /* If not enough space in output buffer for block, fill up buffer, then
  *   write buffer to file and then reset buffer */
    for( ; ; )
    {   charf *dest = Misc_bufs[OUT_BUF].start + Misc_bufs[OUT_BUF].len;
        size_t nfree      = Load_len - Misc_bufs[OUT_BUF].len;
        size_t bytes_left = len - nwrote;
        if(nfree >= bytes_left)
        {   _fmemcpy(dest, &src[nwrote], bytes_left);
            Misc_bufs[OUT_BUF].len += bytes_left;
            break;
        }
        else
        {   _fmemcpy(dest, &src[nwrote], nfree);
            nwrote += nfree;
            Files_write(Files_ptr, Misc_bufs[OUT_BUF].start, Load_len);
            Misc_bufs[OUT_BUF].off += Load_len;
            Misc_bufs[OUT_BUF].len  = 0;
        }
    }
}

/* Call proc_fn on all the regions bounded by the offsets start & end, return
 *  FALSE if end of file reached, TRUE otherwise */
int Buffers_Apply_fn(off_t start, off_t end,
                     void (*proc_fn)(const char far *, size_t len))
{
    off_t off = start;
    while(off < end)
    {   BufPtr buf_ptr = BufPtr_goto(off);
        size_t blk_len = (size_t)(miN(end, buf_end_off(buf_ptr)) - off);
        if(BufPtr_eof(buf_ptr))
            return FALSE;

        proc_fn(buf_ptr->curr, blk_len);
        off += blk_len;
    }
    return TRUE;
}

BufPtr BufPtr_set(BufPtr buf_ptr, char far *ch_ptr)
{   buf_ptr->curr = ch_ptr;
    return buf_ptr;
}

charf *BufPtr_curr(BufPtr buf_ptr)             { return buf_ptr->curr;   }

BufPtr BufPtr_dec(BufPtr buf_ptr)
{
    if(buf_ptr->curr-- > buf_ptr->start)
        return buf_ptr;
    else if(buf_ptr->off)
        return BufPtr_goto(buf_ptr->off - 1);
    else
        return NULL;
}

/* Increment of pointer pointing to a buffer */
BufPtr BufPtr_inc(BufPtr buf_ptr)
{   if(buf_ptr->curr++ < buf_ptr->start + buf_ptr->len - 1)
        return buf_ptr;
    else
        return BufPtr_goto(buf_ptr->off + buf_ptr->len);
}

/* Test if buf_ptr points to the end of the file, will only happen when
 * the current location is at the end of the buffer */
int BufPtr_eof(BufPtr buf_ptr)
{
    return buf_ptr->curr == buf_ptr->start + buf_ptr->len;
}

/* return the offset corresponding to a pointer, it is assumend that
 *  Buf_ptr points to the buffer in which the pointed to location is found */
off_t BufPtr_off(BufPtr buf_ptr)
{
    return buf_ptr->off + (unsigned)(buf_ptr->curr - buf_ptr->start);
}

/* return a pointer to the end of the buffer */
charf *BufPtr_end(BufPtr buf_ptr) { return buf_ptr->start + buf_ptr->len; }

/* move to the following buffer */
BufPtr BufPtr_next(BufPtr buf_ptr)
{
    return BufPtr_goto(buf_ptr->off + buf_ptr->len);
}

/* return the line number corresponding to the location of buf_ptr */
BufPtr BufPtr_line_num(off_t line_num, off_t off, off_t *line_num_result)
{
    LineLoc far *line_loc = LineNums_nearest(&Line_nums, line_num, off);
    BufPtr buf_ptr = BufPtr_goto(line_loc->off);
    off_t nls_found;

 /* If using string buffer, do not use Line_num structure, just count
  *   line numbers in string */
    if(String_buf.start)
    {   size_t end_off = miN((size_t)off, String_buf.len);
        char far *line_start = String_buf.start;
        *line_num_result = 1;
        while(*line_num_result < line_num)
        {   line_start = _fmemchr(line_start, NL,
                                  end_off - (line_start - String_buf.start));
            if(!line_start)
            {   String_buf.curr = &String_buf.start[end_off];
                return &String_buf;
            }
            ++*line_num_result;
            ++line_start;
        }
        String_buf.curr = line_start;
        return &String_buf;
    }

    if(line_loc != LineNums_last(&Line_nums))
    {   off_t nls_needed = line_num - line_loc->line_num;
        buf_ptr = BufPtr_find_nl(buf_ptr, off, nls_needed, &nls_found);
        *line_num_result = line_loc->line_num + nls_found;
    }
 /* Add the location of every NL_CACHE_GAPth line beginning at start_off
  * and ending at end_off to the line number cache, return the line number
  * of the end of the region */
    else if(LineNums_total(&Line_nums) == -1)
    {   off_t curr_line = line_loc->line_num;
        while(curr_line < line_num)
        {   off_t nls_needed = miN(NL_CACHE_GAP, line_num - curr_line);
            buf_ptr = BufPtr_find_nl(buf_ptr, off, nls_needed, &nls_found);
            curr_line += nls_found;
            if(nls_found != nls_needed || BufPtr_eof(buf_ptr))
                break;
            LineNums_add(&Line_nums, curr_line, BufPtr_off(buf_ptr), FALSE);
        }
        *line_num_result = curr_line;
     /* If at end of file, set total number of lines to line count */
        if(BufPtr_eof(buf_ptr))
        {   off_t fsize = Files_size(Files_ptr);
            if(fsize != -1)
                LineNums_add(&Line_nums, curr_line, fsize + Fsize_diff, TRUE);
        }
     /* Move back to the start of the last line in the region and cache*/
        else if(curr_line != line_num)
        {   off_t orig_off = BufPtr_off(buf_ptr);
            buf_ptr = BufPtr_find_nl(buf_ptr, line_loc->off, -1, &nls_found);
            if(nls_found == -1)
                LineNums_add(&Line_nums, curr_line,BufPtr_off(buf_ptr),FALSE);
            buf_ptr = BufPtr_goto(orig_off);
        }
    }
 /* If offset at end of file, set line number to total number of lines in file*/
    else
        *line_num_result = LineNums_total(&Line_nums);

    return buf_ptr;
}

/* store the line number in the line number cache */
void Buffers_Record_line(off_t line_num, off_t off, int last_line)
{
    LineLoc far *nearest = LineNums_nearest(&Line_nums, line_num, OFF_MAX);
    if(nearest < LineNums_last(&Line_nums) || nearest->line_num == line_num)
        return;

    if(!last_line)
    {   off_t nl_result;
        BufPtr buf_ptr = BufPtr_find_nl(BufPtr_goto(off), 0, -1, &nl_result);
        off = BufPtr_off(buf_ptr);
    }
    LineNums_add(&Line_nums, line_num, off, last_line);
}

/* return the number of lines in the file, return -1 if # of lines unknown */
off_t Buffers_Nlines(void)
{
    return LineNums_total(&Line_nums);
}

/* Count the number of lines in the buffer, return FALSE if unable to
 * determine number of lines, TRUE otherwise */
void Buffers_Count_lines(void)
{
    off_t line_nums;
    if(LineNums_total(&Line_nums) == -1)
        BufPtr_line_num(OFF_MAX, OFF_MAX, &line_nums);
}

/* Return the offset pcnt percentage from the start of the file, return -1
 *   if the file size cannot be determined pcnt is assumed to be between 0
 *   and 100 */
off_t Buffers_Pcnt_to_off(int pcnt)
{
    off_t fsize = end_off(TRUE), off;

    if(fsize == -1)
        return -1;

 /* Use fixed point to calculate desired address */
    off = fsize < OFF_MAX / 100 ? (pcnt * fsize) / 100 : (fsize / 100) * pcnt;

    if(off >= fsize)
        off = fsize - 1;
    return off;
}

/* Return the percentage into the file corresponding to the offset off */
int Buffers_Off_to_pcnt(off_t off)
{
    int pcnt;
    off_t fsize = end_off(FALSE);
    if(fsize <= 0)
        return -1;

 /* Set percentage to twice its value, then divide & round */
    pcnt = (int)((off > 10000000) ? off / (fsize/200) : (off*200) / fsize);

    return pcnt/2 + (pcnt & 1);
}

/* Return the end offset of the file, or string buffer if it is being used*/
static off_t end_off(int advance_end)
{
    if(String_buf.start)
        return String_buf.len;
    else
    {   off_t fsize = Files_size(Files_ptr);
 /* If file size unknown, move to end of file to try to determine file size */
        if(fsize == -1 && advance_end)
        {   BufPtr_goto(OFF_MAX);
         /* Return -1 if file size still unknown */
            fsize = Files_size(Files_ptr);
        }
        return (fsize == -1) ? -1 : fsize + Fsize_diff;
    }
}

BufPtr BufPtr_find_nl(BufPtr buf_ptr, off_t stop_off,
                      off_t nl_num,  off_t *nl_result)
{   off_t nl_cnt = 0, start_off = BufPtr_off(buf_ptr);
    int last_buf = FALSE; /* when TRUE, no more buffers will be scanned */
    charf *scan_ptr = NULL, *scan_end;

 /* Make sure input values are logical */
    if(stop_off < 0)
        stop_off = 0;
    if(!nl_num || (nl_num < 0 && start_off <= stop_off)
               || (nl_num > 0 && start_off >= stop_off))
    {   *nl_result = 0;
        return buf_ptr;
    }

 /* If nl_num is negative search for \ns backwards */
    while(!last_buf && !BufPtr_eof(buf_ptr))
    {/* Determine where to stop scan, if stop_off is in current buffer
      *   stop at location corresponding to stop_off, otherwise stop
      *   at start or end of buffer depending on direction */
        if(buf_ptr->off <= stop_off && stop_off <= buf_end_off(buf_ptr))
        {   scan_end = buf_ptr->start + (size_t)(stop_off - buf_ptr->off);
            last_buf = TRUE;
        }
        else if(nl_num < 0)
            scan_end = buf_ptr->start;
        else
            scan_end = BufPtr_end(buf_ptr);

        if(nl_num < 0)
        {  if((buf_ptr = BufPtr_dec(buf_ptr)) == NULL)
                break;

         /* Scan buf for \ns, stop when enough \n found or end reached*/
            scan_ptr = buf_ptr->curr;
            do
            {   if(*scan_ptr == NL && --nl_cnt == nl_num)
                    goto enough_nl;

            }while(scan_ptr-- > scan_end);
            buf_ptr->curr = scan_ptr;
        }
     /* If nl_num is positive search forwards */
        else
        {   scan_ptr = buf_ptr->curr;

         /* Scan region for \n using memchr, stop when enough \n found or
          *   memchr returns NULL (indicating no more \n in region) */
            while( (scan_ptr =
                   _fmemchr(scan_ptr, NL, (scan_end -scan_ptr))) != NULL)
            {   if(++nl_cnt == nl_num)
                    goto enough_nl;

                ++scan_ptr;  /* Move to character after \n */
            }
         /* Move to next buffer if not enough \ns found */
            if(!last_buf)
                buf_ptr  = BufPtr_goto(buf_end_off(buf_ptr));
        }
    }
 /* If enough \n found, set buf_ptr to location after \n */
enough_nl:
    if(nl_cnt == nl_num)
    {   buf_ptr->curr = scan_ptr;
        buf_ptr = BufPtr_inc(buf_ptr);
    }
    else
        buf_ptr = BufPtr_goto(stop_off);

    *nl_result    = nl_cnt;
    return buf_ptr;
}

/* Return a pointer a buffer containing the offset off. The curr member
 *  of the buffer will point to the location corresponding to off */
Buffer *BufPtr_goto(off_t off)
{
    Buffer *buf_ptr = NULL;
    assert(off >= 0);

 /* If string buffer used, go to offset in string buffer */
    if(String_buf.start != NULL)
    {   String_buf.curr = String_buf.start + (  off < (off_t)String_buf.len
                                              ? (size_t)off
                                              : String_buf.len);
        return &String_buf;
    }
 /* Otherwise scan through buffers */
    if(off >= Nwrote)
    {   int buf_i = Wrapped ? (Wrap_buf_i - 1) : Load_buf_i;
        for(; buf_i >= 0 && buf_ptr == NULL; --buf_i)
            buf_ptr = off_in_buf(&In_bufs[buf_i], off);
        if(!buf_ptr)
        {   buf_ptr = off_in_buf(&Overlap_buf, off);
            if(!buf_ptr)
                buf_ptr = off_in_buf(&Misc_bufs[OVERWRITE_BUF], off);
        }
    }
    else
        buf_ptr = off_in_buf(&Misc_bufs[OUT_BUF], off);

 /* If offset not found in any current buffers, attempt to load in a buffer
  * which contains it, after making sure offset is not greater than file size */
    if(!buf_ptr)
    {   if((buf_ptr = off_in_buf(&Misc_bufs[SCROLL_BUF1], off)) != NULL)
            ;
        else if((buf_ptr = off_in_buf(&Misc_bufs[SCROLL_BUF2], off)) != NULL)
            ;
        else if(off < Nwrote)
            buf_ptr = load_scroll_buf(off);
        else
        {   if(off >= In_bufs[Load_buf_i].off && !Files_eof(Files_ptr))
                buf_ptr = load_ahead(off);

            if(!buf_ptr && Files_size(Files_ptr) != -1)
            {   off_t fsize = Files_size(Files_ptr) + Fsize_diff;
                if(off >= fsize)
                {   off = fsize;
                    if(buf_end_off(&In_bufs[Load_buf_i]) == fsize)
                        buf_ptr = &In_bufs[Load_buf_i];
                }
                if(!buf_ptr)
                    buf_ptr = load_scroll_buf(off);
            }
        }
    }
 /* Set result to point to character at offset */
    if(buf_ptr)
        buf_ptr->curr = buf_ptr->start + (int)(off - buf_ptr->off);

    return buf_ptr;
}

/* Return the buf if the offset off can be found in buf, NULL if not */
static BufPtr off_in_buf(BufPtr buf, off_t off)
{
    return (buf->off <= off && off < buf->off + (off_t)buf->len) ? buf : NULL;
}

/* Treat a string as if it were a buffer. After this function is called,
 * BufPtr_goto and end_off will operate on the string buffer until
 * this function is called with a NULL argument */
BufPtr BufPtr_str_to_buf(char far *str)
{
    String_buf.start = String_buf.curr = str;
    if(!str)
        return NULL;
    else
    {   String_buf.off = 0;
        String_buf.len = _fstrlen(str);
        return &String_buf;
    }
}

static Buffer *load_scroll_buf(off_t off)
{
    static int Prev_buf_i;
 /* Use scroll buffer not previously loaded */
    BufPtr buf_ptr = &Misc_bufs[SCROLL_BUF1 + (Prev_buf_i == SCROLL_BUF1)];

    if(off < Nwrote)
        buf_ptr->len = Files_read_off(Files_mod(Files_ptr), off,buf_ptr->start,
                                      Load_len, &buf_ptr->off, 1);
    else {
        buf_ptr->len = Files_read_off(Files_ptr, off - Fsize_diff,
                                      buf_ptr->start, Load_len,
                                      &buf_ptr->off, 0);
        buf_ptr->off += Fsize_diff;
    }
    Prev_buf_i = buf_ptr - Misc_bufs;
    return buf_ptr;
}

/* If desired offset greater than offset of last read byte, keep loading
 * buffers until the desired offset is contained in them. Stop when loading
 * buffers will overwrite the buffer currently being scanned */
static Buffer *load_ahead(off_t off)
{
    BufPtr load_ptr = &In_bufs[Load_buf_i], buf_ptr = NULL;

    while(!Files_eof(Files_ptr))
    {   if(BufPtr_end(load_ptr) == load_ptr->end)
        {   int next_load_i = next_in_buf(Load_buf_i);
            if(next_load_i == Scan_buf_i && Strlen_max > 0)
                break;

            init_load_buf(next_load_i, buf_end_off(load_ptr));
            load_ptr = &In_bufs[Load_buf_i];
        }
        load_buffer();

        if(buf_end_off(load_ptr) > off)
        {   buf_ptr = load_ptr;
            break;
        }
    }
    return buf_ptr;
}

#ifdef BUFFERS_TEST

int main(int argc, char *argv[])
{
    Files *files_ptr;
    char *scan_start, *scan_end, *buf_end;
    argc = argc;
    Files_Init(&argv[1], argc - 1, FALSE, NULL, NULL, NULL, NULL, NULL, NULL);
    Buffers_Init(16, "");

    while((files_ptr = Files_Next()) != NULL)
    {   Buffers_Reset(files_ptr);
        do
        {   BufPtr buf_ptr;
            off_t off;
            Buffers_Load(&scan_start, &scan_end, &buf_end, 0);
            buf_ptr = (!In_overlap
                       ? BufPtr_goto(Buffers_Match_off(scan_start))
                       : &Overlap_buf);
            off = buf_ptr->off + (scan_start - (charn *)buf_ptr->start);
            printf("%s %p %ld %ld %ld", Files_name(files_ptr), scan_start,
                   off,  off + (scan_end - scan_start),
                     (buf_end != scan_end)
                   ? off + (buf_end - scan_start)
                   : off + (buf_ptr->end - scan_start));
            if(Misc_bufs[OVERWRITE_BUF].len)
                printf("\t%p, %ld %d", Misc_bufs[OVERWRITE_BUF].start,
                       Misc_bufs[OVERWRITE_BUF].off,
                       Misc_bufs[OVERWRITE_BUF].len);
            putchar(NL);
        }while(scan_end < buf_end);
        Buffers_Close();
        Files_close(files_ptr);
    }
    return EXIT_SUCCESS;
}
#endif /* BUFFERS_TEST */
