/* backup.c - generate backup name for a file
 * Copyright (C) 1995-99 Andrew Pipkin (minitrue@pagesz.net)
 * MiniTrue is free software released with no warranty. See COPYING for details
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "minitrue.h"
#include "fileops.h"

static char *Backup_ext;
static int Backup_ext_len; /* length of backup extension */
static int Replace_ext;    /* set if backup extension should replace current
                            * filename instead of being appended */
static char *Backup_tail;     /* Characters following backup number */
static int Backup_digits;  /* Mininum # of digits in backup number */

/* Parse the backup extension */
void Backup_Init(const char *ext)
{
    int ext_i, backup_num_i = -1;
 /* If backup name starts with two dots, replace the extension instead of
  * appending to it and only use the first dot */
    if(ext[0] == '.' && ext[1] == '.')
    {   ++ext;
        Replace_ext = 1;
    }
    Backup_ext     = x_strdup(ext);
    Backup_ext_len = strlen(Backup_ext);
 /* Look for * embedded in file name, indicating a backup number should be
  * inserted there number of * indicate minimum length of number */
    for(ext_i = 0; ext_i < Backup_ext_len && Backup_ext[ext_i] != '*';++ext_i)
        ;
    if((backup_num_i = ext_i) < Backup_ext_len)
    {   for( ; Backup_ext[ext_i] == '*'; ++ext_i)
            ++Backup_digits;
    }
    if(backup_num_i != -1)
    {   Backup_tail              = &Backup_ext[backup_num_i + Backup_digits];
        Backup_ext[backup_num_i] = '\0';
    }
}

/* generate the backup filename */
static char *Backup_buf;   /* Buffer for backup filename */
static size_t Backup_buf_len;
const char *Backup_Fname(const char *path, int fname_max_len)
{
    char *fname, *ext, *dest, *num_start;
    char *ext_limit; /* earliest possible start of backup extension */
    int backup_num = 1, path_len = strlen(path);
    size_t new_len = path_len + Backup_ext_len + 64, max_ext_len;

 /* Assume MS-DOS file system if maximum name length 12 */
    int dos_fname = (fname_max_len <= 12);

 /* Copy path into buffer, allocating more memory if necessary */
    if(new_len > Backup_buf_len)
        Backup_buf = x_realloc(Backup_buf, Backup_buf_len = new_len);

    memcpy(Backup_buf, path, path_len + 1);
    fname = parse_path(Backup_buf, &ext, &path_len);

 /* If extension present, see if it should be replaced */
    dest = (ext && Replace_ext) ? ext : Backup_buf + path_len;

 /* Deal with "subtleties" of 8+3 filenames */
    if(dos_fname)
    {   if(ext)
        {   ext_limit = ext + 1;
         /* Replace extension if backup begins with dot */
            if(*Backup_ext == '.')
                dest = ext;
        }
     /* Must have dot if one does not exist */
        else
        {   ext_limit = dest + 1;
            if(*Backup_ext != '.')
                *dest++ = '.';
        }
        max_ext_len = 3;
    }
    else
    {   ext_limit   = fname;
        max_ext_len = fname_max_len;
    }
 /* Write extension up to where backup numbers start */
    num_start = copy_str(dest, Backup_ext);

 /* Write backup number if desired */
    while(Backup_digits)
    {   char *num_end, num_text[20];
        size_t num_len;
     /* Convert the backup number to text & append to extension */
        sprintf(num_text, "%0*d", Backup_digits, backup_num);

        if(strlen(num_text) > max_ext_len)
            return NULL;
     /* Truncate extension if too long */
        if((num_len = strlen(num_text)) > max_ext_len)
            return NULL;
        if((num_start - ext_limit) + num_len > max_ext_len)
            num_start = ext_limit + max_ext_len - num_len;

        num_end = copy_str(num_start, num_text);
     /* Copy chars following backup number to new filename */
        strcpy(num_end, Backup_tail);

     /* If file exists, try next backup number */
        if(!x_exists(Backup_buf))
           break;
        ++backup_num;
    }
    return Backup_buf;
}

void Backup_Kill(void)
{
    free(Backup_ext);
    free(Backup_buf);
}
#ifdef BACKUP_TEST
int main(int argc, char *argv[])
{
    int arg_i;
    Backup_Init(argv[1]);
    for(arg_i = 2; arg_i < argc; ++arg_i)
        printf("%s\n", Backup_Fname(argv[arg_i], fname_max(1)));
    Backup_Kill();
    return 0;
}
#endif
