/* attribs.c - determines appearance of parts of screen
 * Copyright (C) 1995-99 Andrew Pipkin (minitrue@pagesz.net)
 * MiniTrue is free software released with no warranty. See COPYING for details
 */

#include "minitrue.h"
#include "attribs.h"

static int parse_pair(const char *src);
static int parse_abbrev(const char *src);
static void copy_bkg(attr_t *dest, int attr);

enum colors
{   BLACK,       BLUE,            GREEN,         CYAN,          /*  dark */
    RED,         MAGENTA,         BROWN,         LIGHTGRAY,
    DARKGRAY,    LIGHTBLUE,       LIGHTGREEN,    LIGHTCYAN,     /*  light */
    LIGHTRED,    LIGHTMAGENTA,    YELLOW,        WHITE
};
#define attR(fg, bg) ((fg) + (bg) * 16)

/* default background & foreground attrs */
enum { BKG = BLUE, FG = LIGHTGRAY, MATCH_BKG = LIGHTGRAY,
    PAREN_BKG = LIGHTGRAY, STAT_BKG = BLACK};

/* Attributes for monochrome mode, console.c will use termcap to translate
 * these attributes appropriately */
static attr_t Mono_attrs[] =
{   NORM_ATTR, NORM_ATTR, REV_ATTR, NORM_ATTR,
    REV_ATTR, REV_ATTR, REV_ATTR, BOLD_ATTR, BOLD_ATTR,
    BOLD_ATTR, UL_ATTR, REV_ATTR, NORM_ATTR, BOLD_ATTR,
    REV_ATTR, BOLD_ATTR, BOLD_ATTR, NORM_ATTR, BOLD_ATTR,
    REV_ATTR, REV_ATTR, REV_ATTR, REV_ATTR
};

/* Use these attributes to write monochrome colors directly to screen */
enum
{   NORM = attR(LIGHTGRAY, BLACK), REV = attR(BLACK, LIGHTGRAY),
    BOLD = attR(WHITE, BLACK)
};

static attr_t Mono_dos_attrs[] =
{   NORM, NORM, REV, NORM,
    REV, REV, REV, BOLD, BOLD,
    BOLD, BOLD, REV, NORM, BOLD,
    REV, BOLD, BOLD, NORM, BOLD,
    REV, REV, REV, REV
};

/* Text attributes for color mode, color terminals are assumed to be ANSI */
static attr_t Color_attrs[] =
{   attR(FG, BKG), attR(CYAN, BKG), attR(BKG, FG), attR(LIGHTGREEN, BKG),
    attR(YELLOW, MAGENTA), attR(BLACK, CYAN), attR(MAGENTA, LIGHTGRAY),
    attR(LIGHTGRAY, BLACK), attR(LIGHTMAGENTA, BKG), attR(WHITE, BKG),
    attR(YELLOW, BKG), attR(LIGHTGRAY, STAT_BKG), attR(YELLOW, LIGHTGRAY),
    attR(LIGHTGREEN, LIGHTGRAY), attR(LIGHTBLUE, FG),
    attR(YELLOW, STAT_BKG), attR(LIGHTGRAY, STAT_BKG),
    attR(LIGHTGRAY, STAT_BKG), attR(WHITE, STAT_BKG),
    attR(RED, PAREN_BKG), attR(YELLOW, PAREN_BKG),
    attR(LIGHTGREEN, PAREN_BKG), attR(BLUE, PAREN_BKG)
};
static attr_t *Attr_vect = Color_attrs;

void Attribs_Init(int have_color, int use_tty, const char *colors)
{
 /* Use monochrome attributes if color not desired */
    if(!have_color)
        Attr_vect = use_tty ? Mono_attrs : Mono_dos_attrs;

    if(colors)
    {   int attr_i;
        int bkgs[]     = { BKG * 16, MATCH_BKG * 16, STAT_BKG * 16};
        int attrs[][2] =
        {   {NORMAL, 0}, {CURR_MATCH, 1}, {MATCH, 1}, {WRAPPED, 0},
            {UNKNOWN_START, 0}, {STAT_LINE, 2}, {SUBST, -1}, {EMPH, 0},
            {UNDERLINED, 0}, {ERR_MSG, 2}, {CTRL, -1}, {CTRL_MATCH, -1},
            {EOFILE, -1}, {SYMB, 0}, {REP_TEXT, 1}, {REP_CTRL, 1},
            {PROMPT_TEXT, 2}, {ENTERED_TEXT, 2}, {COMMAND_CH, 2},
            {PAREN1, 1}, {PAREN2, 1}, {PAREN3, 1}
        };

        for(attr_i = 0; *colors == ':' && attr_i < 7; ++attr_i)
        {   int bkg_i    = attrs[attr_i][1];
         /* If color pair not present, set the background color to the
          *   default*/
            if(colors[1] == ':')
            {   if(bkg_i != -1)
                    copy_bkg(&Color_attrs[ attrs[attr_i][0] ], bkgs[bkg_i]);
                ++colors;
            }
            else
            {   int new_attr = parse_pair(colors + 1);
                Color_attrs[ attrs[attr_i][0] ] = new_attr;
                if(bkg_i != -1)
                    bkgs[bkg_i] = new_attr;
                colors += 5;
            }
        }
        if(*colors != '\0')
            invalid_param('v', *colors);

     /* Set the background colors of the attributes which were not
      * user-defined to the default */
        for(; attr_i < NTEXT_TYPES; ++attr_i)
        {   int bkg_i = attrs[attr_i][1];
            if(bkg_i != -1)
                copy_bkg(&Color_attrs[ attrs[attr_i][0] ], bkgs[bkg_i]);
        }
    }
}

/* Copy the background color of attr to *dest */
static void copy_bkg(attr_t *dest, int attr)
{
    *dest &= 0xf;
    *dest |= (attr & ~0xf);
}

attr_t Attrib(int text_type) { return Attr_vect[text_type]; }

/* Convert the background/foreground pair of color abbreviations to the
 * corresponding integer value */
static int parse_pair(const char *src)
{
    char error_buf[80];
    int bg = parse_abbrev(src), fg;
    if(bg == -1 || bg >= 8)
    {   sprintf(error_buf, "%.2s is not a valid background color", src);
        error_msg(error_buf);
    }
    fg = parse_abbrev(src + 2);
    if(fg == -1)
    {   sprintf(error_buf, "%.2s is not a valid foreground color", src);
        error_msg(error_buf);
    }
    return attR(fg, bg);
}

/* Convert the color abbreviation to its correponding color code, return -1
 *  if not a valid color */
static int parse_abbrev(const char *src)
{
    char abbrevs[] = "bkblgncyremabrGrgrBlGnCyReMayewh";
    int i;
    if(!*src)
        return -1;
    for(i = 0; i < 32; i += 2)
    {   if(src[0] == abbrevs[i] && src[1] == abbrevs[i + 1])
            return i/2;
    }
    return -1;
}

