/* ADLF.C V1.1 ADlib FM programming interface.

   (C)1994 Daniel Chouinard  -Shareware

   Comments are welcome.  I can be contacted at:
   
   Internet: daniel@cam.org
   CIS:      74271,1465
   Phone:    (514) 768-6290 (Please! EST 18:00-22:00, not collect)
             Don't call after Sep. 15th 1994.

   Please, if you're going to distribute this package, get it right with
   all four files non-modified.  Thanks!

   ADLF is distributed to the world through Internet so that other authors may
   enjoy it and share their own programming experience with the FM chips.
   For instance, I'd like to add stereo left_or_right capability for the
   soundblaster cards but haven't got programming info for the SB16 (Stereo FM
   doesn't work like the Pro.  Annoying.)

   So, if you've got technical info or nice examples you want to share with
   the rest of the world, send me a copy; I'll put it in the next release and
   mention your contribution.

   This code compiles with Borland's Turbo C but should compile with just about
   any C out there.

   Files in this package:

    ADLF     C       20202  - Source
    ADLF     COM     20202  - Executable
    ADLFDEMO BAT     20202  - Execute for a demo.
    SBPROG   DOC     20202  - Jeffrey S. Lee's tech info.

   - Daniel Chouinard

*/

/* Nothing special except inportb and outportb */
#include <stdio.h>

#define ADLIBaddr 0x0388   /* Address register in mono mode */
#define ADLIBdata 0x0389   /* Data register in mono mode */
#define BYTE unsigned char /* All ADlib registers are addressed as 8bit bytes */
#define BOOL unsigned char /* Will use single byte for booleans */

/* Globals */
BYTE debug=0;  /* Debug level */
BYTE linep;    /* Character counter in current command line */
BOOL proginfo=0; /* Output programming code */

/* ADLIB Gobals */
unsigned int freq=0;
BYTE operator=0, channel=0, am=0, vibrato=0, eg=0, ksr=0, modfreq=0;
BYTE totallevel=0, attack=0, decay=0, sustain=0, release=0;
BYTE keyon=0, octave=0, feedback=0, algorithm=0, waveform=0, lks=0;
BYTE amdepth=0, vibdepth=0, rythm=0, bassdrum=0, snaredrum=0, tomtom=0;
BYTE cymbal=0, hihat=0;
BOOL g1=0,g2=0,g3=0,g4=0,g5=0,g6=0,g7=0,g8=0; /* Group booleans */

/* Examples for ADLF -[0-9] */
char *examples[]= {
  "z~1j1-c0o0u1v1e1k1m12a15d1s0r0w0t63-c0o1u0v1e0k0m8a13d1s4r1w0t63-f1023p7n1-n0",
  "z-ma14d1s1r5t63-o1m2a14d1s1r6t55-g1p4f600n1-n",
  "z-m4a10t55k1-p5f410g1n1-x14-d10r10n",
  "zv1m9a14sw2t50-j1f430p7n1g1-n",
  "z-e1m1t30a15d1s1rgw-e1o1wm15t63a15d1s1r-p0f160n1-n0-e1c1o0m1t30a15d1s1r0g0w0-e1c1o1w0m15t63a15d1s1r0-p0f162n1-n0",
  "z-m2a12d1s1r4t55-o1m1a12d1s1r4t55-g1p5f300n1-x5n0",
  "z-m4a10d1s1r3t55-o1m3a10d1s1r3t55-g1p2f514n1-x5n0",
  "z-v1a14d1s1r0t63w1-o1v1m15a14d1s1r3t63w0-p0f670n1-n0",
  "z{1-c7o1a15d12r3t53f140p1w1n1-n",
  "zm7a1st63-o1m8a1st63-g1f1023p1b1-n1"
};

void waitforkey() {
  int ch;
  printf("Press a key: ");
  ch=getch(); /* Get a key, erase prompt */
  printf("\b\b\b\b\b\b\b\b\b\b\b\b\b             \b\b\b\b\b\b\b\b\b\b\b\b\b");
  if(ch==3) {  /* Key == CTRL-C? */
    printf("Stopped.\n");
    exit(1);
  }
}

/* Should've have used delay() or a timer register. */
/* Instead, I sync to VGA refresh.  This might be a problem on non-VGA cards */
/* Didn't like using timer reg. because other TSR or process might use em and
   didn't want to mess with interrupts yet */
void xsleep(ticks)
int ticks; {
  int times;
  for(times=0;times<ticks && kbhit()==0;times++) {
    while((inportb(0x3da)&8)==0) ; /* Sync to VGA refresh */
    while(inportb(0x3da)&8) ; /* Let it flip */
  }
}

/* Parse decimal value at line[linep] up to non-numeric character, option c */
int parsevalue(line,c) 
char *line;
char c; {
  register unsigned int value;
  value=0;
  while(line[linep] >= '0' && line[linep] <= '9') {
    if(debug>9) printf("Parsevalue-digit:%c\n",line[linep]);
    value=value*10+line[linep++]-'0';
  }
  if (debug>2) printf("Parsed %c=%d up to character %d\n",c,value,linep);
  return(value);
}

/* Show where we are in current line */
void showwhere(line)
char *line; {
  register int a;
  printf("%s\n",line);  /* Print command line */
  for(a=0;a<linep;a++) putchar(' ');  /* Print spaces up to current point */
  printf("T\n"); /* Where we are */
}

/* Parse value with check for limits */
int parseminmax(line,min,max,c)
char *line;
char c;
unsigned int min;
unsigned int max; {
  register unsigned int value;
  value=parsevalue(line,c);
  if(value>max || value<min) {
    showwhere(line);
    printf("Error! %d <= %c <= %d, not %d\n",min,c,max,value);
    exit(2);
  }
  return(value);
}

/* Program ADLib register reg with byte */
int progadlib(reg,byte)
BYTE reg,byte; {
  register int cnt,dummy;
  if(debug>4) printf("Programming %x in register %x\n",byte,reg);
  if(proginfo) printf("\tprogadlib(0x%02X,0x%02X);\n",reg,byte);
  outportb(ADLIBaddr,reg);
  /* Delay a bit for register addressing */
  for(cnt=0;cnt<6;cnt++) dummy=inportb(ADLIBaddr);
  outportb(ADLIBdata,byte);
  /* Delay a lot for data */
  for(cnt=0;cnt<35;cnt++) dummy=inportb(ADLIBdata);
  return(dummy); /* Return status (not used yet) */
}

/* ADLib total SOFT (silent) reset */
void resetadlib() {
  register int a,b,or;
  or=debug;
  if(or<11) debug=0;
  if(or>1) printf("[Reset]!\n");
  for(b=0;b<3;b++) {
    for(a=0;a<0x15;a++) {
      progadlib(0x060+a,255); /* Max attack and decay all channels and ops */
      progadlib(0x080+a,255); /* Max Sustain and release all channels and ops */
    }
  }
  for(a=0;a<0x15;a++) {
    progadlib(0x0b0+a,0); /* Key off all channels and ops */
    progadlib(0xc0+a,0); /* Feedback and connection type all channels and ops */
    progadlib(0xe0+a,0); /* Waveform select all channels and ops */
  }
  progadlib(0xbd,0); /* AM,Vib,RYTHM... */
  for(b=0;b<3;b++) {
    for(a=0;a<0x15;a++) {
      progadlib(0x060+a,0); /* Min attack and decay all channels and ops */
      progadlib(0x080+a,0); /* Min Sustain and release all channels and ops */
    }
  }
  for(a=0;a<0x15;a++) {
    progadlib(0x40+a,63); /* Turn-off volume */
  }
  progadlib(1,0x20); /* Set bit 5 of register one */
  /* Reset internal globals */
  freq=0; operator=0; channel=0; am=0; vibrato=0; eg=0; ksr=0;
  totallevel=0; attack=0; decay=0; sustain=0; release=0; modfreq=0;
  keyon=0; octave=0; feedback=0; algorithm=0; waveform=0; lks=0;
  amdepth=0; vibdepth=0; rythm=0; bassdrum=0; snaredrum=0; tomtom=0;
  cymbal=0; hihat=0;
  g1=0;g2=0;g3=0;g4=0;g5=0;g6=0;g7=0;g8=0;
  debug=or;
}

/* Flush all accumulated data to all registers mentionned by groups */
void doitall() {
  register int value,reg;
  BYTE base[11] = { 0,1,2,8,9,10,16,17,18,3,4 };
  /* Has any data been mentionned yet? */
  if(debug>2 && (g1+g2+g3+g4+g5+g6+g7+g8)!=0) printf("Flushing data\n");
  /* For each group: program if necessary */
  if(g1) {
    if(debug>2) printf("Group 1\n");
    progadlib(0x20+3*operator+base[channel],modfreq|(ksr<<4)|(eg<<5)|(vibrato<<6)|(am<<7));
  }
  if(g2) {
    if(debug>2) printf("Group 2\n");
    /* Total level is logically inversed. 63=silent, 0=loud */
    /* so we inverse it here.  This makes programming friendlier */
    progadlib(0x40+3*operator+base[channel],(63-totallevel)|(lks<<6));
  }
  if(g3) {
    if(debug>2) printf("Group 3\n");
    progadlib(0x60+3*operator+base[channel],decay|(attack<<4));
  }
  if(g4) {
    if(debug>2) printf("Group 4\n");
    progadlib(0x80+3*operator+base[channel],release|(sustain<<4));
  }
  if(g5) {
    if(debug>2) printf("Group 5\n");
    progadlib(0x0b0+channel,((freq & 0x0300)>>8)|(octave<<2)|(keyon<<5));
    progadlib(0x0a0+channel,freq & 0x0ff);
  }
  if(g6) {
    if(debug>2) printf("Group 6\n");
    progadlib(0x0c0+channel,algorithm|(feedback<<1));
  }
  if(g7) {
    if(debug>2) printf("Group 7\n");
    progadlib(0xe0+3*operator+base[channel],waveform);
  }
  if(g8) {
    if(debug>2) printf("Group 8\n");
    progadlib(0xbd,(amdepth<<7)|(vibdepth<<6)|(rythm<<5)|(bassdrum<<4)| \
                   (snaredrum<<3)|(tomtom<<2)|(cymbal<<1)|hihat );
  }
  /* All flushed, reset group indicators */
  g1=0; g2=0; g3=0; g4=0; g5=0; g6=0; g7=0; g8=0;
}

/* Parse one line for options and arguments and execute */
int doline(line,name)
char *line,*name; {
  register int c,ticks;
  char cmd[80];  /* For system calls (!cmd!) */
  if(strlen(line)) {
    /* If line non-empty, remove possible eol so printing command line */
    /* Does not induce extra line feed */
    if(line[0]) {
      if(line[strlen(line)-1]=='\n') line[strlen(line)-1]=0;
    }
  }
  if(debug==1) printf("-> %s\n",line);
  linep=0;
  while (linep<strlen(line)) {
    c=line[linep];
    if(debug>1) showwhere(line); /* Print current pos in debug level 2 & up */
    switch(c) { /* Check single character options, parse arguments */
      case '~':
        linep++;  /* Got ~, go to next character */
        amdepth=parseminmax(line,0,1,c); /* Parse numerical arg, 0-1 */
        if(debug>1) printf("AM depth set to %d\n",amdepth);  /* Print info */
        g8=1; /* Mark group 8 as modified so it'll be flushed */
        break; /* End switch */
      case 'j':
        linep++;
        vibdepth=parseminmax(line,0,1,c);
        if(debug>1) printf("Vibrato depth set to %d\n",vibdepth);
        g8=1;
        break;
      case '{':
        linep++;
        rythm=parseminmax(line,0,1,c);
        if(debug>1) printf("Rythm set to %d\n",rythm);
        g8=1;
        break;
      case 'q':
        linep++;
        bassdrum=parseminmax(line,0,1,c);
        if(debug>1) printf("Bass drum set to %d\n",bassdrum);
        g8=1;
        break;
      case '}':
        linep++;
        snaredrum=parseminmax(line,0,1,c);
        if(debug>1) printf("Snare drum set to %d\n",snaredrum);
        g8=1;
        break;
      case 'y':
        linep++;
        tomtom=parseminmax(line,0,1,c);
        if(debug>1) printf("Tomtom set to %d\n",tomtom);
        g8=1;
        break;
      case '%':
        linep++;
        cymbal=parseminmax(line,0,1,c);
        if(debug>1) printf("Cymbal set to %d\n",cymbal);
        g8=1;
        break;
      case 'h':
        linep++;
        hihat=parseminmax(line,0,1,c);
        if(debug>1) printf("Hihat set to %d\n",hihat);
        g8=1;
        break;
      case '!':
        linep++;
        ticks=0; /* Use ticks (why not?) as temp pointer in line */
        /* Up to another '!' or eol */
        while(line[linep]!='!' && line[linep]!=0) cmd[ticks++]=line[linep++];
        if(line[linep]) linep++; /* If we got anything, step over it */
        cmd[ticks]=0; /* and mark eol */
        system(cmd);  /* Shell the command */
        break;
      case 'x':
        linep++;
        ticks=parseminmax(line,0,32767,c);
        if(ticks==0) waitforkey();
        else {
          if(debug>1) printf("Sleeping %d ticks\n",ticks);
          xsleep(ticks);
        }
        break;
      case 'c':
        linep++;
        channel=parseminmax(line,0,8,c);
        if(debug>1) printf("Channel set to %d\n",channel);
        break;
      case 'o':
        linep++;
        operator=parseminmax(line,0,1,c);
        if(debug>1) printf("Operator set to %d\n",operator);
        break;
      case 'u':
        linep++;
        am=parseminmax(line,0,1,c);
        g1=1;
        if(debug>1) printf("Amplitude modulation set to %d\n",am);
        break;
      case 'v':
        linep++;
        vibrato=parseminmax(line,0,1,c);
        g1=1;
        if(debug>1) printf("Vibrato set to %d\n",vibrato);
        break;
      case 'e':
        linep++;
        eg=parseminmax(line,0,1,c);
        g1=1;
        if(debug>1) printf("Eg set to %d\n",eg);
        break;
      case 'k':
        linep++;
        ksr=parseminmax(line,0,1,c);
        g1=1;
        if(debug>1) printf("Keyboard Scalig rate set to %d\n",ksr);
        break;
      case 'm':
        linep++;
        modfreq=parseminmax(line,0,15,c);
        g1=1;
        if(debug>1) printf("Modulator frequency set to %d\n",modfreq);
        break;
      case 'l':
        linep++;
        lks=parseminmax(line,0,3,c);
        g2=1;
        if(debug>1) printf("Level key scaling set to %d\n",lks);
        break;
      case 't':
        linep++;
        totallevel=parseminmax(line,0,63,c);
        g2=1;
        if(debug>1) printf("Total level set to %d\n",totallevel);
        break;
      case 'a':
        linep++;
        attack=parseminmax(line,0,15,c);
        g3=1;
        if(debug>1) printf("Attack level set to %d\n",attack);
        break;
      case 'd':
        linep++;
        decay=parseminmax(line,0,15,c);
        g3=1;
        if(debug>1) printf("Decay set to %d\n",decay);
        break;
      case 's':
        linep++;
        sustain=parseminmax(line,0,15,c);
        g4=1;
        if(debug>1) printf("Sustain set to %d\n",sustain);
        break;
      case 'r':
        linep++;
        release=parseminmax(line,0,15,c);
        g4=1;
        if(debug>1) printf("Release set to %d\n",release);
        break;
      case 'f':
        linep++;
        freq=parseminmax(line,0,1023,c);
        g5=1;
        if(debug>1) printf("Frequency set to %d\n",freq);
        break;
      case 'n':
        linep++;
        keyon=parseminmax(line,0,1,c);
        g5=1;
        if(debug>1) printf("Key on set to %d\n",keyon);
        break;
      case 'b':
        linep++;
        feedback=parseminmax(line,0,7,c);
        g6=1;
        if(debug>1) printf("Feedback set to %d\n",feedback);
        break;
      case 'g':
        linep++;
        algorithm=parseminmax(line,0,1,c);
        g6=1;
        if(debug>1) printf("Algorithm set to %d\n",algorithm);
        break;
      case 'w':
        linep++;
        waveform=parseminmax(line,0,3,c);
        g7=1;
        if(debug>1) printf("Waveform set to %d\n",waveform);
        break;
      case 'p':
        linep++;
        octave=parseminmax(line,0,7,c);
        g7=1;
        if(debug>1) printf("Octave set to %d\n",octave);
        break;
      case '-' :
        if(debug>1) printf("-\n");
        doitall(); /* Flush it all */
        linep++;
        break;
      case '+' :
        linep++;
        proginfo=parseminmax(line,0,1,c);
        if(debug>1) printf("Output program code: %d\n",proginfo);
        break;
      case 'i' :
        linep++;
        debug=parseminmax(line,0,11,c);
        if(debug>1) printf("Debugging set to %d\n",debug);
        break;
      case 'z':
        linep++;
        resetadlib();
        break;
      case '@' :
      case '#' :
        /* Rest of line is comment */
        /* '@' is comment marker so that even batch files can be read as in:
          @echo off
          @adlf -f thisbatch.bat
          @goto exit
          x0c0s0t45...
          zi3...
          @@
          :exit
        */
        if(line[linep+1]=='@') {
          /* Exit if line starts with @@ */
          if(debug>4) printf("Double-@, exit!\n");
          exit(0);
        }
        /* Step over to eol */
        linep=strlen(line);
        break;
      /* Step over newlines, carriage-returns, spaces and tabs */
      case '\n':
      case '\r':
      case ' ' : 
      case '\t' : 
        linep++;
        break;
      /* User has no clue */
      default :
        printf("%s: Option %c not recognized.  Exit.\n",name,c);
        exit(3);
        break;
    }
  }
  doitall(); /* Flush so that we don't have to end command lines with '-' */
  channel=0; operator=0; /* Don't remember c&o over lines so user doesn't abuse it */
}

/* HELP */
void usage(name,code) 
char *name;
int code; {
  /* I'm using tabs here.  Hope everyone's got them set at ever 8 chrs... */
  printf("%s (C)1994 Daniel Chouinard\n",name);
  printf("Usage: %s options...\n",name);
  printf("\n");
  printf("Group\tOption\tArg.\tEffect\n");
  printf("-----\t------\t-------\t-----------------------------\n");
  printf("\tz\t\tReset all ADLib channels\n");
  printf("\tx\t0-32767\tWait arg 70th sec. (VGA refresh) (0=Must press key)\n");
  printf("\ti\t0-11\tSet debugging level (0=Quiet, 11=Babbling)\n");
  printf("\t+\t0-1\tOutput programming code\n");
  printf("\t# or @\t\tRest of line is comment (double @ stops file execution)\n");
  printf("\t-\t\tFlush data to FM chip\n");
  printf("\t!cmd!\t\tExecute dos command\n");
  printf("\tc\t0-8\tSelect channel\n");
  printf("\to\t0-1\tSelect operator\n");
  printf("1\tv\t0-1\tOperator: Set vibrato\n");
  printf("1\te\t0-1\tOperator: Set EG type\n");
  printf("1\tu\t0-1\tOperator: Amplitude modulation\n");
  printf("1\tm\t0-15\tOperator: Set modulator frequency\n");
  printf("1\tk\t0-1\tOperator: Keyboard scaling rate\n");
  printf("2\tl\t0-3\tOperator: Set level key scaling\n");
  printf("2\tt\t0-63\tOperator: Set total level (volume)\n");
  waitforkey();
  printf("3\ta\t0-15\tOperator: Set attack\n");
  printf("3\td\t0-15\tOperator: Set decay\n");
  printf("4\ts\t0-15\tOperator: Set sustain\n");
  printf("4\tr\t0-15\tOperator: Set release\n");
  printf("5\tf\t0-1023\tChannel : Set frequency\n");
  printf("5\tn\t0-1\tChannel : Set key depressed\n");
  printf("5\tp\t0-7\tChannel : Set octave\n");
  printf("6\tb\t0-7\tChannel : Set feedback\n");
  printf("6\tg\t0-1\tChannel : Set FM/Additive mode (0=op0 mod op1, 1=Add)\n");
  printf("7\tw\t0-3\tOperator: Select waveform\n");
  printf("8\t~\t0-1\tGlobal  : Set AM depth (0=1db, 1=4.8db)\n");
  printf("8\tj\t0-1\tGlobal  : Set Vibrato depth (0=7%, 1=14%)\n");
  printf("8\t{\t0-1\tGlobal  : Rythm enable\n");
  printf("8\tq\t0-1\tGlobal  : Bass drum enable\n");
  printf("8\t}\t0-1\tGlobal  : Snare drum enable\n");
  printf("8\ty\t0-1\tGlobal  : Tom Tom enable\n");
  printf("8\t%\t0-1\tGlobal  : Cymbal enable\n");
  printf("8\th\t0-1\tGlobal  : Hi Hat enable\n");
  printf("\n");
  waitforkey();
  printf("\n");
  printf("Usage examples:\n\n");
  printf("%s\t\t\tThis help.\n",name);
  printf("%s z-t1m1u1...\tExecute z-t1m1... commands.\n",name);
  printf("%s -f filename\tExecute commands in 'filename'.\n",name);
  printf("%s -\t\t\tExecute commands from stdin (keyboard).\n",name);
  printf("%s -[0-9]\t\tInternal pre-programmed sounds.\n",name);
  printf("\n\nFor a programming example, type %sDEMO.BAT\n",name);
  exit(code);
}

int main(argc,argv)
int argc;
char *argv[]; {
  char line[256]; /* Holds one command line of options and arguments */
  char myname[128]; /* Holds executable's name at runtime */ 
  FILE *fp;  /* File pointer for command file reading */
  int p=0;
  linep=0;
  myname[0]=0; /* Start with empty name */
  for(linep=0;linep<strlen(argv[0]);linep++) {
    /* Look for last path separator */
    if(argv[0][linep]=='\\' || argv[0][linep]=='/') p=linep+1;
  }
  linep=p;
  /* Copy name up to dot or eol */
  while(argv[0][linep]!='.' && argv[0][linep]!=0) {
    myname[strlen(myname)+1]=0;
    myname[strlen(myname)]=argv[0][linep++];
  }
  linep=0;
  if(argc==2) {
    if(strcmp(argv[1],"-")==0) {
      /* Execute stdin */
      while((gets(line))!=NULL) doline(line,myname);
      exit(0);
    }
    if(argv[1][1]>='0' && argv[1][1]<='9' && argv[1][0]=='-') {
      /* Do an example */
      doline(examples[0-atoi(argv[1])]);
      exit(0);
    }
  }
  if(argc==3||argc<2) {
    /* Two args.  First should be -f */
    if(strcmp(argv[1],"-f")!=0 && strcmp(argv[1],"-F")!=0) usage(myname,4);
    /* Second arg should be file.  Open */
    if((fp=fopen(argv[2],"r"))==NULL) {
      printf("%s: Can't open %s\n",myname,argv[2]);
      exit(5);
    }
    while(fgets(line,255,fp)) doline(line,myname); /* Parse whole file */
    fclose(fp);
    exit(0);
  }
  if(argc!=2) usage(myname,4); /* If not one or two args, print help */
  doline(argv[1],myname); /* First argument is commands */
  exit(0);
}
