/*****************************************************************************
*                           BC 3.2 PROGRAM                                   *
*                     LF SPECTRUM ANALYSER FOR SB-16                         *
*                              FFT.CPP                                       *
*                                                                            *
*  This product is copyrighted (C) 1996 by Henk Thomassen. Permission is     *
*  granted for non-commercial use and distribution.                          *
******************************************************************************
* REMARK:  Display is used in the VGA 640x480 resolution.                    *
*          To eliminate flikker when writing to the screen, double           *
*          buffering is used for the display arrea with screen[] as          *
*          display buffer.                                                   *
*          The display is (more or less) corrected for my AWE-32, don't know *
*          if SB-16 has equal input gain.                                    *
*                                                                            *
*          Compiler : Borland C++ v3.2                                       *
*          Options  : Model           = LARGE                                *
*                     Floting point   = 80287/80387                          *
*                     Instruction set = 80386                                *
*                                                                            *
*          v1.0 Jan 1996                                                     *
*                                                                            *
******************************************************************************
* If you have any questions or remarks please contact me                     *
* Henk Thomassen, thomass@IAEhv.nl                                           *
*****************************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <mem.h>
#include <malloc.h>
#include <conio.h>
#include <ctype.h>
#include <math.h>
#include <dos.h>
#include <graphics.h>

#include "graphics.h"
#include "sb16.h"
#include "mouse.h"

#define FALSE           (0)
#define TRUE            (!FALSE)
#define PI2              2 * M_PI
#define PI4		 4 * M_PI

/*****************************************************************************
* ENUMS AND TEXT FOR ENUMS        				     	     *
*****************************************************************************/
char *plot_text[]   = { "Lines", "Dots"};
char *y_loglin_text[]={ "Linear", "Logaritmic"};
char *window_text[] = { "Rectangle", "Barlett", "Hanning", "Hamming", "Blackman" };
char *fmode_text[]  = { "Scope","Absolute", "Argument", "Real", "Imaginary" };
char *dmode_text[]  = { "Scope","Spectrum","Spectrum","Spectrum","Spectrum","Spectrum" };
char *refresh_text[]= { "NO refresh", "Refresh" };
char *input_text[]  = { "SB16 8b","SB16 16b","Internal"};
char *peak_text[]   = { "Peak ON","Peak OFF"};

enum { lines, dots };
enum { linear, logaritmic };
enum { none, barlett, hanning, hamming, blackman };
enum { scope, absolute, fase, real, imag };
enum { no_refresh, yes_refresh };
enum { sb16_8, sb16_16,internal};
enum { on_peak,off_peak};

#include "fft_data.inc"  // fft_data is using enums.

/*****************************************************************************
* DEFINITION OF BUFFERS / ARREAYS					     *
*****************************************************************************/
typedef struct
 {
   float re, im;
 } complex;

complex vector[4096+1];                 // vector.re = input for FFT or SCOPE
					//   vector.re/im = output FFT
int dplot[DISPLAY_W+1];                 // Stores actual didplay data
float window_shape[4096+1];             // Stores actual window shape

/*****************************************************************************
* GENERAL BASIC SETUP							     *
*****************************************************************************/
float      system_adjustment = 0.0249;  // adjustment for AWE_32

float      Y_scaler;                    // adjust Y for display window
float	   Y_offset;                    // adjust Y for display window offset
float	   Y2_offset;                   // adjust Y for screen[] offset
float	   X_scaler;                    // adjust X for display window
float	   X_offset;                    // adjust X for display window offset

int        program_end       = FALSE;   // if TRUE exit program

int        mode              = absolute;// actual mode ( see enum scope..imag)

void       (*processing)();             // actual processing function, change
					//   with input

float      peak_signal;                 // detected peak_signal, freq and
int        peak_freq;                   //   position (Y) on display
int        peak_y;                      //

int        sysamp;                      // system amplification in dBm
					//   position volume control and
					//   amplifiers on SB-16/AWE

/*****************************************************************************
* FUNCTION PROTOTYPES                					     *
*****************************************************************************/
int matherr(struct exception *a);// ignore DOMAIN errors
void cursor_mode(void);          // stop processing input signal, and
				 //   start signal mesurament with mouse_cursor
void write_displaytext(int c);   // write text on X/Y axes
void write_divisions();          // make division lines in display
void build_display(void);        // build fixed part of user interface
void write_menutext();           // write/update text in knop_windows
void up_down_x(int ud);          // switch x_division
void up_down_y(int ud );         // switck y_divisions
void shift_y(int ud);            // shift display in Y direction
extern "C" {                     // external fft routine (ASM)
  int  fft(complex x[], unsigned n);
  }
void makekeys();                 // init all key's
void init(void);                 // program initialisation
void system_switch(int n);       // switch between display modes
void prep_display(void);         // calculate scalig and offsets
void make_window(void);          // calculate window for sample
void switch_input(int new_mode );// switch input mode
void make_dplot();               //   calculate all points for next display
				 // in dplot[]
void display_sample(void);       // draw all lines for next display in
				 //   screen[] and copy screen[] to display
void processing_16sb16();        // iput processing for SB_16 in 16 bit mode
void processing_8sb16();         // iput processing for SB_16 in  8 bit mode
void processing_internal();      // iput processing for internal function

/*****************************************************************************
*                                                                            *
*****************************************************************************/
int matherr (struct exception *a)
{
  if (a->type == DOMAIN)
    return(1);
  else
    return(0);
}
/*****************************************************************************
*                                                                            *
*****************************************************************************/
void cursor_mode(void)
{
  int  ok=FALSE,d,x, x1, y, y1,k;
  float f, p,x0=X_offset,y0=Y_offset;
  char dummy[80];
  int relon=FALSE,mx,my,mk;

  unsigned savemask;
  savemask=LastMask;

  int peaksave=s.peakon;
  s.peakon=off_peak;
  display_sample();

  setmouseevent(MEVENT_KRD+MEVENT_KLD);
  settextstyle(DEFAULT_FONT,HORIZ_DIR,1);
  targetmouse();

  mousebox(DISPLAY_X,DISPLAY_Y,DISPLAY_X+DISPLAY_W,DISPLAY_Y+DISPLAY_H);
  do
   {
      if (event())
	{
	  getevent(mx,my,mk);
	  if (1&mk)
	  {
	    hidemouse();
	    outpw(0x3c4,0x0402);  // select VGA plane 0
	    outpw(0x3ce,0x0204);
	    clearDisplay0();
	    hLine(DISPLAY_X,DISPLAY_X+DISPLAY_W,my);
	    vLine(mx,DISPLAY_Y,DISPLAY_Y+DISPLAY_H);
	    outpw(0x3c4,0x0f02);  // select VGA plane all
	    showmouse();
	    x0=mx;
	    y0=my;
	    x1=x-1; // force redisplay
	    relon=TRUE;
	  }
	  if (2&mk)
	  {
	    ok=TRUE;
	  }
	}

      x = (int)mousex();
      y = (int)mousey();
      if ((x1!=x)||(y1!=y))
       {
	 x1=x;
	 y1=y;

	 setcolor(3);
	 hidemouse();

	 do {} while((inportb(0x3DA) & 8)==0);
	 do {} while((inportb(0x3DA) & 8)!=0);
	 outpw(0x3c4,0x0102);  // select VGA plane 0
	 outpw(0x3ce,0x0004);
	 copydisplaybuf();
	 outpw(0x3c4,0x0f02);  // select all VGA planes

	 f = (x-X_offset) / X_scaler;
	 p = (y - Y_offset) / Y_scaler * ((s.y_loglin==linear) ? s.ylin_scalemul[s.ylin_scale]:1.0);
	 if(s.y_loglin==linear)
	      if(mode==scope)
		sprintf(dummy, "Relative: %1.*f%s/%1.*f", o_prex(),f, s.xlin_scaletxt,o_preylin(),p);
	      else
		sprintf(dummy, "Relative: %1.*f%s/%1.*f%s", o_prex(),f, s.xlin_scaletxt,o_preylin(),p,s.ylin_scaletxt[s.ylin_scale]);
	 else
	   sprintf(dummy, "Cursor: %1.*f%s/%1.*f%s", o_prex(),f, s.xlin_scaletxt,o_preylog(),p,s.ylog_scaletxt );
	 outtextxy(100,30,dummy);

	 if(relon)
	 {
	   f = (x-x0) / X_scaler;
	   p = (y-y0)/Y_scaler * ((s.y_loglin==linear) ? s.ylin_scalemul[s.ylin_scale]:1.0);
	   if(s.y_loglin==linear)
	      if(mode==scope)
		sprintf(dummy, "Relative: %1.*f%s/%1.*f", o_prex(),f, s.xlin_scaletxt,o_preylin(),p);
	      else
		sprintf(dummy, "Relative: %1.*f%s/%1.*f%s", o_prex(),f, s.xlin_scaletxt,o_preylin(),p,s.ylin_scaletxt[s.ylin_scale]);
	   else
	      sprintf(dummy, "Relative: %1.*f%s/%1.*f%s", o_prex(),f, s.xlin_scaletxt,o_preylog(),p,s.ylog_scaletxt );
	   outtextxy(326,30,dummy);
	 }

	 write_divisions();
	 showmouse();
       } // if
      //processing();
   }  // do
   while(!ok);
   setcolor(7);
   hidemouse();

   arrowmouse();
   mousebox(0,0,getmaxx(),getmaxy());
   setmouseevent(savemask);

   s.peakon=peaksave;
   outpw(0x3c4,0x0402);  // select VGA plane 2
   outpw(0x3ce,0x0204);
   clearDisplay0();
   outpw(0x3c4,0x0f02);  // select VGA plane all
   write_divisions();
   showmouse();
}

/*****************************************************************************
*                                                                            *
*****************************************************************************/
void write_displaytext(int c)
{
  int  m,n;
  char dummy[50];

  hidemouse();

  settextjustify(RIGHT_TEXT, TOP_TEXT);
  settextstyle(SMALL_FONT,HORIZ_DIR,0);
  setusercharsize(1,1,1,1);

  setcolor(c);
  for(m = 0; m<=s.x_divisions; m++)
  {
    n=(int)DISPLAY_X+m*DISPLAY_W / s.x_divisions;
    sprintf(dummy, "%3.*f\0x0", prex(),(s.xlin_min + m)*xdiv());
    outtextxy(n+(textwidth(dummy)/2), DISPLAY_Y+DISPLAY_H+5, dummy);
  }

  for(m = 0; m<=s.y_divisions; m++)
   {
     n=(int)DISPLAY_Y+(s.y_divisions-m)*DISPLAY_H / s.y_divisions;
     if(s.y_loglin==linear)
       sprintf(dummy,"%3.*f\0x0", preylin(),(float)(s.ylin_min + m)*ydivlin());
     else
       sprintf(dummy,"%3.*f\0x0", preylog(),(float)(s.ylog_min + m)*ydivlog());
     outtextxy(DISPLAY_X-3 , n - 6*(m == 0), dummy);
   }

  settextstyle(SMALL_FONT,VERT_DIR,0);
  if(s.y_loglin==linear)
      outtextxy(DISPLAY_X-30, DISPLAY_Y+DISPLAY_H/2-textwidth(s.ylin_scaletxt[s.ylin_scale])/2,s.ylin_scaletxt[s.ylin_scale]);
  else
    outtextxy(DISPLAY_X-30, DISPLAY_Y+DISPLAY_H/2-5,s.ylog_scaletxt);

  settextstyle(SMALL_FONT,HORIZ_DIR,0);
  settextjustify(LEFT_TEXT, TOP_TEXT);

  outtextxy(DISPLAY_X+DISPLAY_W/2-5, DISPLAY_Y+DISPLAY_H+15, s.xlin_scaletxt);

  showmouse();
  settextstyle(DEFAULT_FONT,HORIZ_DIR,1);
}

/*****************************************************************************
*                                                                            *
*****************************************************************************/
void write_divisions()
{
  int n, m;
  hidemouse();

  setlinestyle(USERBIT_LINE, 0x1111, 1);

    setcolor(8);
  for(m = 0; m<=s.x_divisions; m++)
  {
    n=(int)DISPLAY_X+m*DISPLAY_W / s.x_divisions;
    if((m!=0)&&(m!=s.x_divisions))
       line(n, DISPLAY_Y,n, DISPLAY_Y+DISPLAY_H);
  }

  for(m = 0; m<=s.y_divisions; m++)
   {
     n=(int)DISPLAY_Y+(s.y_divisions-m)*DISPLAY_H / s.y_divisions;
     if((m!=0)&&(m!=s.y_divisions))
       line(DISPLAY_X, n, DISPLAY_X+DISPLAY_W, n);
   }
  setlinestyle(SOLID_LINE, 0, 1);
  showmouse();
}


/*****************************************************************************
*                                                                            *
*****************************************************************************/
void build_display(void)
{
  hidemouse();
  cleardevice();
  cabinet(DISPLAY_X,DISPLAY_Y,DISPLAY_H,DISPLAY_W);

  settextstyle(DEFAULT_FONT,HORIZ_DIR,0);

//----------------------------------------- mode block
  emboss(455,385,514,397);
  setknop(0,knopin(0));       // window

  emboss(455,400,514,412);
  setknop(2,knopin(2));       // mode

  emboss(455,415,514,427);
  setknop(5,knopin(5));       // display

  kader(435,365,525,435);
  setcolor(1);
  outtextxy(440,370,"MODE");

//----------------------------------------- input block
  emboss(455,455,514,467);
  setknop(7,knopin(7));       // input

  kader(435,440,525,475);
  setcolor(1);
  outtextxy(440,445,"INPUT");

//----------------------------------------- display block
  emboss(550,385,614,397);
  setknop(4,knopin(4));      // peak

  emboss(550,400,614,412);
  setknop(6,knopin(6));      // s.plot

  emboss(550,415,614,427);
  setknop(9,knopin(9));      // s.refresh

  kader(530,365,624,435);
  setcolor(1);
  outtextxy(535,370,"DISPLAY");

//----------------------------------------- y-axis block
  set_pmknop(0);             // y division +/-
  set_pmknop(1);             // y lin/log
  kader(245,365,335,475);
  emboss(251,388,329,405);
  setcolor(1);
  outtextxy(267,410,"-    +");
  outtextxy(259,440,"Log  Lin");
  outtextxy(250,370,"Y-SCALE");

//----------------------------------------- x-axis block
  emboss(346,388,424,405);
  set_pmknop(2);             // x division +/-
  kader(340,365,430,475);
  setcolor(1);
  outtextxy(362,410,"-    +");
  outtextxy(345,370,"X-SCALE");

//----------------------------------------- Exit switch
  setknop(12,knopin(8));
  setcolor(1);
  outtextxy(535,445,"EXIT");
  kader(530,440,624,475);

//----------------------------------------- General
  setcolor(1);
  outtextxy(64,13,"SPECTRUM ANALYSER (0..20 KHz)");
  outtextxy(10,370,"SOUND BLASTER-16 CONTROL");

  kader(5,5,624,360);   // display
  kader(5,365,240,475); // sb16 mixer

  settextjustify(RIGHT_TEXT, TOP_TEXT);
  settextstyle(SMALL_FONT,VERT_DIR,0);
  setusercharsize(1,1,1,1);
  setcolor(6);
  outtextxy(635,5,"v1.0, (c) Henk Thomassen, Jan 1996");
  set_pmknop(3);

  showmouse();
}

/*****************************************************************************
*                                                                            *
*****************************************************************************/
void write_menutext()
{
  int n, m, d;
  char dummy[50];

  hidemouse();

  switch(mode)
  {
    case fase:
    case real:
    case imag:
      settextstyle(SMALL_FONT,HORIZ_DIR,0);
      setusercharsize(1,1,1,1);
      setcolor(3);
//------------------------------------------------------- mode block
      fbar(457,386,513,396,2); // clear window indicator
      outtextxy(460,384,window_text[s.window_mode]);

      fbar(457,401,513,411,2); // clear mode indicator
      outtextxy(460,399,fmode_text[mode]);
//------------------------------------------------------- display block
      fbar(552,386,613,396,7); // clear peak indicator
//------------------------------------------------------- y-axis block
      fbar(253,389,328,404,2);
      if(s.y_loglin==linear)
	sprintf(dummy,"%1.*f %s/div",preylin(),ydivlin(),s.ylin_scaletxt[s.ylin_scale]);
      else
	sprintf(dummy,"%1.*f %s/div",preylog(),ydivlog(),s.ylog_scaletxt);
      outtextxy(291-textwidth(dummy)/2,390,dummy);
      break;
    case absolute:
      settextstyle(SMALL_FONT,HORIZ_DIR,0);
      setusercharsize(1,1,1,1);
      setcolor(3);
//------------------------------------------------------- mode block
       fbar(457,386,513,396,2); // clear window indicator
      outtextxy(460,384,window_text[s.window_mode]);

      fbar(457,401,513,411,2); // clear mode indicator
      outtextxy(460,399,fmode_text[mode]);
//------------------------------------------------------- display block
      fbar(552,386,613,396,2); // clear peak indicator
      outtextxy(555,384,peak_text[s.peakon]);
//------------------------------------------------------- y-axis block
      fbar(253,389,328,404,2);
      if(s.y_loglin==linear)
	sprintf(dummy,"%1.*f %s/div",preylin(),ydivlin(),s.ylin_scaletxt[s.ylin_scale]);
      else
	sprintf(dummy,"%1.*f %s/div",preylog(),ydivlog(),s.ylog_scaletxt);
      outtextxy(291-textwidth(dummy)/2,390,dummy);
      break;

    case scope:
      settextstyle(SMALL_FONT,HORIZ_DIR,0);
      setusercharsize(1,1,1,1);
      setcolor(13);
//------------------------------------------------------- mode block
      fbar(457,386,513,396,7); // clear window indicator
      fbar(457,401,513,411,7); // clear mode indicator
//------------------------------------------------------- display block
      fbar(552,386,613,396,7); // clear peak indicator
//------------------------------------------------------- y-axis block
      fbar(253,389,328,404,2);
      sprintf(dummy,"%s",s.ylin_scaletxt[s.ylin_scale]);
      outtextxy(291-textwidth(dummy)/2,390,dummy);
      break;

  } // case


//------------------------------------------------------- input block
   fbar(457,456,513,466,2); // clear input indicator
   outtextxy(460,455,input_text[s.input_mode]);

//------------------------------------------------------- mode block
   fbar(457,416,513,426,2); // clear display indicator
   outtextxy(460,414,dmode_text[mode]);

//------------------------------------------------------- display block
   fbar(552,401,613,411,2); // clear s.plot indicator
   outtextxy(555,399,plot_text[s.plot]);

   fbar(552,416,613,426,2); // clear s.refresh indicator
   outtextxy(555,414,refresh_text[s.refresh]);

//------------------------------------------------------- x-axis block
   fbar(348,389,423,404,2);
   sprintf(dummy,"%1.*f %s/div",prex(),xdiv(),s.xlin_scaletxt);
   outtextxy(386-textwidth(dummy)/2,390,dummy);

   showmouse();
}

/*****************************************************************************
*                                                                            *
*****************************************************************************/
void up_down_x(int ud)
{
  int xlin=s.xlin_scale;

  if((!ud)&&(xlin<(s.xlin_scalemax-1)))
     xlin++;
  else
    if(ud&&(xlin>0)) xlin--;


  if(xlin!=s.xlin_scale)
  {
    write_displaytext(7);
    s.xlin_scale=xlin;
    if(mode>scope)
    {
      m[absolute].xlin_scale=xlin;
      m[fase].xlin_scale=xlin;
      m[real].xlin_scale=xlin;
      m[imag].xlin_scale=xlin;
    }
    sbsettc(s.sample_freq[s.xlin_scale]);
    prep_display();
    write_displaytext(0);
    write_menutext();
  }
}

/*****************************************************************************
*                                                                            *
*****************************************************************************/
void up_down_y(int ud)
{
  float temp = s.ylog_division[s.ylog_scale]*s.ylog_min;
  int ylin = s.ylin_scale;
  int ylog = s.ylog_scale;


  if(s.y_loglin==linear)
  {
    if((!ud)&&(ylin>0))
      ylin--;
    else
      if(ud&&(ylin<(s.ylin_scalemax-1)))
	ylin++;
  }
  else
  {
    if((!ud)&&(ylog>0))
      ylog--;
    else
      if(ud&&(ylog<(s.ylog_scalemax-1)))
	ylog++;
  }
  if((ylog!=s.ylog_scale)||(ylin!=s.ylin_scale))
  {
    write_displaytext(7);
    s.ylog_min = (int) (temp / s.ylog_division[ylog]);
    s.ylog_max = s.ylog_min + s.y_divisions;
    s.ylog_scale=ylog;
    s.ylin_scale=ylin;
    prep_display();
    write_displaytext(0);
    write_menutext();
  }
}

/*****************************************************************************
*                                                                            *
*****************************************************************************/
void shift_y(int ud)
{
  write_displaytext(7);
  if(s.y_loglin==linear)
  {
    if((!ud)&&(s.ylin_min>s.ylin_downmax))
    {
      s.ylin_min--;
      s.ylin_max--;
    }
    else
      if(ud&&(s.ylin_max<s.ylin_upmax))
      {
	 s.ylin_min++;
	 s.ylin_max++;
      }
  }
  else
  {
    if((!ud)&&(s.ylog_min>s.ylog_downmax))
    {
      s.ylog_min--;
      s.ylog_max--;
    }
    else
      if(ud&&(s.ylog_max<s.ylog_upmax))
      {
	s.ylog_min++;
	s.ylog_max++;
      }
  }
  prep_display();
  write_displaytext(0);
  write_menutext();
}


/*****************************************************************************
*                                                                            *
*****************************************************************************/
void makekeys()
{
  init_knop(0,440,385,12,12,6,10,FALSE);  // Window
  init_knop(2,440,400,12,12,6,10,FALSE);  // Mode
  init_knop(5,440,415,12,12,6,10,FALSE);  // Display

  init_knop(7,440,455,12,12,6,10,FALSE);  // input

  init_knop(4,535,385,12,12,6,10,FALSE);  // peak
  init_knop(6,535,400,12,12,6,10,FALSE);  // s.plot
  init_knop(9,535,415,12,12,6,10,FALSE);  // s.refresh

  init_pmknop(0,270,420,40,15,0);  // y div +/-
  init_pmknop(1,270,450,40,15,0);  // lin / log

  init_pmknop(2,365,420,40,15,0);  // y div +/-

  init_knop(12,555,457,49,12,6,10,FALSE); // on/off

  init_pmknop(3,580,160,40,40,1); // up/down
}

/*****************************************************************************
*                                                                            *
*****************************************************************************/
void init(void)
{
  s=m[mode];

  sbsettc(s.sample_freq[s.xlin_scale]);
  make_window();
  setcolors();
  makekeys();
  prep_display();
  build_display();
  init_sb16mix(10,377,sb16_present);
  write_divisions();
  write_displaytext(0);
  write_menutext();
}

/*****************************************************************************
*                                                                            *
*****************************************************************************/
void system_switch(int n)
{

  switch(s.input_mode)
  {
    case sb16_8  :while(dmacount8()!=0xFFFF);  // wacht op einde DMA cycle
		  break;
    case sb16_16 :while(dmacount16()!=0xFFFF);
		  break;
    case internal:break;
  }

  s.refresh=TRUE;
  m[mode]=s;
  mode=n;
  s=m[mode];

  sbsettc(s.sample_freq[s.xlin_scale]);
  make_window();
  prep_display();

  switch(s.input_mode)
  {
    case sb16_8 :  processing=processing_8sb16;
		   sbrec8(s.samples[s.xlin_scale]);
		   break;
    case sb16_16:  processing=processing_16sb16;
		   sbrec16(s.samples[s.xlin_scale]);
		   break;
    case internal :processing=processing_internal;
		   break;
  }
}

/*****************************************************************************
*                                                                            *
*****************************************************************************/
void prep_display(void)
{
  if (s.y_loglin==linear)
  {
  Y_scaler  =  (float)DISPLAY_H / (s.ylin_min - s.ylin_max)/s.ylin_division[s.ylin_scale];
  Y_offset  =  (float)DISPLAY_Y - (float)DISPLAY_H * s.ylin_max / (s.ylin_min - s.ylin_max);
  Y2_offset = -(float)DISPLAY_H * s.ylin_max / (s.ylin_min - s.ylin_max);
  }
  else
  {
  Y_scaler  =  (float)DISPLAY_H / (s.ylog_min - s.ylog_max)/s.ylog_division[s.ylog_scale];
  Y_offset  =  (float)DISPLAY_Y - (float)DISPLAY_H * s.ylog_max / (s.ylog_min - s.ylog_max);
  Y2_offset = -(float)DISPLAY_H * s.ylog_max / (s.ylog_min - s.ylog_max);
  }

  X_scaler =  (float)DISPLAY_W / (s.x_divisions *s.xlin_division[s.xlin_scale]);
  X_offset =  (float)DISPLAY_X - (float)DISPLAY_W* s.xlin_min / s.x_divisions;
}

/*****************************************************************************
*                                                                            *
*****************************************************************************/
void make_window(void)
{
  unsigned n,smpls;

  smpls =s.samples[s.xlin_scale]-1;
  for(n = 1; n < smpls; n++)
   {
     switch(s.window_mode)
      {
	case none    : window_shape[n]=0.5;
		       break;
	case barlett : if((n-1)<=(smpls/2))
			 window_shape[n]=2.0*(n-1)/smpls;
		       else
			 window_shape[n]=2.0-2.0*(n-1)/smpls;
		       break;
	case hanning : window_shape[n]=0.5-0.5*cos(PI2*(n-1)/smpls);
		       break;
	case hamming : window_shape[n]=0.54-0.46*cos(PI2*(n-1)/smpls);
		       break;
	case blackman: window_shape[n]=0.42-0.5*cos(PI2*(n-1)/smpls)+0.08*cos(PI4*(n-1)/smpls);
		       break;
      }

     switch(s.input_mode)
     {
       case sb16_16: window_shape[n]*=system_adjustment/smpls/256.0;
		     break;
       case sb16_8:  window_shape[n]*=system_adjustment/smpls;
		     break;
       case internal:window_shape[n]*=system_adjustment/smpls/256.0;
		     break;
     }
   }
}

/*****************************************************************************
*                                                                            *
*****************************************************************************/
void switch_input(int new_mode )
{
  switch(s.input_mode)
  {
    case sb16_8 : while(dmacount8()!=0xFFFF);  // wacht op einde DMA cycle
		  break;
    case sb16_16: while(dmacount16()!=0xFFFF);
		  break;
    case internal:break;
  }

  s.input_mode=new_mode;
  for(int i=0; i<5; i++) m[i].input_mode=s.input_mode;

  switch(s.input_mode)
  {
    case sb16_8  :make_window();   // part of display scaling is included
		  processing=processing_8sb16;
		  sbrec8(s.samples[s.xlin_scale]);
		  break;
    case sb16_16 :make_window();   // part of display scaling is included
		  processing=processing_16sb16;
		  sbrec16(s.samples[s.xlin_scale]);
		  break;
    case internal:make_window();   // part of display scaling is included
		  processing=processing_internal;
		  break;
  }
}

/*****************************************************************************
*                                                                            *
*****************************************************************************/
void make_dplot()
{
  int u,n,m;
  float r,sysamplin,sysampv,y;

  peak_signal=-5000;
  peak_freq=0;
  peak_y=0;
  sysamplin=pow(10,-sysamp/10.0);
  sysampv=sqrt(sysamplin);

  for(m = 1,n = 1; m <= DISPLAY_W * s.x_multiplier[s.xlin_scale]; m+=s.x_multiplier[s.xlin_scale],n++)
   {
     switch(mode)
      {
	case scope      : r=(vector[m].re-1);
			  y = r*Y_scaler + Y2_offset+1;
			  break;
	case absolute   : r = (vector[m].re*vector[m].re + vector[m].im*vector[m].im)*sysamplin;
			  if(s.y_loglin==logaritmic)
			  {
			     r = (r <= 1.e-37) ? -1000 : 10*log10(r);
			     y = Y_scaler * r + Y2_offset;
			  }
			  else
			     y = Y_scaler * r + Y2_offset;
			  break;
	case fase       : r = atan2(vector[m].im, vector[m].re);
			  y = Y_scaler * r + Y2_offset;
			  break;
	case real       : r = vector[m].re*0.7746;
			  y = Y_scaler * r * sysampv + Y2_offset;
			  break;
	case imag       : r = vector[m].im*0.7746;
			  y = Y_scaler * r * sysampv + Y2_offset;
			  break;
      }


     if(y < 0) y = 0;
     else if(y > DISPLAY_H) y = DISPLAY_H;
     dplot[n]=(int)y;

     if((r>peak_signal)&&(m>4)) { peak_signal=r; peak_freq=m-1; peak_y=y;}
   }
}

/*****************************************************************************
*                                                                            *
*****************************************************************************/
void display_sample(void)
{
  unsigned x,y,n,m;
  char dummy[80];


  if(s.refresh==yes_refresh) cleardisplaybuf();

  for(m = 2, n = 0; m <= DISPLAY_W/s.x_magnifier[s.xlin_scale]; n+=s.x_magnifier[s.xlin_scale], m++)
   {
     switch(s.plot)
      {
	case lines :	linebuf(n,dplot[m-1],n+s.x_magnifier[s.xlin_scale],dplot[m]);
			break;
	case dots  :    putdotbuf(n,dplot[m]);
			break;
      }
   }

  if((mode==absolute)&&(s.peakon==on_peak)&&(s.refresh==yes_refresh))
  {
    x=peak_freq;
    if(x>(DISPLAY_W-80)) x=DISPLAY_W-80;
    if(peak_y<20) peak_y=20;

    if(s.y_loglin==logaritmic)
      sprintf(dummy,(" %1.1f %s"),peak_signal,s.ylog_scaletxt);
    else
      sprintf(dummy,(" %1.2f %s"),(float)peak_signal*s.ylin_scalemul[s.ylin_scale],s.ylin_scaletxt[s.ylin_scale]);
    textxy(x,peak_y-10,dummy);

    sprintf(dummy,(" %i %s\0x00"),(int)(peak_freq*s.sample_freq[s.xlin_scale] / (s.samples[s.xlin_scale] -1)),"Hz");
    textxy(x,peak_y-20,dummy);
  }

  do {} while((inportb(0x3DA) & 8)==0);
  do {} while((inportb(0x3DA) & 8)!=0);
  outpw(0x3c4,0x0102);  // select VGA plane 0
  outpw(0x3ce,0x0004);
  copydisplaybuf();
  outpw(0x3c4,0x0f02);  // select all VGA planes
}

/*****************************************************************************
*                                                                            *
*****************************************************************************/
void processing_8sb16()
{
  unsigned Spos=0, n;

  sysamp=(mode!=fase) ? gettotalamp():0;
  if(dmacount8()==0xFFFF)
  {
    switch(mode)
    {
      case fase:
      case real:
      case imag:
      case absolute: for(n = 1; n < s.samples[s.xlin_scale]; n++)
		     {
		       vector[n].re = (float)(raw_8[n]-1)*window_shape[n];
		       vector[n].im = 0;
		     }
		     sbrec8(s.samples[s.xlin_scale]);  // start DMA cycle
		     fft(vector, s.power[s.xlin_scale]);
		     break;
      case scope:    for (Spos=30; Spos<500;Spos++)
		       if(((raw_8[Spos  ]+raw_8[Spos+1]+raw_8[Spos+2]+raw_8[Spos+3])>0)&&
			((raw_8[Spos+4]+raw_8[Spos+5]+raw_8[Spos+6]+raw_8[Spos+7])<0))
			 break;
		     for(n=0;n<s.samples[s.xlin_scale];n++)
			vector[n].re = raw_8[Spos+n];
		     sbrec8(s.samples[s.xlin_scale]+500);  // start DMA cycle
		     break;
    }
    make_dplot();
    display_sample();
  }
}

/*****************************************************************************
*                                                                            *
*****************************************************************************/
void processing_16sb16()
{
  long d;
  unsigned Spos=0, n;

  if(dmacount16()==0xFFFF)
  {
    sysamp=(mode!=fase) ? gettotalamp():0;

    switch(mode)
    {
      case fase:
      case real:
      case imag:
      case absolute: for(n = 1; n < s.samples[s.xlin_scale]; n++)
		     {
		       vector[n].re = (float)((raw_16[n])-256)*window_shape[n];
		       vector[n].im = 0;
		     }
		     sbrec16(s.samples[s.xlin_scale]);
		     fft(vector, s.power[s.xlin_scale]);
		     break;
      case scope:    for (Spos=0; Spos<(s.samples[s.xlin_scale]-30);Spos++)
		       if((((long)raw_16[Spos  ]+raw_16[Spos+1]+raw_16[Spos+2]+raw_16[Spos+3])>0)&&
			 (((long)raw_16[Spos+4]+raw_16[Spos+5]+raw_16[Spos+6]+raw_16[Spos+7])<0))
		     break;
		     for(n=0;n<s.samples[s.xlin_scale];n++)
			vector[n].re =(raw_16[Spos+n] / 256.0);
		     sbrec16(s.samples[s.xlin_scale]+500);
		     break;
    }
    make_dplot();
    display_sample();
  }
}

/*****************************************************************************
*                                                                            *
*****************************************************************************/
void processing_internal()
{
  long d;
  unsigned Spos=0, n;
  float dtime=1.0/s.sample_freq[s.xlin_scale];
  float dt = dtime*random(50)/5000.0;
  int i;

  for(i=0; i<(s.samples[s.xlin_scale]+500); i++)
     raw_16[i] = (int)fabs((sin(i*(dtime+dt)*2.0*M_PI*1235.0)*21200))-10600+random(30)-15;

  sysamp=(mode!=fase) ? 0:0;

    switch(mode)
    {
      case fase:
      case real:
      case imag:
      case absolute: for(n = 1; n < s.samples[s.xlin_scale]; n++)
		     {
		       vector[n].re = (float)((raw_16[n])-256)*window_shape[n];
		       vector[n].im = 0;
		     }
		     fft(vector, s.power[s.xlin_scale]);
		     break;
      case scope:    for (Spos=0; Spos<(s.samples[s.xlin_scale]-30);Spos++)
		       if((((long)raw_16[Spos  ]+raw_16[Spos+1]+raw_16[Spos+2]+raw_16[Spos+3])>0)&&
			 (((long)raw_16[Spos+4]+raw_16[Spos+5]+raw_16[Spos+6]+raw_16[Spos+7])<0))
		      break;
		     for(n=0;n<s.samples[s.xlin_scale];n++)
			vector[n].re =(raw_16[Spos+n] / 256.0);
		     break;
    }
    make_dplot();
    display_sample();
}

/*****************************************************************************
*  MAIN: Program start and main program loop                                 *
*****************************************************************************/
void main(void)
{
  int m;
  int my, mx, mk;

  // reserve space for DMA buffer on page break
  if(!sbmalloc())
  {
     printf("SB - Memory Allocation Error\n");
     exit(1);
  }

  // detect sb-16, if sb-16 not found program will run but all
  //  sb_16 routines will be disabeld with sb16_present=FALSE
  if (!detect_blaster())
     printf("ERROR: BLASTER string not found, using default values \n\n");

  printf("SB settings - Base i/o    = %4x \n",sb_baseio);
  printf("              Irq         = %4x \n",sb_irq);
  printf("              DMA low     = %4x \n",sb_dma);
  printf("              DMA high    = %4x \n",sb_dma16);

  if (!sbinit(sb_baseio, sb_irq, sb_dma, sb_dma16))
  {
    printf("ERROR: No SB16 found on port %4x  \n",sb_baseio);
  }
  else
    printf("              DSP version = %4i \n\n\n",sb_dspversion);

  // program need mouse, no keyboard control included.
  initmouse();
  if(!mouse_present)
  {
    printf("ERROR: mouse not found \n");
    exit(3);
  }
  else
    printf("mouse found \n");

  // delay to read text
  delay(500);

  // switch to graphics mode
  if (!opengraph())
  {
    printf("ERROR: graphics error");
    closegraph();
    exit(1);
  }

  // create slides for volume control sb_16
  if (!openslides())
  {
    printf("ERROR: memmory alocation error");
    closegraph();
    exit(1);
  }

  // setup display
  init();

  setmouseevent(MEVENT_KLD+MEVENT_KLU);
  showmouse();
  program_end = FALSE;

  // select proper processing routine and get first sample
  switch(s.input_mode)
  {
    case sb16_16  :processing=processing_16sb16;
		   sbrec16(s.samples[s.xlin_scale]);
		   break;
    case sb16_8   :processing=processing_8sb16;
		   sbrec8(s.samples[s.xlin_scale]);
		   break;
    case internal :processing=processing_internal;
		   break;
  }

  // main program loop
  while(!program_end)
  {
     processing();

     while(kbhit()) if(getch()==27) program_end=TRUE;

     if(getevent(mx,my,mk)&&(mk&1))
     {
	if (mousein(mx,my,DISPLAY_X,DISPLAY_Y,DISPLAY_X+DISPLAY_W,DISPLAY_Y+DISPLAY_H))
	  cursor_mode();
	// 0 = window mode
	else if (xy_onknop(mx,my,0)&&(mode!=scope))
	     {
	       s.window_mode++;
	       if(s.window_mode>blackman) s.window_mode=none;
	       make_window();
	       write_menutext();
	       moveknop(0,processing);
	     }
	// 2 = analyser mode select ( absolute .. real )
	else if (xy_onknop(mx,my,2)&&(mode!=scope))
	     {
	       write_displaytext(7);
	       if(mode<imag)
		 system_switch(mode+1);
	       else
		 system_switch(absolute);
	       write_displaytext(0);
	       write_menutext();
	       moveknop(2,processing);
	     }
	// 5 = scope/analyser mode select
	else if (xy_onknop(mx,my,5))
	     {
	       write_displaytext(7);
	       if(mode==scope)
		 system_switch(absolute);
	       else
		 system_switch(scope);
	       write_displaytext(0);
	       write_menutext();
	       moveknop(5,processing);
	     }
	// 7 = input selection
	else if (xy_onknop(mx,my,7))
	     {
	       m=s.input_mode+1;
	       if(m>internal) m=sb16_8;
	       switch_input(m);
	       write_menutext();
	       moveknop(7,processing);
	     }
	// 4 = peak on/off
	else if (xy_onknop(mx,my,4)&&(mode!=scope)&&(mode!=fase))
	     {
	       if(s.peakon==on_peak) s.peakon=off_peak; else s.peakon=on_peak;
	       write_menutext();
	       moveknop(4,processing);
	     }
	// 6 = lines / dot switch
	else if (xy_onknop(mx,my,6))
	     {
	       if(s.plot==lines) s.plot=dots; else s.plot=lines;
	       write_menutext();
	       moveknop(6,processing);
	     }
	// 9 = refresh on/off switch
	else if (xy_onknop(mx,my,9))
	     {
	       if(s.refresh==yes_refresh) s.refresh=no_refresh; else s.refresh=yes_refresh;
	       write_menutext();
	       moveknop(9,processing);
	     }
	// 12 = exit program
	else if (xy_onknop(mx,my,12))
	     {
	       program_end=TRUE;
	       moveknop(12,processing);
	     }
        // 0-0/1 = inc / dec Y-division
	else if (xy_onpmknop(mx,my,0))
	     {
	       switch(pmknopin(0))
	       {
		 case 0: up_down_y(FALSE);
			 movepmknop(0,processing);
			 break;
		 case 1: up_down_y(TRUE);
			 movepmknop(0,processing);
			 break;
	       }
	     }
	// 1-0/1 = log / lin Y-scale
	else if (xy_onpmknop(mx,my,1))
	     {
	       write_displaytext(7);
	       switch(pmknopin(1))
	       {
		 case 0: if((s.ylog_scalemax>0)&&(s.y_loglin!=logaritmic))
			   s.y_loglin=logaritmic;
			 break;
		 case 1: if((s.ylin_scalemax>0)&&(s.y_loglin!=linear))
			   s.y_loglin=linear;
			 break;
	       }
	       prep_display();
	       write_displaytext(0);
	       write_menutext();
	       movepmknop(1,processing);
	     }
	// 2-0/1 = inc/dec X-division
	else if (xy_onpmknop(mx,my,2))
	     {
	       switch(pmknopin(2))
	       {
		 case 0: up_down_x(FALSE);
			 break;
		 case 1: up_down_x(TRUE);
			 break;
	       }
	       prep_display();
	       movepmknop(2,processing);
	     }
	// 3-0/1 up/down display
	else if (xy_onpmknop(mx,my,3))
	     {
	       switch(pmknopin(3))
	       {
		 case 0: shift_y(FALSE);
			 break;
		 case 1: shift_y(TRUE);
			 break;
	       }
	       prep_display();
	       movepmknop(3,processing);
	     }
	else
	 if (sb16_present)
	 {
	   sb16_mixer(mx,my);              // mixer switches
	   moveslide(0,mx,my,processing);  // speaker volume
	   moveslide(1,mx,my,processing);  // aux volume
	   moveslide(2,mx,my,processing);  // cd volume
	   moveslide(3,mx,my,processing);  // mic volume
	 }
      } // if event
  } // while (main loop)

  hidemouse();
  setmouseevent(MEVENT_OFF);
  closeslides();
  closegraph();
  sbfree();
}

/****************************** END *****************************************/



