{$R+}

(* KAYESTIM.PAS

This Turbo Pascal program for standard MS-DOS estimates the internal
rate of return by implementing the discrete version by Salmi and
Luoma (1981) of Kay's (1976) originally continuous model.

Kay, J.A. (1976), "Accountants, too, could be happy in a golden age:
the accountants rate of profit and the internal rate of return",
Oxford Economic Papers (New Series) 28/3, 447-460.

Kay, J.A. (1978), "Accounting rate of profit and internal rate of
return; a reply", Oxford Economic Papers 30/3, 469-470.

Salmi, T., and Luoma, M. (1981), "Deriving the internal rate of
return from accountant's rate of profit: analysis and empirical
estimation", Finnish Journal of Business Economics 1, 20-45.

This is the estimation formula (11) from Salmi and Luoma (1981).
page 25.

           n            t
             p(t)/(1+a)
          t=1
(11)  a = ---------------
           n             t
            v(t-1)/(1+a)
          t=1

where
  p(t) = accountant's profit (operating income) in year t
  v(t) = book value of the firm's assets at the end of year t
  n    = number of years under observation

The a from (11) is solved by the secant method familiar from
numerical analysis.

The program utilizes auxiliary routines from the Turbo Pascal unit
SIMUTPU.

The header of the program
procedure HEADER (progName, progDesc, progVers,
                  progDate, prefix : string);

Delete trailing white spaces from a string
function TRAILFN (sj : string) : string;

Get substrings from a string
function PARSERFN (sj : string; PartNumber : integer) : string;

Generalized power function for Turbo Pascal
function GENPOWFN (number, exponent : real) : real;

Test whether a file exists
function FILEXIST (name : string) : boolean;

Usage of the program: KAYESTIM DATAFILE

DATAFILE contents. All the lines with (#) are comments. Likewise,
empty lines are ignored. Hence the estimation in the numerical
example below would be based on seven observations.

#SIMUANNU; Simulate observations; annuity depreciation
#Copyright (c) by Timo Salmi and Ilkka Virtanen (Ver. 1.2) 4-Mar-95
#Department of Accounting and Business Finance; Department of
#Mathematics and Statistics, University of Vaasa, Finland
#First year: 1980
#Years to simulate: 11
#Growth rate: 0.0800
#Amplitude of cycles: 0.0000
#Shock coefficient: 2.5000
#Shock timing: 9999
#First capital expenditure: 40.0000
#   1    0.7000
#   2    0.6000

#Year     Capital   FundsFrom     Annuity   Operating        Book
#        expendit  operations   depreciat      income       value
#            g(t)        f(t)        d(t)        p(t)        v(t)
#1980     40.0000      0.0000      0.0000      0.0000     40.0000
#1981     43.2000     28.0000     20.0000      8.0000     63.2000
#1982     46.6560     54.2400     41.6000     12.6400     68.2560
#1983     50.3885     58.5792     44.9280     13.6512     73.7165
 1984     54.4196     63.2655     48.5222     14.7433     79.6138
 1985     58.7731     68.3268     52.4040     15.9228     85.9829
 1986     63.4750     73.7929     56.5963     17.1966     92.8615
 1987     68.5530     79.6964     61.1240     18.5723    100.2905
 1988     74.0372     86.0721     66.0140     20.0581    108.3137
 1989     79.9602     92.9578     71.2951     21.6627    116.9788
 1990     86.3570    100.3945     76.9987     23.3958    126.3371

#True internal rate of return = 20.0000

*)

program KayEstimProgram;  { Discrete form of Kay's IRR estimation }

uses SIMUTPU;  { A unit of general auxiliary routines }

const progDesc = 'Estimate IRR with Salmi & Luoma '
               + 'interepration of KAY''s algorithm';
      progName = 'KAYESTIM';
      progDate = '4-Mar-95';
      progVers = '1.2';
      maxYears = 50;

var con : text;  { For non-redirectable screen output }

type realVectorType = array [1..maxYears] of real;

var p,  { accountant's profit }
    v   { book value }
     : realVectorType;

var n,          { number of years }
    iter        { number of iterations }
     : integer;

var irrEstimate { the result }
     : real;

(* Read and display the input data *)
procedure READDATA (var p, v : realVectorType; var n : integer);
var s : string;   { read line as string }
    k : integer;  { error index }
    j : integer;  { position of the comment initiator '#' }
    year : integer;
    filePointer : text;
    flag : boolean;
begin
  {}
  {... open the input file ...}
  if ParamCount > 0 then begin
    if not FILEXIST (ParamStr(1)) then begin
      writeln;
      writeln ('File ', ParamStr(1), ' not found');
      halt;
    end; {if}
    assign (filePointer, ParamStr(1));
  end
  else begin
    writeln;
    writeln ('Usage: ', progName, ' InputFileName');
    halt;
  end; {endif}
  reset (filePointer);
  {}
  {... read the annual data from the data file ...}
  n := 1;
  flag := true;
  while not eof(filePointer) do begin
    readln (filePointer, s);
    s := TRAILFN(s);
    if s = 'end' then break;
    j := Pos('#',s);
    if (Length(s) > 0) and ((j = 0) or (j > 1)) then begin
      if j > 1 then s := TRAILFN (Copy (s, 1, j-1));
      Val (PARSERFN(s,1), year, k);
      Val (PARSERFN(s,5), p[n], k);
      Val (PARSERFN(s,6), v[n], k);
      if flag then begin
        writeln ('Year   Operating income   Book value');
        flag := false;
      end; {if}
      writeln (year : 4, p[n]:19:4, v[n]:13:4);
      Inc (n);
    end {if}
      else writeln (s);
  end; {while}
  {}
  close (filePointer);
  n := n - 1;
  writeln ('Number of years: ', n);
end;  (* readdata *)

(* A routine for the function of sums *)
function FFN (a : real; p, v : realVectorType; n : integer) : real;
var sum : real;
    t   : integer;
begin
  sum := 0.0;
  for t := 2 to n do
    sum := sum + p[t] / GENPOWFN (1.0 + a, t)
         - a * v[t-1] / GENPOWFN (1.0 + a, t);
  ffn := sum;
end;  (* ffn *)

(* Solve Kay's a estimate of the internal rate of return,
   secant method *)
function IRRESTFN (p, v : realVectorType;
                   n : integer;
                   var iter : integer) : real;
const maxIter = 30;
var a, a1, a2, f1, f2 : real;
    i : integer;
begin
  a1 := 0.0;
  a2 := 0.2;
  for i := 1 to maxIter do begin
    f1 := FFN (a1, p, v, n);
    f2 := FFN (a2, p, v, n);
    a  := (a1 * f2 - a2 * f1) / (f2 - f1);
    a1 := a2;
    a2 := a;
    irrestfn := a2;
    iter := i;
    if (abs(a2-a1) < 0.000001) then exit;
  end; {for}
  writeln ('The algorithm failed in ', maxIter, ' iterations');
  halt;
end;  (* irrestfn *)

(* Main program *)
begin
  FileMode := 0;
  HEADER (progName, progDesc, progVers, progDate, '');
  READDATA (p, v, n);
  irrEstimate := IRRESTFN (p, v, n, iter);
  writeln;
  writeln ('Estimated internal rate of return a = ',
            100.0*irrEstimate:0:4, '%');
  writeln ('Number of iterations = ', iter);
end.  (* KayEstimProgram *)
