{$R+}  {Range check}

(* FINDNEGB.PAS

Find the negative binomial contribution distribution that gives the
target internal rate of return for a given lifespan *)

program FindNegBinProgram;

uses SIMUTPU;

const progDesc = 'Find negative binomial contribution distribution for target IRR';
      progName = 'FINDNEGB';
      progDate = '14-Mar-95';
      progVers = '1.0';
      maxYears = 50;

(* Two of the four parameters given as constants for simplicity *)
const r =  2;             { location parameter for the distribution }
      q =  0.85;          { shape parameter for the distribution }

type realVectorType = array [0..maxYears] of real;

var lifeSpan
     : integer;

var targetIrr
     : real;

var  b                    { the sought contribution coefficients }
     : realVectorType;

(* Get the parameters *)
procedure GETPARAM (var lifeSpan  : integer;
                    var targetIRR : real);
var s : string;
    k : integer;
    target : real;
begin
  if ParamCount < 2 then begin
    writeln ('Usage: FINDNEGB lifeSpan targetIRR%');
    halt;
  end;
  s := ParamStr(1);
  Val (s, lifeSpan, k);
  if k > 0 then begin writeln ('Error in value ', s); halt; end;
  s := ParamStr(2);
  Val (s, target, k);
  if k > 0 then begin writeln ('Error in value ', s); halt; end;
  targetIRR := target / 100.0;
end; (* getparam *)

(* Auxiliary function to calculate net present value *)
function NPVFN (b        : realVectorType;
                rate     : real;
                lifeSpan : integer) : real;
var y : real;
    i : integer;
begin
  y := -1.0;
  for i := 0 to lifeSpan do
    y := y + b[i] / GENPOWFN (1.0 + rate, i);
  npvfn := y;
end;  (* npvfn *)

(* Internal rate of return for the contribution distribution *)
function IRRFN (b : realVectorType; lifeSpan : integer) : real;
const maxIter = 30;
var x, x1, x2, y1, y2 : real;
    j : integer;
begin
  x1 := 0.0;
  x2 := 0.2;
  for j := 1 to maxIter do begin
    y1 := NPVFN (b, x1, lifeSpan);
    y2 := NPVFN (b, x2, lifeSpan);
    x := (x1*y2 - x2*y1) / (y2 - y1);
    x1 := x2;
    x2 := x;
    irrfn := x;
    if abs(x2-x1) < 0.000001 then exit;
  end; {for}
  writeln ('The IRRFN algorithm failed in ', maxIter, ' iterations');
  halt;
end;  (* irrfn *)

(* Form the negative binomial contribution distribution for given
   parameters: N = lifespan, r = location, q = shape, s = scaling *)
procedure CALCULATE (N : integer;
                     r : real;
                     q : real;
                     s : real;
                     var b : realVectorType);
var t : integer;
    P : realVectorType;   { auxiliary vector }
begin
  FillChar (P, sizeOf(P), 0);
  P[0] := GENPOWFN (1.0-q, r);
  for t := 1 to N do begin
    P[t] := (r + t - 1.0) / t * q * P[t-1];
    b[t] := s * P[t];
  end; {for}
end; (* calculate *)

(* Are the values of different signs *)
function DIFFSGFN (f1, f2 : real) : boolean;
begin
  if (f1 < 0.0) and (f2 > 0.0) then
    diffsgfn := true
  else if (f1 > 0.0) and (f2 < 0.0) then
    diffsgfn := true
  else
    diffsgfn := false;
end;  (* diffsgfn *)

(* Interate with the bisection method *)
procedure BISECT (lifeSpan  : integer;
                  targetIRR : real;
                  var b     : realVectorType);
const maxIter = 30;
      zero    = 0.000001;
var sc, sc1, sc2, f, f1, f2 : real;
    b1, b2 : realVectorType;
    j : integer;
begin
  writeln ('Iterating...');
  FillChar (b, sizeOf(b), 0);
  FillChar (b1, sizeOf(b), 0);
  FillChar (b2, sizeOf(b), 0);
  sc1 := 1.0; { scaling }
  CALCULATE (lifeSpan, r, q, sc1, b1);
  f1 := IRRFN (b1, lifeSpan) - targetIRR;
  writeln ('irr1 = ', 100.0*(f1+targetIRR):13:10, '%');
  {}
  sc2 := 10.0;
  CALCULATE (lifeSpan, r, q, sc2, b2);
  f2 := IRRFN (b2, lifespan) - targetIRR;
  writeln ('irr2 = ', 100.0*(f2+targetIRR):13:10, '%');
  {}
  if not DIFFSGFN(f1,f2) then begin
    writeln ('Failure in BISECT to find initial ',
             'values on different sides of the target');
    halt;
  end; {if}
  {}
  for j := 1 to MaxIter do begin
    sc := (sc1 + sc2) / 2.0;
    CALCULATE (lifeSpan, r, q, sc, b);
    writeln ('scaling factor = ', sc:0:10);
    f := IRRFN (b, lifespan) - targetIRR;
    writeln ('irr = ', 100*(f+targetIRR):0:10, '%');
    if (abs (f - f1) < zero) and (f > 0) then exit;
    {}
    CALCULATE (lifeSpan, r, q, sc1, b1);
    f1 := IRRFN (b1, lifeSpan) - targetIRR;
    CALCULATE (lifeSpan, r, q, sc2, b2);
    f2 := IRRFN (b2, lifespan) - targetIRR;
    {}
    if DIFFSGFN (f, f1) then
      begin
        sc2 := sc;
        b2 := b;
        f2 := f;
      end
    else
      begin
        sc1 := sc;
        b1 := b;
        f1 := f;
      end;
  end; {for}
  writeln ('The BISECT algorithm failed in ', maxIter, ' iterations');
  halt;
end;  (* bisect *)

(* Present the results *)
procedure SHOW (lifeSpan : integer;
                b        : realVectorType);
var irr : real;
    i   : integer;
begin
  writeln;
  irr := IRRFN (b, lifeSpan);
  writeln ('IRR = ', 100.0 * irr :0:8, '%');
  for i := 1 to lifespan do writeln (b[i]:10:8, '  #b[', i, ']');
end;  (* show *)

(* Main program *)
begin
  HEADER (progName, progDesc, progVers, progDate, '');
  writeln;
  GETPARAM (lifeSpan, targetIRR);
  BISECT (lifespan, targetIRR, b);
  SHOW (lifeSpan, b);
end. (* FindNegBinProgram *)
