MODULE ScrDemo;
  (*
    MULTIGRAPH: Graphics Support Library - Screen Level
    -----------------------------------------------------
    Copyright (C) 1993-1995, Alex Iakovlev & Dimon Maslov
    -----------------------------------------------------
    Demonstration Program, Ver M.19
  *)
  (*
    **********************************************************************
    Compilers:  TopSpeed / Stony Brook Modula-2
    ----------------------------------------------------------------------
    A.Iakovlev  Ver  M.1                                      June 90
    A.Iakovlev  Ver  M.2   Multiple fonts                     July 90
    A.Iakovlev  Ver  M.3   Cursors and so on                  August 90
    A.Iakovlev  Ver  M.4   Pages and page origins             August 90
    A.Iakovlev  Ver  M.5   Screen level redesign              November 90
    A.Iakovlev  Ver  M.6   Text cursors...                    January 91
    A.Iakovlev  Ver  M.7   Area cursors...                    March 91
    A.Iakovlev  Ver  M.8   Use Pause key for freezing         July 91
    A.Iakovlev  Ver  M.9   Texts in separate file             August 91
    A.Iakovlev  Ver  M.10  Simplified ScrTxOut                October 91
    A.Iakovlev  Ver  M.11  Changes in Polyline/Polygon        June 92
    A.Iakovlev  Ver  M.12  Dynamic graphics modes in Scr      August 92
    A.Iakovlev  Ver  M.13  Renaming                           June 93
    A.Iakovlev  Ver  M.14  Use modules Strings, Terminal      January 94
    A.Iakovlev  Ver  M.15  Delay and randoms for Stony Brook  February 94
    A.Iakovlev  Ver  M.16  Separate structured constants      April 94
    A.Iakovlev  Ver  M.17  High and true colors               March 95
    A.Iakovlev  Ver  M.18  No structured constants            March 95
    A.Iakovlev  Ver  M.19  United modules ScrOut/ScrCur       March 95
    **********************************************************************
    TopSpeed Compiler dependencies:
    ------ predefined type LONGCARD
    ------ options
    Stony Brook Compiler dependencies:
    ------ predefined type LONGCARD
    ------ options
    TopSpeed library dependencies:
    ------ import from Strings: Assign, Delete, Copy, Concat, Length
    ------ import from Terminal: Write, WriteString, WriteLn
    Stony Brook library dependencies:
    ------ import from Strings: Assign, Delete, Copy, Concat, Length
    ------ import from Terminal: Write, WriteString, WriteLn
  *)

  IMPORT
    SYSTEM, Strings, Terminal,
    SrvLex, Geo, Res, DrvKbd, DrvClk, DrvDsk,
    Scr, ScrOut, ScrCur, ScrFnt;

  (* ======================================= Main constants and globals *)

  CONST
    TextStrLen = 64;  (* --- maximum length of info string *)
    BorderSize = 22;  (* --- width of the screen border *)
    TimeOut = 5;      (* --- timeout in seconds *)
    HeadStrNmb = 16;
    ModeStrNmb = 21;

  TYPE
    Info =
    (
      adapter, press, resolution,
      noMemory, noPalettes, noShades, redXXX, greenXXX, blueXXX,
      noOrgs, org600
    );
    Feature =
    (
      title,
      dotMarkers, plusMarkers, asteriskMarkers,
      circleMarkers, crossMarkers, squareMarkers,
      rectangles, solidRectangles, stippledRectangles,
      ellipses, solidEllipses, stippledEllipses,
      polylines, solidPolygons, stippledPolygons,
      texts,
      dotCursors, plusCursors, asteriskCursors,
      circleCursors, crossCursors, squareCursors,
      rectCursors, charCursors, stringCursors,
      copyingReplace, copyingInvert,
      palettes, shades, origins
    );

  TYPE
    TextString = ARRAY [0 .. TextStrLen - 1] OF CHAR;

  VAR
    vpm: Scr.PIXMAP;
    colNmb: LONGCARD;
    fontsNmb, hatchesNmb: CARDINAL;
    mainFont, hatchFont: ScrOut.FONT;
    scrnWdt, scrnHgt, charWdt, charHgt: CARDINAL;
    endMark: CARDINAL;
    infoStr: ARRAY Info OF TextString;
    ftreStr: ARRAY Feature OF TextString;
    headStr: ARRAY [0 .. HeadStrNmb - 1] OF TextString;
    modeStr: ARRAY [0 .. ModeStrNmb - 1] OF TextString;
    escPressed: BOOLEAN;
    ftre: Feature;
    bkAttrs: ScrOut.Attrs;
    scrnV, viewV: Scr.View;
    nullPt: Geo.Point;

  (* =========================================== Errors reporting *)

  PROCEDURE Error(file, msg: ARRAY OF CHAR; halt: BOOLEAN);
    VAR
      key: CARDINAL;
    BEGIN
      IF Scr.Graph() THEN Scr.Clear(vpm, 0); Scr.TermGraph() END;
      Terminal.WriteLn();
      Terminal.WriteString(file);
      Terminal.WriteString(msg);
      Terminal.WriteLn();
      Terminal.WriteString("To continue press any key");
      Terminal.WriteLn();
      REPEAT UNTIL DrvKbd.Get(key);
      IF halt THEN HALT() END;
    END Error;

  (* ======================================================== Files loading *)

  PROCEDURE LoadString(filename: ARRAY OF CHAR; VAR str: ARRAY OF CHAR);
    CONST
      NoFileMsg = " --- file not found";
      LargeFileMsg = " --- file too large";
      BadFileMsg = " --- bad file";
      WordShift = 10000H;
    VAR
      f: DrvDsk.File;
      low, hgh, read: CARDINAL;
      size: LONGCARD;
    BEGIN
      f := DrvDsk.Open(filename);
      IF DrvDsk.result = 0 THEN
        size := DrvDsk.Size(f);
        hgh := VAL(CARDINAL, size DIV WordShift);
        low := VAL(CARDINAL, size MOD WordShift);
        IF (hgh = 0) AND (low <= HIGH(str) + 1) THEN
          read := DrvDsk.Read(f, SYSTEM.ADR(str), low);
          IF read = low THEN
            IF low <= HIGH(str) THEN str[low] := 0C END;
            DrvDsk.Close(f);
          ELSE
            Error(filename, BadFileMsg, TRUE);
          END;
        ELSE
          Error(filename, LargeFileMsg, TRUE);
        END;
      ELSE
        Error(filename, NoFileMsg, TRUE);
      END;
    END LoadString;

  PROCEDURE LoadTexts();
    CONST
      TxtFileLen = 6000;
      TxtFileName = "SCRDEMO.T";
      InfoNmb = 20;
      FuncNmb = 50;
    VAR
      str: ARRAY [0 .. TxtFileLen - 1] OF CHAR;
      len, i0, i1, itemNo: CARDINAL;
    BEGIN
      LoadString(TxtFileName, str);
      len := Strings.Length(str);
      i0 := 0;
      WHILE i0 < len - 1 DO
        i0 := SrvLex.Skip(str, i0);
        i1 := SrvLex.ExtractTypedNumber(str, i0, SrvLex.dec, itemNo);
        i0 := SrvLex.Skip(str, i1) + 1;
        i1 := SrvLex.PosLineBreak(str, i0);
        IF i1 > len THEN i1 := len END;
        IF itemNo < InfoNmb THEN
          Strings.Copy(str, i0, i1 - i0, infoStr[VAL(Info, itemNo)]);
        ELSIF itemNo < InfoNmb + FuncNmb THEN
          Strings.Copy(str, i0, i1 - i0, ftreStr[VAL(Feature, itemNo - InfoNmb)]);
        ELSIF itemNo < InfoNmb + FuncNmb + HeadStrNmb THEN
          Strings.Copy(str, i0, i1 - i0, headStr[itemNo - InfoNmb - FuncNmb]);
        ELSIF itemNo < InfoNmb + FuncNmb + HeadStrNmb + ModeStrNmb THEN
          Strings.Copy(str, i0, i1 - i0, modeStr[itemNo - InfoNmb - FuncNmb - HeadStrNmb]);
        END;
        i0 := i1 + 2;
      END;
    END LoadTexts;

  (* ========================================================= Randoms *)

  (*
     ------ Sorry, the code for randoms was taken from JPI's library
     ------ as Stony Brook doesn't contain them...
  *)

  CONST
    HistoryMax = 54;

  VAR
    history: ARRAY [0 .. HistoryMax] OF CARDINAL;
    historyPtr, lowerPtr: CARDINAL;

  (*$O-*)
  (*/NOCHECK:O*)

  PROCEDURE SetUpHistory(seed: CARDINAL);
    VAR
      x: LONGCARD;
      i: CARDINAL;
    BEGIN
      historyPtr := HistoryMax;
      lowerPtr := 23;
      x := VAL(LONGCARD, seed);
      i := 0;
      REPEAT
        x := x * 3141592621 + 17;
        history[i] := VAL(CARDINAL, x DIV 10000H);
        INC(i);
      UNTIL i > HistoryMax;
    END SetUpHistory;

  PROCEDURE CardRandom(range: CARDINAL): CARDINAL;
    VAR
      res: CARDINAL;
    BEGIN
      IF historyPtr = 0 THEN
        IF lowerPtr = 0 THEN
          SetUpHistory(12345);
        ELSE
          historyPtr := HistoryMax;
          DEC(lowerPtr);
        END;
      ELSE
        DEC(historyPtr);
        IF lowerPtr = 0 THEN
          lowerPtr := HistoryMax;
        ELSE
          DEC(lowerPtr);
        END;
      END;
      res := history[historyPtr] + history[lowerPtr];
      history[historyPtr] := res;
      IF range = 0 THEN
        RETURN res;
      ELSE
        RETURN res MOD range;
      END;
    END CardRandom;

  (*/REVERT*)
  (*$O=*)

  PROCEDURE LongCardRandom(range: LONGCARD): LONGCARD;
    VAR
      res: LONGCARD;
    BEGIN
      IF range <= 10000H THEN
        RETURN VAL(LONGCARD, CardRandom(VAL(CARDINAL, range)));
      ELSE
        res := VAL(LONGCARD, CardRandom(0)) * 10000H + VAL(LONGCARD, CardRandom(0));
        RETURN res MOD range;
      END;
    END LongCardRandom;

  PROCEDURE Randomize();
    BEGIN
      SetUpHistory(DrvClk.Time());
    END Randomize;

  (* =========================================================== Timing *)

  PROCEDURE SetEndMark(i: CARDINAL);
  (* --------------- set time mark in future for end of repeating *)
    BEGIN
      endMark := DrvClk.Time() + i;
    END SetEndMark;

  PROCEDURE EndRepeat(): BOOLEAN;
    CONST
      TimeRange = 20;  (* --- check time with probability 1/20 *)
    VAR
      key: CARDINAL;
    BEGIN
      IF DrvKbd.Get(key) THEN
        escPressed := (key = DrvKbd.Escape);
        RETURN TRUE;
      ELSE
        RETURN (CardRandom(TimeRange) = 0) AND (DrvClk.Time() >= endMark);
      END;
    END EndRepeat;

  (* ===================================== Box utilities *)

  TYPE
    Box =
      RECORD
        s0: CARDINAL;
        r0: Geo.Rectangle;
        dx, dy: [-1 .. +1];
      END;

  PROCEDURE InitBox(r: Geo.Rectangle; s: CARDINAL; VAR p: Geo.Point; VAR b: Box);
    VAR
      wx, wy, dmax, d0: CARDINAL;
    BEGIN
      b.r0 := r;
      (* ---------- determine rectangle sizes *)
      wx := b.r0.ixMax - b.r0.ixMin + 1;
      wy := b.r0.iyMax - b.r0.iyMin + 1;
      (* ------------------------- set initial point *)
      p.ix := b.r0.ixMin + VAL(INTEGER, CardRandom(wx));
      p.iy := b.r0.iyMin + VAL(INTEGER, CardRandom(wy));
      (* ---------------- determine rectangle halfsizes *)
      wx := wx DIV 2;
      wy := wy DIV 2;
      (* ---------- dmax is maximum width of return zone! *)
      IF wx < wy THEN
        dmax := wx;
      ELSE
        dmax := wy;
      END;
      (* -------- find width of return zone *)
      IF s - 1 <= dmax THEN
        d0 := s - 1;
      ELSE
        d0 := dmax;
      END;
      (* --------- correct rectangle *)
      INC(b.r0.ixMin, d0);
      DEC(b.r0.ixMax, d0);
      INC(b.r0.iyMin, d0);
      DEC(b.r0.iyMax, d0);
      (* ------------ set maximum step *)
      b.s0 := d0 + 1;
      (* ------------------ set initial directions *)
      b.dx := 2 * VAL(INTEGER, CardRandom(2)) - 1;
      b.dy := 2 * VAL(INTEGER, CardRandom(2)) - 1;
    END InitBox;

  PROCEDURE ChangePoint(VAR p: Geo.Point; VAR b: Box);
    CONST
      ProbDirChange = 60;
    BEGIN
      IF p.ix < b.r0.ixMin THEN
        b.dx := + 1;
      ELSIF p.ix > b.r0.ixMax THEN
        b.dx := - 1;
      ELSIF CardRandom(ProbDirChange) = 0 THEN
        b.dx := - b.dx;
      END;
      IF p.iy < b.r0.iyMin THEN
        b.dy := + 1;
      ELSIF p.iy > b.r0.iyMax THEN
        b.dy := - 1;
      ELSIF CardRandom(ProbDirChange) = 0 THEN
        b.dy := - b.dy;
      END;
      INC(p.ix, b.dx * VAL(INTEGER, CardRandom(b.s0)));
      INC(p.iy, b.dy * VAL(INTEGER, CardRandom(b.s0)));
    END ChangePoint;

  PROCEDURE BoundRect(VAR r: Geo.Rectangle; max: CARDINAL);
    BEGIN
      WHILE r.ixMax - r.ixMin + 1 > VAL(INTEGER, max) DO
        INC(r.ixMin);
        DEC(r.ixMax);
      END;
      WHILE r.iyMax - r.iyMin + 1 > VAL(INTEGER, max) DO
        INC(r.iyMin);
        DEC(r.iyMax);
      END;
    END BoundRect;

  (* ================================================ String utilities *)

  PROCEDURE Append(VAR res: ARRAY OF CHAR; src: ARRAY OF CHAR);
    BEGIN
      Strings.Concat(res, src, res);
    END Append;

  PROCEDURE LongCardToStr(x: LONGCARD; VAR s: ARRAY OF CHAR; VAR ok: BOOLEAN);
    VAR
      x0: LONGCARD;
      j, len: CARDINAL;
    BEGIN
      x0 := x;
      len := 0;
      WHILE x0 > 0 DO x0 := x0 DIV 10; INC(len) END;
      IF len = 0 THEN INC(len) END;
      IF len - 1 <= HIGH(s) THEN
        IF len - 1 < HIGH(s) THEN s[len] := 0C END;
        FOR j := len - 1 TO 0 BY -1 DO
          s[j] := CHR(VAL(CARDINAL, x MOD 10) + ORD("0"));
          x := x DIV 10;
        END;
        ok := TRUE;
      ELSE
        ok := FALSE;
      END
    END LongCardToStr;

  PROCEDURE CardToStr(x: CARDINAL; VAR s: ARRAY OF CHAR; VAR ok: BOOLEAN);
    BEGIN
      LongCardToStr(VAL(LONGCARD, x), s, ok);
    END CardToStr;

  (* ================================================ Font utilities *)

  PROCEDURE Font(): ScrOut.FONT;
    CONST
      MinLinesNmb = 25;
      MaxLinesNmb = 30;
    VAR
      j, w, h, ln: CARDINAL;
      font: ScrOut.FONT;
    BEGIN
      FOR j := 2 TO fontsNmb - 1 DO
        font := ScrOut.StdFont(j);
        ScrOut.Dims(font, w, h);
        ln := scrnHgt DIV h;
        IF (MinLinesNmb <= ln) AND (ln <= MaxLinesNmb) THEN
          RETURN font;
        END;
      END;
      RETURN ScrOut.StdFont(0);
    END Font;

  PROCEDURE RandomFont(): ScrOut.FONT;
    BEGIN
      IF fontsNmb > 2 THEN
        RETURN ScrOut.StdFont(CardRandom(fontsNmb - 2) + 2);
      ELSE
        RETURN ScrOut.StdFont(0);
      END;
    END RandomFont;

  PROCEDURE CharsNmb(f: ScrOut.FONT): CARDINAL;
    VAR
      s: ScrOut.CharSet;
      count: CARDINAL;
      ch: CHAR;
    BEGIN
      ScrOut.Chars(f, s);
      count := 0;
      FOR ch := MIN(CHAR) TO MAX(CHAR) DO
        IF ch IN s THEN INC(count) END;
      END;
      RETURN count;
    END CharsNmb;

  (* ======================================= Colors determination *)

  PROCEDURE RandomColor(): Scr.Pixel;
    BEGIN
      RETURN VAL(Scr.Pixel, 1 + LongCardRandom(colNmb - 1));
    END RandomColor;

  PROCEDURE TitleColor(): Scr.Pixel;
    BEGIN
      RETURN Scr.RGBColor(0, 128, 0);
    END TitleColor;

  PROCEDURE MsgColor(): Scr.Pixel;
    BEGIN
      RETURN Scr.RGBColor(0, 128, 128);
    END MsgColor;

  (* =================================== Headers and bottom info *)

  VAR
    oldLen: CARDINAL;

  PROCEDURE Rect(n: CARDINAL; VAR r: Geo.Rectangle);
    BEGIN
      r.ixMin := (scrnWdt - n * charWdt) DIV 2 - 1;
      r.ixMax := (scrnWdt + n * charWdt) DIV 2 + 1;
      r.iyMin := (BorderSize - charHgt) DIV 2 - 1;
      r.iyMax := (BorderSize + charHgt) DIV 2 + 1;
    END Rect;

  PROCEDURE NoHeader();
    VAR
      r: Geo.Rectangle;
    BEGIN
      Rect(oldLen, r);
      Scr.SetClip(vpm, FALSE);
      ScrOut.FillRectangle(vpm, bkAttrs, r);
      Scr.SetClip(vpm, TRUE);
    END NoHeader;

  PROCEDURE Header(f: Feature);
    VAR
      len: CARDINAL;
      r: Geo.Rectangle;
      p: Geo.Point;
      a: ScrOut.Attrs;
    BEGIN
      NoHeader();
      IF f # title THEN
        len := Strings.Length(ftreStr[f]);
        Rect(len, r);
        Geo.RectBasePt(r, p);
        INC(p.ix, 2);
        INC(p.iy, 2);
        ScrOut.SetTextAttrs
        (
          a,
          Scr.copy, MsgColor(), 0, ScrOut.fillStippled, mainFont
        );
        Scr.SetClip(vpm, FALSE);
        Scr.SolidRect(vpm, r, 0, Scr.copy);
        ScrOut.String(vpm, a, p, ftreStr[f]);
        Scr.SetClip(vpm, TRUE);
        oldLen := len;
      END;
    END Header;

  PROCEDURE BottomInfo();
    CONST
      XStr = " x ";
    VAR
      r: Geo.Rectangle;
      p: Geo.Point;
      s, num: TextString;
      ok: BOOLEAN;
      n: CARDINAL;
      a: ScrOut.Attrs;
    BEGIN
      Strings.Assign(infoStr[resolution], s);
      Append(s, " ");
      CardToStr(scrnWdt, num, ok);
      Append(s, num);
      Append(s, XStr);
      CardToStr(scrnHgt, num, ok);
      Append(s, num);
      Append(s, XStr);
      LongCardToStr(colNmb, num, ok);
      Append(s, num);
      Rect(Strings.Length(s), r);
      Geo.ShiftRect(r, 0, scrnHgt - BorderSize);
      Geo.RectBasePt(r, p);
      INC(p.ix, 2);
      INC(p.iy, 2);
      ScrOut.SetTextAttrs
      (
        a,
        Scr.copy, MsgColor(), 0, ScrOut.fillStippled, mainFont
      );
      Scr.SetClip(vpm, FALSE);
      Scr.SolidRect(vpm, r, 0, Scr.copy);
      ScrOut.String(vpm, a, p, s);
      Scr.SetClip(vpm, TRUE);
    END BottomInfo;

  (* ============================================= Demo procedures *)

  PROCEDURE Title();
    VAR
      len, j, up: CARDINAL;
      a: ScrOut.Attrs;
      p: Geo.Point;
    BEGIN
      (* ----------------------------------------- set attributes *)
      ScrOut.SetTextAttrs
      (
        a,
        Scr.copy, TitleColor(), 0, ScrOut.fillStippled, mainFont
      );
      (* ---------------------------------------------- set up margin *)
      up := (scrnHgt - HeadStrNmb * charHgt) DIV 2;
      (* ------------------------------------------------------ drawing *)
      FOR j := 0 TO HeadStrNmb - 1 DO
        len := Strings.Length(headStr[j]);
        IF len > 0 THEN
          (* --------------------------------------------- set point *)
          p.ix := (scrnWdt - charWdt * len) DIV 2;
          p.iy := up + charHgt * j;
          (* ---------------------------------------- draw string *)
          ScrOut.String(vpm, a, p, headStr[j]);
        END;
      END;
      (* --------------------------------------------- pause *)
      SetEndMark(TimeOut);
      REPEAT UNTIL EndRepeat();
    END Title;

  PROCEDURE Markers(mt: ScrOut.MarkerStyle);
    CONST
      MarkerColorMaxLen = 20;
      SizeMax = 24;
    VAR
      p: Geo.Point;
      a: ScrOut.Attrs;
      j: CARDINAL;
    BEGIN
      (* ----------------------- drawing *)
      SetEndMark(TimeOut);
      REPEAT
        (* -------------------------- set attributes *)
        ScrOut.SetMarkAttrs
        (
          a,
          Scr.copy, RandomColor(), mt, 1 + CardRandom(SizeMax)
        );
        IF mt = ScrOut.dot THEN
          a.function := Scr.xor;
        END;
        (* ------------------------------------------------- output *)
        FOR j := 0 TO CardRandom(MarkerColorMaxLen) DO
          p.ix := VAL(INTEGER, CardRandom(scrnWdt));
          p.iy := VAL(INTEGER, CardRandom(scrnHgt));
          ScrOut.Marker(vpm, a, p);
        END;
      UNTIL EndRepeat();
    END Markers;

  PROCEDURE Texts();
    CONST
      TextColorMaxLen = 10;
      String = "AbCdEfGhIjKlMnOpQrStUvWxYz";
      StrMax = 28;
    VAR
      p: Geo.Point;
      str: ARRAY [0 .. StrMax] OF CHAR;
      a: ScrOut.Attrs;
      j: CARDINAL;
    BEGIN
      (* ------------------------------- drawing *)
      SetEndMark(TimeOut);
      REPEAT
        (* --------------------------------------------- set attributes *)
        ScrOut.SetTextAttrs
        (
          a,
          Scr.copy, RandomColor(), 0, ScrOut.fillStippled, RandomFont()
        );
        (* -------------------------------------- get random part of string *)
        Strings.Copy(String, CardRandom(StrMax), CardRandom(StrMax), str);
        (* ---------------------------------------------------------- output *)
        FOR j := 0 TO CardRandom(TextColorMaxLen) DO
          p.ix := VAL(INTEGER, CardRandom(scrnWdt));
          p.iy := VAL(INTEGER, CardRandom(scrnHgt));
          ScrOut.String(vpm, a, p, str);
        END;
      UNTIL EndRepeat();
    END Texts;

  PROCEDURE Polylines();
    CONST
      PolylineColorMaxLen = 10;
      PolylineSpeed = 3;
      NodesNmb = 8;
    VAR
      bs: ARRAY [0 .. NodesNmb - 1] OF Box;
      ps: ARRAY [0 .. NodesNmb - 1] OF Geo.Point;
      st: Res.Status;
      a: ScrOut.Attrs;
      j, k: CARDINAL;
    BEGIN
      (* -------------------- init attributes *)
      ScrOut.SetLineAttrs(a, Scr.copy, 0);
      (* --------------------------------- set boxes *)
      FOR k := 0 TO NodesNmb - 1 DO
        InitBox(viewV.port, PolylineSpeed, ps[k], bs[k]);
      END;
      (* -------------------------------------------- drawing *)
      SetEndMark(TimeOut);
      Scr.SetClip(vpm, FALSE);
      REPEAT
        (* ---------------------------------------- set attributes *)
        a.color := RandomColor();
        (* ---------------------------------------------------- output *)
        FOR j := 0 TO CardRandom(PolylineColorMaxLen) DO
          st := ScrOut.Polyline(vpm, a, ps, 0, NodesNmb - 1, TRUE);
          FOR k := 0 TO NodesNmb - 1 DO
            ChangePoint(ps[k], bs[k]);
          END;
        END;
      UNTIL EndRepeat();
      Scr.SetClip(vpm, TRUE);
    END Polylines;

  PROCEDURE Rectangles();
    CONST
      RectSpeed = 12;
    VAR
      rect: Geo.Rectangle;
      b1, b2: Box;
      p1, p2: Geo.Point;
      a: ScrOut.Attrs;
    BEGIN
      (* -------------------- init attributes *)
      ScrOut.SetLineAttrs(a, Scr.copy, 0);
      (* --------------------------------- set boxes *)
      InitBox(viewV.port, RectSpeed, p1, b1);
      InitBox(viewV.port, RectSpeed, p2, b2);
      Geo.SetRectPt(rect, p1, p2);
      (* ------------------------------------------ drawing *)
      SetEndMark(TimeOut);
      Scr.SetClip(vpm, FALSE);
      REPEAT
        (* --------------------------------- set attributes *)
        a.color := RandomColor();
        (* --------------------------------------- output *)
        ScrOut.Rectangle(vpm, a, rect);
        (* ---------------------- set next rectangle *)
        ChangePoint(p1, b1);
        ChangePoint(p2, b2);
        Geo.SetRectPt(rect, p1, p2);
      UNTIL EndRepeat();
      Scr.SetClip(vpm, TRUE);
    END Rectangles;

  PROCEDURE Ellipses();
    CONST
      EllipseColorMaxLen = 12;
      RadFactor = 4;
    VAR
      p: Geo.Point;
      a: ScrOut.Attrs;
      j, a0, b0, radMax: CARDINAL;
    BEGIN
      (* --------------------------- init attributes *)
      ScrOut.SetLineAttrs(a, Scr.copy, 0);
      (* -------------------------------------------- set limut for radius *)
      radMax := (viewV.port.iyMax - viewV.port.iyMin + 1) DIV RadFactor + 1;
      (* ----------------------------------------------------------- drawing *)
      SetEndMark(TimeOut);
      REPEAT
        (* ----------------------------------------- set attributes *)
        a.color := RandomColor();
        (* ----------------------------------------- output *)
        FOR j := 0 TO CardRandom(EllipseColorMaxLen) DO
          p.ix := VAL(INTEGER, CardRandom(scrnWdt));
          p.iy := VAL(INTEGER, CardRandom(scrnHgt));
          a0 := CardRandom(radMax);
          b0 := CardRandom(radMax);
          ScrOut.Ellipse(vpm, a, p, a0, b0);
        END;
      UNTIL EndRepeat();
    END Ellipses;

  PROCEDURE FillRectangles(ft: ScrOut.FillStyle);
    CONST
      RectSpeed = 12;
    VAR
      rect: Geo.Rectangle;
      b1, b2: Box;
      p1, p2: Geo.Point;
      f: ScrOut.Attrs;
    BEGIN
      (* ---------------------- init attributes *)
      ScrOut.SetFillAttrs
      (
        f,
        Scr.copy, 0, 0, ft, hatchFont, nullPt, 0
      );
      (* ------------------------------------------------- set boxes *)
      InitBox(viewV.port, RectSpeed, p1, b1);
      InitBox(viewV.port, RectSpeed, p2, b2);
      Geo.SetRectPt(rect, p1, p2);
      (* -------------------------------------- drawing *)
      SetEndMark(TimeOut);
      Scr.SetClip(vpm, FALSE);
      REPEAT
        (* -------------- set fill attributes *)
        f.color := RandomColor();
        f.fillStippleNo := CardRandom(hatchesNmb);
        (* ------------------------------------ output *)
        ScrOut.FillRectangle(vpm, f, rect);
        (* ------------- set the next rectangle *)
        ChangePoint(p1, b1);
        ChangePoint(p2, b2);
        Geo.SetRectPt(rect, p1, p2);
      UNTIL EndRepeat();
      Scr.SetClip(vpm, TRUE);
    END FillRectangles;

  PROCEDURE FillEllipses(ft: ScrOut.FillStyle);
    CONST
      RadFactor = 4;
    VAR
      p: Geo.Point;
      a0, b0, radMax: CARDINAL;
      f: ScrOut.Attrs;
    BEGIN
      (* ----------------------- init attributes *)
      ScrOut.SetFillAttrs
      (
        f,
        Scr.copy, 0, 0, ft, hatchFont, nullPt, 0
      );
      (* --------------------------------------------- set limut for radius *)
      radMax := (viewV.port.iyMax - viewV.port.iyMin + 1) DIV RadFactor + 1;
      (* ----------------------------------------------------------- drawing *)
      SetEndMark(TimeOut);
      REPEAT
        (* ---------------------------------- set fill attributes *)
        f.color := RandomColor();
        f.fillStippleNo := CardRandom(hatchesNmb);
        (* ------------------------------------- output *)
        p.ix := VAL(INTEGER, CardRandom(scrnWdt));
        p.iy := VAL(INTEGER, CardRandom(scrnHgt));
        a0 := CardRandom(radMax);
        b0 := CardRandom(radMax);
        ScrOut.FillEllipse(vpm, f, p, a0, b0);
      UNTIL EndRepeat();
    END FillEllipses;

  PROCEDURE FillPolygons(ft: ScrOut.FillStyle);
    CONST
      NodesNmb = 5;
    VAR
      ps: ARRAY [0 .. NodesNmb - 1] OF Geo.Point;
      st: Res.Status;
      f: ScrOut.Attrs;
      k: CARDINAL;
    BEGIN
      (* ------------------- init attributes *)
      ScrOut.SetFillAttrs
      (
        f,
        Scr.copy, 0, 0, ft, hatchFont, nullPt, 0
      );
      (* ------------------------------- drawing *)
      SetEndMark(TimeOut);
      REPEAT
        (* --------------------- set fill attributes *)
        f.color := RandomColor();
        f.fillStippleNo := CardRandom(hatchesNmb);
        (* -------------------------------------- set nodes *)
        FOR k := 0 TO NodesNmb - 1 DO
          ps[k].ix := VAL(INTEGER, CardRandom(scrnWdt));
          ps[k].iy := VAL(INTEGER, CardRandom(scrnHgt));
        END;
        (* ------------------------------------------------ output *)
        st := ScrOut.FillPolygon(vpm, f, ps, 0, NodesNmb - 1);
      UNTIL EndRepeat();
    END FillPolygons;

  PROCEDURE MarkerCursors(mt: ScrOut.MarkerStyle);
    CONST
      CursNmb = 100;
      CursSpeed = 4;
      SizeMax = 24;
    TYPE
      Cursor =
        RECORD
          m: ScrCur.MARKERCURSOR;
          p: Geo.Point;
          b: Box;
          f: BOOLEAN;
        END;
    VAR
      curs: ARRAY [0 .. CursNmb - 1] OF Cursor;
      a: ScrOut.Attrs;
      n, j: CARDINAL;
    BEGIN
      (* ------------ init cursors number *)
      IF mt = ScrOut.dot THEN
        n := CursNmb;
      ELSE
        n := CursNmb DIV 10;
      END;
      (* ----------------------------- make cursors *)
      FOR j := 0 TO n - 1 DO
        (* ------------------------------- set attributes *)
        ScrOut.SetMarkAttrs
        (
          a,
          Scr.xor, RandomColor(), mt, 1 + CardRandom(SizeMax)
        );
        (* ----------------------------------- initialize cursor *)
        WITH curs[j] DO
          InitBox(viewV.port, CursSpeed, p, b);
          IF
            ScrCur.NewMarker(vpm, a, p, m) = Res.done
          THEN
            ScrCur.ShowMarker(m);
            f := TRUE;
          ELSE
            f := FALSE;
          END;
        END;
      END;
      (* ------------------------------------ drawing *)
      SetEndMark(TimeOut);
      REPEAT
        FOR j := 0 TO n - 1 DO
          WITH curs[j] DO
            IF f THEN
              ChangePoint(p, b);
              ScrCur.MoveMarker(m, p);
            END;
          END;
        END;
      UNTIL EndRepeat();
      (* ----------------------- dispose cursors *)
      FOR j := n - 1 TO 0 BY - 1 DO
        WITH curs[j] DO
          IF f THEN
            ScrCur.DisposeMarker(m);
          END;
        END;
      END;
    END MarkerCursors;

  PROCEDURE RectCursors();
    CONST
      CursNmb = 3;
      CursSpeed = 8;
    TYPE
      Cursor =
        RECORD
          c: ScrCur.RECTCURSOR;
          p1, p2: Geo.Point;
          b1, b2: Box;
          f: BOOLEAN;
        END;
    VAR
      curs: ARRAY [0 .. CursNmb - 1] OF Cursor;
      a: ScrOut.Attrs;
      r: Geo.Rectangle;
      j: CARDINAL;
    BEGIN
      (* ---------------------- make cursors *)
      FOR j := 0 TO CursNmb - 1 DO
        (* ------------------------------ set attributes *)
        ScrOut.SetLineAttrs(a, Scr.xor, RandomColor());
        (* -------------------------------- initialize cursor *)
        WITH curs[j] DO
          InitBox(viewV.port, CursSpeed, p1, b1);
          InitBox(viewV.port, CursSpeed, p2, b2);
          Geo.SetRectPt(r, p1, p2);
          IF
            ScrCur.NewRectangle(vpm, a, r, c) = Res.done
          THEN
            ScrCur.ShowRectangle(c);
            f := TRUE;
          ELSE
            f := FALSE;
          END;
        END;
      END;
      (* ------------------------------------ drawing *)
      SetEndMark(TimeOut);
      REPEAT
        FOR j := 0 TO CursNmb - 1 DO
          WITH curs[j] DO
            IF f THEN
              ChangePoint(p1, b1);
              ChangePoint(p2, b2);
              Geo.SetRectPt(r, p1, p2);
              ScrCur.MoveRectangle(c, r);
            END;
          END;
        END;
      UNTIL EndRepeat();
      (* ----------------------- dispose cursors *)
      FOR j := CursNmb - 1 TO 0 BY - 1 DO
        WITH curs[j] DO
          IF f THEN
            ScrCur.DisposeRectangle(c);
          END;
        END;
      END;
    END RectCursors;

  PROCEDURE CharCursors();
    CONST
      CursNmb = 12;
      CursSpeed = 8;
    TYPE
      Cursor =
        RECORD
          c: ScrCur.CHARCURSOR;
          p: Geo.Point;
          b: Box;
          f: BOOLEAN;
        END;
    VAR
      curs: ARRAY [0 .. CursNmb - 1] OF Cursor;
      a: ScrOut.Attrs;
      j: CARDINAL;
      ch: CHAR;
    BEGIN
      (* ----------------------------- make cursors *)
      FOR j := 0 TO CursNmb - 1 DO
        (* ----------------------------------- set attributes *)
        ScrOut.SetTextAttrs
        (
          a,
          Scr.xor, RandomColor(), 0, ScrOut.fillStippled, RandomFont()
        );
        (* --------------------------------------- initialize cursors *)
        WITH curs[j] DO
          InitBox(viewV.port, CursSpeed, p, b);
          ch := CHR(CardRandom(ORD(MAX(CHAR))));
          IF
            ScrCur.NewChar(vpm, a, p, ch, c) = Res.done
          THEN
            ScrCur.ShowChar(c);
            f := TRUE;
          ELSE
            f := FALSE;
          END;
        END;
      END;
      (* ------------------------------------ drawing *)
      SetEndMark(TimeOut);
      REPEAT
        FOR j := 0 TO CursNmb - 1 DO
          WITH curs[j] DO
            IF f THEN
              ChangePoint(p, b);
              ScrCur.MoveChar(c, p);
            END;
          END;
        END;
      UNTIL EndRepeat();
      (* ----------------------- dispose cursors *)
      FOR j := CursNmb - 1 TO 0 BY - 1 DO
        WITH curs[j] DO
          IF f THEN
            ScrCur.DisposeChar(c);
          END;
        END;
      END;
    END CharCursors;

  PROCEDURE StringCursors();
    CONST
      CursNmb = 3;
      CursSpeed = 5;
      StrLen = 6;
      InitString = "AbCdEf";
      MinChar = 48;
      MaxChar = 126;
      NMove = 150;
      NInsert = 20;
      NDelete = 40;
    TYPE
      Cursor =
        RECORD
          c: ScrCur.STRINGCURSOR;
          p: Geo.Point;
          b: Box;
          f: BOOLEAN;
        END;
    VAR
      curs: ARRAY [0 .. CursNmb - 1] OF Cursor;
      a: ScrOut.Attrs;
      j: CARDINAL;
      ch: CHAR;
    BEGIN
      (* ---------------------------- make cursors *)
      FOR j := 0 TO CursNmb - 1 DO
        (* ---------------------------------- set attributes *)
        ScrOut.SetTextAttrs
        (
          a,
          Scr.xor, RandomColor(), 0, ScrOut.fillStippled, RandomFont()
        );
        (* ------------------------------------- initialize cursors *)
        WITH curs[j] DO
          InitBox(viewV.port, CursSpeed, p, b);
          IF
            ScrCur.NewString(vpm, a, p, InitString, c) = Res.done
          THEN
            ScrCur.ShowString(c);
            f := TRUE;
          ELSE
            f := FALSE;
          END;
        END;
      END;
      (* ------------------------------------ drawing *)
      SetEndMark(TimeOut);
      REPEAT
        FOR j := 0 TO CursNmb - 1 DO
          WITH curs[j] DO
            IF f THEN
              CASE CardRandom(NMove + NInsert + NDelete) OF
              | 0 .. NMove - 1:
                ChangePoint(p, b);
                ScrCur.MoveString(c, p);
              | NMove .. NMove + NInsert - 1:
                ch := CHR(MinChar + CardRandom(MaxChar - MinChar + 1));
                ScrCur.ModSubString(c, ch, CardRandom(StrLen), TRUE);
              | NMove + NInsert .. NMove + NInsert + NDelete - 1:
                ScrCur.DelSubString(c, CardRandom(StrLen), 1);
              END;
            END;
          END;
        END;
      UNTIL EndRepeat();
      (* ----------------------- dispose cursors *)
      FOR j := CursNmb - 1 TO 0 BY - 1 DO
        WITH curs[j] DO
          IF f THEN
            ScrCur.DisposeString(c);
          END;
        END;
      END;
    END StringCursors;

  PROCEDURE RectCopying(func: Scr.Function);
    CONST
      XMin = 161; XMax = 190; YMin = 101; YMax = 130;
    VAR
      mem: Scr.PIXMAP;
      srcR, dstR: Geo.Rectangle;
      p: Geo.Point;
      f: ScrOut.Attrs;
      e: ScrOut.Attrs;
      a: ScrOut.Attrs;
      wdt, hgt: CARDINAL;
    BEGIN
      (* -------------------- set source rectangle *)
      Geo.SetRect(srcR, XMin, XMax, YMin, YMax);
      (* --------------------- allocate the memory pixmap *)
      IF Scr.New(srcR, mem) = Res.done THEN
        (* ------------------------ draw on the memory pixmap *)
        ScrOut.SetFillAttrs
        (
          f,
          Scr.copy, RandomColor(), RandomColor(),
          ScrOut.fillOpaqueStippled, hatchFont,
          nullPt, CardRandom(hatchesNmb)
        );
        ScrOut.SetLineAttrs(e, Scr.copy, RandomColor());
        ScrOut.FillRectangle(mem, f, srcR);
        ScrOut.Rectangle(mem, e, srcR);
        (* ------------------------------------ get some parameters *)
        Geo.RectBasePt(srcR, p);
        wdt := srcR.ixMax - srcR.ixMin;
        hgt := srcR.iyMax - srcR.iyMin;
        (* -------------------------------------------- copying *)
        SetEndMark(TimeOut);
        REPEAT
          dstR.ixMin := CardRandom(scrnWdt - wdt);
          dstR.ixMax := dstR.ixMin + VAL(INTEGER, wdt);
          dstR.iyMin := CardRandom(scrnHgt - hgt);
          dstR.iyMax := dstR.iyMin + VAL(INTEGER, hgt);
          Scr.RectCopy(vpm, dstR, mem, p, func);
        UNTIL EndRepeat();
        (* ------------ deallocate the memory pixmap *)
        Scr.Dispose(mem);
      ELSE
        (* ---------------------------- no memory: draw message and wait *)
        p.ix := (scrnWdt - Strings.Length(infoStr[noMemory]) * charWdt) DIV 2;
        p.iy := (scrnHgt - charHgt) DIV 2;
        ScrOut.SetTextAttrs
        (
          a,
          Scr.copy, MsgColor(), 0, ScrOut.fillStippled, mainFont
        );
        ScrOut.String(vpm, a, p, infoStr[noMemory]);
        SetEndMark(TimeOut DIV 2);
        REPEAT UNTIL EndRepeat();
      END;
    END RectCopying;

  PROCEDURE Palettes();
    CONST
      PalDelay = TimeOut * 2000;
    VAR
      s: ARRAY [0 .. 1] OF CHAR;
      palNmb, old, current: CARDINAL;
      a: ScrOut.Attrs;
      p: Geo.Point;
    BEGIN
      (* ------------------------ prologue *)
      palNmb := Scr.PalettesNmb();
      (* ------------------------------- set attributes *)
      ScrOut.SetTextAttrs
      (
        a,
        Scr.copy, MsgColor(), 0, ScrOut.fillOpaqueStippled, mainFont
      );
      (* ---------------------------------------- set base point *)
      p.ix := (scrnWdt - Strings.Length(s) * charWdt) DIV 2;
      p.iy := (scrnHgt - charHgt) DIV 2;
      (* --------------------------------- check number of palettes *)
      IF palNmb > 1 THEN
        (* --------------------------------------- store palette *)
        old := Scr.Palette();
        (* --------------------------------- change palette *)
        current := 0;
        SetEndMark(TimeOut * 2);
        REPEAT
          s[0] := CHR(ORD("0") + current DIV 10);
          s[1] := CHR(ORD("0") + current MOD 10);
          ScrOut.String(vpm, a, p, s);
          Scr.SetPalette(current);
          DrvClk.Delay(PalDelay DIV palNmb);
          current := (current + 1) MOD palNmb;
        UNTIL EndRepeat();
        (* ------------------- restore palette *)
        Scr.SetPalette(old);
      ELSE
        (* ---------------------------- no palettes: draw message and wait *)
        p.ix := (scrnWdt - Strings.Length(infoStr[noPalettes]) * charWdt) DIV 2;
        ScrOut.String(vpm, a, p, infoStr[noPalettes]);
        SetEndMark(TimeOut DIV 2);
        REPEAT UNTIL EndRepeat();
      END;
    END Palettes;

  PROCEDURE Shades();
    CONST
      ShColor = 255;
      ShMaxShades = 256;
      ShDelay = TimeOut * 100;
      ShStrLen = 12;
    TYPE
      ShString = ARRAY [0 .. ShStrLen - 1] OF CHAR;
      ShStringBlock = ARRAY Scr.PureColor OF ShString;
    VAR
      s: ShStringBlock;
      temp: ShString;
      forward: BOOLEAN;
      nsh, sh, shInc: INTEGER;
      count, j, shCard: CARDINAL;
      old: ARRAY Scr.PureColor OF CARDINAL;
      pc: Scr.PureColor;
      a: ScrOut.Attrs;
      p: Geo.Point;
    BEGIN
      (* ------------------------ load shade strings *)
      Strings.Assign(infoStr[redXXX  ], s[Scr.red  ]);
      Strings.Assign(infoStr[greenXXX], s[Scr.green]);
      Strings.Assign(infoStr[blueXXX ], s[Scr.blue ]);
      (* ---------------------------------------- set attributes *)
      ScrOut.SetTextAttrs
      (
        a,
        Scr.copy, MsgColor(), ShColor, ScrOut.fillOpaqueStippled, mainFont
      );
      (* ------------------------------------- check number of shades *)
      IF (Scr.PaletteShadesNmb(Scr.red) > 1) OR
         (Scr.PaletteShadesNmb(Scr.green) > 1) OR
         (Scr.PaletteShadesNmb(Scr.blue) > 1) THEN
        (* ------------------------------------- store and modify shades *)
        FOR pc := MIN(Scr.PureColor) TO MAX(Scr.PureColor) DO
          old[pc] := Scr.PaletteShade(0, ShColor, pc);
          Scr.SetPaletteShade(0, ShColor, pc, 0);
        END;
        (* ------------------------ fill the viewport by selected color *)
        Scr.SolidRect(vpm, viewV.port, ShColor, Scr.copy);
        (* ------------------------------------------- change shades *)
        forward := TRUE;
        SetEndMark(TimeOut * 3);
        LOOP
          FOR pc := MIN(Scr.PureColor) TO MAX(Scr.PureColor) DO
            nsh := Scr.PaletteShadesNmb(pc);
            shInc := ShMaxShades DIV nsh;
            sh := Scr.PaletteShade(0, ShColor, pc);
            FOR count := 0 TO VAL(CARDINAL, 2 * nsh - 2) DO
              IF forward THEN
                IF sh < ShMaxShades - shInc THEN
                  INC(sh, shInc);
                ELSE
                  forward := FALSE;
                END;
              END;
              IF NOT forward THEN
                IF sh > 0 THEN
                  DEC(sh, shInc);
                ELSE
                  forward := TRUE;
                END;
              END;
              p.ix := (scrnWdt - ShStrLen * charWdt) DIV 2;
              p.iy := (scrnHgt - charHgt * 6) DIV 2 + ORD(pc) * charHgt * 2;
              temp := s[pc];
              shCard := VAL(CARDINAL, sh);
              FOR j := ShStrLen - 1 TO ShStrLen - 3 BY - 1 DO
                temp[j] := CHR(ORD("0") + shCard MOD 10);
                shCard := shCard DIV 10;
              END;
              ScrOut.String(vpm, a, p, temp);
              Scr.SetPaletteShade(0, ShColor, pc, sh);
              DrvClk.Delay(ShDelay DIV nsh);
              IF EndRepeat() THEN EXIT END;
            END;
          END;
        END;
        (* -------------------------------------- restore shades *)
        FOR pc := MIN(Scr.PureColor) TO MAX(Scr.PureColor) DO
          Scr.SetPaletteShade(0, ShColor, pc, old[pc]);
        END;
      ELSE
        (* ----------------------------- no shades: draw message and wait *)
        p.ix := (scrnWdt - Strings.Length(infoStr[noShades]) * charWdt) DIV 2;
        p.iy := (scrnHgt - charHgt ) DIV 2;
        ScrOut.String(vpm, a, p, infoStr[noShades]);
        SetEndMark(TimeOut DIV 2);
        REPEAT UNTIL EndRepeat();
      END;
    END Shades;

  PROCEDURE Origins();
    CONST
      OrgDelay = TimeOut * 500;
      OrgLine = 600;
    VAR
      orgNmb: CARDINAL;
      current, delta: INTEGER;
      a: ScrOut.Attrs;
      p: Geo.Point;
      old: BOOLEAN;
    BEGIN
      (* ---------------------------- prologue *)
      orgNmb := Scr.PageOrgsNmb();
      (* ------------------------------------- set attributes *)
      ScrOut.SetTextAttrs
      (
        a,
        Scr.copy, MsgColor(), 0, ScrOut.fillStippled, mainFont
      );
      (* --------------------------------- check number of origins *)
      IF orgNmb > 20 THEN
        (* --------------------------------------------- draw message *)
        p.ix := (scrnWdt - Strings.Length(infoStr[org600]) * charWdt) DIV 2;
        p.iy := OrgLine;
        Scr.SwapClip(vpm, FALSE, old);
        ScrOut.String(vpm, a, p, infoStr[org600]);
        Scr.SetClip(vpm, old);
        (* --------------------------- change origin *)
        current := 0;
        SetEndMark(TimeOut * 2);
        REPEAT
          IF current = VAL(INTEGER, orgNmb - 1) THEN
            delta := - 1;
          ELSIF current = 0 THEN
            delta := 1;
          END;
          INC(current, delta);
          Scr.SetPageOrg(current);
          DrvClk.Delay(OrgDelay DIV orgNmb);
        UNTIL EndRepeat();
        (* ------------------- restore origin *)
        Scr.SetPageOrg(0);
      ELSE
        (* ------------------------------ no origins: draw message and wait *)
        p.ix := (scrnWdt - Strings.Length(infoStr[noOrgs]) * charWdt) DIV 2;
        p.iy := (scrnHgt - charHgt) DIV 2;
        ScrOut.String(vpm, a, p, infoStr[noOrgs]);
        SetEndMark(TimeOut DIV 2);
        REPEAT UNTIL EndRepeat();
      END;
    END Origins;

  (* ================================= Tour through all the demos *)

  PROCEDURE Tour();
    BEGIN
      (* ----------------------- prologue *)
      Randomize();
      nullPt.ix := 0;
      nullPt.iy := 0;
      colNmb := Scr.ColorsNmb();
      Scr.Dims(scrnWdt, scrnHgt);
      Geo.SetRect(scrnV.port, 0, scrnWdt - 1, 0, scrnHgt - 1);
      scrnV.clip := TRUE;
      fontsNmb := ScrOut.StdFontsNmb();
      mainFont := Font();
      hatchFont := ScrOut.StdFont(1);
      hatchesNmb := CharsNmb(hatchFont);
      ScrOut.Dims(mainFont, charWdt, charHgt);
      ScrOut.SetFillAttrs
      (
        bkAttrs,
        Scr.copy, RandomColor(), 0,
        ScrOut.fillOpaqueStippled, hatchFont,
        nullPt, CardRandom(hatchesNmb)
      );
      oldLen := 0;
      viewV := scrnV;
      INC(viewV.port.ixMin, BorderSize);
      INC(viewV.port.iyMin, BorderSize);
      DEC(viewV.port.ixMax, BorderSize);
      DEC(viewV.port.iyMax, BorderSize);
      vpm := Scr.PagePixmap();
      (* ------------------------ attempt to initialize adapter *)
      IF Scr.InitGraph() = Res.done THEN
        (* ---------------- success: set viewport and draw the bottom info *)
        Scr.SetPalette(0);
        Scr.SetDirectPaletteShades(0);
        ScrOut.FillRectangle(vpm, bkAttrs, scrnV.port);
        Scr.SetViewport(vpm, viewV);
        BottomInfo();
        (* ----------------------- main cycle *)
        escPressed := FALSE;
        ftre := title;
        REPEAT
          (* ------------- clear the viewport and draw title *)
          Scr.SolidRect(vpm, viewV.port, 0, Scr.copy);
          Header(ftre);
          (* ---------------------------- run the demo procedure *)
          CASE ftre OF
          | title:
            Title();
          | dotMarkers:
            Markers(ScrOut.dot);
          | plusMarkers:
            Markers(ScrOut.plus);
          | asteriskMarkers:
            Markers(ScrOut.asterisk);
          | circleMarkers:
            Markers(ScrOut.circle);
          | crossMarkers:
            Markers(ScrOut.cross);
          | squareMarkers:
            Markers(ScrOut.square);
          | rectangles:
            Rectangles();
          | solidRectangles:
            FillRectangles(ScrOut.fillSolid);
          | stippledRectangles:
            FillRectangles(ScrOut.fillStippled);
          | ellipses:
            Ellipses();
          | solidEllipses:
            FillEllipses(ScrOut.fillSolid);
          | stippledEllipses:
            FillEllipses(ScrOut.fillStippled);
          | polylines:
            Polylines();
          | solidPolygons:
            FillPolygons(ScrOut.fillSolid);
          | stippledPolygons:
            FillPolygons(ScrOut.fillStippled);
          | texts:
            Texts();
          | dotCursors:
            MarkerCursors(ScrOut.dot);
          | plusCursors:
            MarkerCursors(ScrOut.plus);
          | asteriskCursors:
            MarkerCursors(ScrOut.asterisk);
          | circleCursors:
            MarkerCursors(ScrOut.circle);
          | crossCursors:
            MarkerCursors(ScrOut.cross);
          | squareCursors:
            MarkerCursors(ScrOut.square);
          | rectCursors:
            RectCursors();
          | charCursors:
            CharCursors();
          | stringCursors:
            StringCursors();
          | copyingReplace:
            RectCopying(Scr.copy);
          | copyingInvert:
            RectCopying(Scr.xor);
          | palettes:
            Palettes();
          | shades:
            Shades();
          | origins:
            Origins();
          ELSE
          END;
          IF ftre < MAX(Feature) THEN INC(ftre) ELSE ftre := title END;
        UNTIL escPressed;
        (* ----------------------------- epilogue *)
        Scr.Clear(vpm, 0);
        Scr.TermGraph();
      ELSE
        (* ---------------- adapter doesn't support this mode *)
        Error(infoStr[adapter], "", FALSE);
      END;
    END Tour;

  (* ===================================================== Main *)

  VAR
    key, lineNo: CARDINAL;
    mode: INTEGER;

  BEGIN
    (* --------------- load texts *)
    LoadTexts();
    (* ----------------------- main loop *)
    LOOP
      (* ------------------------ write header *)
      Terminal.Write(033C);
      Terminal.WriteString("[2J");
      FOR lineNo := 0 TO ModeStrNmb - 1 DO
        Terminal.WriteLn();
        Terminal.WriteString(modeStr[lineNo]);
      END;
      (* ---------------------------------- request the mode *)
      REPEAT
        REPEAT UNTIL DrvKbd.Get(key);
        IF key = DrvKbd.Escape THEN
          EXIT;
        ELSE
          mode := VAL(INTEGER, ORD(key)) - VAL(INTEGER, ORD("a"));
        END;
      UNTIL (0 <= mode) AND (mode <= VAL(INTEGER, ORD(MAX(Scr.Type))));
      (* ------------------------------ set the mode and run the demo *)
      Scr.SetMode(VAL(Scr.Type, mode));
      Tour();
    END;
  END ScrDemo.
