{ SORTPARA.PAS : Sort text paragraphs

  Title   : SORTPARA
  Version : 2.3
  Date    : Feb 13, 2000
  Author  : J.R. Ferguson
  Language: Turbo Pascal v4.0 or higher
  Usage   : Refer procedure Help
  Remarks : In memory sort
  Download: http://hello.to/ferguson
            To compile this sourcefile you will need units from the Pascal
            library JRFPAS that can be found on the same Internet site.
  E-mail  : j.r.ferguson@iname.com
}

{$V-}

program SORTPARA;


uses DefLib, ArgLib, StpLib, ChrLib;


const
  MAXFNM    = 79;         { Max filespec length (integrated environment) }
  DFLMSG    = 'CON';      { Default message output destination }
  INPBUFSIZ = 16 * 1024;  { Input buffer size in bytes }
  OUTBUFSIZ = 16 * 1024;  { Output buffer size in bytes }
  MAXPARA   = 5000;       { Maximum number of paragraphs we can handle }

  { Error codes and messages: }
  ERROK     = 0;
  ERRARG    = 1;
  ERRFNF    = 2;
  ERRCRE    = 3;
  ERRREA    = 4;
  ERRWRI    = 5;
  ERRMAX    = 6;
  ERRMEM    = 7;

  ERRMSG    : array[ERRFNF..ERRMEM] of StpTyp =
 ('File not found',
  'File creation error',
  'Read error',
  'Write error',
  'Too many paragraphs',
  'Out of memory'
 );

type
  InpBufTyp = array[1..INPBUFSIZ] of char;
  OutBufTyp = array[1..OUTBUFSIZ] of char;
  LinePtr   = ^LineRec;
  LineRec   = record nxt: LinePtr; txt: StpTyp end;
  PtrArrInd = 0..MAXPARA;
  PtrArrTyp = array[1..MAXPARA] of LinePtr;
  KeyDescTyp= record LinBeg,LinCnt: integer; ColBeg,ColCnt: StpInd end;

var
  ErrCod    : integer;
  InpFnm,
  OutFnm    : StpTyp;
  Msg,
  Inp,
  Out       : Text;
  InpBuf    : InpBufTyp;
  OutBuf    : OutBufTyp;
  InpOpn,
  OutOpn    : boolean;
  PtrArr    : PtrArrTyp;
  ParaCnt   : PtrArrInd;
  KeyDesc   : KeyDescTyp;
  OptHlp    : boolean;
  OptLex    : boolean;
  OptIgn    : boolean;
  OptDes    : integer; { -1 = descending; +1 = ascending }


{--- General routines ---}


procedure Help;
  procedure wi(i: integer); begin write  (Msg,i) end;
  procedure wr(s: StpTyp);  begin write  (Msg,s) end;
  procedure wl(s: StpTyp);  begin writeln(Msg,s) end;
begin
wl('SORTPARA v2.3 - Sort text paragaphs (blocks of text lines separated by');
wl('one or more blank lines). Output paragraphs are separated by one blank');
wr('line. Max line length is '); wi(MaxStp); wl('.');
wl('');
wl('Usage  : SORTPARA [<in] [>out] [/option[...] [...]]');
wl('Options:');
wl('  K[lineref][;colref]           Define Key line(s)/column(s).');
wl('     with lineref = [l1][,l2]   First line l1 [1], last line   l2 [l1].');
wl('                 or [l1][,+l2]  First line l1 [1], nr of lines l2 [1].');
wr('          colref  = [c1][,c2]   First col  c1 [1], last col    c2 [');
  wi(MaxStp); wl('].');
wr('                 or [c1][,+c2]  First col  c1 [1], nr of cols  c2 [');
  wi(MaxStp); wl('].');
wl('  D  Sort in descending order.');
wl('  I  Ignore upper/lower case. Implied by /L.');
wr('  L  Sort in lexical order (control,punctuation,digits,letters).');
wl      (' Implies /I.');
wl('  H  Send this help text to (redirected) output.');
end;


{$F+} function HeapFunc(Size: word): integer; {$F-}
{ Make New() and GetMem() return a nil pointer when the heap is full }
begin HeapFunc:= 1; end;



{--- Command line parsing routines ---}


function ReadUns(var arg: StpTyp): integer;
var n: integer; c: char;
begin
  n:= 0; c:= StpcRet(arg,1);
  while IsDigit(c) do begin
    StpDel(arg,1,1);
    n:= 10 * n + (ord(c) - ord('0'));
    c:= StpcRet(arg,1);
  end;
  ReadUns:= n;
end;


procedure ReadKeyDesc(var arg: StpTyp);
var Cbeg,Ccnt: integer;

  procedure ReadRef(var start,count: integer);
  begin
    if IsDigit(StpcRet(arg,1)) then start:= ReadUns(arg);
    if StpcRet(arg,1) = ',' then begin
      StpDel(arg,1,1);
      if StpcRet(arg,1) = '+' then begin
        StpDel(arg,1,1);
        if IsDigit(StpcRet(arg,1)) then count:= ReadUns(arg)
        else ErrCod:= ERRARG;
      end
      else begin
        if not IsDigit(StpcRet(arg,1)) then ErrCod:= ERRARG
        else begin
          count:= ReadUns(arg) - pred(start);
          if count < 0 then ErrCod:= ERRARG;
        end;
      end;
    end;
  end;

begin with KeyDesc do begin {ReadKeyDesc}
  ReadRef(LinBeg,LinCnt);
  if (ErrCod = ERROK) and (StpcRet(arg,1) = ';') then begin
    StpDel(arg,1,1);
    CBeg:= ColBeg; CCnt:= ColCnt; ReadRef(CBeg,CCnt);
    if (CBeg > MaxStp) or (CCnt > MaxStp) then ErrCod:= ERRARG
    else begin ColBeg:= CBeg; ColCnt:= CCnt; end;
  end;
end end;


procedure ReadOpt(var arg: StpTyp);
var nextopt: boolean;
begin
  StpDel(arg,1,1);
  repeat
    if StpEmpty(arg) or (StpcRet(arg,1) = '/') then ErrCod:= ERRARG
    else begin
      nextopt:= false;
      while (ErrCod=ERROK) and not nextopt and not StpEmpty(arg) do
      case StpcGet(arg) of
        'K': ReadKeyDesc(arg);
        'D': OptDes:= -1;
        'I': OptIgn:= true;
        'L': OptLex:= true;
        'H': OptHlp:= true;
        '/': nextopt:= true;
        else ErrCod:= ERRARG;
      end;
    end;
  until (ErrCod <> ERROK) or not nextopt;
end;


procedure ReadArgs;
var i   : ArgInd;
    arg : StpTyp;
begin
  GetArgs; i:= 0;
  while (ErrCod = ERROK) and (i < ArgC) do begin
    Inc(i); StpCpy(arg,ArgV[i]); StpUpp(arg);
    case StpcRet(arg,1) of
      '/' : ReadOpt(arg);
      '<' : StpSub(InpFnm,arg,2,MAXFNM); {simulate MS-DOS in Integrated Env}
      '>' : StpSub(OutFnm,arg,2,MAXFNM); {simulate MS-DOS in Integrated Env}
      else  ErrCod:= ERRARG;
    end;
  end;
  if OptHlp then if ErrCod = ERROK then ErrCod:= ERRARG else OptHlp:= false;
end;


{--- Low-level I/O routines ---}


procedure OpenMsg;
begin
  if OptHlp then Assign(Msg,OutFnm) else Assign(Msg,DFLMSG);
  rewrite(Msg);
end;


procedure CloseMsg;
begin Close(Msg); end;


procedure OpenInp;
begin
  Assign(Inp,InpFnm); SetTextBuf(Inp,InpBuf);
  {$I-} reset(Inp); {$I+}
  if IOresult <> 0 then ErrCod:= ERRFNF else InpOpn:= true;
end;


procedure CloseInp;
begin
  {$I-} Close(Inp); {$I+}
  if IOresult = 0 then InpOpn:= false;
end;


procedure OpenOut;
begin
  Assign(Out,OutFnm); SetTextBuf(Out,OutBuf);
  {$I-} rewrite(Out); {$I+}
  if IOresult <> 0 then ErrCod:= ERRCRE else OutOpn:= true;
end;


procedure CloseOut;
begin
  {$I-} Close(Out); {$I+}
  if IOresult = 0 then OutOpn:= false else begin
    if ErrCod = ERROK then ErrCod:= ERRWRI;
  end;
end;



{--- Processing routines ---}



procedure ReadText;
var line: StpTyp; InPara: boolean; tail: LinePtr;
begin
  ParaCnt:= 0; InPara:= false;
  while (ErrCod=ERROK) and not eof(Inp) do begin
    {$I-} readln(Inp,line); {$I+}
    if IOresult <> 0 then ErrCod:= ERRREA else begin
      StpRTS(line);
      if line = '' then InPara:= false
      else begin
        if InPara then begin
          GetMem(tail^.nxt, SizeOf(LinePtr) + succ(Length(line)));
          tail:= tail^.nxt;
          if tail = nil then ErrCod:= ERRMEM else with tail^ do begin
            nxt:= nil; Move(line, txt, succ(Length(line)));
          end;
        end
        else begin
          if ParaCnt = MAXPARA then ErrCod:= ERRMAX else begin
            Inc(ParaCnt);
            GetMem(PtrArr[ParaCnt], SizeOf(LinePtr) + succ(Length(line)));
            tail:= PtrArr[ParaCnt];
            if tail = nil then ErrCod:= ERRMEM else with tail^ do begin
              nxt:= nil; Move(line, txt, succ(Length(line)));
            end;
          end;
          InPara:= true;
        end;
      end;
    end;
  end;
end;


procedure WriteText;
var i: PtrArrInd; p: LinePtr;
begin
  i:= 0;
  while (ErrCod = ERROK) and (i < ParaCnt) do begin
    Inc(i); p:= PtrArr[i];
    if i > 1 then begin
      {$I-} writeln(Out); {$I+}
      if IOresult <> 0 then ErrCod:= ERRWRI;
    end;
    while (ErrCod = ERROK) and (p <> nil) do begin
      {$I-} writeln(Out,p^.txt); {$I+}
      if IOresult <> 0 then ErrCod:= ERRWRI else p:= p^.nxt;
    end;
  end;
end;


function ParaCompare(p1,p2: LinePtr): integer;
{ result < 0 if p1^ < p2^,
         = 0 if p1^ = p2^,
         > 0 if p1^ > p2^;
}
var i: integer; tmp1,tmp2: StpTyp; result: integer;
begin with KeyDesc do begin
  for i:= 2 to LinBeg do begin
    if p1 <> nil then p1:= p1^.nxt;
    if p2 <> nil then p2:= p2^.nxt;
  end;
  i:= 0; result:= 0;
  while (i<LinCnt) and (result=0) and (p1<>nil) and (p2<>nil) do begin
    inc(i);
    StpSub(tmp1,p1^.txt,ColBeg,ColCnt);
    StpSub(tmp2,p2^.txt,ColBeg,ColCnt);
    if OptLex then result:= StpLexCmp(tmp1,tmp2)
    else begin
      if OptIgn then begin StpUpp(tmp1); StpUpp(tmp2); end;
      result := StpCmp(tmp1,tmp2);
    end;
    p1:= p1^.nxt; p2:= p2^.nxt;
  end;
  if (i < LinCnt) and (result = 0) then begin { partial key(s) }
    if (p1=nil) and (p2=nil) then { result:= 0 }
    else begin
      if      p1 = nil then result:= -1
      else if p2 = nil then result:= +1;
    end;
  end;
  ParaCompare:= OptDes * result;
end end;


procedure SortPointers;
{ Insertion sort with binary search }
var i,l,r,m: PtrArrInd; p: LinePtr;
begin
  i:= 1;
  while i < ParaCnt do begin
    Inc(i); l:= 1; r:= pred(i); p:= PtrArr[i];
    while l <= r do begin
      m:= (l + r) shr 1;
      if ParaCompare(PtrArr[m],p) > 0 then r:= pred(m) else l:= succ(m);
    end;
    if l < i then begin
      Move(PtrArr[l], PtrArr[succ(l)], (i-l) * SizeOf(LinePtr));
      PtrArr[l]:= p;
    end;
  end;
end;



{--- Main line ---}


procedure MainProcess;
var Heap: pointer;
begin
  Mark(Heap);
  ReadText;
  if ErrCod = ERROK then begin
    SortPointers;
    WriteText;
  end;
  Release(Heap);
end;


procedure MainInit;
begin
  ErrCod:= ERROK;
  StpCreate(InpFnm); InpOpn:= false;
  StpCreate(OutFnm); OutOpn:= false;
  HeapError:= @HeapFunc; {install function to catch out-of-heap condition}
  OptHlp:= false; OptLex:= false; OptIgn:= false; OptDes:= +1;
  with KeyDesc do begin
    LinBeg:= 1; LinCnt:= 1;
    ColBeg:= 1; ColCnt:= MaxStp;
  end;
  ReadArgs;
  if ErrCod = ERROK then OpenInp;
  if ErrCod = ERROK then OpenOut;
end;


procedure MainTerm;
begin
  if InpOpn then CloseInp;
  if OutOpn then CloseOut;
  if ErrCod <> ERROK then begin
    OpenMsg;
    if (ErrCod=ERRARG) then Help
    else writeln(Msg,'SORTPARA: ',ERRMSG[ErrCod]);
    CloseMsg;
  end;
end;


begin { Main program }
  MainInit;
  if ErrCod = ERROK then MainProcess;
  MainTerm;
end.
