#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <dir.h>

#define FA_SUBDIR  16
#define FA_FILE    39
#define MAXCMD     80
#define MAXLINE    24
#define MAXNAME    13
#define MAXTMP     1
#define OFF        0
#define ON         1

typedef struct filespec
{
    unsigned char drive;
    char filename[MAXNAME];
    struct filespec *next;
} FILESPEC;  /* structure for filespecs */

typedef struct drivespec
{
    char drive[MAXDRIVE];
    struct drivespec *next;
} DRIVESPEC;  /* structure for drivespecs */

typedef struct
{
    unsigned day   : 5;
    unsigned month : 4;
    unsigned year  : 7;
} DOSDATE;  /* structure for DOS date */

typedef struct
{
    unsigned twosec  : 5;
    unsigned minutes : 6;
    unsigned hours   : 5;
} DOSTIME;  /* structure for DOS time */

unsigned char line;  /* global variables */
unsigned char lon = OFF, cd = OFF, more = OFF, inter = OFF;
char oldir[MAXDIR], cmdbuf[MAXCMD];  /* embedded DOS command buffer */
long totbyte;

/* function prototypes */

long int finddir(char *path, FILESPEC *sp);
long int findfile(char *path, FILESPEC *sp);
void filefound(char *path, struct ffblk fb);
void printlong(char *path, struct ffblk fb);
FILESPEC *addfile(FILESPEC *s, unsigned char drive, char *file, char *ext);
DRIVESPEC *adddrive(DRIVESPEC *d, char *drive);
char drivefound(DRIVESPEC *d, char *drive);
void domore(void);
unsigned char getkey(void);
unsigned char getdrive(void);
void setdrive(unsigned char drive);

int main(int argc, char *argv[])
{
    char drive[MAXDRIVE], dir[MAXDIR], file[MAXFILE], ext[MAXEXT];
    long int found;
    DRIVESPEC *dr;
    FILESPEC *sp;
    int i;

    if(argc < 2 || ((argv[1][0] == '/' || argv[1][0] == '-') &&
    argv[1][1] == '?'))
    {
        fprintf(stderr,
            "WHEREIS  Version 2.2  \26  A simple file finder utility\r\n" \
            "Copyright (c) 1993  Hung Vu\r\n\n"                           \
            "Usage: whereis <drive:><file><.ext>... [DOS cmd] /PCLI\r\n"  \
            "       /P      displays one screen at a time.\r\n"           \
            "       /C      change directory.\r\n"                        \
            "       /L      long format.\r\n"                             \
            "     [DOS cmd] embedded DOS command enclosed as [...]\r\n"   \
            "       /I      interative DOS cmd.\r\n");
        return 0;
    }
    for(i = 1; i < argc; i++) strupr(argv[i]);
    while(argv[argc-1][strlen(argv[argc-1])-1] == ']' ||
    argv[argc-1][0] == '/' || argv[argc-1][0] == '-')  /* parse options */
    {
        switch(argv[argc-1][strlen(argv[argc-1])-1])
        {
            case 'P' :
                if(!more && !cd && !inter) more = ON;
                argc--;
                break;
            case 'L' :
                if(!lon && !cd && !inter) lon = ON;
                argc--;
                break;
            case 'C' :
                if(!cd && !more && !lon && !inter) cd = ON;
                argc--;
                break;
            case 'I' :
                if(!inter && !more && !lon && !cd) inter = ON;
                argc--;
                break;
            case ']' :
                argc--;
                i = argc;
                while(argv[i][0] != '[' && i > 1) i--;
                if(argv[i][0] == '[')
                {
                    argc = i;
                    strcpy(cmdbuf, &argv[i][1]);
                    for(;;)
                    {
                        i++;
                        strcat(cmdbuf, " ");
                        strcat(cmdbuf, argv[i]);
                        if(cmdbuf[strlen(cmdbuf)-1] == ']')
                        {
                            cmdbuf[strlen(cmdbuf)-1] = 0;
                            break;
                        }
                    }
                    if(!inter && !more && !lon && !cd) inter = 2;
                }
                break;
            default :
                argc--;
        }
    }
    if(strlen(cmdbuf) == 0) inter = OFF;  /* make sure inter is off if */
                                          /* no embedded DOS cmd found */
    dr = (DRIVESPEC *)NULL;  /* initialise drive list */
    sp = (FILESPEC *)NULL;   /* initialise filespec list */
    for(i = 1; i < argc; i++)  /* parse filespecs and drivespecs */
    {
        fnsplit(argv[i], drive, dir, file, ext);
        if(strlen(drive) == 0)           /* if no drive then use */
        {                                /* current drive */
            drive[0] = 65+(getdrive());
            drive[1] = ':';
            drive[2] = 0;
        }
        sp = addfile(sp, drive[0]-65, file, ext);
        if(!drivefound(dr, drive)) dr = adddrive(dr, drive);
    }
    line = 0;
    totbyte = 0L;
    found = 0L;
    while(dr)
    {
        fprintf(stderr, "Scanning DRIVE %s\r\n", dr->drive);
        if(more) domore();
        found += finddir(dr->drive, sp);  /* find dir on each drive in */
        dr = dr->next;                    /* drive list */
    }
    if(lon && totbyte)
    {
        fprintf(stdout, "%15s%9ld bytes\r\n", "", totbyte);
        if(more) domore();
    }
    fprintf(stderr, "%9ld file(s) found.\r\n", found);
    return 0;
}

long int finddir(char *path, FILESPEC *sp)
{
    char newpath[MAXPATH], srchpath[MAXPATH];
    struct ffblk fb;
    long int found = 0L;
    int done = 0;

    found += findfile(path, sp);  /* find files in current dir */
    strcpy(newpath, path);
    strcat(newpath, "\\*.*");
    done = findfirst(newpath, &fb, FA_SUBDIR);  /* find first subdir */
    if(done) return found;  /* if no subdir, return found count */
    if(fb.ff_attrib == FA_SUBDIR &&
    strcmp(fb.ff_name, "..") && strcmp(fb.ff_name, "."))
    {
        strcpy(srchpath, path);
        strcat(srchpath, "\\");
        strcat(srchpath, fb.ff_name);    /* if valid subdir then call */
        found += finddir(srchpath, sp);  /* finddir on subdir */
    }
    for(;;)  /* infinite loop to find remaining subdirs */
    {
        done = findnext(&fb);  /* find next subdir */
        if(done) break;
        if(fb.ff_attrib != FA_SUBDIR) continue;
        if(strcmp(fb.ff_name, "..") && strcmp(fb.ff_name, "."))
        {
            strcpy(srchpath, path);
            strcat(srchpath, "\\");
            strcat(srchpath, fb.ff_name);    /* if valid subdir then call */
            found += finddir(srchpath, sp);  /* finddir on subdir */
        }
    }
    return found;  /* return found count */
}

long int findfile(char *path, FILESPEC *sp)
{
    char newpath[MAXPATH];
    struct ffblk fb;
    long int found = 0L;
    int done;

    while(sp)
    {
        if(sp->drive == (path[0]-65))  /* if filespec is for current */
        {                              /* search drive, process */
            strcpy(newpath, path);
            strcat(newpath, "\\");
            strcat(newpath, sp->filename);
            done = findfirst(newpath, &fb, FA_FILE);  /* find first file */
            if(!done)
            {
                filefound(path, fb);  /* process file found */
                found++;
                for(;;)  /* infinite to find remaining files */
                {
                    done = findnext(&fb);  /* find next file */
                    if(done) break;  /* if no more file break out of loop */
                    filefound(path, fb);  /* process file found */
                    found++;
                }
            }
        }
        sp = sp->next;  /* next filespec in list */
    }
    return found;
}

void filefound(char *path, struct ffblk fb)
{
    char *ret, name[MAXPATH], cmd[MAXCMD];
    unsigned char ch;

    strcpy(name, path);
    strcat(name, "\\");
    strcat(name, fb.ff_name);
    if(inter)  /* if interactive then process cmd buffer */
    {
        ret = strchr(cmdbuf, '%');
        ch = cmdbuf[ret-cmdbuf+1];
        cmdbuf[ret-cmdbuf+1] = 's';  /* replaces F or D with s */
        if(ch == 'D') sprintf(cmd, cmdbuf, path);  /* substitute path or */
        else sprintf(cmd, cmdbuf, name);      /* filename as appropriate */
        cmdbuf[ret-cmdbuf+1] = ch;  /* restore F or D to cmd buffer */
        fprintf(stderr, "%s", cmd);
        if(inter == ON)  /* if interative is on then prompt user */
        {
            fprintf(stderr, "  (Y/N)?");
            ch = getkey();
        } else ch = 0;
        fprintf(stderr, "\r\n");
        if(!ch || ch == 'Y' || ch == 'y') system(cmd);  /* execute cmd */
    }
    else if(cd)  /* if cd then change directory */
    {
        fprintf(stderr, "%s  (CD)?", name);
        ch = getkey();
        fprintf(stderr, "\r\n");
        if(ch == 'Y' || ch == 'y' || ch == 'C' || ch == 'c')
        {
            if((path[0]-65) != (getdrive())) setdrive(path[0]-65);
            chdir(path);
            exit(0);
        }
    }
    else if(lon) printlong(path, fb);
    else fprintf(stdout, "%s\r\n", name);
    if(more) domore();
}

void printlong(char *path, struct ffblk fb)
{
    char tmp[MAXTMP], file[MAXFILE], ext[MAXEXT];
    DOSDATE *d;
    DOSTIME *t;

    fnsplit(fb.ff_name, tmp, tmp, file, ext);
    if(strlen(oldir) == 0 || strcmp(oldir, path))
    {                         /* if new path, print path */
        if(totbyte)
        {
            fprintf(stdout, "%15s%9ld bytes\r\n", "", totbyte);
            totbyte = 0L;
            if(more) domore();
        }
        fprintf(stdout, "Directory: %s\r\n", path);
        strcpy(oldir, path);  /* save path */
        if(more) domore();
    }
    d = (DOSDATE *)&fb.ff_fdate;
    t = (DOSTIME *)&fb.ff_ftime;
    totbyte += fb.ff_fsize;
    fprintf(stdout, "%-8s%-7s%9ld %02u/%02u/%02u  %2u:%02u\r\n", file, ext,
    fb.ff_fsize, d->day, d->month, d->year+80, t->hours, t->minutes);
}

FILESPEC *addfile(FILESPEC *s, unsigned char drive, char *file, char *ext)
{
    FILESPEC *tmp;

    tmp = (FILESPEC *)malloc(sizeof(FILESPEC));
    if(!tmp)
    {
        fprintf(stderr, "\r\nMemory allocation error!\r\n");
        exit(1);
    }
    tmp->drive = drive;
    strcpy(tmp->filename, file);
    strcat(tmp->filename, ext);
    tmp->next = s;
    return tmp;
}

DRIVESPEC *adddrive(DRIVESPEC *d, char *drive)
{
    DRIVESPEC *tmp;

    tmp = (DRIVESPEC *)malloc(sizeof(DRIVESPEC));
    if(!tmp)
    {
        fprintf(stderr, "\r\nMemory allocation error!\r\n");
        exit(1);
    }
    strcpy(tmp->drive, drive);
    tmp->next = d;
    return tmp;
}

char drivefound(DRIVESPEC *d, char *drive)
{
    while(d)
    {
        if(strcmp(d->drive, drive) == 0) return 1;
        d = d->next;
    }
    return 0;
}

void domore(void)
{
    line++;
    if(line < MAXLINE) return;
    fprintf(stderr, "-- MORE --\r");
    getkey();
    fprintf(stderr, "          \r");
    line = 0;
}

unsigned char getkey(void)
{
    asm mov  ah,01h
    asm int  21h
    return _AL;
}

unsigned char getdrive(void)
{
    asm mov  ah,19h
    asm int  21h
    return _AL;
}

void setdrive(unsigned char drive)
{
    _DL = drive;
    asm mov  ah,0eh
    asm int  21h
}
