/**************************************************************************
 *       OCRPRO.C                                                         *
 *                                                                        *
 *                 OCRPRO 1.0, Command Line Version.                      *
 *                 Copyright (c) 1993 Amr A. Awadallah.                   *
 *                                                                        *
 *    "OcrPro" an Optical Character Recognition program based on the      *
 *     center of gravity algorithm with step by step recognition, so      *
 *     that the student can observe each of the main phases for           *
 *     recognition (Preprocessing, normalization, segmentation, etc.)     *
 *                                                                        *
 *                                                                        *
 *       You are allowed to use this source code, in part or in whole,    *
 *       But, please, would you kindly remeber to mention me.             *
 *                                                                        *
 *       I hope you benefit from it, and that it opens in front of you a  *
 *       a zillion gates for new creative ideas.                          *
 *                                                                        *
 *       Coded By:                                                        *
 *                  Amr A. Awadallah,            May 1993                 *
 *                  Computer Department,                                  *
 *                  Faculty Of Engineering,                               *
 *                  Cairo University,                                     *
 *                  Egypt.                                                *
 *                  e-mail : "amr@frcu.eun.eg" or "a.awadallah@ieee.org"  *
 *                  phone  : (202) 330-4738                               *
 *                                                                        *
 **************************************************************************/


#include <conio.h>
#include <stdio.h>
#include <graphics.h>
#include <math.h>
#include <process.h>
#include <string.h>
#include <values.h>
#include <dir.h>

#define MAXHEIGHT 50
#define MAXWIDTH  50
#define MAXLIBSIZE 512
#define MAXLINES 100
#define MAXCHARACTERS 200
#define GRAPHICSTEXTCOLOR WHITE

struct line
{
	int ymin;
	int ymax;
	int space;
};

struct chr
{
	int ymin;
	int ymax;
	int xmin;
	int xmax;
	int space;
};

struct cog
{
	char x;
	char y;
};

struct pattern
{
	struct cog cogs[9];
	char ch;
};

struct pattern lib[MAXLIBSIZE];
int   bmax,updatelib=0,fast,learn;
int   xmax,ymax,colormax,midx,midy;
FILE  *outfileptr;
char libfile[MAXPATH];

char  getkey(void);
void  processCommandLine(int,char* [],char[],char[],char[]);
void  printUsage();
void  initGraphics();
void  loadImage(char[]);
void  drawTextFile(char *);
void  initLines(struct line[]);
int   lineDetermination(struct line[],int*);
void  initChrs(struct chr[]);
int   chrDetermination(struct line,struct chr[],int*);
void  chrFinetune(struct line,struct chr*);
char  recognizeChr(struct chr);
void  genChrMat(struct chr,int*,int*,ch ar[][MAXWIDTH]);
void  dispChrMat(int,int,char[][MAXWIDTH]);
void  dispLineInfo(struct line myline,int,int);
void  dispChrInfo(struct chr);
void  dispStatus(char[]);
void  dispLearn();
void  clacChrCogs(int,int,char[][MAXWIDTH],struct cog[]);
void  dispChrCogs(struct cog[]);
void  saveLib(char[]);
int   getLib(char[]);
void  addChrToLib(struct pattern);
int   getChrFromLib(struct pattern*);
int   calcError(struct cog[], struct cog[]);


main(int argc, char* argv[])
{
	struct line lines[MAXLINES];
	int lspace,cspace;
	int l,lmax;
	char infile[MAXPATH],outfile[MAXPATH];
	int i,s=0; // generic counter & accumulator
	char name[]="OCRPRO 1.0, Command Line Version.\nCopyright (c) 1993 Amr A. Awadallah.\n";
	printf(name);
	for(i=0;i<strlen(name);i++) s^=name[i];
	if(s!=76) abort();
	processCommandLine(argc,argv,infile,outfile,libfile);
	bmax=getLib(libfile);
	initGraphics();
	dispLearn();
	loadImage(infile);
	lmax=lineDetermination(lines,&lspace);
	outfileptr=fopen(outfile,"wb");
	for(l=0;l<lmax;l++)
	{
		struct chr chrs[MAXCHARACTERS];
		int c,cmax;
		char string[MAXCHARACTERS]="\0";
		int yy=445,xx=10;
		setfillstyle(1,DARKGRAY);
		bar(1,438,399,462);
		cmax=chrDetermination(lines[l],chrs,&cspace);
		dispLineInfo(lines[l],lmax,cmax);
		for(c=0;c<cmax;c++)
		{
			char ch;
			char chh[]=" ";
			int i;
			chrFinetune(lines[l],&chrs[c]);
			ch=recognizeChr(chrs[c]);
			setcolor(WHITE);
			for(i=(chrs[c].space)/(4*cspace);i>0;i--)
			{
				outtextxy(xx,yy," "); xx+=8;
				strcat(string," ");
			}
			chh[0]=ch;
			outtextxy(xx,yy,chh); xx+=8;
			strcat(string,chh);
		}
		fputs(string,outfileptr);
		fputs("\n\r",outfileptr);
		getkey();
	}
	closegraph();
	fclose(outfileptr);
	saveLib(libfile);
	return 0;
}

char getkey()
{
	char ch=0;
	do
	{
		if(!fast) ch=getch();
		else if(kbhit()) ch=getch();
		switch(ch)
		{
			case 13 :   learn=!(learn);
							dispLearn();
							break;
			case 27 :   closegraph();
							saveLib(libfile);
							fclose(outfileptr);
							exit(0);
							break;
		}
	} while(ch==13);
	return ch;
}

void  processCommandLine(int argc,char* argv[],char infile[]
														,char outfile[],char libfile[])
{
	int i=0,j,k;
	FILE  *tempfileptr;
	// Set Defaults
	strcpy(outfile,"output.txt");
	strcpy(libfile,"ocrpro.lib");
	learn=0; fast=0;
	if(argc<2)
	{
		printf("\nERROR: Too few arguments, at least input file name must be given.\n");
		printUsage();
		exit(0);
	}
	if(argv[1][0]=='/')
		if(argv[1][1]=='?')
		{
			printUsage();
			exit(0);
		}
		else
		{
			printf("\nERROR: Input filename must be specefied before any options.\n");
			printUsage();
			exit(0);
		}
	i++;
	strcpy(infile,argv[i]);
	i++;
	if(i<argc && argv[i][0]!='/') {strcpy(outfile,argv[i]); i++;}
	for(;i<argc;i++)
	{
		for(j=1;j<strlen(argv[i]);j++)
		{
			switch(argv[i][j])
			{
				case 'l' : learn=1; break;
				case 'f' :
				case 'F' : fast=1; break;
				case 'L' : if(argv[i][j+1]=='\0')
								{
									printf("\nERROR: Missing Library Name.\n");
									printUsage();
									exit(0);
								}
								for(j++,k=0;j<strlen(argv[i]);j++,k++) libfile[k]=argv[i][j];
							  break;
				case '?' : printUsage();
							  printf("\nPRESS ANY KEY TO CONTINE");
							  getch();
							  break;
				default : printf("\nERROR: Invalid option /%c\n",argv[i][j]);
							 printUsage();
							 exit(0);
			}
		}
	}
	if ((tempfileptr=fopen(infile,"r")) == NULL)
	{
		printf("\nError Opening Input Text File: %s\n",infile);
		exit(1);
	}
	fclose(tempfileptr);
	if ((tempfileptr=fopen(outfile,"wb")) == NULL)
	{
		printf("\nError Opening Output Text File: %s\n",outfile);
		exit(1);
	}
	fclose(tempfileptr);
}

void printUsage()
{
	printf("\nUsage: ocrpro infile.txt [outfile.txt] [options]\n");
	printf("Options :\n");
	printf("/l : Enable LEARN mode. (Default LEARN=OFF)\n");
	printf("/f : Enable FAST mode. (Default DETAILED mode)\n");
	printf("/? : Display help screen.\n");
	printf("/Llibname : Use auxiliary library file given by 'libname'.\n\n");
	printf("You could use 'ENTER' key to toggle learning ON/OFF during DETAILED mode.\n");
	printf("The Default library file name is 'ocrpro.lib'.\n");
	printf("The Default output file name is 'output.txt'.\n");
}

void initGraphics()
{
	int gdriver = DETECT, gmode, errorcode;
	initgraph(&gdriver, &gmode, "");
	errorcode = graphresult();
	if (errorcode != grOk)  /* an error occurred */
	{
		printf("Graphics error: %s\n", grapherrormsg(errorcode));
		printf("Requires VGA Card or Compatible\n");
		getch();
		exit(1);             /* return with error code */
	}
	// Initialize Graphics Variables
	xmax=getmaxx(); midx=xmax>>1;
	ymax=getmaxy(); midy=ymax>>1;
	colormax=getmaxcolor();
	dispStatus("INITIALIZING GRAPHICS");
	setcolor(WHITE);
	if(!fast)
	{
		setfillstyle(1,BLACK);     // Graphics File Area
		bar(0,0,xmax-1,midy-1);
		rectangle(0,midy,xmax,ymax);
		line(400,midy,400,ymax);   // Mid Line
		setfillstyle(1,BLACK);     // DispChrMat Area
		line(400,midy+16,xmax,midy+16);
		bar(401,midy+1,xmax-1,ymax-1);
		setfillstyle(1,GREEN);     // DispLineInfo
		bar(1,midy+1,399,307);
		line(0,308,400,308);
		setfillstyle(1,BLUE);      // DispChrInfo
		bar(1,309,399,399);
	}
	rectangle(0,400,400,ymax);
	setfillstyle(1,RED);       // RecognizeInfo
	bar(1,401,399,436);
	line(1,437,399,437);
	setfillstyle(1,DARKGRAY);     // OutTextLine
	bar(1,438,399,462);
	line(0,463,399,463);
	setfillstyle(1,BROWN);     // StatusInfo
	bar(1,464,296,ymax-1);
	line(297,464,297,ymax);
	setfillstyle(1,LIGHTGRAY); // LearnInfo
	bar(298,464,399,ymax-1);
}

void loadImage(char filename[])
{
	dispStatus("LOADING IMAGE FILE TO BUFFER");
	setfillstyle(1,BLACK);     // Graphics File Area
	bar(0,0,xmax-1,midy-1);
	drawTextFile(filename);
}

void drawTextFile(char *filename)
{
	// Open File For Read
	FILE  *fptr;
	char string[81];
	int i,endy;
	fptr=fopen(filename,"r");
	// Print Input Text File In Graphics Mode
	setcolor(GRAPHICSTEXTCOLOR);
	settextstyle(3,HORIZ_DIR, 4);
	moveto(0,0);
	if(!fast) endy=midy; else endy=400;
	i=0;
	while(fgets(string,80,fptr)!=NULL && (i+textheight(string)+4)<endy)
	{
		outtextxy(0,i,string);
		i+=textheight(string)+4;
	}
	fclose(fptr);
}

void initLines(struct line lines[])
{
	int l;
	for(l=0;l<MAXLINES;l++)
		lines[l].ymax=lines[l].ymin=-1;
}

int lineDetermination(struct line lines[],int *lspacep)
{
	int l,s;
	int x,y;
	int endy;
	dispStatus("LINE DETERMINATION");
	initLines(lines);
	l=0;s=0;
	(*lspacep)=MAXINT;
	if(!fast) endy=midy; else endy=400;
	for(y=0;y<endy;y++)  // midy should be ymax for full page scan
	{
		for(x=0;x<xmax;x++)
		{
			if(getpixel(x,y)==GRAPHICSTEXTCOLOR)
			{
				if(lines[l].ymin==-1)
				{
					lines[l].ymin=y;
					lines[l].space=s;
					if(s < (*lspacep) && l>0) (*lspacep)=s;
					s=0;
				}
				break;
			}
			else
			{
				putpixel(x,y,BLUE);
				if(lines[l].ymin!=-1 && x==xmax-1)
				{
					lines[l].ymax=y;
					l++;
				}
			}
		}
		if(x==xmax) s++;
	}
	return l;
}

void initChrs(struct chr chrs[])
{
	int c;
	for(c=0;c<MAXCHARACTERS;c++)
		chrs[c].ymin=chrs[c].ymax=chrs[c].xmin=chrs[c].xmax=chrs[c].space=-1;
}

int chrDetermination(struct line myline,struct chr chrs[],int *cspacep)
{
	int c=0,s=0;
	int x,y;
	dispStatus("CHARACTER DETERMINATION");
	initChrs(chrs);
	(*cspacep)=MAXINT;
	for(x=0;x<xmax;x++)
	{
		for(y=myline.ymin;y<myline.ymax;y++)
		{
			if(getpixel(x,y)==GRAPHICSTEXTCOLOR)
			{
				if(chrs[c].xmin==-1)
				{
					chrs[c].xmin=x;
					chrs[c].space=s;
					if(s < (*cspacep) && c>0) (*cspacep)=s;
					s=0;
				}
				break;
			}
			else
			{
				putpixel(x,y,GREEN);
				if(chrs[c].xmin!=-1 && y==myline.ymax-1)
				{
					chrs[c].xmax=x;
					c++;
				}
			}
		}
		if(y==myline.ymax) s++;
	}
	return c;
}

void chrFinetune(struct line myline,struct chr* mychrptr)
{
	int x,y;
	dispStatus("CHARACTER FINETUNE");
	// Get ymin for the character from above
	for(y=myline.ymin;y<myline.ymax;y++)
	{
		for(x=mychrptr->xmin;x<mychrptr->xmax;x++)
		{
			if(getpixel(x,y)==GRAPHICSTEXTCOLOR)
			{
				if(mychrptr->ymin==-1) mychrptr->ymin=y;
				break;
			}
			else putpixel(x,y,RED);
		}
		if(mychrptr->ymin!=-1) break;
	}
	// Get ymax for the character from below
	for(y=myline.ymax-1;y>=myline.ymin;y--)
	{
		for(x=mychrptr->xmin;x<mychrptr->xmax;x++)
		{
			if(getpixel(x,y)==GRAPHICSTEXTCOLOR)
			{
				if(mychrptr->ymax==-1) mychrptr->ymax=y+1;
				break;
			}
			else putpixel(x,y,RED);
		}
		if(mychrptr->ymax!=-1) break;
	}
}

char recognizeChr(struct chr mychr)
{
	int h,w;
	char letter[MAXHEIGHT][MAXWIDTH];
	char msg[80];
	char ch;
	struct pattern pat;
	int emin;
	int temp; //temp variable to keep value of fast
	int xx=10,yy;
	yy=410;
	dispStatus("CHARACTER RECOGNITION & LEARNING");
	setfillstyle(1,RED);
	bar(1,401,399,436);
	genChrMat(mychr,&h,&w,letter);
	dispChrInfo(mychr);
	dispChrMat(h,w,letter);
	clacChrCogs(h,w,letter,pat.cogs);
	dispChrCogs(pat.cogs);
	pat.ch=''; //Default character is HAPPY FACE.
	emin=getChrFromLib(&pat);
	if(emin)
		if(learn)
		{
			dispStatus("USER INPUT, LEARN MODE");
			setcolor(colormax);
			settextstyle(0, HORIZ_DIR,1);
			if(emin==-1)
				sprintf(msg,"LIBRARY FILE IS EMPTY, Can't make any guesses,");
			else
				sprintf(msg,"Suggested Character = '%c', with error = %d",pat.ch,emin);
			outtextxy(xx,yy,msg);
			outtextxy(xx,yy+=10,"Enter Correct Character, or 'SPACE' to skip: ");
			temp=fast;
			fast=0;
			ch=getkey();
			fast=temp;
			if(ch!=' ')
			{
				pat.ch=ch;
				addChrToLib(pat);
			}
		}
		else
		{
			dispStatus("PRESS ANY KEY TO CONTINUE");
			setcolor(colormax);
			settextstyle(0, HORIZ_DIR,1);
			if(emin==-1)
				sprintf(msg,"LIBRARY FILE IS EMPTY, Can't make any guesses.");
			else
				sprintf(msg,"Guessed Character = '%c' , with error = %d",pat.ch,emin);
			outtextxy(xx,yy,msg);
			getkey();
		}
	else
	{
		dispStatus("PRESS ANY KEY TO CONTINUE");
		setcolor(colormax);
		settextstyle(0, HORIZ_DIR,1);
		sprintf(msg,"Perfect Match Character = '%c'",pat.ch);
		outtextxy(xx,yy,msg);
		getkey();
	}
	return pat.ch;
}

void genChrMat(struct chr mychr,int *hp,int *wp,char letter[][MAXWIDTH])
{
	int i=0,j,x,y;
	dispStatus("GENERATING CHARACTER MATRIX");
	for(y=mychr.ymin;y<mychr.ymax;y++)
	{
		j=0;
		for(x=mychr.xmin;x<mychr.xmax;x++)
		{
			if(getpixel(x,y)==GRAPHICSTEXTCOLOR) letter[i][j]=1;
			else letter[i][j]=0;
			j++;
		}
		i++;
	}
	(*hp)=i;
	(*wp)=j;
}

void dispChrMat(int h,int w,char letter[][MAXWIDTH])
{
	if(!fast)
	{
		int x,y,i,j;
		int xx=410;
		int yy;
		yy=midy+10;
		setfillstyle(1,BLACK);
		bar(xx-9,yy-9,xmax-1,ymax-1);
		settextstyle(0, HORIZ_DIR,1);
		setcolor(WHITE);
		outtextxy(xx+10,yy-5,"CHARACTER ANALYSIS MATRIX");
		line(xx-10,yy+6,xmax,yy+6);
		yy+=15;
		i=0;
		for(y=0;y<h;y++)
		{
			j=0;
			for(x=0;x<w;x++)
			{
				if(letter[y][x])
				{
					setcolor(GREEN);
					outtextxy(x+j+xx,y+i+yy,"");
				}
				else
				{
					setcolor(DARKGRAY);
					outtextxy(x+j+xx,y+i+yy,"");
				}
				j+=8;
			}
			i+=8;
		}
	}
}

void dispLineInfo(struct line myline,int lmax,int cmax)
{
	if(!fast)
	{
		char msg[80];
		int xx=10;
		int yy;
		yy=midy+6;
		setfillstyle(1,GREEN);
		bar(1,midy+1,399,307);
		setcolor(colormax);
		settextstyle(0,0,1);
		sprintf(msg,"Total number of lines = %d",lmax);
		outtextxy(xx,yy,msg);
		outtextxy(xx,yy+=14,"Current Line Information:");
		sprintf(msg,"Number of characters = %d",cmax);
		outtextxy(xx,yy+=12,msg);
		sprintf(msg,"(ymax,ymin) = (%d,%d)",myline.ymax,myline.ymin);
		outtextxy(xx,yy+=10,msg);
		sprintf(msg,"Height = %d, Space before = %d",
							myline.ymax-myline.ymin,myline.space);
		outtextxy(xx,yy+=10,msg);
	}
}

void dispChrInfo(struct chr mychr)
{
	if(!fast)
	{
		char msg[80];
		int xx=10,yy;
		yy=312;
		setfillstyle(1,BLUE);
		bar(1,309,399,399);
		setcolor(colormax);
		settextstyle(0,0,1);
		outtextxy(xx,yy,"Character Information:");
		sprintf(msg,"(ymax,ymin) = (%d,%d)",mychr.ymax,mychr.ymin);
		outtextxy(xx,yy+=11,msg);
		sprintf(msg,"(xmax,xmin) = (%d,%d)",mychr.xmax,mychr.xmin);
		outtextxy(xx,yy+=10,msg);
		sprintf(msg,"Height = %d, Width = %d",
							mychr.ymax-mychr.ymin,mychr.xmax-mychr.xmin);
		outtextxy(xx,yy+=10,msg);
		sprintf(msg,"Space before = %d",mychr.space);
		outtextxy(xx,yy+=10,msg);
	}
}

void dispStatus(char str[])
{
	setfillstyle(1,BROWN);     // StatusInfo
	bar(1,464,296,ymax-1);
	settextstyle(0, HORIZ_DIR,1);
	setcolor(YELLOW);
	outtextxy(10,468,str);
}

void dispLearn()
{
	setfillstyle(1,LIGHTGRAY);
	bar(298,464,399,ymax-1);
	settextstyle(0, HORIZ_DIR,1);
	setcolor(BLACK);
	outtextxy(300,468,"LEARN = ");
	setcolor(RED);
	if(learn) outtextxy(372,468,"ON"); else outtextxy(372,468,"OFF");
}

void clacChrCogs(int h,int w,char letter[][MAXWIDTH],struct cog cogs[])
{
	int x,y;
	int mx,my,mmx,mmy,t;
	int xx=410,yy;
	yy=midy+25;
	dispStatus("CALCULATING CHARACTER COGS");
	mx=my=t=0;
	for(y=0;y<h;y++)
		for(x=0;x<w;x++)
			if(letter[y][x])
			{
				t++;
				mx+=x;
				my+=y;
			}
	if(t) {mx/=t; my/=t;}
	cogs[0].x=mx;
	cogs[0].y=my;

	setcolor(LIGHTGRAY);
	settextstyle(0, HORIZ_DIR,1);
	mmx=mmy=t=0;
	for(x=0;x<mx;x++)
	{
		if(!fast) outtextxy(xx+x*9,yy+my*9,"*");
		if(letter[my][x])
		{
			t++;
			mmx+=x;
		}
	}
	if(t) mmx/=t;
	cogs[1].x=t?mmx:mx;
	cogs[1].y=my;

	mmx=mmy=t=0;
	for(y=0;y<my;y++)
	 for(x=0;x<mx;x++)
		 if(y==x-mx+my)
		 {
			 if(!fast) outtextxy(xx+x*9,yy+y*9,"*");
			 if(letter[y][x])
			 {
				 t++;
				 mmx+=x;
				 mmy+=y;
			 }
		 }
	if(t) {mmx/=t; mmy/=t;}
	cogs[2].x=t?mmx:mx;
	cogs[2].y=t?mmy:my;

	mmx=mmy=t=0;
	for(y=0;y<my;y++)
	{
		if(!fast) outtextxy(xx+mx*9,yy+y*9,"*");
		if(letter[y][mx])
		{
			t++;
			mmy+=y;
		}
	}
	if(t) mmy/=t;
	cogs[3].x=mx;
	cogs[3].y=t?mmy:my;

	mmx=mmy=t=0;
	for(y=0;y<my;y++)
	 for(x=mx+1;x<w;x++)
		 if(y==-x+mx+my)
			{
			 if(!fast) outtextxy(xx+x*9,yy+y*9,"*");
			 if(letter[y][x])
			 {
				 t++;
				 mmx+=x;
				 mmy+=y;
			 }
		 }
	if(t) {mmx/=t; mmy/=t;}
	cogs[4].x=t?mmx:mx;
	cogs[4].y=t?mmy:my;

	mmx=mmy=t=0;
	for(x=mx+1;x<w;x++)
	{
		if(!fast) outtextxy(xx+x*9,yy+my*9,"*");
		if(letter[my][x])
		{
			t++;
			mmx+=x;
		}
	}
	if(t) mmx/=t;
	cogs[5].x=t?mmx:mx;
	cogs[5].y=my;

	mmx=mmy=t=0;
	for(y=my+1;y<h;y++)
	 for(x=mx+1;x<w;x++)
		if(y==x-mx+my)
		{
			 if(!fast) outtextxy(xx+x*9,yy+y*9,"*");
			 if(letter[y][x])
			 {
				 t++;
				 mmx+=x;
				 mmy+=y;
			 }
		 }
	if(t) {mmx/=t; mmy/=t;}
	cogs[6].x=t?mmx:mx;
	cogs[6].y=t?mmy:my;

	mmx=mmy=t=0;
	for(y=my+1;y<h;y++)
	{
		if(!fast) outtextxy(xx+mx*9,yy+y*9,"*");
		if(letter[y][mx])
		{
			t++;
			mmy+=y;
		}
	}
	if(t) mmy/=t;
	cogs[7].x=mx;
	cogs[7].y=t?mmy:my;

	mmx=mmy=t=0;
	for(y=my+1;y<h;y++)
	 for(x=0;x<mx;x++)
		 if(y==-x+mx+my)
		 {
			 if(!fast) outtextxy(xx+x*9,yy+y*9,"*");
			 if(letter[y][x])
			 {
				 t++;
				 mmx+=x;
				 mmy+=y;
			 }
		 }
	if(t) {mmx/=t; mmy/=t;}
	cogs[8].x=t?mmx:mx;
	cogs[8].y=t?mmy:my;
}

void dispChrCogs(struct cog cogs[])
{
	if(!fast)
	{
		int i;
		char msg[80],msg1[80];
		int xxx=410,yyy;
		int xx=10,yy;
		yy=365;
		yyy=midy+25;
		setcolor(WHITE);
		settextstyle(0, HORIZ_DIR,1);
		outtextxy(xx,yy,"Centers of gravity =");
		strcpy(msg,"\0");
		for(i=0;i<5;i++)
		{
			sprintf(msg1,"(%d,%d),",cogs[i].x,cogs[i].y);
			strcat(msg,msg1);
		}
		outtextxy(xx,yy+=10,msg);
		strcpy(msg,"\0");
		for(i=5;i<9;i++)
		{
			sprintf(msg1,"(%d,%d),",cogs[i].x,cogs[i].y);
			strcat(msg,msg1);
		}
		outtextxy(xx,yy+=10,msg);
		setcolor(BLUE);
		for(i=1;i<9;i++)
		{
			outtextxy(xxx+cogs[i].x*9,yyy+cogs[i].y*9,"*");
		}
		setcolor(RED);
		outtextxy(xxx+cogs[0].x*9,yyy+cogs[0].y*9,"*");
	}
}

void saveLib(char libfile[])
{
	if(updatelib)
	{
		FILE  *libFilePtr;
		if(bmax>0)
		{
			if((libFilePtr=fopen(libfile,"wb"))==NULL)
			{
				perror("Error Opening Library File for output\n");
				exit(1);
			}
			fwrite(lib,sizeof(lib[0]),bmax,libFilePtr);
			fclose(libFilePtr);
		}
	}
}

int getLib(char libfile[])
{
	FILE  *libFilePtr;
	int b=0;
	if((libFilePtr=fopen(libfile,"rb"))!=NULL)
	{
		while(fread(&lib[b],sizeof(lib[b]),1,libFilePtr)==1) b++;
		fclose(libFilePtr);
	}
	return b;
}

void addChrToLib(struct pattern pat)
{
	memcpy(&lib[bmax++],&pat,sizeof(pat));
	updatelib=1;
}

int getChrFromLib(struct pattern* patp)
{
	int e,emin=-1;
	int b;
	if(bmax)
	{
		dispStatus("SEARCHING FOR CHARACTER");
		emin=calcError(patp->cogs,lib[0].cogs);
		patp->ch=lib[0].ch;
		for(b=1;b<bmax;b++)
		{
			e=calcError(patp->cogs,lib[b].cogs);
			if(e<emin) {emin=e; patp->ch=lib[b].ch;}
		}
	}
	return emin;
}

int calcError(struct cog cogs1[], struct cog cogs2[])
{
	int xerror=0,yerror=0,terror;
	int i;
	for(i=0;i<9;i++)
	{
		xerror+=(cogs1[i].x-cogs2[i].x)*(cogs1[i].x-cogs2[i].x);
		yerror+=(cogs1[i].y-cogs2[i].y)*(cogs1[i].y-cogs2[i].y);
	}
	terror=xerror+yerror;
	return terror;
}
