{ WASTED.EXE - Version 1.8    }
{ Created: 05/17/1996         }
{ Writen by Tim Jones         }
{ tjones@wpogate.ssc.nasa.gov }

{ Modified 4 June, 1996 by Jason Hood:                              }
{ used comma-separated numbers and adjusted the display accordingly }

{ Modified 31 October, 1996 by Jason Hood:                          }
{ added the /F option to display wastage of files                   }
{ added the ability to select file specifications                   }
{ added the /NS option to prevent recursing into subdirectories     }

{ Modified 28 to 30 March, 1997 by Jason Hood:			    }
{ added the /C option to compare cluster sizes                      }
{ took directory wastage into account				    }

{$M 32767,0,655360}          {set up a large stack for recursion}

Program wasted;
uses
  dos;                       {for things like findfirst/findnext, intr() etc.}
const
  version='1.8';             {current version of the program}
  lastrptm=2;                {number of report methods in the program}
  clusternum=8; 	     {number of clusters for comparison}
  direntry=32;		     {number of bytes for a directory entry}

type ClusterSizes = record
		      cluster:longint;	{cluster size}
		      total:longint;  	{total number of clusters}
		      size:longint;	{total number of bytes for files}
		      dirsize:longint;	{total number of bytes used in dirs}
		    end;

var
  fspec:string;              {file spec. for selecting files}
  param:string;              {holding variable for each cmd-line parameter}
  startDir,origDir:string;   {dir to start checking at and the dir we are in}
  totclust:longint;          {total number of clusters on the disk}
  clust,usrclust:longint;    {cluster size, user cluster size}
  totalbs:longint;           {total bytesize for all directories}
  totalrs:longint;           {total realsize for all directories}
  totalds:longint;           {total dirsize  for all directories}
  totalfiles:longint;	     {total number of files}
  totaldirs:longint;	     {total number of subdirectories}
  clustcmp:array [1..clusternum] of ClusterSizes; {clusters for comparison}
  cmpclust:boolean;	     {true if comparing clusters}
  regs:registers;            {variable for holding DOS Interrupt registers}
  i:byte;                    {loop variable}
  fcount:byte;               {# of files specified on the cmd-line}
  sdrive:byte;               {source drive in numerical format A=1, B=2 etc.}
  s1name,source:dirstr;      {user specified dir and actual dir to be used}
  sdir:dirstr;               {pieces parts for the split source dir}
  sname:namestr;             {pieces parts for the split source dir}
  sext:extstr;               {pieces parts for the split source dir}
  error:integer;             {string to numeric conversion error flag}
  files:boolean;	     {display only files in selected directory}
  showtotals:boolean;        {holds user choice for showing totals or not}
  recurse:boolean;	     {recurse into subdirectories or not}
  nt:boolean;		     {flag to indicate /NT has been used}
  pause:boolean;             {holds user choice for screen pausing}
  linecount:byte;            {holds the number of screen lines displayed}
  quitit:boolean;            {if quitit is true then the program will end}
  usrrptm:integer;           {user specified report method}
  tmp:string;                {used to hold numbers for conversion to strings}


{This function will convert a given string into an all upper case string}
function upper(s:string):string;
var
  i:Integer;
begin
  for i := 1 to Length(s) do
    s[i] := UpCase(s[i]);
  upper:=s;
end;


{This function will "pad right" a string to a given length with a given char.}
function padr(s:string; l:integer; c:char):string;
  var j:integer;
begin
  while length(s) < l do
  begin
    s:=s+c;
  end;
  padr:=s;
end;


{This function will return the left portion of a string (chars 1 to p)}
function left(s:string; p:integer):string;
begin
  left:=copy(s, 1, p);
end;


{This function will return the right portion of a string (chars strlen to p)}
function right(s:string; p:integer):string;
begin
  right:=copy(s, length(s)-p+1, p);
end;


{This function will return a comma-separated number}
function sep(num:longint):string;
var
  s: string;
  j: integer;
begin
  str(num, s);
  j := length(s) - 2;
  while ( j > 1) do begin
    insert(',', s, j);
    DEC(j,3);
  end;
  sep := s;
end;


{This function invokes DOS interrupt 21h function 1Ch to return the cluster}
{size for the given drive}
{bytedrv = numerical equivalent of the drive letter A=0, B=1, C=2 etc.}
function getClustSize(bytedrv:byte):word;
var
  regs:registers;
begin
  regs.AH:=$1C;                     {Call to Get FAT info}
  regs.DL:=bytedrv;                 {function 1C expects A=1, B=2, C=3 etc.}
  intr($21,regs);                   {perform actual interrupt call}
  getClustSize:=regs.al*regs.cx;    {calculate/return cluster size}
end;


{This function invokes DOS interrupt 21h function 36h to return the number}
{of used clusters on the given drive}
{bytedrv = numerical equivalent of the drive letter A=0, B=1, C=2 etc.}
function getTotalClust(bytedrv:byte):word;
var
  regs:registers;
begin
  regs.AH:=$36;                     {Call to Get Free Disk Space}
  regs.DL:=bytedrv;                 {function 36 expects A=1, B=2, C=3 etc.}
  intr($21,regs);                   {perform actual interrupt call}
  getTotalClust:=regs.dx;           {calculate/return no. of used clusters}
end;


{This function will return the actualsize given the bytesize and clustersize.}
function getActualSize(bytesize,cluster:longint):longint;
  var actualsize:longint;
begin
  actualsize:=(bytesize div cluster)*cluster;
  if bytesize mod cluster > 0 then
    INC(actualsize,cluster);
  getActualSize:=actualsize;
end;


{This function will make sure that the given directory has at least one}
{directory specified and will chop off the trailing backslash if needed}
{This function was necessary because the pascal CHDIR() function does not}
{appear to work properly.  ex. chdir('C:\') works but chdir('C:\BP\BIN\')}
{does not work.  (go figure)}
function dir(sdir:string):string;
begin
  if right(sdir,1)='\' then        {check for an ending backslash}
  begin
    delete(sdir, length(sdir), 1); {remove the ending backslash}
    if right(sdir,1)=':' then      {check for the case of "C:\"}
      sdir:=sdir+'\';              {put the ending backslash back on}
  end;
  dir:=sdir;  {return the new directory}
end;

{This procedure will display the syntax for the program and also display an}
{error message if one was supplied.  This procedure always halts the program}
{with an error level 0 (no error) or 1 (error)}
procedure syntax(errmsg:string);
begin
  writeln(' Purpose:');
  writeln('   WASTED was written to quickly traverse a harddrive and report how much');
  writeln('   diskspace each directory really uses based upon the cluster size.');
  writeln(' Output:');
  writeln('   bytes  usedbytes  usedbytes-bytes  percentage');
  writeln('   where percentage is based upon the report method used. (see below)');
  writeln(' Syntax:');
  writeln('   WASTED [directory][spec1[;spec2[;...]]] [parameters]');
  writeln('   where directory is the directory to start at (default=current directory)');
  writeln('         specn is the file specification to use (must have * or ?)');
  writeln(' Parameters: (prefix / and - are valid)');
  writeln('   /? = This help text');
  writeln('   /C:n = Sets the cluster size to n (where n > 0)');
  writeln('   /C   = Compare cluster sizes');
  writeln('   /F   = Display each file, with an overall total');
  writeln('   /NT  = Turns off the displaying of the Total line (No Totals)');
  writeln('   /NS  = Selected directory only (No Subdirectories)');
  writeln('   /P   = Pause between screens');
  writeln('   /R:n = Report using method n');
  writeln('          n=1 = Wasted percentage based upon disk size');
  writeln('          n=2 = Wasted percentage based upon used disk space (default)');
  writeln(' optional parameters may be specified in any order');
  writeln;
  if errmsg <> '' then
  begin
    writeln;
    writeln('Error: '+errmsg);
    halt(1);
  end;
  halt(0);
end;


{This function takes a given drive/directory and figures out the cluster info}
function init(startdir:string):string;
var
  source:string[80];
  posn:integer;
begin
  {replace / with \ for those of us who like UN?X style}
  for posn := 1 to length(startdir) do
    if startdir[posn] = '/' then startdir[posn]:='\';
  {generate the source drive (default = current drive\dir)}
  source:=fexpand(startdir);       {expands the dir to a fully qualified dir}
  if right(source,1)<>'\' then begin {not explicitly a directory}
    posn:=pos(';',source);
    if posn <> 0 then begin        {multiple file specifications}
      repeat DEC(posn);            {find the start of the first one}
      until source[posn] = '\';
      fspec:=copy(source,posn+1,length(source));
    end
    else begin
      fspec:='';
      {assume a directory if there's no wildcards}
      if (pos('*',source) = 0) and (pos('?',source) = 0) then
	source:=source+'\';	   {append an ending \}
    end;
  end;
  fsplit(source,sdir,sname,sext);  {splits the dir into pieces parts}
  if fspec='' then begin	   {will be *.* if no spec. given}
    if sname='' then fspec:='*' else fspec:=sname;
    if sext='' then fspec:=fspec+'.*' else fspec:=fspec+sext;
  end;
  sdrive:=ord(upcase(sdir[1]))-64; {store the source drive as a number A=1}
  clust:=getClustSize(sdrive);     {store the cluster size}
  totclust:=getTotalClust(sdrive); {store the total number of clusters}
  if usrclust > 0 then             {if the user specified a cluster size...}
  begin
    {calc how many user clusters will fit on the drive}
    totclust:=(totclust*clust) div usrclust;
    clust:=usrclust;               {use the users cluster size}
  end
  else if cmpclust then
  begin
    usrclust:=512;             	   {first cluster size for comparison}
    for i:=1 to clusternum do
      with clustcmp[i] do
      begin
	cluster:=usrclust;
	total:=(totclust*clust) div usrclust;
	size:=0;
	dirsize:=0;
	usrclust:=usrclust*2;
      end;
  end;
  totalbs:=0;                      {initialize the total bytesize}
  totalrs:=0;                      {initialize the total realsize}
  totalds:=0;			   {initialize the total dirsize}
  totalfiles:=0;		   {initialize the total file count}
  totaldirs:=0;			   {initialize the total subdirectory count}
  linecount:=0;                    {initialize the number of lines displayed}
  quitit:=false;                   {allow recursion to start}
  init:=sdir;                      {return the drive/dir we just got info on}
end;


{This function invokes DOS interrupt 21h function 07h to wait for user input}
{from STDIN.  The reason I just didn't use Pascal's Readkey or Keypressed}
{functions is because if you include the CRT unit, then you cannot perform}
{redirection of output on the dos commandline ex. WASTED C:\ > FULLDISK.TXT}
function dosPause:byte;
var
  regs:registers;
begin
  regs.AH:=$07;                     {Call to Direct STDIN Input function}
  intr($21,regs);                   {perform actual interrupt call}
  if regs.AL = $1B then             {check for ESC key}
    quitit:=true;                   {set flag to break out of recursion loop}
end;


procedure checkpause(lines:byte);
var
  k:char;
begin
  if pause = true then
  begin
    if linecount+lines > 23 then
    begin
      dospause;
      linecount:=0;
    end;
    INC(linecount,lines);
  end;
end;


{This procedure writes the path (and filename), padding as necessary and}
{checking for pause}
procedure writepath(path:string);
begin
  if length(path) > 33 then begin
    checkpause(2);		 {add 2 to linecount for long paths}
    writeln(path);               {display it on its own line}
    write(' ':33);		 {and move the next line over to the counts}
  end
  else begin
    checkpause(1);	         {add 1 to the linecount and check for pause}
    write(padr(path,33,' '));
  end;
end;

{This procedure writes all the values}
procedure writevalues(bytesize, realsize: longint);
begin
  write(sep(bytesize):12,' ',sep(realsize):12,' ',sep(realsize-bytesize):12,' ');
  if (totclust = 0) or (realsize = 0) then write(0.0:6:2) {avoid div by zero}
  else
    case usrrptm of
      1:write(((((realsize-bytesize)/clust)/totclust)*100):6:2);
      2:write(100*((realsize-bytesize)/realsize):6:2);
    end;
  writeln('%');
end;


{This procedure will calculate the amount of wasted space taken up by the}
{files in the current directory. fspec is a string containing possible multiple}
{specifications, eg: "*.exe;*.com".}
procedure calcWaste(fspec:string);
var
  dirinfo:searchrec; {search record for findfirst/findnext functions}
  bytesize:longint;  {combined byte size of the files as reported by DOS}
  realsize:longint;  {combined byte size of the files based upon clusters}
  actualsize:longint;{byte size of a file based upon clusters}
  filename:string;   {full path and filename}
  spec:string;	     {file spec for testing zero count}
  filecount:integer; {number of files and subdirectories in this directory}
  i:integer;	     {loop for comparing clusters}
begin
  bytesize:=0;  {initialize byte size of the files as reported by DOS}
  realsize:=0;  {initialize byte size of the files based upon clusters}
  filecount:=0; 
  fspec:=fspec+';';	{to simplify processing}
  repeat        {for each file spec - could duplicate files}
    spec:=left(fspec,pos(';',fspec)-1);
    findfirst(spec,hidden+readonly+sysfile+archive+directory,dirinfo);
    while doserror = 0 do                       {while files exist in the dir}
    begin
      INC(filecount);
      if dirinfo.attr and directory <> directory then
      begin
	INC(totalfiles);
	INC(bytesize,dirinfo.size);             {add byte sizes}
	if cmpclust then
	  for i:=1 to clusternum do
	    with clustcmp[i] do
	      INC(size,getActualSize(dirinfo.size,cluster))
	else
	begin
	  actualsize:=getActualSize(dirinfo.size,clust);
	  INC(realsize,actualsize);  	        {add clust sizes}
	  if files then begin
	    filename:=fexpand(dirinfo.name);
	    writepath(filename);
	    writevalues(dirinfo.size, actualsize);
	  end;
	end;
      end;
      findnext(dirinfo);                        {find the next file}
    end;
    delete(fspec,1,pos(';',fspec));		{find the next spec.}
  until fspec='';

  if (cmpclust) and (spec = '*.*') then		{add the space taken by}
  begin                                         {the directory itself}
    actualsize:=filecount*direntry;
    INC(totalds,actualsize);
    for i:=1 to clusternum do
      with clustcmp[i] do
	INC(dirsize,getActualSize(actualsize,cluster));
  end;

  if not files and not cmpclust then begin {totals line will take care of files}
    {if a filespec has been given then ignore directories not containing it}
    if (bytesize = 0) and (spec <> '*.*') then
      write(chr(13))	{will be safely overwritten with spaces}
    else begin
      if (spec = '*.*') then begin
	INC(bytesize,filecount*direntry);
	INC(realsize,getActualSize(filecount*direntry,clust));
      end;
      writevalues(bytesize, realsize);
    end;
  end;
  INC(totalbs,bytesize); {add this directory bytesize to the total}
  INC(totalrs,realsize); {add this directory realsize to the total}
end;


{This procedure will display the dir we are working on, calculate the space}
{being wasted in that dir, and finally, recurse into any subdirectories}
procedure showWasted(theDir:string);
var
  nextDir:string;      {var to hold the next dir before recursing into it}
  dirinfo2:searchrec;  {searchrec for findfirst/findnext}
begin
  if not files and not cmpclust then writepath(theDir);
  calcwaste(fspec);    {calculate and display wasted space}
  if not recurse then exit;
  {find the first subdir in this dir}
  findfirst('*.*',hidden+sysfile+readonly+archive+directory,dirinfo2);
  while (quitit = false) and (doserror = 0) do {while no ESC key and a file was found (no error)}
  begin
    if dirinfo2.attr and directory = directory then  {if the file found is a directory...}
    begin
      nextDir:=dirinfo2.name;   {store the name as the next dir to recurse}
      {only recurse non . and .. directories}
      if (nextDir <> '.') and (nextDir <> '..') then
      begin
	INC(totaldirs);            {update the subdirectory count}
	nextDir:=fexpand(nextDir); {expand the next dir for display later}
	ChDir(nextDir);         {change into the next directory}
	showWasted(nextDir);    {call this procedure(showWasted) recursivly}
	ChDir('..');            {change back to this directory}
      end;
    end;
    findnext(dirinfo2);         {find the next file/directory}
  end;
end;


{This procedure will display some initial stats and spawn off the initial}
{call to the recursion procedure}
{Munge, munj, n. To process information; think; muttle over.}
procedure munge;
  var i:integer;

  function verb(number:longint):string;
  begin
    if number = 0 then verb := ''
    else if number = 1 then verb := 'is'
    else verb := 'are';
  end;

  function plural(number:longint; word:string):string;
  begin
    if number = 1 then plural := word
    else
    begin
      if right(word,1) = 'y' then plural:=left(word,length(word)-1)+'ies'
      {else if right(word,1) = 'x' then plural:=word+'es'}
      else plural:=word+'s';
    end;
  end;

begin
  GetDir(0,origDir);       {find where we are on the drive so we can come back}
  if s1name <> '' then     {if the user specified a starting point...}
    startDir:=init(s1name) {init drive info for the users drive}
  else
    startDir:=init(origDir); {init drive info for the current drive}
  {display cluster info and column headers}
  writeln('Cluster size  :  ',sep(clust):12);
  writeln('# of Clusters :  ',sep(totclust):12);
  writeln('Est. Disk Size:  ',sep(totclust*clust):12);
  writeln;
  if not cmpclust then
  begin
    if files then write('File     ')
    else          write('Directory');
    writeln('                            Bytesize     Realsize       Wasted');
  end;
  linecount:=6;
  {$I-}
  ChDir(dir(startDir));         {go into the starting directory}
  if IOResult = 3 then syntax('Path not found: '+startDir);
  {$I+}
  showWasted(startDir);       {display wasted space in this dir and its subdirs}
  if cmpclust then
  begin
  if (totalfiles = 0) then
    writeln('There are no files matching "',startdir,fspec,'".')
  else begin
    write('   In ',startdir,fspec);
    if (totaldirs <> 0) then write(' and the ',sep(totaldirs),' ',plural(totaldirs,'directory'), ' below it');
    writeln(' there ',verb(totalfiles));
    write('   ',sep(totalfiles),' ',plural(totalfiles,'file'));
    write(' occupying ',sep(totalbs),' ',plural(totalbs,'byte'),'.');
    writeln;
    if fspec = '*.*' then
      writeln('   It requires ', sep(totalds), ' bytes to store the directory entries.');
    writeln;
    if fspec = '*.*' then begin
      writeln('                    Realsize                         Wasted');
      writeln('   Cluster        Files  Directories       Files      Dirs.       Total');
    end
    else writeln('   Cluster     Realsize       Wasted');
    for i:=1 to clusternum do
      with clustcmp[i] do
      begin
	if clust = cluster then write(startdir[1]+':=>')
	else write('    ');
	write(sep(cluster):6,' ',sep(size):12,' ');
	if fspec = '*.*' then
	  write(sep(dirsize):12,' ',sep(size-totalbs):11,' ',
		sep(dirsize-totalds):10,' ',
		sep(size-totalbs + dirsize-totalds):11,' ')
	else
	  write(sep(size-totalbs):12,' ');
	if (total = 0) or (size = 0) then write(0.0:6:2) {avoid div by zero}
	else
	  case usrrptm of
	    1:write(((((size-totalbs+dirsize-totalds)/cluster)/total)*100):6:2);
	    2:write(100*((size-totalbs+dirsize-totalds)/(size+dirsize)):6:2);
	  end;
	writeln('%');
      end
    end
  end
  else
  if showtotals = true then   {display the total line if the user hasn't specified otherwise}
  begin
    write('Total:', ' ':27);
    writevalues(totalbs, totalrs);
  end;
  ChDir(origDir);          {go back to where we originally started from}
end;


begin
  {display the program name, version, and author(s)}
  writeln;
  writeln('WASTED.EXE v',version,' - written by Tim Jones and Jason Hood');
  writeln;
  fcount:=0;                 {set file counter to zero}
  usrclust:=0;               {set user defined clustersize to 0}
  cmpclust:=false;	     {assume not comparing cluster sizes}
  {change the usrrptm value to a 1 to make the second report method the default}
  usrrptm:=2;                {set the report method to 2 (default)}
  files:=false;		     {directories only}
  showtotals:=true;          {set the flag to show the totals}
  recurse:=true;	     {all subdirectories}
  nt:=false;		     {no /NT}
  tmp:='';                   {init the tmp number holder}
  for i:=1 to paramcount do  {loop through each command-line parameter}
  begin
    param:=upper(paramstr(i));  {store off the current parameter in uppercase}
    if (left(param,1) = '/') or (left(param,1) = '-') then  {is it an option?}
    begin
      case param[2] of   {check the second character of the parameter}
	'?':syntax('');  {help text}
	'C':begin    {user specified cluster size}
	      {check for a colon separator and report any error}
	      if param[3] <> ':' then
		if length(param) > 2 then syntax('Unknown parameter:'+param)
		else cmpclust:=true
	      else
	      begin
		{convert what follows the colon into a number}
		val(right(param, length(param)-3), usrclust, error);
		if error <> 0 then  {if the conversion failed...}
		begin
		  {display the conversion error and quit}
		  syntax('Numeric conversion error in parameter:'+param);
		end;
		if usrclust <= 0 then  {if the cluster size is <=0...}
		begin
		  {display the cluster size error and quit}
		  syntax('Cluster size must be >= 0');
		end;
	      end;
	    end;
	'F':begin    {Listing files}
	      files:=true;
	      if not nt then showtotals:=true; {in case /NS turns it off}
	    end;
	'N':begin    {No ... parameter?}
	      if param[3] = 'S' then begin
		recurse:=false;
		if not files then showtotals:=false; {no totals for one directory}
	      end
	      else if param[3] = 'T' then begin
		showtotals:=false;     {turn off the showing of end totals}
		nt:=true;	       {explicitly turned off}
	      end
	      else syntax('Unknown parameter:'+param);
	    end;
	'P':begin    {Pause screen listing}
	      pause:=true;           {turn on the pausing between screenfulls}
	    end;
	'R':begin    {report method was specified}
	      {check for a colon separator and report any error}
	      if param[3] <> ':' then syntax('Unknown parameter:'+param);
	      {convert what follows the colon into a number}
	      val(right(param, length(param)-3), usrrptm, error);
	      if error <> 0 then  {if the conversion failed...}
	      begin
		{display the conversion error and quit}
		syntax('Numeric conversion error in parameter:'+param);
	      end;
	      if usrrptm <= 0 then  {if the report method is <=0...}
	      begin
		{display the report method error and quit}
		syntax('Report method must be >= 0');
	      end;
	      if usrrptm > lastrptm then  {if the report method is too high...}
	      begin
		{display the report method error and quit}
		str(lastrptm,tmp);  {convert numeric report method to string}
		syntax('Report method must be <= '+tmp);
	      end;
	    end;
	else
	  {the user specified an unknown parameter so display error and quit}
	  syntax('Unknown parameter:'+param);
      end; {case}
    end else
    begin
      {the user must have specified a filename / directory}
      fcount:=fcount+1;
      case fcount of  {this case stmt is here for future expansion}
	1: s1name:=param;  {source 1 name = user specified starting directory}
      else
	{only one file parameter is allowed, display error and quit}
	syntax('Too many file parameters:'+param);
      end;
    end;
  end;
  munge; {let's go!!!}
end.

{Program History:
 v1.0 - first (non public) release limited to 4096 byte cluster size
 v1.1 + added auto calculating of cluster size
      - changed wasted percentage to be percentage of used disk space
 v1.2 - changed wasted percentage to be percentage of total disk space
      - previous versions were fixed to C:
      + added ability to start from any directory
 v1.3 - first public release
      - completely re-worked the checking of commandline parameters
      - fixed a bug which ended the recursion early
      + added help/syntax screen
      + added /? option for display of help/syntax screen
      + added /C:n option to allow the user to change the cluster size
      + added this history list
 v1.4 - fixed a bug which causes Runtime Error 003 (path not found) when
	you are in a subdirectory and did not specify a starting directory.
      + now you can specify paths with or without the trailing backslash
      + added a Total line at the end (default is to show totals)
      + added the switch /NT to disable the showing of totals
      + added the switch /P to enable a pause between screenfulls of info
 v1.5 + added the switch /R:n to allow the user to select the method
	for calculating the wasted percentage
	n=1 = Wasted percentage based upon disk size (default)
	n=2 = Wasted percentage based upon used disk space
      + added more information to the documentation file
 v1.6 - the program will now recurse into hidden subdirectories
      + recompiled to produce a smaller EXE (almost half the original size!)
 v1.61 + used commas in displaying numbers
 v1.7 + added the switch /F to display files
      + added file specifications capability
      + added the switch /NS to prevent recursing into subdirectories
      - general maintenance
 v1.8 + added comparison of cluster sizes by using plain /C
      + added approximate directory wastage
      - used INC and DEC instructions at appropriate places
}
