UNIT _XU_lib;

  {*****************************************************************}
  {*******  XU_Lib.pas  (c) 1996 by Dutra Lacerda <2:362/20> *******}
  {*******        Hereby donated to the public domain        *******}
  {*****************************************************************}
  {*  DESCRIPTION:                                                 *}
  {*                TxuAtom   = Encode/Decode a 3-4 Bytes Block    *}
  {*                TxuBuffer = Encode/Decode a Buffer             *}
  {*                TxuFile   = Encode/Decode a File               *}
  {*  DESCENDANTS:                                                 *}
  {*                Txx?????  = Encode/Decode with XX Table        *}
  {*                Tuu?????  = Encode/Decode with UU Table        *}
  {*****************************************************************}

INTERFACE
Const
   xuMaxBuffSize         = $2000;    { 8K Blocks of 3..4 bytes }
Type
  TxuCodeString  = string[64];
Type
  TxuDecodedAtom = array[1..3] of byte;   {Decoded 3 byte Group}
  TxuEncodedAtom = array[1..4] of char;   {Encoded 4 char Group}
Type
  TxuDecodedBuffer = array[0..xuMaxBuffSize-1] of TxuDecodedAtom;
  TxuEncodedBuffer = array[0..xuMaxBuffSize-1] of TxuEncodedAtom;
Type
  TxuDecodedFile = File;
  TxuEncodedFile = TEXT;
Type
  TxuDecodedText = File;
  TxuEncodedText = TEXT;

  {*****************************************************************}
  {*                   TxuAtom and successors:                     *}
  {*****************************************************************}
  {*  DESCRIPTION:                                                 *}
  {*  -They translate, with AtomEncode/AtomDecode procedures,      *}
  {*   any given Atom between plain code and XX/UU code.           *}
  {*  -The INIT of an object is responsibility of the programmer   *}
  {*   and will NOT be tested!  Reason is performance.             *}
  {*  -The basic TxuAtom INIT receives an encoding table in the    *}
  {*   form of a string with all encoding chars in is position     *}
  {*  -The derivate TxxAtom and TuuAtom Objects also MUST be       *}
  {*   Initialized but with no arguments since their tables ARE    *}
  {*   standard and included in the implementation section.        *}
  {*  -TxuAtom have no variable internal states                    *}
  {*****************************************************************}
  {*  USAGE:                                                       *}
  {*  -Choose the needed object according with the desired method  *}
  {*   witch can bee XX, UU or User-Defined-Encoding-Table         *}
  {*  -Init the Object appropriately.                              *}
  {*  -USE TEncodeAtom and TDecodeAtom type variables to feed the  *}
  {*   AtomEncode and AtomDecode Procedures.                       *}
  {*  -Each object can be used asynchronously.                     *}
  {*  -Each object can be used in several simultaneous tasks.      *}
  {*****************************************************************}

  TxuAtomPtr = ^TxuAtom;
  TxuAtom = {*** USER-DEFINED-Encoding-Table ***}
    object
      procedure Init( CodeString : TxuCodeString );
        {PreCondition: CodeString must have 64 unique Chars   }
        {              with ascii values from #20 to #127     }
        {PosCondition: Both encode and decode tables have     }
        {              been initialized and all procedures    }
        {              are now ready to be used.              }
      procedure InitXX; {*** XX Init equivalent to BASE-64 ***}
        {PreCondition: None                                   }
        {PosCondition: Both encode and decode tables have     }
        {              been initialized with XX data and all  }
        {              procedures are now ready to be used.   }
      procedure InitUU;     {*** UU encoding/decoding Init ***}
        {PreCondition: None                                   }
        {PosCondition: Both encode and decode tables have     }
        {              been initialized with UU data and all  }
        {              procedures are now ready to be used.   }
      procedure EnCodeAtom( A3: TxuDecodedAtom; var A4: TxuEncodedAtom );
        {PreCondition: TxuAtom have been initialized correctly  }
        {PosCondition: A4 will be the encoded version of A3     }
      procedure DeCodeAtom( A4: TxuEncodedAtom; var A3: TxuDecodedAtom );
        {PreCondition: TxuAtom have been initialized correctly  }
        {PosCondition: A3 will be the decoded version of A4     }
    PRIVATE
      EncodeTable : Array[0..63] of Char;
      DecodeTable : Array[20..127] of Byte;
    end;


  {*****************************************************************}
  {*                   TxuBuffer and successors:                   *}
  {*****************************************************************}
  {*  DESCRIPTION:                                                 *}
  {*  -They translate, with BufferEncode/BufferDecode procedures,  *}
  {*   any given Buffer between plain code and XX/UU code.         *}
  {*  -The INIT of an object is responsibility of the programmer   *}
  {*   and will NOT be tested!  Reason is performance.             *}
  {*  -The basic TxuBuffer INIT receives an encoding table in the  *}
  {*   form of a string with all encoding chars in is position     *}
  {*  -The derivate TxxBuffer and TuuBuffer Objects also MUST be   *}
  {*   Initialized but with no arguments since their tables ARE    *}
  {*   standard and included in the implementation section.        *}
  {*  -TxuBuffer have no variable internal states                  *}
  {*****************************************************************}
  {*  USAGE:                                                       *}
  {*  -Choose the needed object according with the desired method  *}
  {*   witch can bee XX, UU or User-Defined-Encoding-Table         *}
  {*  -Init the Object appropriately.                              *}
  {* !-USE TEncodedBuffer and TxuDecodedBuffer type variables to   *}
  {*   feed the BufferEncode and BufferDecode Procedures.          *}
  {* !-Used Buffers MUST be 100% Data        *** PitFall ***       *}
  {*  -Each object can be used asynchronously.                     *}
  {*  -Each object can be used in several simultaneous tasks.      *}
  {*****************************************************************}

  TxuBufferPtr = ^TxuBuffer;
  TxuBuffer = {*** USER-DEFINED-Encoding-Table ***}
    object(TxuAtom)
      procedure EnCodeBuffer( var B3: TxuDecodedBuffer; var B4: TxuEncodedBuffer; var Size: word );
        {PreCondition: TxuBuffer have been initialized correctly     }
        {PosCondition: B4[0..size] will be the encoded version of B3 }
      procedure DeCodeBuffer( var B4: TxuEncodedBuffer; var B3: TxuDecodedBuffer; var Size: word );
        {PreCondition: TxuBuffer have been initialized correctly     }
        {PosCondition: B3[0..size] will be the decoded version of B4 }
    end;


  {*****************************************************************}
  {*                   TxuFile and successors:                     *}
  {*****************************************************************}
  {*  DESCRIPTION:                                                 *}
  {*  -They translate, with FileEncode/FileDecode procedures,      *}
  {*   any given File between plain code and XX/UU code.           *}
  {*  -The INIT of an object is responsibility of the programmer   *}
  {*   and will NOT be tested!  Reason is performance.             *}
  {*  -The basic TxuFile INIT receives an encoding table in the    *}
  {*   form of a string with all encoding chars in is position     *}
  {*  -The derivate TxxFile and TuuFile Objects also MUST be       *}
  {*   Initialized but with no arguments since their tables ARE    *}
  {*   standard and included in the implementation section.        *}
  {*  -TxuFile have no variable internal states                    *}
  {*****************************************************************}
  {*  USAGE:                                                       *}
  {*  -Choose the needed object according with the desired method  *}
  {*   witch can bee XX, UU or User-Defined-Encoding-Table         *}
  {*  -Init the Object appropriately.                              *}
  {* !-USE TEncodeFile and TDecodeFile type variables to feed      *}
  {*   the FileEncode and FileDecode Procedures.                   *}
  {* !-Used Files MUST be 100% Data        *** PitFall ***         *}
  {*  -Each object can be used asynchronously.                     *}
  {*  -Each object can be used in several simultaneous tasks.      *}
  {*  -Terminate the Object appropriately with DONE procedure      *}
  {*****************************************************************}

  TxuFilePtr = ^TxuFile;
  TxuFile = {*** USER-DEFINED-Encoding-Table ***}
    object(TxuBuffer)
      procedure EnCodeFile( var F3: TxuDecodedFile; var F4: TxuEncodedFile; var Size: word );
        {PreCondition: TxuFile have been initialized correctly       }
        {PosCondition: F4[0..size] will be the encoded version of F3 }
      procedure DeCodeFile( var F4: TxuEncodedFile; var F3: TxuDecodedFile; var Size: word );
        {PreCondition: TxuFile have been initialized correctly and   }
        {              File F4 is posicioned at XU-Code begin...     }
        {PosCondition: F3[0..size] will be the decoded version of F4 }
    PRIVATE
      xuDecodedFileSize : longint;
      xuEncodedFileSize : longint;
      xuDecodedFilePos  : longint;
      xuEncodedFilePos  : longint;
    end;


  {*****************************************************************}
  {*                   TxuText and successors:                     *}
  {*****************************************************************}
  {*  DESCRIPTION:                                                 *}
  {*  -They translate, with TextEncode/TextDecode procedures,      *}
  {*   any given File between plain code and XX/UU code.           *}
  {*  -The INIT of an object is responsibility of the programmer   *}
  {*   and will NOT be tested!  Reason is performance.             *}
  {*  -The basic TxuText INIT receives an encoding table in the    *}
  {*   form of a string with all encoding chars in is position     *}
  {*  -The derivate TxxText and TuuText Objects also MUST be       *}
  {*   Initialized but with no arguments since their tables ARE    *}
  {*   standard and included in the implementation section.        *}
  {*  -TxuText have no variable internal states                    *}
  {*****************************************************************}
  {*  USAGE:                                                       *}
  {*  -Choose the needed object according with the desired method  *}
  {*   witch can bee XX, UU or User-Defined-Encoding-Table         *}
  {*  -Init the Object appropriately.                              *}
  {* !-USE TEncodeText and TDecodeText type variables to feed      *}
  {*   the TextEncode and TextDecode Procedures.                   *}
  {* !-Used Files MUST be 100% Data        *** PitFall ***         *}
  {*  -Each object can be used asynchronously.                     *}
  {*  -Each object can be used in several simultaneous tasks.      *}
  {*  -Terminate the Object appropriately with DONE procedure      *}
  {*****************************************************************}

  TxuTextPtr = ^TxuText;
  TxuText = {*** USER-DEFINED-Encoding-Table ***}
    object(TxuFile)
      procedure FindCode( var F4: TxuEncodedText );
        {PreCondition: TxuText have been initialized correctly       }
        {PosCondition: Text is posicioned at the begin of XU-Code    }
      procedure EnCodeText( var F3: TxuDecodedText; var F4: TxuEncodedText; var Size: word );
        {PreCondition: TxuText have been initialized correctly       }
        {PosCondition: F4[0..size] will be the encoded version of F3 }
      procedure DeCodeText( var F4: TxuEncodedText; var F3: TxuDecodedText; var Size: word );
        {PreCondition: TxuText have been initialized correctly and   }
        {              Text F4 is posicioned at XU-Code begin...     }
        {PosCondition: F3[0..size] will be the decoded version of F4 }
    PRIVATE
      xuDecodedTextSize : longint;
      xuEncodedTextSize : longint;
      xuDecodedTextPos  : longint;
      xuEncodedTextPos  : longint;
    end;



IMPLEMENTATION
Const
  XXstring : TxuCodeString =
  ('+-0123456789ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz');
  UUstring : TxuCodeString =
  ('`!"#$%&''()*+,-./0123456789:;<=>?@ABCDEFGHIJKLMNOPQRSTUVWXYZ[\]^_');
  (*                                                               *)
  (*        must be double to be accepted as a ' single char   *)


  {*****************************************************************}
  {*                   TxuAtom and successors:                     *}
  {*****************************************************************}
  {*   Invariant:                                                  *}
  {*     - 2 Tables for Encoding and Decoding will be used.        *}
  {*     - TxuAtom.init receive the encode table as a string of 64 *}
  {*       legal chars with unique values between #20 and #127.    *}
  {*     - The decode table is calculated from the encode table    *}
  {*       on initialization.                                      *}
  {*     - TxxAtom and TuuAtom will be automaticaly receive the XX *}
  {*       and UU table standards. And will not need any arguments *}
  {*****************************************************************}

  {*** Basic TxuAtom Object ***}

  procedure TxuAtom.Init( CodeString : TxuCodeString );
  var
    i  : byte;
    Ch : char;
  begin
    for i := 0 to 63 do
      Self.EncodeTable[i] := CodeString[i];
{$IFDEF DEBUG}
    for i := 32 to 127 do
      Self.DecodeTable[i] := $FF;   {This value should NEVER appear}
{$ENDIF}
    for i := 0 to 63 do
      Self.DecodeTable[ ord(Self.EncodeTable[i]) ] := i;
  end;

  procedure TxuAtom.InitXX;
  begin
    TxuAtom.Init( XXstring );
  end;

  procedure TxuAtom.InitUU;
  begin
    TxuAtom.Init( UUstring );
  end;

  procedure TxuAtom.EnCodeAtom( A3: TxuDecodedAtom; var A4: TxuEncodedAtom );
  begin
    A4[1] := Self.EncodeTable[ A3[1] shr 2 ];
    A4[2] := Self.EncodeTable[ ((A3[1] shl 4) OR (A3[2] shr 4)) AND $3F ];
    A4[3] := Self.EncodeTable[ ((A3[2] shl 2) OR (A3[3] shr 6)) AND $3F ];
    A4[3] := Self.EncodeTable[ A3[3] AND $3F ];
  end;

  procedure TxuAtom.DeCodeAtom( A4: TxuEncodedAtom; var A3: TxuDecodedAtom );
  begin
    A3[1] := (Self.DecodeTable[ord(A4[1])] shl 2) OR (Self.DecodeTable[ord(A4[2])] shr 4);
    A3[2] := (Self.DecodeTable[ord(A4[2])] shl 4) OR (Self.DecodeTable[ord(A4[3])] shr 2);
    A3[3] := (Self.DecodeTable[ord(A4[3])] shl 6) OR  Self.DecodeTable[ord(A4[4])];
  end;


  {********************************************************************}
  {*                   TxuBuffer and successors:                      *}
  {********************************************************************}
  {*   Invariant:                                                     *}
  {*     - 2 Tables for Encoding and Decoding will be used.           *}
  {*     - TxuBuffer.init receive the encode table as a string of 64  *}
  {*       legal chars with unique values between #20 and #127.       *}
  {*     - The decode table is calculated from the encode table       *}
  {*       on initialization.                                         *}
  {*     - TxxBuffer and TuuBuffer will be automaticaly receive the   *}
  {*       XX and UU table standards. And will not need any arguments *}
  {*     - Buffers are assumed to be 100% Data    *** PitFall ***     *}
  {********************************************************************}

  {*** Basic TxuBuffer Object ***}

  procedure TxuBuffer.EnCodeBuffer( var B3: TxuDecodedBuffer; var B4: TxuEncodedBuffer; var Size: word );
  var
    index   : word;
  begin
    index := 0;
    { Size 1 refers to Buffer[0]... So correct index is always size-1 }
    while (index <= Size) do
      begin

      self.EncodeAtom( B3[index], B4[index] );

      inc(index);
      end;
    { Size 1 refers to Buffer[0]... So correct size is always index+1 }
  end;

  procedure TxuBuffer.DeCodeBuffer( var B4: TxuEncodedBuffer; var B3: TxuDecodedBuffer; var Size: word );
  var
    index   : word;
  begin
    index := 0;
    { Size 1 refers to Buffer[0]... So correct index is always size-1 }
    while (index <= Size) do
      begin

      Self.DecodeAtom( B4[index], B3[index] );

      inc(index);
      end;
    { Size 1 refers to Buffer[0]... So correct size is always index+1 }
  end;


  {********************************************************************}
  {*                   TxuFile and successors:                        *}
  {********************************************************************}
  {*   Invariant:                                                     *}
  {*     - 2 Tables for Encoding and Decoding will be used.           *}
  {*     - TxuFile.init receive the encode table as a string of 64    *}
  {*       legal chars with unique values between #20 and #127.       *}
  {*     - The decode table is calculated from the encode table       *}
  {*       on initialization.                                         *}
  {*     - TxxFile and TuuFile will be automaticaly receive the       *}
  {*       XX and UU table standards. And will not need any arguments *}
  {*     - Files are assumed to be already ASSIGNED                   *}
  {********************************************************************}


  procedure TxuFile.EnCodeFile( var F3: TxuDecodedFile; var F4: TxuEncodedFile; var Size: word );
  var
    index   : longint;
    xuB4    : TxuEncodedBuffer;
    xuB3    : TxuDecodedBuffer;
  begin
    while (not eof(F3)) do
      begin

      Self.EncodeBuffer( xuB3, xuB4, size );

      end;
  end;

  procedure TxuFile.DeCodeFile( var F4: TxuEncodedFile; var F3: TxuDecodedFile; var Size: word );
  var
    index   : longint;
    xuB4    : TxuEncodedBuffer;
    xuB3    : TxuDecodedBuffer;
  begin
    while (not eof(F3)) do
      begin

      self.DecodeBuffer( xuB4, xuB3, size );

      end;
  end;


  {********************************************************************}
  {*                   TxuText and successors:                        *}
  {********************************************************************}
  {*   Invariant:                                                     *}
  {*     - 2 Tables for Encoding and Decoding will be used.           *}
  {*     - TxuText.init receive the encode table as a string of 64    *}
  {*       legal chars with unique values between #20 and #127.       *}
  {*     - The decode table is calculated from the encode table       *}
  {*       on initialization.                                         *}
  {*     - TxxText and TuuText will be automaticaly receive the       *}
  {*       XX and UU table standards. And will not need any arguments *}
  {*     - Files are assumed to be already ASSIGNED                   *}
  {********************************************************************}


  procedure TxuText.FindCode( var F4: TxuEncodedText );
  const
    MaxChecks = 16;
  var
    index   : longint;
    line    : string;
    xuB4    : TxuEncodedBuffer;
    xuB3    : TxuDecodedBuffer;

  function Check4Begin : Boolean;
    begin
    end;

  procedure GetParameters;
    begin
    end;

  function Check4Code  : Boolean;
    begin
    end;

  var
    LinesRead       : longint;
    LastLineEmpty   : Boolean;
    PosChecks       : Word;
    FoundBegin      : Boolean;
  begin
    LinesRead   := 0;
    LastLineEmpty   := True;
    PosChecks   := 0;
    FoundBegin      := false;
    while (not eof(F4)) and (PosChecks>=MaxChecks) do
      begin
      readln(F4, line);
      inc(LinesRead);
      if Check4begin then
        begin
        GetParameters;
        PosChecks := 0;
        end
      else
        if Check4Code then inc(PosChecks);
      end;
  end;

  procedure TxuText.EnCodeText( var F3: TxuDecodedText; var F4: TxuEncodedText; var Size: word );
  var
    index   : longint;
    xuF4    : TxuEncodedFile;
    xuF3    : TxuDecodedFile;
  begin
    while (not eof(F3)) do
      begin

      Self.EncodeFile( xuF3, xuF4, size );

      end;
  end;

  procedure TxuText.DeCodeText( var F4: TxuEncodedText; var F3: TxuDecodedText; var Size: word );
  var
    index   : longint;
    xuF4    : TxuEncodedFile;
    xuF3    : TxuDecodedFile;
  begin
    while (not eof(F3)) do
      begin

      Self.DecodeFile( xuF4, xuF3, size );

      end;
  end;


{******************************* Unit Init ********************************}

(* Initialization of the Atom *)
begin
  {Nothing}
end.
