/****************************************************************************
*
*						  Techniques Class Library
*
*                   Copyright (C) 1994 SciTech Software.
*							All rights reserved.
*
* Filename:		$RCSfile: rect.cpp $
* Version:		$Revision: 1.1 $
*
* Language:		C++ 3.0
* Environment:	any
*
* Description:	Implementation for an integer rectangle class.
*
* $Id: rect.cpp 1.1 1994/03/10 11:50:56 kjb release $
*
****************************************************************************/

#include "rect.hpp"

Rect Rect::operator + (const Rect& r) const
/****************************************************************************
*
* Function:		Rect::operator +
* Parameters:	r	- Rectangle to perform union with
* Returns:		Resultant union rectangle.
*
* Description:	Computes the union of two rectangles and returns it.
*
****************************************************************************/
{
	if (isEmpty())
		return r;
	if (r.isEmpty())
		return *this;
	return Rect(MIN(topLeft.x, r.topLeft.x), MIN(topLeft.y, r.topLeft.y),
				MAX(botRight.x, r.botRight.x), MAX(botRight.y, r.botRight.y) );
}

Rect Rect::operator + (const Point& p) const
/****************************************************************************
*
* Function:		Rect::operator +
* Parameters:	p	- Point to include into rectangle
* Returns:		Resultant union rectangle.
*
* Description:	Expands the size of a rectangle to include the specified
*				point.
*
****************************************************************************/
{
	if (isEmpty())
		return Rect(p.x,p.y,p.x+1,p.y+1);
	return Rect(MIN(topLeft.x, p.x), MIN(topLeft.y, p.y),
				MAX(botRight.x, p.x+1), MAX(botRight.y, p.y+1) );
}

Rect Rect::operator & (const Rect& r) const
/****************************************************************************
*
* Function:		Rect::operator &
* Parameters:	r	- Rectangle to perform intersection with
* Returns:		Resultant interseciont rectangle.
*
* Description:	Computes the intersection of two rectangles and returns it.
*
****************************************************************************/
{
	Rect result(MAX(topLeft.x, r.topLeft.x), MAX(topLeft.y, r.topLeft.y),
				MIN(botRight.x, r.botRight.x), MIN(botRight.y, r.botRight.y) );
	if (result.isEmpty())
		result.empty();
	return result;
}

Rect& Rect::operator += (const Rect& r)
/****************************************************************************
*
* Function:		Rect::operator +=
* Parameters:	r	- Rectangle to perform union with
*
* Description:	Computes the union of two rectangles (faster).
*
****************************************************************************/
{
	if (isEmpty())
		*this = r;
	else if (!r.isEmpty()) {
		topLeft.x = MIN(topLeft.x, r.topLeft.x);
		topLeft.y = MIN(topLeft.y, r.topLeft.y);
		botRight.x = MAX(botRight.x, r.botRight.x);
		botRight.y = MAX(botRight.y, r.botRight.y);
		}
	return *this;
}

Rect& Rect::operator += (const Point& p)
/****************************************************************************
*
* Function:		Rect::operator +=
* Parameters:	p	- Point to include in rectangle
*
* Description:	Expands the size of a rectangle to include the specified
*				point.
*
****************************************************************************/
{
	if (isEmpty()) {
		topLeft = p;
		botRight.x = p.x+1;
		botRight.y = p.y+1;
		}
	else {
		topLeft.x = MIN(topLeft.x, p.x);
		topLeft.y = MIN(topLeft.y, p.y);
		botRight.x = MAX(botRight.x, p.x+1);
		botRight.y = MAX(botRight.y, p.y+1);
		}
	return *this;
}

Rect& Rect::operator &= (const Rect& r)
/****************************************************************************
*
* Function:		Rect::operator &=
* Parameters:	r	- Rectangle to perform intersection with
*
* Description:	Computes the intersection of two rectangles.
*
****************************************************************************/
{
	topLeft.x = MAX(topLeft.x, r.topLeft.x);
	topLeft.y = MAX(topLeft.y, r.topLeft.y);
	botRight.x = MIN(botRight.x, r.botRight.x);
	botRight.y = MIN(botRight.y, r.botRight.y);
	if (isEmpty())
		empty();
	return *this;
}

bool Rect::intersect(const Rect& r) const
/****************************************************************************
*
* Function:		Rect::intersect
* Parameters:	r	- Rectangle to check against
* Returns:		True if the rectangles intersect
*
****************************************************************************/
{
	Rect result(MAX(topLeft.x, r.topLeft.x), MAX(topLeft.y, r.topLeft.y),
				MIN(botRight.x, r.botRight.x), MIN(botRight.y, r.botRight.y) );
	if (result.isEmpty())
		return false;
	return true;
}

ostream& operator << (ostream& o,const Rect& r)
/****************************************************************************
*
* Function:		operator <<
* Parameters:	o	- Stream to send rectangle to
*				r	- Rectangle to display
* Returns:		output stream used
*
****************************************************************************/
{
	return	o << r.topLeft << " - " << r.botRight;
}
