/****************************************************************************
*
*						  Techniques Class Library
*
*                   Copyright (C) 1994 SciTech Software.
*							All rights reserved.
*
* Filename:     $RCSfile: hashtab.hpp $
* Version:      $Revision: 1.1 $
*
* Language:		C++ 3.0
* Environment:	any
*
* Description:	Header file for a HashTable class.
*
* $Id: hashtab.hpp 1.1 1994/03/09 12:17:52 kjb release $
*
****************************************************************************/

#ifndef	__HASHTAB_HPP
#define	__HASHTAB_HPP

#ifndef	__DEBUG_H
#include "debug.h"
#endif

#ifndef	__IOSTREAM_H
#include <iostream.h>
#endif

/*--------------------------- Class Definition ----------------------------*/

//---------------------------------------------------------------------------
// The HashTableNode class is a simple class used to link the objects in
// a single hash table bucket together. To put anything useful into the
// table, you must derive the object place in the table from HashTableNode.
//---------------------------------------------------------------------------

class HashTableNode {
protected:
	HashTableNode	*next;
	HashTableNode	**prev;

	friend class GenHashTable;
	friend	ostream& operator << (ostream& o,GenHashTable& h);
public:
			// Constructor to satisfy some compilers :-(
			HashTableNode() {};

			// Virtual destructor to delete a hash table node
	virtual	~HashTableNode();

			// Virtual member to compute the hash value of a symbol
	virtual	uint hash() const = 0;

			// Virtual equality operator for a hash table node and a key
	virtual bool operator == (const HashTableNode& key) const = 0;

			// Virtual member to display a hash table node
	virtual void printOn(ostream& o) const = 0;

			// Method to display a hash table node
	friend	ostream& operator << (ostream& o,HashTableNode& n)
			{
				n.printOn(o);
				return o;
			};
	};

//---------------------------------------------------------------------------
// The HashTable class is a class designed to hold a number of unordered
// objects together in a hash table for fast access. In the simple form,
// HashTableNode objects contain nothing special. To add an arbitrary class
// to the table, you must derive the class from HashTableNode (either
// through single or multiple inheritance).
//---------------------------------------------------------------------------

class GenHashTable {
protected:
	uint			size;		// Size of hash table
	ulong			count;		// Number of objects in table
	HashTableNode	**table;	// Table of hash table node pointers

public:
			// Constructor
			GenHashTable(uint tabsize = 0);

			// Destructor
			~GenHashTable();

			// Member function to add a symbol to the table
			void add(HashTableNode* node);

			// Member function to remove a symbol from the table
			HashTableNode* remove(HashTableNode* node);

			// Member function to find a symbol in the table
			HashTableNode* find(HashTableNode* key) const;

			// Member function to find a symbol in the table (cached)
			HashTableNode* findCached(HashTableNode* key);

			// Member function to find the next symbol in chain
			HashTableNode* next(HashTableNode* last) const;

			// Empties the entire table by destroying all nodes.
			void empty();

			// Returns the number of items in the table
			ulong numberOfItems() const	{ return count; };

			// Returns true if the table is empty
			bool isEmpty() const		{ return count == 0; };

			// Returns the load factor for the table
			ulong loadFactor() const	{ return (size * 1000L) / count; };

			// Friend to display the contents of the hash table
	friend	ostream& operator << (ostream& o,GenHashTable& h);
	};

//---------------------------------------------------------------------------
// Template wrapper class for declaring Type Safe Hash Tables.
//---------------------------------------------------------------------------

template <class T> class HashTable : public GenHashTable {
public:
			HashTable(uint tabsize = 0)
				: GenHashTable(tabsize){};
			T* remove(T* node)
				{ return (T*)GenHashTable::remove(node); };
			T* find(T* key) const
				{ return (T*)GenHashTable::find(key); };
			T* findCached(T* key)
				{ return (T*)GenHashTable::findCached(key); };
			T* next(T* last) const
				{ return (T*)GenHashTable::next(last); };
	};

/*-------------------------- Function Prototypes --------------------------*/

// Pre-defined hash routines for C style strings

uint	hash_add(uchar *name);		// Simple and fast
uint	hash_pjw(uchar *name);		// Better distribution of symbols
uint	hash_sa(uchar *name);		// From Sedgewick's Algorithms
uint	hash_ge(uchar *name);		// From Gosling's Emac's
uchar	hash_8(uchar *name);		// Randomised 8 bit hash value
ushort	hash_16(uchar *name);		// Randomised 16 bit hash value
ulong	hash_32(uchar *name);		// Randomised 32 bit hash value

#endif	// __HASHTAB_HPP
