/****************************************************************************
*
*						MegaGraph Graphics Library
*
*                   Copyright (C) 1994 SciTech Software.
*							All rights reserved.
*
* Filename:		$RCSfile: mgldemo.c $
* Version:		$Revision: 1.3 $
*
* Language:		ANSI C
* Environment:	IBM PC (MS DOS)
*
* Description:	Demo program for the MegaGraph graphics library. This
*				program is very similar to the graphics demo program
*				that comes with Borland's range of compiler, but has
*				been extended to fully test the MGL library.
*
*				This is also useful as a comparison between the MGL and
*				Borland's BGI environment.
*
*				NOTE:	Fonts are searched for relative to the directory
*						where the driver files are found. This demo
*						expects the driver files to be in the current
*						directory or in the previous subdirectory, and all
*						fonts to be in the directory 'fonts' relative to
*						the directory where the drivers were found.
*
*						If the driver files are linked with the program
*						code, fonts are searched relative to the path
*						specified to MGL_init().
*
*				If you wish to compile the flood fill demo, define the
*				macro FLOODFILL_DEMO, but you will also need to compile
*				and link the program with the 'ffill.c' source file.
*
*
* $Id: mgldemo.c 1.3 1994/03/10 09:25:52 kjb release $
*
****************************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <stdarg.h>
#include <string.h>
#include <dos.h>
#include <ctype.h>
#include <conio.h>
#include <math.h>

#include "mgraph.h"
#include "mouse.h"
#include "getopt.h"

/*--------------------------- Macros and typedefs -------------------------*/

#define ESC		0x1B

/* Macro to swap two integer values */

#define SWAP(a,b)	{ a^=b; b^=a; a^=b; }

#define M_PI        3.14159265358979323846

/*----------------------------- Global Variables --------------------------*/

int				driver;				/* Graphics device driver type		*/
int				mode;				/* Selected graphics mode			*/
color_t			maxcolor;			/* Maximum color value				*/
int				palsize;			/* Size of video palette			*/
int				aspect;				/* Video mode aspect ratio			*/
double			aspectRatio;		/* Aspect ratio of pixels			*/
int				err;				/* Error code						*/
palette			far *pal;			/* Pointer to palette on heap		*/
text_settings	defaultTextSettings;/* Default text settings			*/
font			far *largeFont;		/* Large font for labelling			*/
int				forceDriver = grDETECT;
bool			slowpalette = false;
bool			biospalette = false;
int				snowlevel = -1;

/* Viewport rectangles used throught program	*/

rect	fullView;					/* Full screen viewport				*/
rect	titleView;					/* Title text viewport				*/
rect	statusView;					/* Status line viewport				*/
rect	demoView;					/* Viewport to draw demo stuff in	*/

char *markerStyle[] = {				/* Marker style names				*/
	"MARKER_SQUARE",
	"MARKER_CIRCLE",
	"MARKER_X"
	};

char *writeMode[] = {				/* Write mode operation names		*/
	"REPLACE_MODE",
	"AND_MODE",
	"OR_MODE",
	"XOR_MODE"
	};

char *fillStyle[] = {				/* Fill style names					*/
	"SOLID_PATTERN",
	"BITMAP_PATTERN_OPAQUE",
	"BITMAP_PATTERN_TRANSPARENT",
	"PIXMAP_PATTERN"
	};

char *polyStyle[] = {				/* Polygon style names				*/
	"CONVEX_POLYGON",
	"COMPLEX_POLYGON",
	"AUTO_POLYGON"
	};

char *textDirection[] = {			/* Text direction names				*/
	"LEFT_DIR",
	"UP_DIR",
	"RIGHT_DIR",
	"DOWN_DIR"
	};

char *horizJust[] = {				/* Horizontal justification names	*/
	"LEFT_TEXT",
	"CENTER_TEXT",
	"RIGHT_TEXT"
	};

char *vertJust[] = {				/* Vertical justification names		*/
	"TOP_TEXT",
	"CENTER_TEXT",
	"BOTTOM_TEXT"
	};

/* Names of all fonts on disk */

#define	NUMFONTS	(sizeof(fontFilename) / sizeof(fontFilename[0]))

char *fontFilename[] = {
	"astrolog.fnt",
	"cyrillc.fnt",
	"gotheng.fnt",
	"gothger.fnt",
	"gothita.fnt",
	"greekc.fnt",
	"greekcs.fnt",
	"greeks.fnt",
	"italicc.fnt",
	"italiccs.fnt",
	"italict.fnt",
	"japanese.fnt",
	"little.fnt",
	"mathbig.fnt",
	"mathsmal.fnt",
	"musical.fnt",
	"romanc.fnt",
	"romancs.fnt",
	"romand.fnt",
	"romans.fnt",
	"romant.fnt",
	"scriptc.fnt",
	"scripts.fnt",
	"symbol.fnt",
	};

/* Different mouse cursors */

cursor CHECK = {
	0xFFF0, 0xFFE0, 0xFFC0, 0xFF81,
	0xFF03, 0x0607, 0x000F, 0x001F,
	0x803F, 0xC07F, 0xE0FF, 0xF1FF,
	0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF,

	0x0000, 0x0006, 0x000C, 0x0018,
	0x0030, 0x0060, 0x70C0, 0x3980,
	0x1F00, 0x0E00, 0x0400, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000,
	0x0005, 0x000A };

cursor CROSS = {
	0xF01F, 0xE00F, 0xC007, 0x8003,
	0x0441, 0x0C61, 0x0381, 0x0381,
	0x0381, 0x0C61, 0x0441, 0x8003,
	0xC007, 0xE00F, 0xF01F, 0xFFFF,

	0x0000, 0x07C0, 0x0920, 0x1110,
	0x2108, 0x4004, 0x4004, 0x783C,
	0x4004, 0x4004, 0x2108, 0x1110,
	0x0920, 0x07C0, 0x0000, 0x0000,
	0x0007, 0x0007 };

cursor GLOVE = {
	0xF3FF, 0xE1FF, 0xE1FF, 0xE1FF,
	0xE1FF, 0xE049, 0xE000, 0x8000,
	0x0000, 0x0000, 0x07FC, 0x07F8,
	0x9FF9, 0x8FF1, 0xC003, 0xE007,

	0x0C00, 0x1200, 0x1200, 0x1200,
	0x1200, 0x13B6, 0x1249, 0x7249,
	0x9249, 0x9001, 0x9001, 0x8001,
	0x4002, 0x4002, 0x2004, 0x1FF8,
	0x0004, 0x0000 };

cursor IBEAM = {
	0xF39F, 0xFD7F, 0xFEFF, 0xFEFF,
	0xFEFF, 0xFEFF, 0xFEFF, 0xFEFF,
	0xFEFF, 0xFEFF, 0xFEFF, 0xFEFF,
	0xFEFF, 0xFEFF, 0xFD7F, 0xF39F,

	0x0C60, 0x0280, 0x0100, 0x0100,
	0x0100, 0x0100, 0x0100, 0x0100,
	0x0100, 0x0100, 0x0100, 0x0100,
	0x0100, 0x0100, 0x0280, 0x0C60,
	0x0007, 0x0008 };

/* Array of useful patterns */

#define	NUMPATS	(sizeof(bitpat) / sizeof(bitpat[0]))

pattern bitpat[] = {
	{0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00},
	{0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF},
	{0x7F, 0xFF, 0xF7, 0xFF, 0x7F, 0xFF, 0xF7, 0xFF},
	{0xDD, 0x77, 0xDD, 0x77, 0xDD, 0x77, 0xDD, 0x77},
	{0xAA, 0x55, 0xAA, 0x55, 0xAA, 0x55, 0xAA, 0x55},
	{0x55, 0xFF, 0x55, 0xFF, 0x55, 0xFF, 0x55, 0xFF},
	{0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA},
	{0xEE, 0xDD, 0xBB, 0x77, 0xEE, 0xDD, 0xBB, 0x77},
	{0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88},
	{0xB1, 0x30, 0x03, 0x1B, 0xD8, 0xC0, 0x0C, 0x8D},
	{0x80, 0x10, 0x02, 0x20, 0x01, 0x08, 0x40, 0x04},
	{0xFF, 0x88, 0x88, 0x88, 0xFF, 0x88, 0x88, 0x88},
	{0xFF, 0x80, 0x80, 0x80, 0xFF, 0x08, 0x08, 0x08},
	{0x80, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00},
	{0x80, 0x40, 0x20, 0x00, 0x02, 0x04, 0x08, 0x00},
	{0x82, 0x44, 0x39, 0x44, 0x82, 0x01, 0x01, 0x01},
	{0xF8, 0x74, 0x22, 0x47, 0x8F, 0x17, 0x22, 0x71},
	{0x55, 0xA0, 0x40, 0x40, 0x55, 0x0A, 0x04, 0x04},
	{0x20, 0x50, 0x88, 0x88, 0x88, 0x88, 0x05, 0x02},
	{0xBF, 0x00, 0xBF, 0xBF, 0xB0, 0xB0, 0xB0, 0xB0},

	{0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00},
	{0x80, 0x00, 0x08, 0x00, 0x80, 0x00, 0x08, 0x00},
	{0x88, 0x00, 0x22, 0x00, 0x88, 0x00, 0x22, 0x00},
	{0x88, 0x22, 0x88, 0x22, 0x88, 0x22, 0x88, 0x22},
	{0xAA, 0x00, 0xAA, 0x00, 0xAA, 0x00, 0xAA, 0x00},
	{0xFF, 0x00, 0xFF, 0x00, 0xFF, 0x00, 0xFF, 0x00},
	{0x11, 0x22, 0x44, 0x88, 0x11, 0x22, 0x44, 0x88},
	{0xFF, 0x00, 0x00, 0x00, 0xFF, 0x00, 0x00, 0x00},
	{0x01, 0x02, 0x04, 0x08, 0x10, 0x20, 0x40, 0x80},
	{0xAA, 0x00, 0x80, 0x00, 0x88, 0x00, 0x80, 0x00},
	{0xFF, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80},
	{0x08, 0x1C, 0x22, 0xC1, 0x80, 0x01, 0x02, 0x04},
	{0x88, 0x14, 0x22, 0x41, 0x88, 0x00, 0xAA, 0x00},
	{0x40, 0xA0, 0x00, 0x00, 0x04, 0x0A, 0x00, 0x00},
	{0x03, 0x84, 0x48, 0x30, 0x0C, 0x02, 0x01, 0x01},
	{0x80, 0x80, 0x41, 0x3E, 0x08, 0x08, 0x14, 0xE3},
	{0x10, 0x20, 0x54, 0xAA, 0xFF, 0x02, 0x04, 0x08},
	{0x77, 0x89, 0x8F, 0x8F, 0x77, 0x98, 0xF8, 0xF8},
	{0x00, 0x08, 0x14, 0x2A, 0x55, 0x2A, 0x14, 0x08},
	{0x00, 0x08, 0x14, 0x2A, 0x55, 0x2A, 0x14, 0x08},

	{0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00},
	{0x80, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00},
	{0x80, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00},
	{0x88, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00},
	{0x88, 0x00, 0x00, 0x00, 0x88, 0x00, 0x00, 0x00},
	{0x88, 0x00, 0x20, 0x00, 0x88, 0x00, 0x00, 0x00},
	{0x88, 0x00, 0x20, 0x00, 0x88, 0x00, 0x02, 0x00},
	{0x88, 0x00, 0x22, 0x00, 0x88, 0x00, 0x02, 0x00},
	{0x88, 0x00, 0x22, 0x00, 0x88, 0x00, 0x22, 0x00},
	{0xA8, 0x00, 0x22, 0x00, 0x88, 0x00, 0x22, 0x00},
	{0xA8, 0x00, 0x22, 0x00, 0x8A, 0x00, 0x22, 0x00},
	{0xAA, 0x00, 0x22, 0x00, 0x8A, 0x00, 0x22, 0x00},
	{0xAA, 0x00, 0x22, 0x00, 0xAA, 0x00, 0x22, 0x00},
	{0xAA, 0x00, 0xA2, 0x00, 0xAA, 0x00, 0x22, 0x00},
	{0xAA, 0x00, 0xA2, 0x00, 0xAA, 0x00, 0x2A, 0x00},
	{0xAA, 0x00, 0xAA, 0x00, 0xAA, 0x00, 0x2A, 0x00},
	{0xAA, 0x00, 0xAA, 0x00, 0xAA, 0x00, 0xAA, 0x00},
	{0xAA, 0x40, 0xAA, 0x00, 0xAA, 0x00, 0xAA, 0x00},
	{0xAA, 0x40, 0xAA, 0x00, 0xAA, 0x04, 0xAA, 0x00},
	{0xAA, 0x44, 0xAA, 0x00, 0xAA, 0x04, 0xAA, 0x00},
	{0xAA, 0x44, 0xAA, 0x00, 0xAA, 0x44, 0xAA, 0x00},
	{0xAA, 0x44, 0xAA, 0x10, 0xAA, 0x44, 0xAA, 0x00},
	{0xAA, 0x44, 0xAA, 0x10, 0xAA, 0x44, 0xAA, 0x01},
	{0xAA, 0x44, 0xAA, 0x11, 0xAA, 0x44, 0xAA, 0x01},
	{0xAA, 0x44, 0xAA, 0x11, 0xAA, 0x44, 0xAA, 0x11},
	{0xAA, 0x54, 0xAA, 0x11, 0xAA, 0x44, 0xAA, 0x11},
	{0xAA, 0x54, 0xAA, 0x11, 0xAA, 0x45, 0xAA, 0x11},
	{0xAA, 0x55, 0xAA, 0x11, 0xAA, 0x45, 0xAA, 0x11},
	{0xAA, 0x55, 0xAA, 0x11, 0xAA, 0x55, 0xAA, 0x11},
	{0xAA, 0x55, 0xAA, 0x51, 0xAA, 0x55, 0xAA, 0x11},
	{0xAA, 0x55, 0xAA, 0x51, 0xAA, 0x55, 0xAA, 0x15},
	{0xAA, 0x55, 0xAA, 0x55, 0xAA, 0x55, 0xAA, 0x15},
	{0xAA, 0x55, 0xAA, 0x55, 0xAA, 0x55, 0xAA, 0x55},
	{0xEA, 0x55, 0xAA, 0x55, 0xAA, 0x55, 0xAA, 0x55},
	{0xEA, 0x55, 0xAA, 0x55, 0xAE, 0x55, 0xAA, 0x55},
	{0xEE, 0x55, 0xAA, 0x55, 0xAE, 0x55, 0xAA, 0x55},
	{0xEE, 0x55, 0xAA, 0x55, 0xEE, 0x55, 0xAA, 0x55},
	{0xEE, 0x55, 0xBA, 0x55, 0xEE, 0x55, 0xAA, 0x55},
	{0xEE, 0x55, 0xBA, 0x55, 0xEE, 0x55, 0xAB, 0x55},
	{0xEE, 0x55, 0xBB, 0x55, 0xEE, 0x55, 0xAB, 0x55},
	{0xEE, 0x55, 0xBB, 0x55, 0xEE, 0x55, 0xBB, 0x55},
	{0xFE, 0x55, 0xBB, 0x55, 0xEE, 0x55, 0xBB, 0x55},
	{0xFE, 0x55, 0xBB, 0x55, 0xEF, 0x55, 0xBB, 0x55},
	{0xFF, 0x55, 0xBB, 0x55, 0xEF, 0x55, 0xBB, 0x55},
	{0xFF, 0x55, 0xBB, 0x55, 0xFF, 0x55, 0xBB, 0x55},
	{0xFF, 0x55, 0xFB, 0x55, 0xFF, 0x55, 0xBB, 0x55},
	{0xFF, 0x55, 0xFB, 0x55, 0xFF, 0x55, 0xBF, 0x55},
	{0xFF, 0x55, 0xFF, 0x55, 0xFF, 0x55, 0xBF, 0x55},
	{0xFF, 0x55, 0xFF, 0x55, 0xFF, 0x55, 0xFF, 0x55},
	{0xFF, 0xD5, 0xFF, 0x55, 0xFF, 0x55, 0xFF, 0x55},
	{0xFF, 0xD5, 0xFF, 0x55, 0xFF, 0x5D, 0xFF, 0x55},
	{0xFF, 0xDD, 0xFF, 0x55, 0xFF, 0x5D, 0xFF, 0x55},
	{0xFF, 0xDD, 0xFF, 0x55, 0xFF, 0xDD, 0xFF, 0x55},
	{0xFF, 0xDD, 0xFF, 0x75, 0xFF, 0xDD, 0xFF, 0x55},
	{0xFF, 0xDD, 0xFF, 0x75, 0xFF, 0xDD, 0xFF, 0x57},
	{0xFF, 0xDD, 0xFF, 0x77, 0xFF, 0xDD, 0xFF, 0x57},
	{0xFF, 0xDD, 0xFF, 0x77, 0xFF, 0xDD, 0xFF, 0x77},
	{0xFF, 0xFD, 0xFF, 0x77, 0xFF, 0xDD, 0xFF, 0x77},
	{0xFF, 0xFD, 0xFF, 0x77, 0xFF, 0xDF, 0xFF, 0x77},
	{0xFF, 0xFF, 0xFF, 0x77, 0xFF, 0xDF, 0xFF, 0x77},
	{0xFF, 0xFF, 0xFF, 0x77, 0xFF, 0xFF, 0xFF, 0x77},
	{0xFF, 0xFF, 0xFF, 0xF7, 0xFF, 0xFF, 0xFF, 0x77},
	{0xFF, 0xFF, 0xFF, 0xF7, 0xFF, 0xFF, 0xFF, 0x7F},
	{0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0x7F},
	{0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF}
	};

/*---------------------------- Implementation -----------------------------*/

void help(void)
/****************************************************************************
*
* Function:		help
*
* Description:	Provide usage information about the program.
*
****************************************************************************/
{
	int		i,driver,mode;
	int		*modeList;

	MGL_detectGraph(&driver,&mode);
	if (forceDriver < driver && forceDriver != grDETECT)
		driver = forceDriver;
	modeList = MGL_availableModes(driver);

	printf("Usage: mgldemo [-ega|vga|vesa] [-hfbsp] [mode]\n\n");
	printf("-ega    Force system to work as standard 256k EGA\n");
	printf("-vga    Force system to work as standard VGA\n");
	printf("-vesa   Force system to work with VESA VBE if available. By default the\n");
	printf("        system will try to use the accelerated drivers\n");
	printf("-f      Turn off special SuperVGA performance enhancements.\n");
	printf("-b      Program the palette via the BIOS (XGA and Non VGA style SuperVGA's).\n");
	printf("-s      Use slower palette routines for VLB compatability.\n");
	printf("-p<arg> Set the palette snow level factor (defaults to 100).\n");
	printf("-h      Provide this usage information.\n\n");
	printf("Valid modes to specify are:\n\n");

	i = 0;
	while(modeList[i] != -1) {
		printf("    \"%s\" - %d page\n",MGL_modeName(modeList[i]),
			MGL_availablePages(driver,modeList[i]));
		i++;
		}

	printf("\nFor example, to run the demo with EGA 320x200 16 color mode, type the\n");
	printf("following:\n\n");
	printf("    mgldemo \"320 x 200 EGA 16 color\"\n");

	exit(1);
}

void cdecl exitFunc(void)
/****************************************************************************
*
* Function:		exitFunc
*
* Description:	Exit function to ensure that the graphics system is shut
*				down correctly.
*
****************************************************************************/
{
	MGL_exit();
}

void fatalError(char *msg)
/****************************************************************************
*
* Function:		fatalError
* Parameters:	msg	- Message to display on exit
*
* Description:	This routine is called when a fatal error occurs and
*				the program must be exited gracefully.
*
****************************************************************************/
{
	MGL_exit();
	printf(msg);
	exit(EXIT_FAILURE);
}

void startGraphics(char *modeName)
/****************************************************************************
*
* Function:		startGraphics
*
* Description:	Attempts to initialise the graphics system in the
*				specified manner. Reports any intialisation errors if
*				the occur.
*
****************************************************************************/
{
	int		i,height;
	int		far *modeList;

#ifdef	USE_LINKED
	MGL_registerDriver(EGANAME,EGA_driver);
	MGL_registerDriver(VGANAME,VGA_driver);
	MGL_registerDriver(VGA256NAME,VGA256_driver);
	MGL_registerDriver(VGAXNAME,VGAX_driver);
	MGL_registerDriver(SVGA256NAME,SVGA256_driver);
	MGL_registerDriver(SVGA32KNAME,SVGA32K_driver);
	MGL_registerDriver(SVGA64KNAME,SVGA64K_driver);
	MGL_registerDriver(SVGA16MNAME,SVGA16M_driver);
	MGL_registerDriver(SVGAS3NAME,SVGAS3_driver);
#endif

	/* Detect the video subsystem type, and if determine which video mode
	 * to use.
	 */

	MGL_detectGraph(&driver,&mode);
	if (forceDriver < driver && forceDriver != grDETECT) {
		if ((driver = forceDriver) == grEGA)
			mode = grEGA_640x350x16;
		}
	if (modeName != NULL) {
		modeList = MGL_availableModes(driver);

		/* Check if the user asked for help */

		if (modeName[0] == '-' && modeName[1] == 'h')
			help();

		/* Look up the mode number given it's name */

		i = 0;
		while (true) {
			if ((mode = modeList[i++]) == -1)
				break;
			if (strcmp(MGL_modeName(mode),modeName) == 0)
				break;
			}
		if (mode == -1) {
			printf("Invalid video mode selected\n");
			exit(1);
			}
		}

	atexit(exitFunc);				/* Register exit routine			*/

	_installMouse = true;			/* Install mouse handling stuff		*/

#ifdef	USE_LINKED
	MGL_init(&driver,&mode,"");
#else
	MGL_init(&driver,&mode,"..\\");
#endif

	if ((err = MGL_result()) != grOK) {
		printf("Graphics error: %s\n",MGL_errorMsg(err));
		printf("Driver: %s, Mode: %s\n",
			MGL_driverName(driver),
			MGL_modeName(mode));
		exit(1);
		}

	/* Change default startup options from command line */

	if (snowlevel != -1) MGL_setPaletteSnowLevel(snowlevel);

	/* Load a large font for labelling */

	largeFont = MGL_loadFont("fonts\\romant.fnt");
	if ((err = MGL_result()) != grOK) {
		MGL_exit();
		printf("Graphics error: %s\n",MGL_errorMsg(err));
		exit(1);
		}

	/* Obtain a few useful constant */

	maxcolor = MGL_maxColor();
	aspect = MGL_getAspectRatio();	/* Aspect ratio for display mode	*/
	aspectRatio = 1000.0 / aspect;
	MGL_getTextSettings(&defaultTextSettings);

	/* Allocate space for the palette, and read the default palette		*/

	if ((palsize = MGL_getPaletteSize()) != -1) {
		pal = malloc(palsize * sizeof(palette));
		if (pal == NULL)
			fatalError("Out of memory!\n");
		MGL_getDefaultPalette(pal);
		}
	else pal = NULL;

	/* Determine the size of the viewports required */

	MGL_getViewport(&fullView);
	height = MGL_textHeight();		/* Height of default font			*/

	titleView = fullView;
	titleView.bottom = titleView.top + height + 5;

	statusView = fullView;
	statusView.top = statusView.bottom - height - 5;

	demoView = fullView;
	demoView.top = titleView.bottom;
	demoView.bottom = statusView.top+1;
}

int gprintf(char *fmt, ... )
/****************************************************************************
*
* Function:		gprintf
* Parameters:	fmt		- Format string
*				...		- Standard printf style parameters
* Returns:		Number of items converted successfully.
*
* Description:	Simple printf style output routine for sending text to
*				the current viewport. It begins drawing the string at
*				the current CP location, and move the CP to the start of
*				the next logical line.
*
****************************************************************************/
{
	va_list	argptr;					/* Argument list pointer			*/
	char	buf[255];				/* Buffer to build sting into		*/
	int		cnt;					/* Result of SPRINTF for return 	*/
	point	CP;

	va_start(argptr,fmt);

	cnt = vsprintf(buf,fmt,argptr);
	MGL_getCP(&CP);
	MGL_drawStr(buf);				/* Display the string				*/
	CP.y += MGL_textHeight();		/* Advance to next line				*/
	MGL_moveTo(CP);

	va_end(argptr);

	return cnt;						/* Return the conversion count		*/
}

#ifdef __MSC__

int random(int max)
{
    return ((long)rand() * (long)max) / RAND_MAX;
}

#endif

#if defined(__MSC__) || defined(__SYMC__)

void delay(int milliseconds)
{
    int i,j,k;

    for (i = 0; i < milliseconds; i++)
        for (j = 0; j < 1000; j++)
            k += i * j;
}

#endif

long randoml(long max)
/****************************************************************************
*
* Function:		randoml
* Parameters:	max	- Maximum random number value+1
* Returns:		Random long between 0 and max-1.
*
* Description:	Routine to compute random long numbers. We do this simply
*				by expanding a random integer into the correct range,
*				which means some values will never get generated :-(,
*				but hell, it works for what we want (it will also fail
*				for large numbers).
*
****************************************************************************/
{
    return ((float)rand() * max) / (float)RAND_MAX;
}

void drawBorder(void)
/****************************************************************************
*
* Function:		drawBorder
*
* Description:	Draws a border around the currently active viewport. The
*				border is drawn just inside the bounds of the viewport.
*
****************************************************************************/
{
	rect	view;

	MGL_getViewport(&view);
	MGL_setViewport(fullView);
	MGL_rect(view);
	MGL_setViewport(view);
}

void statusLine(char *msg)
/****************************************************************************
*
* Function:		statusLine
* Parameters:	msg	- Message to display in status bar
*
* Description:	Displays a message in the status bar. The status bar is
*				cleared before the message is displayed.
*
****************************************************************************/
{
	text_settings	tset;

	MS_hide();						/* Hide mouse cursor				*/

	MGL_pushViewport();
	MGL_insetRect(statusView,1,1);
	MGL_setViewport(statusView);
	MGL_clearViewport();

	MGL_getTextSettings(&tset);
	MGL_setTextSettings(&defaultTextSettings);
	MGL_setTextJustify(CENTER_TEXT,CENTER_TEXT);
	MGL_drawStrXY(MGL_maxx()/2, MGL_maxy()/2, msg);
	MGL_setTextSettings(&tset);

	MGL_insetRect(statusView,-1,-1);
	MGL_popViewport();

	MS_show();						/* Show mouse cursor				*/
}

void pause(void)
/****************************************************************************
*
* Function:		pause
*
* Description:	Pause until a key is pressed. If the key is the ESC key,
*				the exit the program.
*
****************************************************************************/
{
	static char	msg[] = "Esc aborts or press a key...";
	int			ch;

	statusLine(msg);		/* Display status message					*/

	ch = getch();

	if (ch == ESC)
		exit(1);

	if (ch == 0)			/* Swallow extended key codes 				*/
		ch = getch();
}

void mainWindow(char *heading)
/****************************************************************************
*
* Function:		mainWindow
* Parameters:	heading	- Text describing the demo
*
* Description:	Displays a heading for the demo, sets the viewport for
*				the demo code and draws a border around the viewport. We
*				also restore the default attributes.
*
****************************************************************************/
{
	text_settings	tset;
	rect			r;

	MS_hide();						/* Hide mouse cursor				*/

	MGL_defaultAttributes();		/* Restore default attributes		*/
	MGL_clearDevice();				/* Clear the graphics screen		*/

	MGL_setViewport(titleView);

	MGL_getTextSettings(&tset);
	MGL_setTextSettings(&defaultTextSettings);
	MGL_setTextJustify(CENTER_TEXT,CENTER_TEXT);
	MGL_drawStrXY(MGL_maxx()/2, MGL_maxy()/2, heading);
	MGL_setTextSettings(&tset);

	MGL_setViewport(statusView);
	drawBorder();
	MGL_setViewport(demoView);
	drawBorder();

	r = demoView;
	MGL_insetRect(r,1,1);
	MGL_setViewport(r);

	MS_show();						/* Show mouse cursor				*/
}

void reportStatus(void)
/****************************************************************************
*
* Function:		reportStatus
*
* Description:	Report the current graphics system configuration after
*				graphics has been initialised.
*
****************************************************************************/
{
	attributes		attr;
	text_settings	tset;

	mainWindow("Status report after MGL_init");

	MGL_moveToCoord(10,4);

	/* Get required information */

	MGL_getAttributes(&attr);

	gprintf("Graphics device    : (%d) %s", driver, MGL_driverName(driver));
	gprintf("Graphics mode      : (%d) %s", mode, MGL_modeName(mode));
	gprintf("Screen resolution  : ( 0, 0, %d, %d )",MGL_sizex(),MGL_sizey());
	gprintf("View port          : ( %d, %d, %d, %d )",
		attr.viewport.left,attr.viewport.top,
		attr.viewport.right,attr.viewport.bottom);
	gprintf("Clipping rectangle : ( %d, %d, %d, %d )",
		attr.clipRect.left,attr.clipRect.top,
		attr.clipRect.right,attr.clipRect.bottom);
	gprintf("Clipping           : %s", attr.clip ? "ON" : "OFF");
	gprintf("Current position   : ( %d, %d )", MGL_getX(),MGL_getY());
	gprintf("Colors available   : %ld", maxcolor+1);
	gprintf("Color              : %ld", attr.color);
	gprintf("Background color   : %ld", attr.backColor);
	gprintf("Border dark color  : %ld", attr.bdr_dark);
	gprintf("Border bright color: %ld", attr.bdr_bright);
	gprintf("Marker color       : %ld", attr.markerColor);
	gprintf("Marker size        : %d", attr.markerSize);
	gprintf("Marker style       : %s", markerStyle[attr.markerStyle]);
	gprintf("Write mode         : %s", writeMode[attr.writeMode]);
	gprintf("Pen height         : %d", attr.penHeight);
	gprintf("Pen width          : %d", attr.penWidth);
	gprintf("Fill style         : %s", fillStyle[attr.penStyle]);
	gprintf("Polygon style      : %s", polyStyle[attr.poly_type]);
	gprintf("Maximum Page Index : %d", MGL_maxPage());

	tset = attr.tsettings;		/* Make an alias */

	gprintf("");
	gprintf("Text settings");
	gprintf("");

	gprintf("Current font       : %s", tset.fnt->name);
	gprintf("Text direction     : %s", textDirection[tset.dir]);
	gprintf("Horizontal justify : %s", horizJust[tset.horiz_just]);
	gprintf("Vertical justify   : %s", vertJust[tset.vert_just]);
	gprintf("Horizontal size    : ( %d / %d )", tset.sz_numerx,tset.sz_denomx);
	gprintf("Vertical size      : ( %d / %d )", tset.sz_numery,tset.sz_denomy);
	gprintf("Space extra        : %d", tset.space_extra);

	pause();
}

void colorDemo(void)
/****************************************************************************
*
* Function:		colorDemo
*
* Description:	Display the current color palette on the screen.
*
****************************************************************************/
{
	int		color,width,height,x,y,i,j,top,bottom,start;
	palette	pal[256],temp[256];

	mainWindow("Color Demonstration");

	if (maxcolor <= 15) {
		/* Simple color demonstration for 16 color displays */

		width = 2 * ((MGL_maxx()+1) / 16);
		height = 2 * ((MGL_maxy()-10)  / 10);

		x = width / 2;
		y = height / 2;
		color = 1;
		for (j = 0; j < 3; j++) {
			for (i = 0; i < 5; i++) {
				MGL_setColor(color++);
				MGL_fillRectCoord(x,y,x+width,y+height);
				x += (width/2) * 3;
				}
			y += (height / 2) * 3;
			x = width / 2;
			}
		}
	else if (maxcolor == 255) {
		/* Color demonstration for large numbers of colors */

		width = 2 * ((MGL_maxx()+1) / 46);
		height = 2 * ((MGL_maxy()-10)  / 47);

		x = width / 2;
		y = height / 2;
		color = 1;
		for (j = 0; j < 16; j++) {
			for (i = 0; i < 16; i++) {
				MGL_setColor(color++);
				MGL_fillRectCoord(x,y,x+width,y+height);
				x += (width/2) * 3;
				}
			y += (height / 2) * 3;
			x = width / 2;
			}

		MGL_setColor(MGL_defaultColor());
		statusLine("Press a key to rotate palette");
		getch();
		statusLine("Press a key to rotate other direction");

		// Palette rotations

		MGL_getPalette(pal,palsize,0);
		while (!kbhit()) {
			MGL_rotatePalette(pal+1,palsize-1,PAL_ROTATE_UP);
			MGL_setPalette(pal+1,palsize-1,1);
			}
		getch();
		statusLine("Press a key to fade palette");

		while (!kbhit()) {
			MGL_rotatePalette(pal+1,palsize-1,PAL_ROTATE_DOWN);
			MGL_setPalette(pal+1,palsize-1,1);
			}
		getch();

		// Palette fade out

		temp[0] = pal[0];
		for (i = 31; i >= 0; i--) {
			MGL_fadePalette(temp+1,pal+1,palsize-1,(uchar)(i*8));
			MGL_setPalette(temp+1,palsize-1,1);
			}

		// Palette fade in
		for (i = 0; i <= 31; i++) {
			MGL_fadePalette(temp+1,pal+1,palsize-1,(uchar)(i*8));
			MGL_setPalette(temp+1,palsize-1,1);
			}
		}
	else {
		/* Color demonstration for HiColor and TrueColor modes */

		start = MGL_maxx()/8;
		width = MGL_maxx() - start*2;
		MGL_beginDrawing();

		top = MGL_maxy()/8;
		bottom = MGL_maxy()/8 + MGL_maxy()/5;
		for (x = 0; x < width; x++) {
			MGL_setColor(MGL_packColor((x * 255L) / width,0,0));
			MGL_lineCoordFast(start+x,top,start+x,bottom);
			}

		top += MGL_maxy()/5;
		bottom += MGL_maxy()/5;
		for (x = 0; x < width; x++) {
			MGL_setColor(MGL_packColor(0,(x * 255L) / width,0));
			MGL_lineCoordFast(start+x,top,start+x,bottom);
			}

		top += MGL_maxy()/5;
		bottom += MGL_maxy()/5;
		for (x = 0; x < width; x++) {
			MGL_setColor(MGL_packColor(0,0,(x * 255L) / width));
			MGL_lineCoordFast(start+x,top,start+x,bottom);
			}

		top += MGL_maxy()/5;
		bottom += MGL_maxy()/5;
		for (x = 0; x < width/2; x++) {
			MGL_setColor(MGL_packColor((((width/2)-x) * 255L) / (width/2),
				(x * 255L) / (width/2),0));
			MGL_lineCoordFast(start+x,top,start+x,bottom);
			}
		for (; x < width; x++) {
			MGL_setColor(MGL_packColor(0,
				( ((width/2) - (x - (width/2))) * 255L) / (width/2),
				((x-width/2) * 255L) / (width/2) ));
			MGL_lineCoordFast(start+x,top,start+x,bottom);
			}
		MGL_endDrawing();
		}

	MGL_setColor(MGL_defaultColor());
	pause();
	MGL_setDefaultPalette();
}

void paletteDemo(void)
/****************************************************************************
*
* Function:		paletteDemo
*
* Description:	Change the palette of the screen until the user has had
*				enough.
*
****************************************************************************/
{
	int 	i,j,x,y,color,height,width;

	if (maxcolor > 255)
		return;

	mainWindow("Palette Demonstration");
	statusLine("Press any key to continue, ESC to Abort");

	width  = (MGL_maxx()+1) / 15;		/* Compute width of boxes		*/
	height = (MGL_maxy()+1) / 10;		/* Compute height of boxes		*/

	x = y = 0;							/* Start in upper corner		*/
	color = 1;							/* Begin width 1st color		*/

	for (j = 0; j < 10; j++) {			/* For 10 rows of boxes 		*/
		for (i = 0; i < 15; i++) {		/* For 15 columns of boxes		*/
			MGL_setColor(color++);
			MGL_fillRectCoord(x,y,x+width+1,y+height+1);
			x += width + 1;				/* Advance to next col			*/

			/* Choose color out of order */

			color = 1 + (color % 14);
			}
		x = 0;
		y += height + 1;
		}

	while (!kbhit()) {					/* Until user hits a key...		*/
		MGL_setPaletteEntry(1+random(14),
			(uchar)random(PALMAX+1),
			(uchar)random(PALMAX+1),
			(uchar)random(PALMAX+1));
		}

	MGL_setColor(MGL_defaultColor());
	MGL_setDefaultPalette();
	pause();
}

void pixelDemo(void)
/****************************************************************************
*
* Function:		pixelDemo
*
* Description:	Display a pattern of random dots on the screen and
*				pick them back up again.
*
****************************************************************************/
{
	int		seed = 1958;
	int		i,x,y,maxx,maxy;
	color_t	color;

	mainWindow("MGL_pixel / MGL_getPixel Demonstration");

	maxx = MGL_maxx();
	maxy = MGL_maxy();

	srand(seed);			/* Seed random number gen with known value	*/

	MGL_beginPixel();
	for (i = 0; i < 10000; i++) {		/* Put 5000 pixels on screen	*/
		x = 1 + random(maxx);			/* Generate a random location	*/
		y = 1 + random(maxy);
		MGL_setColor(1 + randoml(maxcolor));
		MGL_pixelCoord(x,y);
		}

	srand(seed);			/* Reseed random number gen with same value	*/

	MGL_setColor(MGL_realColor(BLACK));			/* Draw over in black 			*/

	for (i = 0; i < 10000; i++) {		/* Take the 5000 pixels off		*/
		x = 1 + random(maxx);			/* Generate a random location	*/
		y = 1 + random(maxy);
		color = MGL_getPixelCoord(x,y);
		if (color == 1 + randoml(maxcolor))	/* Keep random in sync		*/
			MGL_pixelCoord(x,y);
		}
	MGL_endPixel();

	MGL_setColor(MGL_defaultColor());
	pause();
}

#define	RADIUS		20					/* Dimensions of saucer			*/
#define	STARTX		100
#define	STARTY		50
#define	PAUSETIME	60

void imageDemo(void)
/****************************************************************************
*
* Function:		imageDemo
*
* Description:	Moves the image of a saucer randomly around the screen,
*				using very simple XOR animation.
*
****************************************************************************/
{
	rect	extent;
	void	far *saucer;
	int		i,x,y,width,height,maxx,maxy,step;
	size_t	size;

	mainWindow("MGL_getImage / MGL_putImage Demonstration");
	statusLine("Press any key to continue, ESC to Abort");

	maxx = MGL_maxx();
	maxy = MGL_maxy();

	/* Draw the flying saucer */

	extent.left = STARTX - RADIUS;
	extent.top = STARTY - ((RADIUS/3)+2);
	extent.right = STARTX + RADIUS + 1;
	extent.bottom = STARTY + ((RADIUS/3)+2) + 1;

	MGL_fillEllipse(extent);

	MGL_lineCoord(STARTX+7, STARTY-6, STARTX+10, STARTY-12);
	MGL_lineCoord(STARTX-7, STARTY-6, STARTX-10, STARTY-12);

	extent.left = STARTX-10 - 2;
	extent.top = STARTY-12 - 2;
	extent.right = STARTX-10 + 2 + 1;
	extent.bottom = STARTY-12 + 2 + 1;

	MGL_ellipse(extent);
	MGL_offsetRect(extent,20,0);
	MGL_ellipse(extent);

	/* Read the image of the saucer */

	extent.left = STARTX-(RADIUS+1);
	extent.top = STARTY-14;
	extent.right = STARTX+(RADIUS+1) + 1;
	extent.bottom = STARTY+(RADIUS/3)+3 + 1;
	width = extent.right - extent.left;
	height = extent.bottom - extent.top;

	size = MGL_imageSize(extent);
	saucer = malloc(size);

	if (saucer == NULL)
		fatalError("Out of memory!\n");

	MGL_getImage(extent,saucer);
	MGL_putImage(extent,saucer,XOR_MODE);

	/* Plot some "stars" for the background */

	MGL_beginPixel();
	for (i = 0; i < 1000; i++) {
		MGL_setColor(randoml(maxcolor)+1);
		MGL_pixelCoord(random(maxx),random(maxy));
		}
	MGL_endPixel();

	/* Start in the middle of the viewport */

	x = maxx/2;		y = maxy/2;

	/* Repeat moving the saucer until a key is pressed */

	while (!kbhit()) {
		/* Draw the Saucer, pause for a while and erase it */

		MGL_putImageCoord(x,y,x+width+1,y+height+1,saucer,XOR_MODE);
		delay(PAUSETIME);
		MGL_putImageCoord(x,y,x+width+1,y+height+1,saucer,XOR_MODE);

		/* Move Saucer to a new location */

		step = random(2*RADIUS);
		if ((step/2) % 2 == 0)
			x += step;
		else
			x -= step;

		step = random(RADIUS);
		if ((step/2) % 2 == 0)
			y += step;
		else
			y -= step;

		/* Ensure that is has not moved off the edge of the viewport */

		if (x + width-1 > maxx)
			x = maxx - width+1;
		else if (x < 0)
			x = 0;

		if (y + height-1 > maxy)
			y = maxy - height+1;
		else if (y < 0)
			y = 0;
		}
	free((void*)saucer);

	MGL_setColor(MGL_defaultColor());
	pause();
}

void rectangleDemo(void)
/****************************************************************************
*
* Function:		rectangleDemo
*
* Description:	Display a number of random rectangles with random
*				attributes.
*
****************************************************************************/
{
	int		maxx,maxy,val;
	rect	r;

	mainWindow("Random rectangles");
	statusLine("Press any key to continue, ESC to Abort");

	maxx = MGL_maxx();
	maxy = MGL_maxy();

	while (!kbhit()) {
		r.left = random(maxx);
		r.right = random(maxx);
		r.top = random(maxy);
		r.bottom = random(maxy);

		/* Fix the rectangle so it is not empty */

		if (r.right < r.left)
			SWAP(r.left,r.right);
		if (r.bottom < r.top)
			SWAP(r.top,r.bottom);

		MGL_setColor(randoml(maxcolor)+1);
		MGL_setBackColor(randoml(maxcolor)+1);

		if ((val = random(3)) == 0) {
			MGL_setPenStyle(BITMAP_PATTERN_TRANSPARENT);
			MGL_setPenBitmapPattern(&bitpat[random(NUMPATS)+1]);
			}
		else if (val == 1) {
			MGL_setPenStyle(BITMAP_PATTERN_OPAQUE);
			MGL_setPenBitmapPattern(&bitpat[random(NUMPATS)+1]);
			}
		else {
			MGL_setPenStyle(SOLID_PATTERN);
			}

		MGL_fillRect(r);
		}

	MGL_defaultAttributes();
	pause();
}

void circleDemo(void)
/****************************************************************************
*
* Function:		circleDemo
*
* Description:	Display a random pattern of circles on the screen
*				until the user hits a key.
*
****************************************************************************/
{
	int 	maxx,maxy,val;
	rect	r;

	mainWindow("Circle Demonstration");
	statusLine("Press any key to continue, ESC to Abort");

	maxx = MGL_maxx();
	maxy = MGL_maxy();

	while (!kbhit()) {
		r.left = random(maxx-100);
		r.top = random(maxy-100);
		r.right = r.left + random(100);
		r.bottom = r.top + random(100);

		MGL_setColor(randoml(maxcolor)+1);
		MGL_setBackColor(randoml(maxcolor)+1);

		MGL_setPenSize(random(5)+1,random(5)+1);

		if ((val = random(10)) == 0) {
			MGL_setPenStyle(BITMAP_PATTERN_TRANSPARENT);
			MGL_setPenBitmapPattern(&bitpat[random(NUMPATS)+1]);
			}
		else if (val == 1) {
			MGL_setPenStyle(BITMAP_PATTERN_OPAQUE);
			MGL_setPenBitmapPattern(&bitpat[random(NUMPATS)+1]);
			}
		else {
			MGL_setPenStyle(SOLID_PATTERN);
			}

		MGL_ellipse(r);
		}

	MGL_defaultAttributes();
	pause();
}

void filledCircleDemo(void)
/****************************************************************************
*
* Function:		filledCircleDemo
*
* Description:	Display a random pattern of filled circles on the screen
*				until the user hits a key.
*
****************************************************************************/
{
	int 	maxx,maxy,val;
	rect	r;

	mainWindow("Filled circle Demonstration");
	statusLine("Press any key to continue, ESC to Abort");

	maxx = MGL_maxx();
	maxy = MGL_maxy();

	while (!kbhit()) {
		r.left = random(maxx-100);
		r.top = random(maxy-100);
		r.right = r.left + random(100);
		r.bottom = r.top + random(100);

		MGL_setColor(randoml(maxcolor)+1);
		MGL_setBackColor(randoml(maxcolor)+1);

		if ((val = random(5)) == 0) {
			MGL_setPenStyle(BITMAP_PATTERN_TRANSPARENT);
			MGL_setPenBitmapPattern(&bitpat[random(NUMPATS)+1]);
			}
		else if (val == 1) {
			MGL_setPenStyle(BITMAP_PATTERN_OPAQUE);
			MGL_setPenBitmapPattern(&bitpat[random(NUMPATS)+1]);
			}
		else {
			MGL_setPenStyle(SOLID_PATTERN);
			}

		MGL_fillEllipse(r);
		}

	MGL_defaultAttributes();
	pause();
}

void arcDemo(void)
/****************************************************************************
*
* Function:		arcDemo
*
* Description:	Display a random pattern of elliptical arc's on the screen
*				until the user hits a key.
*
****************************************************************************/
{
	int 	maxx,maxy,startAngle,endAngle;
	rect	r;

	mainWindow("Arc Demonstration");
	statusLine("Press any key to continue, ESC to Abort");

	maxx = MGL_maxx();
	maxy = MGL_maxy();

	while (!kbhit()) {
		r.left = random(maxx-100);
		r.top = random(maxy-100);
		r.right = r.left + random(100);
		r.bottom = r.top + random(100);
		startAngle = random(360);
		endAngle = random(360);

		MGL_setColor(randoml(maxcolor)+1);
		MGL_ellipseArc(r,startAngle,endAngle);
		}

	MGL_defaultAttributes();
	pause();
}

void filledArcDemo(void)
/****************************************************************************
*
* Function:		filledArcDemo
*
* Description:	Display a random pattern of filled arcs on the screen
*				until the user hits a key.
*
****************************************************************************/
{
	int 	maxx,maxy,startAngle,endAngle,val;
	rect	r;

	mainWindow("Filled Arc Demonstration");
	statusLine("Press any key to continue, ESC to Abort");

	maxx = MGL_maxx();
	maxy = MGL_maxy();

	while (!kbhit()) {
		r.left = random(maxx-100);
		r.top = random(maxy-100);
		r.right = r.left + random(100);
		r.bottom = r.top + random(100);
		startAngle = random(360);
		endAngle = random(360);

		MGL_setColor(randoml(maxcolor)+1);
		MGL_setBackColor(randoml(maxcolor)+1);

		if ((val = random(5)) == 0) {
			MGL_setPenStyle(BITMAP_PATTERN_TRANSPARENT);
			MGL_setPenBitmapPattern(&bitpat[random(NUMPATS)+1]);
			}
		else if (val == 1) {
			MGL_setPenStyle(BITMAP_PATTERN_OPAQUE);
			MGL_setPenBitmapPattern(&bitpat[random(NUMPATS)+1]);
			}
		else {
			MGL_setPenStyle(SOLID_PATTERN);
			}

		MGL_fillEllipseArc(r,startAngle,endAngle);
		}

	MGL_defaultAttributes();
	pause();
}

#define adjasp(y)	((int)(aspectRatio * (double)(y)))
#define torad(d)	(( (double)(d) * M_PI ) / 180.0 )

void pieDemo(void)
/****************************************************************************
*
* Function:		pieDemo
*
* Description:	Draws a simple pie chart on the screen.
*
****************************************************************************/
{
	int			xcenter,ycenter,radius,lradius;
	int 		x,y;
	double		radians,piesize;
	arc_coords	coords;

	mainWindow("Pie Chart Demonstration");

	xcenter = MGL_maxx()/2;				/* Center the Pie horizontally	*/
	ycenter = MGL_maxy()/2 + 20;		/* Center the Pie vertically	*/
	radius  = MGL_maxx()/5;				/* It will cover 2/5ths screen	*/
	piesize = MGL_maxy() / 4.0;			/* Optimum height ratio of pie	*/

	while ((aspectRatio*radius) < piesize)
		radius++;

	lradius = radius + (radius/5);		/* Labels placed 20% farther	*/

	MGL_useFont(largeFont);
	MGL_setTextJustify(CENTER_TEXT,TOP_TEXT);
	MGL_drawStrXY(MGL_maxx()/2, 6, "This is a Pie Chart");

	MGL_setTextSize(2,3,2,3);

	MGL_setColor(MGL_realColor(RED));
	MGL_fillEllipseArcCoord(xcenter+10,ycenter-adjasp(10),
		radius,adjasp(radius),0,90);
	MGL_setColor(MGL_realColor(WHITE));
	MGL_setPenStyle(SOLID_PATTERN);
	MGL_ellipseArcCoord(xcenter+10,ycenter-adjasp(10),
		radius,adjasp(radius),0,90);
	MGL_getArcCoords(&coords);
	MGL_lineCoord(coords.x,coords.y,coords.startX,coords.startY);
	MGL_lineCoord(coords.x,coords.y,coords.endX,coords.endY);
	radians = torad(45);
	x = xcenter + (int)(cos(radians) * (double)lradius );
	y = ycenter - (int)(sin(radians) * (double)lradius * aspectRatio );
	MGL_setTextJustify(LEFT_TEXT,BOTTOM_TEXT);
	MGL_drawStrXY(x,y,"25 %");

	MGL_setColor(MGL_realColor(GREEN));
	MGL_setPenStyle(BITMAP_PATTERN_OPAQUE);
	MGL_setPenBitmapPattern(&bitpat[3]);
	MGL_fillEllipseArcCoord(xcenter,ycenter,radius,adjasp(radius),90,135);
	MGL_setColor(MGL_realColor(WHITE));
	MGL_setPenStyle(SOLID_PATTERN);
	MGL_ellipseArcCoord(xcenter,ycenter,radius,adjasp(radius),90,135);
	MGL_getArcCoords(&coords);
	MGL_lineCoord(coords.x,coords.y,coords.startX,coords.startY);
	MGL_lineCoord(coords.x,coords.y,coords.endX,coords.endY);
	radians = torad(113);
	x = xcenter + (int)(cos(radians) * (double)lradius );
	y = ycenter - (int)(sin(radians) * (double)lradius * aspectRatio );
	MGL_setTextJustify(RIGHT_TEXT,BOTTOM_TEXT);
	MGL_drawStrXY(x,y,"12.5 %");

	MGL_setColor(MGL_realColor(YELLOW));
	MGL_setPenStyle(BITMAP_PATTERN_OPAQUE);
	MGL_setPenBitmapPattern(&bitpat[7]);
	MGL_fillEllipseArcCoord(xcenter-10,ycenter,radius,adjasp(radius),135,225);
	MGL_setColor(MGL_realColor(WHITE));
	MGL_setPenStyle(SOLID_PATTERN);
	MGL_ellipseArcCoord(xcenter-10,ycenter,radius,adjasp(radius),135,225);
	MGL_getArcCoords(&coords);
	MGL_lineCoord(coords.x,coords.y,coords.startX,coords.startY);
	MGL_lineCoord(coords.x,coords.y,coords.endX,coords.endY);
	radians = torad(180);
	x = xcenter + (int)(cos(radians) * (double)lradius );
	y = ycenter - (int)(sin(radians) * (double)lradius * aspectRatio );
	MGL_setTextJustify(RIGHT_TEXT,CENTER_TEXT);
	MGL_drawStrXY(x,y,"25 %");

	MGL_setColor(MGL_realColor(LIGHTBLUE));
	MGL_setPenStyle(BITMAP_PATTERN_OPAQUE);
	MGL_setPenBitmapPattern(&bitpat[10]);
	MGL_fillEllipseArcCoord(xcenter,ycenter,radius,adjasp(radius),225,360);
	MGL_setColor(MGL_realColor(WHITE));
	MGL_setPenStyle(SOLID_PATTERN);
	MGL_ellipseArcCoord(xcenter,ycenter,radius,adjasp(radius),225,360);
	MGL_getArcCoords(&coords);
	MGL_lineCoord(coords.x,coords.y,coords.startX,coords.startY);
	MGL_lineCoord(coords.x,coords.y,coords.endX,coords.endY);
	radians = torad(293);
	x = xcenter + (int)(cos(radians) * (double)lradius );
	y = ycenter - (int)(sin(radians) * (double)lradius * aspectRatio );
	MGL_setTextJustify(LEFT_TEXT,TOP_TEXT);
	MGL_drawStrXY(x,y,"37.5 %");

	MGL_defaultAttributes();
	pause();
}

void lineRelDemo(void)
/****************************************************************************
*
* Function:		lineRelDemo
*
* Description:	Display a pattern using relative move and line commands.
*
****************************************************************************/
{
	int 	h,w,dx,dy,cx,cy;
	point	outs[7];

	mainWindow("MGL_moveRel / MGL_lineRel Demonstration");

	cx = MGL_maxx() / 2;				/* Center of the viewport		*/
	cy = MGL_maxy() / 2;
	h  = MGL_maxy() / 8;
	w  = MGL_maxx() / 9;
	dx = 2 * w;
	dy = 2 * h;

	MGL_setBackColor(MGL_realColor(BLUE));
	MGL_clearViewport();

	outs[0].x = cx -  dx;
	outs[0].y = cy -  dy;
	outs[1].x = cx - (dx-w);
	outs[1].y = cy - (dy+h);
	outs[2].x = cx +  dx;
	outs[2].y = cy - (dy+h);
	outs[3].x = cx +  dx;
	outs[3].y = cy +  dy;
	outs[4].x = cx + (dx-w);
	outs[4].y = cy + (dy+h);
	outs[5].x = cx -  dx;
	outs[5].y = cy + (dy+h);
	outs[6].x = cx -  dx;
	outs[6].y = cy -  dy;

	MGL_setColor(MGL_realColor(WHITE));
	MGL_fillPolygon(7,outs,0,0);

	outs[0].x = cx - (w/2);
	outs[0].y = cy + h;
	outs[1].x = cx + (w/2);
	outs[1].y = cy + h;
	outs[2].x = cx + (w/2);
	outs[2].y = cy - h;
	outs[3].x = cx - (w/2);
	outs[3].y = cy - h;
	outs[4].x = cx - (w/2);
	outs[4].y = cy + h;

	MGL_setColor(MGL_realColor(BLUE));
	MGL_fillPolygon(5,outs,0,0);

	/*	Draw a Tesseract object on the screen using the MGL_lineRel and	*/
	/*	MGL_moveRel drawing commands.									*/

	MGL_moveToCoord(cx-dx,cy-dy);
	MGL_lineRelCoord(w,-h);
	MGL_lineRelCoord(3*w,0);
	MGL_lineRelCoord(0, 5*h);
	MGL_lineRelCoord(-w,h);
	MGL_lineRelCoord(-3*w,0);
	MGL_lineRelCoord(0,-5*h);

	MGL_moveRelCoord(w,-h);
	MGL_lineRelCoord(0,5*h);
	MGL_lineRelCoord(w+(w/2),0);
	MGL_lineRelCoord(0,-3*h);
	MGL_lineRelCoord(w/2,-h);
	MGL_lineRelCoord(0,5*h);

	MGL_moveRelCoord(0,-5*h);
	MGL_lineRelCoord(-(w+(w/2)),0);
	MGL_lineRelCoord(0,3*h);
	MGL_lineRelCoord(-w/2,h);

	MGL_moveRelCoord(w/2,-h);
	MGL_lineRelCoord(w,0);

	MGL_moveRelCoord(0,-2*h);
	MGL_lineRelCoord(-w,0);

	MGL_defaultAttributes();
	pause();
}

#define MAXPTS	15

void lineToDemo(void)
/****************************************************************************
*
* Function:		lineToDemo
*
* Description:	Display a pattern using MGL_lineTo and MGL_moveTo commands.
*
****************************************************************************/
{
	point	points[MAXPTS];
	int 	i,j,h,w,xcenter,ycenter;
	int 	radius,angle,step;
	double  rads;

	mainWindow("MGL_moveTo / MGL_lineTo Demonstration");

	w = MGL_maxx();
	h = MGL_maxy();

	xcenter = w / 2;				/* Determine the center of circle	*/
	ycenter = h / 2;
	radius  = (h - 30) / (aspectRatio * 2);
	step	= 360 / MAXPTS;			/* Determine # of increments		*/

	angle = 0;
	for(i = 0; i < MAXPTS; i++) {		/* Determine circle intercepts	*/
		rads = (double)angle * M_PI / 180.0;/* Convert angle to radians	*/
		points[i].x = xcenter + (int)(cos(rads) * radius );
		points[i].y = ycenter - (int)(sin(rads) * radius * aspectRatio);
		angle += step;					/* Move to next increment		*/
		}

	/* Draw bounding circle */

	MGL_ellipseCoord(xcenter,ycenter,radius,(int)(radius*aspectRatio));

	for (i = 0; i < MAXPTS; i++) {		/* Draw the chords to the circle */
		for (j = i; j < MAXPTS; j++) {	/* For each remaining intersect */
			MGL_moveTo(points[i]);		/* Move to beginning of chord	*/
			MGL_lineTo(points[j]); 		/* Draw the chord				*/
			}
		}

	pause();
}

void lineDemo(void)
/****************************************************************************
*
* Function:		lineDemo
*
* Description:	Draw a whole bunch of random lines with random pen sizes
*				and pattern modes.
*
****************************************************************************/
{
	int		maxx,maxy,val;
	point	p1,p2;

	mainWindow("Line Demonstration");
	statusLine("Press any key to continue, ESC to Abort");

	maxx = MGL_maxx();
	maxy = MGL_maxy();

	MGL_beginDrawing();

	while (!kbhit()) {
		p1.x = random(maxx);
		p1.y = random(maxy);
		p2.x = random(maxx);
		p2.y = random(maxy);

		MGL_setColor(randoml(maxcolor)+1);
		MGL_setBackColor(randoml(maxcolor)+1);
		MGL_setPenSize(random(5)+1,random(5)+1);

		if ((val = random(5)) == 0) {
			MGL_setPenStyle(BITMAP_PATTERN_TRANSPARENT);
			MGL_setPenBitmapPattern(&bitpat[random(NUMPATS)+1]);
			}
		else if (val == 1) {
			MGL_setPenStyle(BITMAP_PATTERN_OPAQUE);
			MGL_setPenBitmapPattern(&bitpat[random(NUMPATS)+1]);
			}
		else {
			MGL_setPenStyle(SOLID_PATTERN);
			}

		MGL_lineFast(p1,p2);
		}

	MGL_endDrawing();

	MGL_defaultAttributes();
	pause();
}

void textDump(void)
/****************************************************************************
*
* Function:		textDump
*
* Description:	Displays all the characters in each of the fonts in
*				the distribution.
*
****************************************************************************/
{
	int		i,j,maxx;
	font	far *font;
	char	buf[80];

	maxx = MGL_maxx();

	for (i = 0; i < NUMFONTS; i++) {	/* For each available font		*/
		/* Attempt to load the font file from disk. If it cannot be
		 * found, then go onto the next one
		 */

		strcpy(buf,"fonts\\");
		strcat(buf,fontFilename[i]);
		font = MGL_loadFont(buf);
		err = MGL_result();
		if (err == grFontNotFound)
			continue;

		if ((err = MGL_result()) != grOK) {
			MGL_exit();
			printf("Graphics error: %s\n",MGL_errorMsg(err));
			exit(1);
			}

		sprintf(buf,"Character set (%s)", font->name);
		mainWindow(buf);

		/* Display all characters on the screen */

		MGL_useFont(font);
		buf[1] = '\0';
		for (j = ' '; j < 256; j++) {
			buf[0] = j;
			MGL_drawStr(buf);
			if (MGL_getX() + MGL_maxCharWidth() > maxx) {
				MGL_moveToCoord(0,MGL_getY() + MGL_textHeight());
				}
			}

		/* Unload the font from memory */

		MGL_unloadFont(font);
		MGL_useFont(DEFAULT_FONT);
		pause();
		}
	MGL_defaultAttributes();
}

void textStyleDemo(void)
/****************************************************************************
*
* Function:		textDemo
*
* Description:	Show the large font in several different sizes and
*				orientations.
*
****************************************************************************/
{
	int		i,x,y;
	char	buf[80];

	mainWindow("Text styles Demonstration");

	MGL_useFont(largeFont);

	x = y = 5;
	for (i = 5; i > 1; i--) {
		MGL_setTextSize(2,i,2,i);
		sprintf(buf,"Size (2/%d)",i);
		MGL_drawStrXY(x,y,buf);
		y += MGL_textHeight();
		}

	MGL_setTextSize(2,3,2,3);

	x = MGL_maxx()/2;
	y = MGL_maxy()/2;

	MGL_setTextDirection(LEFT_DIR);
	MGL_drawStrXY(x,y,"This goes left");

	MGL_setTextDirection(UP_DIR);
	MGL_drawStrXY(x,y,"This goes up");

	MGL_setTextDirection(RIGHT_DIR);
	MGL_drawStrXY(x,y,"This goes right");

	MGL_setTextDirection(DOWN_DIR);
	MGL_drawStrXY(x,y,"This goes down");

	MGL_defaultAttributes();
	pause();
}

void CRTModeDemo(void)
/****************************************************************************
*
* Function:		CRTModeDemo
*
* Description:	Demonstrate the ability to switch between text and
*				graphics mode.
*
****************************************************************************/
{
	mainWindow("MGL_restoreCRTMode / MGL_setGraphMode demo");
	MGL_setTextJustify(CENTER_TEXT,CENTER_TEXT);

	MGL_drawStrXY(MGL_maxx()/2,MGL_maxy()/2,"Now you are in graphics mode...");
	statusLine("Press any key to for text mode...");

	getch();

	MGL_restoreCRTMode();
	printf( "Now you are in text mode.\n\n" );
	printf( "Press any key to go back to graphics..." );
	getch();

	MGL_setGraphMode();
	mainWindow("MGL_restoreCRTMode / MGL_setGraphMode demo");
	MGL_setTextJustify(CENTER_TEXT,CENTER_TEXT);
	MGL_drawStrXY(MGL_maxx()/2,MGL_maxy()/2,"Back in graphics mode...");

	pause();
}

void patternDemo(void)
/****************************************************************************
*
* Function:		patternDemo
*
* Description:	Display the set of fill patterns defined in this demo
*				program.
*
****************************************************************************/
{
	int 	i,j,x,y,height,width;

	mainWindow("Pattern Demonstration");

	width  = (MGL_maxx()+1) / 20;		/* Compute width of boxes		*/
	height = (MGL_maxy()+1) / 9;		/* Compute height of boxes		*/

	x = y = 10;							/* Start in upper corner		*/
	MGL_setColor(MGL_realColor(LIGHTGRAY));
	MGL_setPenStyle(BITMAP_PATTERN_TRANSPARENT);


	for (j = 0; j < 7; j++) {			/* For 7 rows of boxes 		*/
		for (i = 0; i < 15; i++) {		/* For 15 columns of boxes		*/
			MGL_setPenBitmapPattern(&bitpat[j*15 + i]);
			MGL_fillRectCoord(x,y,x+width+1,y+height+1);
			x += width + 10;			/* Advance to next col			*/
			}
		x = 10;
		y += height + 10;
		}

	MGL_setColor(MGL_defaultColor());
	pause();
}

#define MaxPts		6		/* Maximum # of pts in polygon	*/

void polyDemo(void)
/****************************************************************************
*
* Function:		polyDemo
*
* Description:	Display a random pattern of polygons on the screen with
*				random fill styles.
*
****************************************************************************/
{
	int		i,maxx,maxy,val;
	point	poly[MaxPts];			/* Space to hold polygon data	*/

	mainWindow("MGL_fillPolygon Demonstration");
	statusLine("Press any key to continue, ESC to Abort");

	maxx = MGL_maxx();
	maxy = MGL_maxy();

	while (!kbhit()) {
		/* Define a random polygon */

		for (i = 0; i < MaxPts; i++) {
			poly[i].x = random(maxx);
			poly[i].y = random(maxy);
			}

		MGL_setColor(randoml(maxcolor)+1);
		MGL_setBackColor(randoml(maxcolor)+1);

		if ((val = random(3)) == 0) {
			MGL_setPenStyle(BITMAP_PATTERN_TRANSPARENT);
			MGL_setPenBitmapPattern(&bitpat[random(NUMPATS)+1]);
			}
		else if (val == 1) {
			MGL_setPenStyle(BITMAP_PATTERN_OPAQUE);
			MGL_setPenBitmapPattern(&bitpat[random(NUMPATS)+1]);
			}
		else {
			MGL_setPenStyle(SOLID_PATTERN);
			}

		MGL_fillPolygon(MaxPts,poly,0,0);
		}

	MGL_defaultAttributes();
	pause();
}

void fastPolyDemo(void)
/****************************************************************************
*
* Function:		fastPolyDemo
*
* Description:	Display a random pattern of convex triangular polygons
*				in replace mode at full speed.
*
****************************************************************************/
{
	int		maxx,maxy;
	point	poly[3];			/* Space to hold polygon data	*/

	mainWindow("MGL_fillPolygonFast Demonstration");
	statusLine("Press any key to continue, ESC to Abort");

	maxx = MGL_maxx();
	maxy = MGL_maxy();

	MGL_beginDrawing();

	while (!kbhit()) {
		/* Define a random polygon */

		poly[0].x = random(maxx);
		poly[0].y = random(maxy);
		poly[1].x = random(maxx);
		poly[1].y = random(maxy);
		poly[2].x = random(maxx);
		poly[2].y = random(maxy);

		MGL_setColor(randoml(maxcolor)+1);
		MGL_fillPolygon(3,poly,0,0);
		}

	MGL_endDrawing();

	MGL_defaultAttributes();
	pause();
}

void gouraudPolyDemo(void)
/****************************************************************************
*
* Function:		gouraudPolyDemo
*
* Description:	Display a random pattern of convex gouraud shaded
*				triangular polygons	at full speed.
*
****************************************************************************/
{
	int		i,maxx,maxy;
	point	poly[3];			/* Space to hold polygon data			*/
	color_t	colors[3];			/* Colors for each vertex of polygon	*/

	/* Gouraud shading is not supported for devices with less than 256
	 * colors.
	 */

	if (maxcolor < 255)	return;

	mainWindow("MGL_fillGouraudPolygon Demonstration");
	statusLine("Press any key to continue, ESC to Abort");

	maxx = MGL_maxx();
	maxy = MGL_maxy();

	/* Setup the palette to show smooth shading between bright red and
	 * bright blue. Note that we leave color 0 as black.
	 */

//	if (maxcolor == 255) {
		pal[0].red = pal[0].green = pal[0].blue = 0;
		for (i = 1; i <= 255; i++) {
			pal[i].red = ((long)PALMAX * i) / 255;
			pal[i].green = 0;
			pal[i].blue = ((long)PALMAX * (255 - i)) / 255;
			}
		MGL_setPalette(pal,MGL_getPaletteSize(),0);
//		}

	MGL_beginDrawing();

	while (!kbhit()) {
		for (i = 0; i < 3; i++) {
			poly[i].x = random(maxx);
			poly[i].y = random(maxy);
			colors[i] = 1 + randoml(maxcolor-1);
			}
		MGL_fillGouraudPolygon(3,poly,colors,0,0);
		}

	MGL_endDrawing();

	MGL_defaultAttributes();
	MGL_setDefaultPalette();
	pause();
}

int fixAngle(int angle)
/****************************************************************************
*
* Function:		fixAngle
* Parameters:	angle
* Returns:		Equivalent angle in the range 0-360 degrees.
*
****************************************************************************/
{
	while (angle < 0)
		angle += 360;
	while (angle >= 360)
		angle -= 360;
	return angle;
}

void pageFlipDemo(void)
/****************************************************************************
*
* Function:		pageFlipDemo
*
* Description:	If page flipping is available, this routine will perform
*				a reasonably complex animation sequence. We use a smart
*				dirty rectangle animation technique to only clear the
*				portion of the display page that has changed to get high
*				speed animation.
*
****************************************************************************/
{
	int		maxx,maxy,stepx,stepy,secAngle,minAngle;
	rect	extent,dirtyRect;

	if (MGL_doubleBuffer()) {
		mainWindow("Page flip animation demo");
		statusLine("Press any key to continue, ESC to Abort");

		maxx = MGL_maxx();
		maxy = MGL_maxy();

		/* Draw title again for hidden page */

		MGL_swapBuffers();
		mainWindow("Page flip animation demo");
		statusLine("Press any key to continue, ESC to Abort");

		extent.left = extent.top = 0;
		extent.right = maxx/5;
		extent.bottom = ((long)extent.right * 1000) / aspect;
		dirtyRect = extent;

		stepx = 1;
		stepy = 1;
		secAngle = minAngle = 90;

		while (!kbhit()) {
			MGL_setColor(MGL_realColor(BLACK));
			MGL_fillRect(dirtyRect);
			MGL_setColor(MGL_realColor(RED));
			MGL_fillEllipse(extent);
			MGL_setColor(MGL_realColor(WHITE));
			MGL_fillEllipseArc(extent,secAngle-5,secAngle);
			MGL_fillEllipseArc(extent,minAngle-5,minAngle);

			/* Swap the display buffers */

			MGL_swapBuffers();

			/* Bounce the clock off the walls */

			dirtyRect = extent;
			MGL_insetRect(dirtyRect,-ABS(stepx),-ABS(stepy));
			if (extent.left + stepx < 0)
				stepx = -stepx;
			if (extent.right + stepx > maxx)
				stepx = -stepx;

			if (extent.top + stepy < 0)
				stepy = -stepy;
			if (extent.bottom + stepy > maxy)
				stepy = -stepy;

			MGL_offsetRect(extent,stepx,stepy);

			/* Update the hand movement */

			secAngle = fixAngle(secAngle - 5);
			if (secAngle == 90)
				minAngle = fixAngle(minAngle - 5);
			}

		MGL_singleBuffer();
		MGL_defaultAttributes();
		pause();
		}
	else {
		mainWindow("Page flip animation demo");

		maxx = MGL_maxx();
		maxy = MGL_maxy();

		MGL_setTextJustify(CENTER_TEXT,CENTER_TEXT);
		MGL_moveToCoord(maxx/2,maxy/2);
		MGL_drawStr("Video mode does not support page flipping");

		pause();
		}
}

void mouseCursorDemo(void)
/****************************************************************************
*
* Function:		mouseCursorDemo
*
* Description:	Display the mouse cursor on the screen, and change it to
*				a number of different styles.
*
****************************************************************************/
{
	/* First check to ensure that a mouse is actually installed in the
	 * system. MS_init() will return 0 if no mouse is installed, or the
	 * number of buttons if it is.
	 */

	if (MS_init() > 0) {
		MS_show();
		mainWindow("Mouse Cursor Demonstration");
		statusLine("Press any key for CHECK cursor");

		getch();
		MS_setCursor(&CHECK);
		statusLine("Press any key for CROSS cursor");

		getch();
		MS_setCursor(&CROSS);
		statusLine("Press any key for GLOVE cursor");

		getch();
		MS_setCursor(&GLOVE);
		statusLine("Press any key for IBEAM cursor");

		getch();
		MS_setCursor(&IBEAM);
		statusLine("Press any key for DEFAULT cursor");

		getch();
		MS_setCursor(&DEF_CURSOR);

		pause();
		MS_hide();
		}
}

#ifdef	FLOODFILL_DEMO

/* Function prototypes for floodfill routines in 'ffill.c' */

void MGL_interiorFill(int x,int y);
void MGL_boundaryFill(int x,int y,color_t bdr);

/* We need a rather large stack to fill high resolution displays! */

#ifdef  __TURBOC__
extern unsigned _stklen = 51200U;
#endif

void boundaryFillDemo(void)
/****************************************************************************
*
* Function:		boundaryFillDemo
*
* Description:	Demonstrate the boundary flood fill routine.
*
****************************************************************************/
{
    int     i,j,maxx,maxy;
    color_t color;
	point	poly[6];

	mainWindow("MGL_boundaryFill Demonstration");

	maxx = MGL_maxx();
	maxy = MGL_maxy();

	srand(200);

	/* Draw a set of polygons */

	for (j = 0; j < 3; j++) {
		color = random(10);				/* Keep random in sync			*/
		color = (j == 0) ? BLUE : (j == 1) ? RED : YELLOW;

		for (i = 0; i < 6; i++) {
			poly[i].x = random(maxx);
			poly[i].y = random(maxy);
			}

		MGL_setColor(MGL_realColor(color));
		MGL_fillPolygon(6,poly,0,0);
		}

	MGL_setColor(MGL_defaultColor());
	statusLine("Press a key for boundary fill");
	getch();

	MGL_setColor(MGL_realColor(GREEN));
    MGL_boundaryFill(10,10,MGL_realColor(RED));

	MGL_setColor(MGL_defaultColor());
	pause();
}

void interiorFillDemo(void)
/****************************************************************************
*
* Function:		interiorFillDemo
*
* Description:	Demonstrate the interior flood fill routine.
*
****************************************************************************/
{
    int     i,j,maxx,maxy;
    color_t color;
	point	poly[6];

	mainWindow("MGL_interiorFill Demonstration");

	maxx = MGL_maxx();
	maxy = MGL_maxy();

	srand(200);

	/* Draw a set of polygons */

	for (j = 0; j < 3; j++) {
		color = random(10);				/* Keep random in sync			*/
		color = (j == 0) ? BLUE : (j == 1) ? RED : YELLOW;

		for (i = 0; i < 6; i++) {
			poly[i].x = random(maxx);
			poly[i].y = random(maxy);
			}

		MGL_setColor(MGL_realColor(color));
		MGL_fillPolygon(6,poly,0,0);
		}

	MGL_setColor(MGL_defaultColor());
	statusLine("Press a key for interior fill");
	getch();

	MGL_setColor(MGL_realColor(GREEN));
	MGL_interiorFill(10,10);

	MGL_setColor(MGL_defaultColor());
	pause();
}

#endif

#define	POINTS	10

int		data1[POINTS] = { 1, 3, 5, 4, 3, 2, 1, 5, 4, 2 };
int		data2[POINTS] = { 4, 6, 10, 2, 6, 4, 8, 10, 6, 2 };
int		data3[POINTS] = { 1, 3, 2, 5, 7, 9, 5, 4, 5, 8 };

void markerDemo(void)
/****************************************************************************
*
* Function:		markerDemo
*
* Description:	Draws a simple graph using markers to show to points on
*				the graph.
*
****************************************************************************/
{
	int		i,stepx,stepy,maxy,maxx;
	point	p[POINTS];

	mainWindow("Marker Demonstration");

	maxx = MGL_maxx();
	maxy = MGL_maxy();
	stepx = maxx / 12;
	stepy = maxy / 12;

	/* Draw the graph axes */

	MGL_lineCoord(stepx,maxy - stepy,stepx*11,maxy-stepy);
	MGL_lineCoord(stepx,maxy - stepy,stepx,maxy - stepy*11);

	for (i = stepx*11; i >= stepx; i -= stepx)
		MGL_lineCoord(i,maxy - stepy - 3,i,maxy - stepy + 3);

	for (i = maxy - stepy; i >= maxy-stepy*11; i -= stepy)
		MGL_lineCoord(stepx-3,i,stepx+3,i);

	/* Draw the first set of data */

	for (i = 0; i < POINTS; i++) {
		p[i].x = stepx + i*stepx;
		p[i].y = maxy - stepy - data1[i]*stepy;
		}

	MGL_setMarkerSize(4);
	MGL_setColor(MGL_realColor(LIGHTRED));
	MGL_polyLine(POINTS,p);
	MGL_setMarkerColor(MGL_realColor(MAGENTA));
	MGL_polyMarker(POINTS,p);

	/* Draw the second set of data */

	for (i = 0; i < POINTS; i++) {
		p[i].x = stepx + i*stepx;
		p[i].y = maxy - stepy - data2[i]*stepy;
		}

	MGL_setColor(MGL_realColor(LIGHTGREEN));
	MGL_polyLine(POINTS,p);
	MGL_setMarkerColor(MGL_realColor(MAGENTA));
	MGL_setMarkerStyle(MARKER_CIRCLE);
	MGL_polyMarker(POINTS,p);

	/* Draw the third set of data */

	for (i = 0; i < POINTS; i++) {
		p[i].x = stepx + i*stepx;
		p[i].y = maxy - stepy - data3[i]*stepy;
		}

	MGL_setColor(MGL_realColor(LIGHTBLUE));
	MGL_polyLine(POINTS,p);
	MGL_setMarkerColor(MGL_realColor(MAGENTA));
	MGL_setMarkerStyle(MARKER_X);
	MGL_polyMarker(POINTS,p);

	MGL_defaultAttributes();
	pause();
}

void finale(void)
/****************************************************************************
*
* Function:		finale
*
* Description:	Display a goodbye message before signing off.
*
****************************************************************************/
{
	mainWindow("Finale");
	MGL_setTextJustify(CENTER_TEXT,CENTER_TEXT);

	MGL_useFont(largeFont);
	MGL_drawStrXY(MGL_maxx()/2,MGL_maxy()/2,"Thats all folks!");
	statusLine("Press any key exit...");
	pause();
}

void cdecl main(int argc,char *argv[])
{
	int		option;
	char	*argument;

	/* Parse command line options */
	do {
		option = getopt(argc,argv,"Ee:Vv:FfBbSsPp:Hh",&argument);
		if (option > 0) option = tolower(option);
		switch(option) {
			case 'e':
				forceDriver = grEGA;
				break;
			case 'v':
				if (tolower(argument[0]) == 'g')
					forceDriver = grVGA;
				else if (tolower(argument[0]) == 'e')
					forceDriver = grSVGA;
				else help();
				break;
			case 'f':
				MGL_slowSuperVGA(true);
				break;
			case 'b':
				MGL_useBIOSPalette(true);
				break;
			case 's':
				MGL_slowPalette(true);
				break;
			case 'p':
				snowlevel = atoi(argument);
				break;
			case INVALID:
			case 'h':
				help();
			}
		} while (option != ALLDONE && option != PARAMETER);

	/* Set the system into graphics mode */

	if (nextargv >= argc)
		startGraphics(NULL);
	else
		startGraphics(argv[nextargv]);

	reportStatus();		/* Report status of graphics system				*/
    colorDemo();
	paletteDemo();
	pixelDemo();
	imageDemo();
	rectangleDemo();
	circleDemo();
	filledCircleDemo();
	arcDemo();
	filledArcDemo();
	polyDemo();
	fastPolyDemo();
	gouraudPolyDemo();
	pieDemo();
	lineDemo();
	lineRelDemo();
	lineToDemo();
	patternDemo();
	markerDemo();
	pageFlipDemo();

#if defined(FLOODFILL_DEMO) && !defined(__MSC__)
	boundaryFillDemo();
	interiorFillDemo();
#endif

	mouseCursorDemo();
	textDump();
	textStyleDemo();
	CRTModeDemo();
	finale();

	MGL_exit();			/* Close graphics system and restore text mode	*/
}
