#include <sys\stat.h>
#include <fcntl.h>
#include <io.h>
#include <dir.h>
#include <ctype.h>
#include <string.h>
#include <stdlib.h>
#include <stdio.h>

/*
 * This software is distributed as shareware by its author, Robin Hilliard
 * of the Software Loft.  You are free to modify the source as you see fit,
 * as long as you retain all of this license text in the modified source.
 *
 * If you find this software useful in your business, then please try and
 * send a postcard from where you live.  (There, that's not too much to
 * ask, now is it?)  Remember this software did take time to write and its
 * author has to live and expand his postcard collection somehow. Please be
 * polite and try and acknowledge your thanks by sending a postcard!  The
 * address to post to is:
 *
 *        Robin Hilliard,
 *        Lough Guitane,
 *        Killarney,
 *        Co. Kerry,
 *        Ireland.
 *
 * Any and all bugs reports are gratefully received and will usually elicit
 * a bug fix.  You can contact me on internet as robin@flapjack.ieunet.ie.
 *
 * I wrote this program one afternoon two years ago during a stint as a
 * lecturer in AutoCAD.  The code was originally written for Turbo C,
 * version 2.0, but can probably be compiled under most other compilers
 * without major modification.  There are few comments in the code, but you
 * should be able to work it out easily enough.
 *
 * thanks!
 *
 * robin hilliard, the software loft.
 */

#define VER_MAJOR "1"
#define VER_MINOR "00"

typedef unsigned long int dword;
typedef unsigned int word;
typedef struct __item {
  char name[32];
  dword ofs;
  dword sz;
  struct __item *next, *prev;
  word extract:1;
  word deleted:1;
} ITEM;
#define SIZEOFITEM (32+sizeof(dword))
ITEM *opfilelist;

char AcadSig[28] = "AutoCAD Slide Library 1.0\x0D\x0A\x1A";

ITEM *FirstSlide, *fadd;
char libname[128];
word adding = 0, deleting = 0, extracting = 0;
dword old_lib_ofs = 0;

static void build_name(char*,char*);
static void usage(void);
static ITEM *llocate(char *, ITEM *);
static void list_contents(word);
static ITEM *ladd(char *, ITEM **);
static word read_contents(void);
static void add_to_add_list(char *);
static void work_out_opers(char **);
static void do_operations(void);
static char *fixname(char *);
static void copy_slide(FILE *, FILE *, ITEM *);
static void add_slide_to_lib(FILE *, ITEM *);
static void write_header(FILE *);
static void fcopy(FILE *, FILE *, dword );
static void check_add_list(void);

void *mycalloc(word sz)
{
  char *p;

  if (!sz) {
    printf("trying to alloc zero bytes!");
    exit(4);
  }
  p = calloc(sz, 1);
  if (!p) {
    printf("Out of memory!");
    exit(5);
  }
  return p;
}


static void usage(void)
{
  printf(
         "Syntax:\tSLib <libnm> [opers...]\n"
         "Where:\n"
         "\tlibnm:\tName of slide library. '.SLB' extension assumed\n"
         "\topers:\tOne of [*|-|*-|-*|+]slide where:\n"
         "\t\t\t* extracts <slide> to current directory\n"
         "\t\t\t- deletes <slide> from <slidelib>\n"
         "\t\t\t+ adds <slide> to <slidelib>\n"
         "\t\tMultiple add, extract and deleted operations are supported\n"
         "\n"
         "Example:\tslib acad +newsld\n"
         "\t\t\tAdds slide <newsld.sld> to <acad.slb>\n"
         "\t\tslib mylib *-oldsld\n"
         "\t\t\tExtracts and removes <oldsld.sld> from <mylib.slb>\n"
         "\t\tslib mylib @opfile ...\n"
         "\t\t\tReads commands from <opfile>"
  );
}


void main(int argc, char *argv[])
{
  word lib_exists;

  _fmode = O_BINARY;
  printf("SLib v" VER_MAJOR "." VER_MINOR ": Replacement for AutoCAD's slidelib\n");
  printf("(C) 1993-1995, The Software Loft.  email to slib@flapjack.ieunet.ie\n");
  if (argc<2) {
    usage();
    exit(1);
  }
  switch (*argv[1]) {
  case '-':
  case '?':
  case '/':
       usage();
       if (argv[1][1]=='#')
         printf("\n<Last compiled: " __TIME__ ", " __DATE__ ">");
       exit(1);
  }
  strlwr(strcpy(libname,argv[1]));
  build_name(libname, ".slb");
  lib_exists = read_contents();
  work_out_opers(&argv[2]);
  if (!lib_exists && (deleting || extracting)) {
    printf("Unable to open slide library %s", libname);
    exit(2);
  }
  if (argc==2) {
    list_contents(1);
    exit(3);
  }
  check_add_list();
  do_operations();
}


static char *fixname(char *p)
{
  while (1) {
    switch (*p) {
    case '-':
    case '*':
    case '+':
         p++;
         continue;
    default:
         return p;
    }
  }
}


static void get_opfile(char *name)
{
  FILE *f;
  char str[100];
  char **ptrs, *p;
  word lines, i;

  f = fopen(&name[1], "rt");
  if (!f) {
    printf("Unable to open opfile %s\n", &name[1]);
    return;
  }
  lines = 0;
  while (fgets(str, sizeof(str), f))
    lines++;
  rewind(f);
  ptrs = mycalloc((lines+1)*sizeof(char*));
  lines = 0;
  while (fgets(str, sizeof(str), f)) {
    while (*str==' ')
      strcpy(str, &str[1]);
    while (*str && ((p=&str[strlen(str)-1]),*p==' ' || *p=='\n'))
      str[strlen(str) - 1] = 0;
    if (!*str)
      continue;
    p = mycalloc(strlen(str)+1);
    strcpy(p, str);
    ptrs[lines++] = p;
  }
  fclose(f);
  ptrs[lines] = NULL;
  work_out_opers(ptrs);
  for (i=0;i<lines;i++)
    free(ptrs[i]);
  free(ptrs);
}


static void work_out_opers(char **snames)
{
  char *p;
  ITEM *item;

  while ((p = *snames)!=NULL) {
tol:switch (*p) {
    case '-':
    case '*':
         if ((item=llocate(fixname(p), FirstSlide))==NULL) {
           printf("Unable to locate slide %s in library %s\n", fixname(p), libname);
           break;
         }
         if (*p=='*') {
           extracting = 1;
           item->extract = 1;
         } else {
           deleting = 1;
           item->deleted = 1;
         }
         p++;
         goto tol;

    case '@':
         get_opfile(p);
         break;

    case '+':
         add_to_add_list(fixname(p));
    }
    snames++;
  }
}


static void get_name(char *filepath, char *fname)
{
  char fndrive[3];
  char fndir[81];
  char fnname[9];
  char fnext[5];

  fnsplit(filepath, fndrive, fndir, fnname, fnext);
  strcpy(fname, fnname);
}


static void check_add_list(void)
{
  ITEM *item, *nextitem;
  char s[100];

  item = fadd;
  while (item) {
    get_name(item->name, s);
    nextitem = item->next;
    if (llocate(s, FirstSlide)) {
      printf("Slide %s is already in library\n", s);
      adding--;
      if (nextitem)
        nextitem->prev = item->prev;
      if (item->prev)
        item->prev->next = nextitem;
      else
        fadd = nextitem;
      free(item);
    }
    item = nextitem;
  }
}


static void add_to_add_list(char *name)
{
  char str[100];
  struct stat st;
  ITEM *item;

  strcpy(str, strlwr(name));
  build_name(str, ".sld");
  if (stat(str, &st)) {
    printf("Unable to locate slide %s\n", str);
    return;
  }
  item = ladd(str, &fadd);
  item->sz = st.st_size;
  adding++;
}



static ITEM *llocate(char *name, ITEM *first)
{
  ITEM *item;

  item = first;
  strlwr(name);
  while (item) {
    if (!strcmp(item->name, name))
      return item;
    item = item->next;
  }
  return NULL;
}


static void list_contents(word docomma)
{
  ITEM *item;
  int wd;

  item = FirstSlide;
  printf("Contents of slide library %s are:\n", libname);
  wd = 0;
  while (item) {
    wd += strlen(item->name);
#if defined(DEBUG)
    printf("[%6lu, %6lu]", item->ofs, item->sz);
    docomma = 0;
#endif
    printf("%s", item->name);
    if (docomma && item->next) {
      printf(", ");
      wd+=2;
      if (wd+strlen(item->next->name) > 77) { /* wrap text intelligently */
        printf("\n");
        wd = 0;
      }
    } else {
      printf("\n");
    }
    item = item->next;
  }
}


void build_name(char*input,char*ext)
{
  char fndrive[3];
  char fndir[81];
  char fnname[9];
  char fnext[5]; /* stores dot also */

  fnsplit(input, fndrive, fndir, fnname, fnext);
  fnmerge(input, fndrive, fndir, fnname, ext);
}


static ITEM *ladd(char *name, ITEM **firstitem)
{
  ITEM *item, *ptitem;

  strlwr(name);
  if (*firstitem==NULL) {
    item = *firstitem = mycalloc(sizeof(ITEM));
    ptitem = item->prev = NULL;
  } else {
    ptitem = item = *firstitem;
    while (item->next) {
      ptitem = item;
      item = item->next;
    }
    item = item->next = mycalloc(sizeof(ITEM));
  }
  item->next = NULL;
  item->prev = ptitem;
  strcpy(item->name, name);
  return item;
}


static word read_contents(void)
{
  FILE *lib;
  char str[0x1C];
  dword d;
  ITEM *item, titem;

  lib = fopen(libname, "rb");
  if (!lib)
    return 0;
  fread(&str, 1, 0x1C, lib);
  if (memcmp(str, AcadSig, sizeof(AcadSig))) {
    printf("File is not an AutoCAD slide library (v1.0)");
    exit(3);
  }
  fread(&d, 1, 4, lib);
  while (1) {
    fread(&titem, 1, SIZEOFITEM, lib);
    if (*titem.name) {
      item = ladd(titem.name, &FirstSlide);
      item->ofs = titem.ofs;
    } else {
      break;
    }
  }
  old_lib_ofs = ftell(lib);
  item = FirstSlide;
  while (item) {
    if (item->next) {
      item->sz = item->next->ofs - item->ofs;
    } else {
      item->sz = filelength(fileno(lib)) - item->ofs;
    }
    item = item->next;
  }
  fclose(lib);
  return 1;
}


static void fcopy(FILE *f, FILE *of, dword sz)
{
  int c;

  while (sz--) {
    if ((c = fgetc(f))==EOF)
      return;
    fputc(c, of);
  }
}


static void extract(FILE *f, ITEM *item)
{
  char bak[100], slide[100];
  FILE *of;

  if (fseek(f, item->ofs, SEEK_SET))
    return;
  strcpy(slide, item->name);
  build_name(slide, ".sld");
  strcpy(bak, slide);
  build_name(bak, ".bak");
  unlink(bak);
  if (!rename(slide, bak))
    printf("Backup slide %s created\n", bak);
  of = fopen(slide, "wb");
  if (!of) {
    printf("Unable to create output slide %s\n", slide);
    return;
  }
  printf("Extracting %s...", slide);
  fcopy(f, of, item->sz);
  printf("ok\n");
  fclose(of);
}


static void copy_slide(FILE *f, FILE *of, ITEM *item)
{
  dword sz;
  int c;

  if (fseek(f, item->ofs, SEEK_SET))
    return;
  sz = item->sz;
  while (sz--) {
    if ((c = fgetc(f))==EOF)
      break;
    fputc(c, of);
  }
}


static void add_slide_to_lib(FILE *of, ITEM *item)
{
  char str[100];
  FILE *f;

  strcpy(str, item->name);
  build_name(str, ".sld");
  f = fopen(str, "rb");
  if (!f) {
    printf("Unable to open input slide %s\n", str);
    return;
  }
  printf("Adding %s...", str);
  fcopy(f, of, item->sz);
  printf("ok\n");
  fclose(f);
}


static void work_out_offsets(void)
{
  dword ofs;
  word items = 0;
  ITEM *item;

  item = FirstSlide;
  ofs = sizeof(AcadSig) + sizeof(dword) + SIZEOFITEM;
  while (item) {
    item->ofs = ofs;
    if (!item->deleted)
      ofs+=item->sz;
    item = item->next;
    items++;
  }
  item = fadd;
  while (item) {
    item->ofs = ofs;
    ofs+=item->sz;
    item = item->next;
    items++;
  }
  item = FirstSlide;
  while (item) {
    item->ofs += items*SIZEOFITEM;
    item = item->next;
  }
  item = fadd;
  while (item) {
    item->ofs += items*SIZEOFITEM;
    item = item->next;
  }
}



static void make_item(ITEM *src, ITEM *dst)
{
  char s[100];

  *dst = *src;
  memset(dst->name, 0, sizeof(dst->name));
  get_name(src->name, s);
  strupr(strcpy(dst->name, s));
}



static void write_header(FILE *of)
{
  ITEM *item, titem;
  dword d = 0;

  fwrite(AcadSig, 1, sizeof(AcadSig), of);
  fwrite(&d, 1, sizeof(d), of);
  item = FirstSlide;
  while (item) {
    make_item(item, &titem);
    fwrite(&titem, 1, SIZEOFITEM, of);
    item = item->next;
  }
  item = fadd;
  while (item) {
    make_item(item, &titem);
    fwrite(&titem, 1, SIZEOFITEM, of);
    item = item->next;
  }
  memset(&titem, 0, sizeof(titem));
  fwrite(&titem, 1, SIZEOFITEM, of);
}



static void do_operations(void)
{
  ITEM *item;
  FILE *f, *of;
  char bak[100];
  word lib_exists = 1;

  item = FirstSlide;
  f = fopen(libname, "rb");
  if (!f) {
    if (!adding) {
      printf("Unable to open %s for reading", libname);
      exit(1);
    } else {
      lib_exists = 0;
      goto somewhere_else;
    }
  }
  while (item) {
    if (item->extract)
      extract(f, item);
    item = item->next;
  }
  fclose(f);
  if (!adding && !deleting) {
    printf("No changes make to slide library %s\n", libname);
    return;
  }

  strcpy(bak, libname);
  build_name(bak, ".bak");
  unlink(bak);
  rename(libname, bak);
  printf("Backup file %s created\n", bak);
  f = fopen(bak, "rb");
  if (!f) {
    printf("Unable to open backup copy of library\n", libname);
    return;
  }
somewhere_else:
  of = fopen(libname, "wb");
  if (!of) {
    printf("Unable to open new library %s\n", libname);
    if (lib_exists)
      fclose(f);
    return;
  }
  work_out_offsets();
  write_header(of);
  if (lib_exists) {
    item = FirstSlide;
    while (item) {
      if (!item->deleted)
        copy_slide(f, of, item);
      item = item->next;
    }
  }
  item = fadd;
  while (item) {
    add_slide_to_lib(of, item);
    item = item->next;
  }
  fclose(of);
  if (lib_exists)
    fclose(f);
}
