;
; Descript.asm
;
; Function: subroutines for descriptor handling
;   Handles descriptor field setting\retrieving
;   Handles descriptor allocation table for global descriptors.
;     Local descriptors are statically allocated by the TSS routines
;
	IDEAL
	P386

include "gdt.asi"
include "boot.asi"
include "segs.asi"
include "page.asi"
include "boot.ase"
	PUBLIC DescriptorInit, GrabDescriptor, ReleaseDescriptor
	PUBLIC GrabGDTDescriptor, ReleaseGDTDescriptor
	PUBLIC DescriptorAddress,FindFreeDescriptor
	PUBLIC SetDescriptorBase, SetDescriptorLimit
	PUBLIC GetDescriptorBase, GetDescriptorLimit
	PUBLIC SetGateOffset, SetGateSelector
	PUBLIC GetGateOffset, GetGateSelector
	PUBLIC GetDescriptorType
	PUBLIC desctable

SEGMENT seg386data
desctable dd	(NUMGDT+31) SHR 5 DUP (-1) ; Table of in-use GDT entries
ENDS	seg386data

SEGMENT seg386
;
; Initialize the descriptor allocation table
;
PROC	DescriptorInit
	mov	edi,offset tGDT	; Find GDT
	mov	ecx,NUMGDT	; Get number of entries
	mov	ebx,0		; At entry 0
idl:
	test	[edi + DESCRIPTOR.TYPE],DT_PRESENT ; See if something there
	jz	short noset	; No, next entry
	btr	[desctable],ebx	; Yes clear the bit to mark present
noset:
	inc	ebx		; Next entry
	add	edi,8		;
	loop	idl		; Loop
	mov	eax,ebx		; Find out if extra bits at end of table
	add	eax,31          ;
	and	eax,0ffffffe0h  ;
lp:
	cmp	eax,ebx		; 
	jz	short done      ; No more
	btr	[desctable],ebx ; Reset all extra bits to mark used
	inc	ebx		;
	jmp	lp              ;

done:
	ret
ENDP	DescriptorInit
;
; Set type bytes for descriptor
;
PROC	GrabDescriptor
	mov	[edi + DESCRIPTOR.HILIM],DT_HIDEF
	or	al, DT_PRESENT
	mov	[edi + DESCRIPTOR.TYPE],al
	ret
ENDP	GrabDescriptor
;
; Mark descriptor not present
;
PROC	ReleaseDescriptor
	and	[edi + DESCRIPTOR.TYPE], not DT_PRESENT
	ret
ENDP	ReleaseDescriptor
;
; Allocate a GDT descriptor
;
PROC	GrabGDTDescriptor
	call	GrabDescriptor	; Set type bytes
	mov	eax,edi		; Find descriptor number
	sub	eax,offset tGDT	;
	shr	eax,3           ;
	btr	[desctable],eax ; Mark unavailable
	ret
ENDP	GrabGDTDescriptor
;
; Deallocate a GDT descriptor
;
PROC	ReleaseGDTDescriptor
	call	ReleaseDescriptor	; Mark not present
	mov	eax,edi			; Find descriptor number
	sub	eax,offset tGDT		;
	shr	eax,3			;
	bts	[desctable],eax		; Mark available
	ret
ENDP	ReleaseGDTDescriptor
;
; Translate a selector to the address of a descriptor
;
PROC	DescriptorAddress
	cwde
	push	ebx
	TEST	eax,ti_local		; See if in LDT
	jz	short getgdt
	sub	ebx,ebx			; If so get LDT selector
	sldt	bx			;
	and	ebx, NOT SEL_STATUS	; Strip off RPL and TI
	mov	edi,ebx			;
	add	edi,offset tGDT         ; Find position in GDT
	call	GetDescriptorBase	; Load up the LDT base address
	ZA	edi			; Segment offset
	jmp	short gotldt		
getgdt:
	mov	edi,offset tGDT         ; Otherwise just get the GDT table
gotldt:
	and	eax,NOT SEL_STATUS	; Strip off RPL and TI of descriptor
	add	edi,eax			; Add in to table base
	pop	ebx
	ret
ENDP	DescriptorAddress
;
; Find a free GDT descriptor
;
PROC	FindFreeDescriptor
	push	ecx
	push	ebx
	sub	eax,eax
	sub	ebx,ebx
	mov	edi,offset desctable	; Get descriptor table
	mov	ecx,(NUMGDT+31) shr 5	; Number of DWORDS to search
nextdesc:
	bsf	ebx,[dword ptr edi]	; Search one
	jnz	gotone			; Found a bit, finish up
	add	edi,4			; Else next word
	inc	eax			;
	loop	nextdesc		; loop
	stc				; Searched everything, nothing loose
	pop	ebx
	pop	ecx
	ret
gotone:
	shl	eax,4			; dWords to bits
	add	eax,ebx			; Add in offset
	shl	eax,SEL_SHIFT		; Make it a selector
	mov	edi,eax			; Make DI point to it
	add	edi,offset tGDT		;
	clc                             ; Success
	pop	ebx
	pop	ecx
	ret
ENDP	FindFreeDescriptor
;
; Set the base fields of a descriptor
;
PROC	SetDescriptorBase
	mov	[edi + DESCRIPTOR.base],ax
	shr	eax,16
	mov	[edi + DESCRIPTOR.MEDBASE],al
	mov	[edi + DESCRIPTOR.HIBASE],ah
	ret
ENDP	SetDescriptorBase
;
; Return the base fields of a descriptor
;
PROC	GetDescriptorBase
	push	eax
	mov	ah,[edi + DESCRIPTOR.HIBASE]
	mov	al,[edi + DESCRIPTOR.MEDBASE]
	shl	eax,16
	mov	ax,[edi + DESCRIPTOR.BASE]
	mov	edi,eax
	pop	eax
	ret
ENDP	GetDescriptorBase
;
; Set the limit fields of a descriptor
;
PROC	SetDescriptorLimit
	and	[edi + DESCRIPTOR.HILIM], NOT DT_HIGR
	test	eax,0fff00000h
	jz	sdl_logran
	or	[edi + DESCRIPTOR.HILIM],DT_HIGR
	shr	eax,12
sdl_logran:
	mov	[edi + DESCRIPTOR.LIMIT],ax
	shr	eax,16
	and	[edi + DESCRIPTOR.HILIM],0f0h
	or	[edi + DESCRIPTOR.HILIM],al
	ret
ENDP	SetDescriptorLimit
;
; Get the limit fields of a descriptor
;
; There is actually an instruction to do this one but I don't use it
;
PROC	GetDescriptorLimit
	mov	al,[edi + DESCRIPTOR.HILIM]
	and	ax,15
	shl	eax,16
	mov	ax,[edi + DESCRIPTOR.LIMIT]
	test	[edi + DESCRIPTOR.HILIM],DT_HIGR
	jz	gdl_logran
	shl	eax,12
	or	eax,PG_SIZE -1
gdl_logran:
	ret
ENDP	GetDescriptorLimit
;
; Set a Gate offset
;
PROC	SetGateOffset
	mov	[edi + GATEDESC.LOOFFSET],ax
	shr	ax,16
	mov	[edi + GATEDESC.HIOFFSET],ax
	ret
ENDP	SetGateOffset
;
; Set a Gate Selector
;
PROC	SetGateSelector
	mov	[edi + GATEDESC.SELECTOR],ax
	ret
ENDP	SetGateSelector
;
; Set a Gate Count field
;
PROC	SetGateCount
	mov	[edi + GATEDESC.COUNT],al
	ret
ENDP	SetGateCount
PROC	GetGateOffset
	mov	ax,[edi + GATEDESC.HIOFFSET]
	shl	eax,16
	mov	ax,[edi + GATEDESC.LOOFFSET]
	ret
ENDP	GetGateOFfset
PROC	GetGateSelector
	mov	ax,[edi + GATEDESC.SELECTOR]
	ret
ENDP	GetGateSelector
;
; Get the Descriptor Type
;
PROC	GetDescriptorType
	mov	al,[edi + DESCRIPTOR.TYPE]
	ret
ENDP	GetDescriptorType
ENDS	seg386
END