;
; Cmos.asm
;
; Function: time/date support
;   Handles 1 Second clock interrupt
;   Handles setting/ getting the time and or date
;

	IDEAL
	P386

include "segs.asi"
include "pic.asi"
include "os.asi"
include "cmos.asi"
include "iodelay.asi"
	PUBLIC	GetTime,SetTime,GetDate,SetDate, GrabTime, InitCMOS, RestoreCMOS
SEGMENT	seg386data
reg0a	db	26h			; Reset/Bios value for reg0a
reg0b	db	82h			; Reset/Bios value for reg0b
timeregs db	80h,82h,84h,87h,88h,89h	; Time/Date regs to read
					; Bit 8 of REG SELECT disables NMI
timevals db	6 DUP (0)		; Hold area for time/date
ENDS	seg386data

SEGMENT	seg386
by10	db	10
;
; Interrupt reads time after clock update
;
PROC	GrabTime
	push	eax			; Push regs
	push	ecx			;
	push	esi			;
	push	edi			;
	push	ds			; Get our data seg
	push	DS386			;
	pop	ds			;
	mov	al,SRC +80h		; Read status reg C to clear Interrupt
	call	ReadCMOS		;
	call	ReadTime		; Read the time
	PICACK				; Acknowledge the pics
	PICACK	PIC1ADR			;
	pop	ds			; Restore regs
	pop	edi			;
	pop	esi			;
	pop	ecx			;
	pop	eax			;
	iretd
ENDP	GrabTime
;
; Read the time and date
;
PROC	ReadTime
	mov	ecx,6			; six regs to read
	mov	esi,offset timeregs	; Get register list
	mov	edi,offset timevals	; Get Time/Date save area
gtlp:
	lodsb				; Get reg to read
	call	ReadCMOS		; Read it
	mov	[edi],al		; Save value
	inc	edi			;
	loop	gtlp			; Next reg
	call	OnNMI			; Reenable NMI
	ret				;
ENDP	ReadTime
;
; Get the BIOS values for CMOS clock control and update for us
;
PROC	InitCMOS
	call	AwaitUpdate		; Wait for update to complete
	call	ReadTime		; Read the time
	PICREAD	1,PIC1ADR		; Enable this interrupt
	and	al,0feh			;
	PICWRITE 1,PIC1ADR		;
	mov	al,SRD			; Read power-down register
	call	ReadCMOSwithNoNMI	;
	bt	eax,7			; See if lost power
	jnc	firstime		; Yes, go initialize
	mov	al,SRA			; Read reg A
	call	ReadCMOSwithNoNMI	;
	mov	[reg0a],al		; Save it for rundown
	mov	al,SRB			; Read reg b
	call	ReadCMOSwithNoNMI	;
	mov	[reg0b],al		; Save it for rundown
	bts	eax,INT_UPDATECOMPLETE  ; Enable update interrupt
	btr	eax,INT_PERIODIC	; Disable other interrupts
	btr	eax,INT_ALARM		;
	mov	ah,al 			;
	mov	al,SRB			; Write the new SRB value
	call	WriteCMOSwithNoNMI	;
	ret
firstime:
	call	RestoreCMOS		; First power up, initialize regs
	mov	ah,0			; Value to write
	mov	al,0			; Initial register
initime:
	push	eax			;
	call	WriteCMOSwithNoNMI	; Write the reg
	pop	eax			;
	inc	al			; Next reg
	cmp	al,10			; See if done
	jc	short initime		; No, loop
	ret
ENDP	InitCMOS
;
; Restore status registers for BIOS
;
PROC	RestoreCMOS
	mov	al,SRA			; Restore SRA
	mov	ah,[reg0a]		;
	call	WriteCMOSwithNoNMI	;
	mov	al,SRB			; Restore SRB
	mov	ah,[reg0b]		;
	call	WriteCMOSwithNoNMI	;
	ret
ENDP	RestoreCMOS
;
; Calculate binary fromBCD
;
PROC	FromBCD
	push	ebx
	lodsb				; Get BCD
	mov	bl,al			; bl = al mod 16
	and	bl,0fh			;
	shr	al,4			; al = al / 16
	mul	[by10]			; Multiply by 10
	add	al,bl			; Add in low nib
	pop	ebx
	ret
ENDP	FromBCD
;
; Calculate BCD from binary and write it
;
PROC	WriteBinary
	push	ebx			; Register in bl
	mov	bl,al
	mov	al,ah			; AX = value
	sub	ah,ah
	div	[by10]			; Divide by 10
	shl	al,4 			; Multiply by 16
	add	al,ah			; Add in low nibble
	daa				;
	mov	ah,al			; restore ah and al to writeable state
	mov	al,bl			;
	pop	ebx			; restore bx
	call	WriteCMOSWithNoNMI	; Write the value
	ret
ENDP	WriteBinary
;
; Return the time
;
PROC	GetTime
	push	esi
	mov	esi,offset timevals
	call	FromBCD			; Second
	mov	cl,al	
	call	FromBCD 		; Minute
	mov	bl,al
	call	FromBCD			; Hour
	mov	bh,al
	pop	esi
	ret
ENDP	GetTime
;
; Return the date
;
PROC	GetDate
	push	esi
	mov	esi,offset timevals + 3
	call	FromBCD
	mov	bl,al			; Day
	call	FromBCD
	mov	bh,al			; Month
	call	FromBCD
	movzx	ecx,al			; Year
	add	ecx,YEARBIAS
	pop	esi
	ret
ENDP	GetDate
;
; Set the time
;
PROC	SetTime
	call	HaltClock		; Halt the clock
	mov	ah,bh			; Hour
	mov	al,04h			;
	call	WriteBinary	;
	mov	ah,bl			; Min
	mov	al,02h			;
	call	WriteBinary	;
	mov	ah,cl			; Second
	mov	al,00h			;
	call	WriteBinary	;
	call	ReadTime		; Update RAM
	call	RunClock		; Start the clock again
	ret
ENDP	SetTime
;
; Set the date
;
PROC	SetDate
	call	HaltClock		; Hold the clock
	sub	ecx,YEARBIAS
	mov	ah,cl			; Year
	mov	al,09h			;
	call	WriteBinary	;
	mov	ah,bh			; Month
	mov	al,08h			;
	call	WriteBinary	;
	mov	ah,bl			; Day
	mov	al,07h			;
	call	WriteBinary	;
	call	ReadTime		; Update RAM
	call	RunClock		; Start the clock again
	ret
ENDP	SetDate
;
; Turn the clock on
;
PROC	RunClock
	mov	al,SRB			; Read SRB
	call	ReadCMOSWithNoNMI	;
	btr	eax,HALTBIT		; Kill the halt bit
	bts	eax,INT_UPDATECOMPLETE	; Turn on updates
	mov	ah,al			;
	mov	al,SRB			; Write SRB
	call	WriteCMOSWITHNoNMI	;
	ret
ENDP	RunClock
;
; Halt the clock
;
PROC	HaltClock
	push	ecx			; Wait for update to complete
	call	AwaitUpdate		;
	pop	ecx			;
	mov	al,SRB			; Read SRB
	call	ReadCMOSwithNoNMI	;
	bts	eax,HALTBIT		; Pull halt line HIGH
	mov	ah,al			;
	mov	al,SRB			; Write SRB
	call	WriteCMOSwithNoNMI	;
	ret
ENDP	HaltClock
;
; Wait for update
;
PROC	AwaitUpdate
	mov	ecx,20000h		; This long at max
awaitUpdate2: 
	mov	al,SRA			; Read SRA
	call	ReadCMOSwithNoNMI	;
	test	eax,UPDATEINPROGRESS	; Check the update bit
	loopnz	awaitUpdate2		; Loop while waiting
	ret
ENDP	AwaitUpdate
;
; Read CMOS
;
PROC	ReadCMOS
	cli				; No interrupts
	out	CMOSADDRESS,al		; Send address
	IODELAY				; IO Delay
	in	al,CMOSDATA		; Read data
	sti				; Interrupts
	ret				;
ENDP	ReadCMOS
;
; Write CMOS
;
PROC	WriteCMOS
	cli				; No interrupts
	out	CMOSADDRESS,al		; Out address
	IODELAY
	mov	al,ah			; Write data
	out	CMOSDATA,al		;
	sti				; Let interrupts
	ret
ENDP	WriteCMOS
;
; Read CMOS with NMI disabled
;
PROC	ReadCMOSwithNoNMI
	or	al,80h			; Set NMI bit
	call	ReadCMOS		; Read CMOS
OnNMI:
	push	eax			; Save results
      	mov	al,SRD			; Read SRD with no NMI disable
	call	ReadCMOS		;
	pop	eax			;
	ret
ENDP	ReadCMOSwithNoNMI
;
; Write CMOS with NMI disabled
;
PROC	WriteCMOSwithNoNMI
	or	al,80h			; Set NMI bit
	call	WriteCMOS		; Write the data
	jmp	short onNMI		; Go enable NMI
ENDP	WriteCMOSwithNoNMI
ENDS	seg386
END