/*
  Dynamic Neural Net v1.00     Khaled Mardam-Bey               24/10/94

  This implementation of a multi-layer perceptron with back-propagation
  was written with the intention of providing a (hopefully) easy to
  understand, and easy to modify, source code for users new to this field.

  All memory for this network is allocated dynamically using linked lists
  and all the necessary structures are generated. The Initialization
  procedure returns a single pointer to the network that has just been
  created.

  By changing only TWO variables, you can create a network with any
  number of layers, and each layer with a different number of neurons.

  This network is fully connected, which is to say that each layer
  is fully connected to the next layer.

  You can also have several different sized networks running simultaneously,
  each functioning independently of the others, or acting as inputs to
  other networks. It should also be very easy to modify the source so that
  neurons (or even whole layers) can be created/pruned during operation
  of the network, thus allowing dynamic expansion/contraction.

  The source code was compiled with Borland C++ 3.1 but should compile
  as it is under any C/C++ compiler.

  This source code is placed in the public domain and may be used,
  hacked, etc. in any way as long as it is not for commercial purposes.

  Please e-mail any questions, bug reports, suggestions, improvements, etc.

  internet address : xhlec@westminster.ac.uk
  www homepage     : http://www.wmin.ac.uk/~xhlec
*/

/***************************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <conio.h>
#include <time.h>
#include <math.h>
#include <alloc.h>
#include <string.h>
#include <dos.h>

/***************************************************************************/
/* function prototypes                                                     */

struct network *initialize_network (int, int[]);
struct layer   *create_layer       (int, int);
struct neuron  *create_neuron      (int);

void   assign_inputs   (struct network *);
void   feed_forward    (struct network *);
void   back_prop_error (struct network *);
void   adjust_weights  (struct network *);

void   print_struct    (struct layer   *);
void   clean_up        (struct network *);


/***************************************************************************/
/* network structures - neuron, layer, network                             */

struct neuron {
  float x;             /* the output of this neuron       */
  float e;             /* the error of this neuron        */
  float far *w;        /* the weights in this neuron      */
  struct neuron *nn;   /* the next neuron                 */
};

struct layer {
  int neurons;         /* number of neurons in this layer */
  struct neuron *fn;   /* the first neuron in this layer  */
  struct layer *nl;    /* the next layer                  */
  struct layer *pl;    /* the previous layer              */
};

struct network {
  struct layer *fl;    /* the first layer in this network */
};

/***************************************************************************/
/* Main routine                                                            */

int main(void)
{
  struct network *nw;    /* pointer to a network structure */

  long i, ch = ' ';      /*  i = number of iterations      */
			 /* ch = stores keypresses         */


  /*
     Two variables, NL and NPL, define the structure of the network.
     To create a different network, simply change their values.
  */

  int NL = 3;            /* number of layers in this network        */
  int NPL[3] = {2,2,1};  /* number of neurons per layer             */

  /*
     Example:

	for a FOUR layer network    NL = 4;

     where the 1st layer (input layer) has 4 neurons (inputs)
	   the 2nd layer has 10 neurons
	   the 3rd layer has 2 neurons
       and the 4th layer (output layer) has 15 neurons (outputs)

     then    NPL[4] = {4,10,2,15};

     The whole structure is automatically allocated and created.

     This network is currently setup to learn the XOR function but the
     parts of the program that are XOR specific are independent of
     the actual network implementation itself and are easy to replace
     with any inputs and functions that the user wishes.
  */


  /* create the whole network based on specifications in NL and NPL */
  /* and return a pointer to this network in 'nw'                   */

  nw = initialize_network(NL,NPL);


  /* iterate until user is satsified with network performance */

  for (i = 1; (i <= 100000) && (ch != 27); i++) {

    /* standard procedure for a multi-layer perceptron with backpropagation */

    assign_inputs(nw);
    feed_forward(nw);
    back_prop_error(nw);
    adjust_weights(nw);


    /* check key presses and print out info */

    if (kbhit()) ch = getch();        /* ESCape key exits the program */

    /* pressing 'p' turns off the onscreen printing of calculations   */
    /* to speed up the computation. Press spacebar to view info again */

    if (ch != 'p') {
      clrscr();
      printf("\nIteration: %ld\n",i);
      printf("Inputs     : %f, %f\n",nw->fl->fn->x, nw->fl->fn->nn->x);
      printf("Output     : %f\n\n",nw->fl->pl->fn->x);
    }

    /* press spacebar to step through the calculation, */
    /* any other key  to zoom through the calculation. */

    if (ch == ' ') {
      while(!kbhit());
      ch = getch();
    }
  }

  /* free all assigned memory for specified network structure */

  clean_up(nw);

  /* Goodbye, and have a nice day :-) */

  return 0;
}

/***************************************************************************/
/* allocate memory for a network structure and create the layers in it     */
/* nl    - number of layers                                                */
/* npl[] - array storing the number of neurons for each layer              */

struct network *initialize_network(int nl, int npl[])
{
  int i;
  struct network *nw;
  struct layer *l;

  randomize();

  if ((nw = malloc(sizeof(struct network))) == NULL) {
    printf("not enough memory to allocate network\n");
    exit(1);
  }

  l = nw->fl = create_layer(npl[0],0);   /* create first (input) layer */

  for (i=1; i < nl; i++) {
    l->nl = create_layer(npl[i],npl[i-1]);
    l->nl->pl = l;
    l = l->nl;
  }

  nw->fl->pl = l;

  return nw;
}

/***************************************************************************/
/* allocate memory for a layer structure and create the neurons in it      */
/* npl = neurons per layer                                                 */
/* wpn = weights per neuron                                                */

struct layer *create_layer(int npl, int wpn)
{
  struct layer *l;
  struct neuron *n;

  if ((l = malloc(sizeof(struct layer))) == NULL) {
    printf("not enough memory to allocate layer\n");
    exit(1);
  }

  l->nl = NULL;
  l->pl = NULL;
  l->neurons = npl;

  n = l->fn = create_neuron(wpn);   /* create first neuron in this layer */

  while (--npl > 0) {
    n->nn = create_neuron(wpn);
    n = n->nn;
  }

  return l;
}

/***************************************************************************/
/* allocate memory for a neuron structure and assign it random weights     */
/* wpn = weights per neuron                                                */

struct neuron *create_neuron(int wpn)
{
  struct neuron *n;

  int i;
  float weight;

  if ((n = malloc(sizeof(struct neuron))) == NULL) {
    printf("not enough memory to allocate neuron\n");
    exit(1);
  }

  n->x = 0.0;
  n->e = 0.0;
  n->nn = NULL;

  /* allocate memory for the weights in this neuron */
  /* and assign them random values                  */

  if (wpn > 0) {
    if ((n->w = (float far *) farcalloc(wpn, sizeof(float))) == NULL) {
      printf("not enough memory to allocate weights\n");
      exit(1);
    }

    for (i=0; i <= wpn; i++) {
      weight = ((rand() % 100)+1);
      weight /= ((rand() % 100)+weight*2.0);
      if ((rand() % 100) > 50) weight *= -1.0;

      n->w[i] = weight;
    }
  }

  return n;
}

/***************************************************************************/
/* free all memory that was assigned for the specified network structure   */

void clean_up(struct network *nw)
{
  struct layer *l, *tl;
  struct neuron *n, *tn;

  l = nw->fl;
  n = nw->fl->fn;

  while (l != NULL) {
    while (n != NULL) {
      if (l != nw->fl) farfree(n->w);
      tn = n;
      n = n->nn;
      free(tn);
    }
    tl = l;
    l = l->nl;
    n = l->fn;
    free(tl);
  }

  free(nw);
}

/***************************************************************************/
/* create two random inputs for the XOR function                           */
/* you can change this to assign inputs specific to the function           */
/* that you are implementing                                               */

void assign_inputs(struct network *nw)
{
  struct neuron *n = nw->fl->fn;

  /* assign output values to the neurons in the first (input) layer */
  /* which will act as inputs to the next layer in the network      */

  while (n != NULL) {
    n->x = rand() % 2;
    n = n->nn;
  }
}

/***************************************************************************/
/* calculate and feedforward outputs from the first layer to the last      */

void feed_forward(struct network *nw)
{
  struct layer *l;
  struct neuron *n, *pln;

  float net;
  int i;

  l = nw->fl->nl;    /* the first hidden layer after the input layer */

  while (l != NULL) {
    n = l->fn;

    while (n != NULL) {
      pln = l->pl->fn;    /* first node in the previous layer */

      net = n->w[0];

      /* the number of neurons in the previous layer is     */
      /* the number of weights in the neurons of this layer */

      for (i = 1; i <= l->pl->neurons; i++) {
	net += n->w[i] * pln->x;
	pln = pln->nn;
      }

      /* sigmoid function */

      n->x = 1.0 / (1.0 + exp(-net));

      n = n->nn;
    }

    l = l->nl;
  }
}

/***************************************************************************/
/* backpropagate error from the output layer through to the first layer    */

void back_prop_error(struct network *nw)
{
  struct layer *l;
  struct neuron *n, *nln;

  float d, E;
  int i, v1, v2;

  /* while loop - calculates errors for neurons in output layer       */
  /* ideally this should be an independent function since it can      */
  /* get quite complicated                                            */

  n = nw->fl->pl->fn;  /* the first neuron in the last (output) layer */

  while (n != NULL) {

    /*
       the next three lines are XOR specific.
       change the evaluation to whatever is required.
       your evaluation should set 'd' (the desired
       value) to 1.0 or 0.0 for each neuron.
    */

    v1 = nw->fl->fn->x;
    v2 = nw->fl->fn->nn->x;

    d = v1^v2;

    /*
      if you replace the above three lines with this line:

      if (nw->fl->fn->x > nw->fl->fn->nn->x) d = 1.0; else d = 0.0;

      the network learns to identify when the first input value
      is larger than the second input value. ie. it learns the
      'larger than' function for 1 bit numbers. It is fairly
      easy to modify it for N bit numbers.
    */

    n->e = n->x * (d - n->x) * (1.0 - n->x);

    n = n->nn;
  }

  /* calculate errors for hidden layers */

  l = nw->fl->pl->pl;  /* the layer before the output layer */

  while (l != nw->fl) {
    n = l->fn;

    for (i = 1; i<= l->neurons; i++) {
      E = 0.0;

      /* use the error of the neurons in the next layer to */
      /* calculate the error of those in this layer        */

      for (nln = l->nl->fn; nln != NULL; nln = nln->nn)
	E += nln->w[i] * nln->e;

      n->e = n->x * (1.0 - n->x) * E;

      n = n->nn;
    }

    l = l->pl;   /* point to the previous layer    */
		 /* remember we're going backwards */
  }
}

/***************************************************************************/
/* update weights for all of the neurons from the first to the last layer  */

void adjust_weights(struct network *nw)
{
  struct layer *l;
  struct neuron *n, *pln;

  float LC = 0.5;     /* learning coefficient */
  int i;

  l = nw->fl->nl;     /* the first hidden layer after the input layer */

  while (l != NULL) {
    n = l->fn;

    while (n != NULL) {
      pln = l->pl->fn;

      n->w[0] += (LC * 1.0 * n->e);

      /* the number of neurons in the previous layer is     */
      /* the number of weights in the neurons of this layer */

      for (i = 1; i <= l->pl->neurons; i++) {
	n->w[i] += (LC * pln->x * n->e);
	pln = pln->nn;
      }

      n = n->nn;
    }

    l = l->nl;
  }
}

/***************************************************************************/
/* print out the current structure of the specified layer                  */
/* Used for debugging purposes but I left it in just in case...  :-)       */

void print_struct(struct layer *l)
{
  struct neuron *n;
  int i;

  printf("\n\nStructure for Layer:\n\n");

  n = l->fn;   /* the first neuron in this layer */

  while (n != NULL) {
    printf("neuron\n");
    printf("  x = %f\n",n->x);
    printf("  e = %f\n",n->e);
    printf("  weights:\n");

    for (i = 0; i <= l->pl->neurons; i++)
      printf("    w[%d] = %f\n",i,n->w[i]);

    n = n->nn;
  }

  printf("\n");
  getch();
}
