/*
** Program fdb - family tree database generation and manipulation
**
** Copyright (C) 1994 Andy Burrows 
**
**            email: cadellin@corum.me.man.ac.uk (130.88.29.14)
**
** This program is free software; you can redistribute it and/or modify it
** under the terms of the GNU General Public Licence as published by the Free
** Software Foundation; either version 1, or any later version.
** 
** This program is distributed in the hope that it will be useful, but
** WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
** or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for
** more details. It should be included in this distribution in a file 
** called COPYING
**
*/

/*
   this file contains the callback functions for the button menus
*/

/*#define DEBUG */

/* standard headers */

#include <stdio.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <dirent.h>
#include <ctype.h>

/* XView headers */

#include <xview/xview.h>
#include <xview/panel.h>
#include <xview/canvas.h>
#include <xview/xv_xrect.h>
#include <xview/font.h>

/* family tree database definitions header */

#include "definitions.h"
#include "prototypes.h"

/* file menu selection callback */

void file_menu_selection(menu, menu_item)
Menu       menu;
Menu_item  menu_item;
{
DIR             *dp;
struct dirent   *dirp;
struct stat      statbuf;
int              i, num_choices;
char             dummy_string[MAX_STRING_LENGTH];
FILE            *fp;

/* ensure the text entries are up to date before a possible save operation */

update_text();

#ifdef DEBUG
    printf("%s selected from file menu\n", (char *)xv_get(menu_item, MENU_STRING));
#endif

if (!strcmp((char *)xv_get(menu_item, MENU_STRING), "Load"))
    {
    /* find out how many elements currently in the scrolling list */

    num_choices = (int) xv_get(load_list, PANEL_LIST_NROWS);

#ifdef DEBUG
    printf("deleting %d elements from load file scrolling list\n",num_choices);
#endif

    /* set the frame busy cursor */

    xv_set(load_frame, FRAME_BUSY, TRUE, NULL);

    /* clear the current contents of the load file scrolling list */

    if(num_choices > 0)
        {
        for( i = num_choices - 1 ; i >= 0 ; i-- )
            xv_set(load_list, PANEL_LIST_DELETE, i, NULL);
        }

    /* open the current directory */

    if((dp = opendir(".")) == NULL)
        {
        printf("unable to open current directory\n");
        return;
        }
    
    /* find the name of the current working directory and set the message */

    (void) getcwd(dummy_string,(size_t) MAX_STRING_LENGTH);
    xv_set(load_message, PANEL_LABEL_STRING, dummy_string, NULL);

    /* read its contents, one entry at a time */

    while((dirp = readdir(dp)) != NULL)
        {
        /* ignore . and .. - they will be added on at the end */
        if((strcmp(dirp->d_name,".") == 0) || (strcmp(dirp->d_name,"..") == 0))
            continue;

        /* get the info on the entry */
        if(lstat(dirp->d_name, &statbuf) < 0 )
            {
            printf("problem with lstat on file %s\n",dirp->d_name);
            continue;
            }

        /* 
           if the entry is a directory, add it to the scrolling list 
           with an extra slash at the end to indicate a directory
        */
        if(S_ISDIR(statbuf.st_mode))
            {
            sprintf(dummy_string,"%s/",dirp->d_name);
            xv_set(load_list, 
                      PANEL_LIST_INSERT,        0, 
                      PANEL_LIST_STRING,        0,    dummy_string,
                      PANEL_LIST_SELECT,        0,    FALSE,
                      NULL);
            }

        /* 
           if entry is a normal file, consider adding it to 
           the scrolling list 
        */
        if((statbuf.st_mode & S_IFMT) == S_IFREG)
            {
#ifdef DEBUG
            printf("%s is a regular file\n",dirp->d_name);
#endif
            /*
               check we can open the file for reading
            */
            if((fp = fopen(dirp->d_name,"r")) != NULL)
                {
                /*
                   read in the first line and search for the strings "cadellinsoft"
                   and "Database"
                */

                fgets(dummy_string, 255, fp);
                if((strstr(dummy_string,"cadellinsoft") != NULL) &&
                   (strstr(dummy_string,"Database") != NULL))
                    {
                    /* if found, add the file name to the scrolling list */
                    xv_set(load_list, 
                              PANEL_LIST_INSERT,        0, 
                              PANEL_LIST_STRING,        0,    dirp->d_name,
                              PANEL_LIST_SELECT,        0,    FALSE,
                              NULL);
                    }
                /* close the file after use */
                fclose(fp);
                }
            }
        }

    /* add . and .. - this ensures that they appear at the top of the list */

    xv_set(load_list, 
              PANEL_LIST_INSERT,        0, 
              PANEL_LIST_STRING,        0,    "../",
              PANEL_LIST_SELECT,        0,    FALSE,
              NULL);

    xv_set(load_list, 
              PANEL_LIST_INSERT,        0, 
              PANEL_LIST_STRING,        0,    "./",
              PANEL_LIST_SELECT,        0,    FALSE,
              NULL);

    /* close the current directory */

    if(closedir(dp) < 0)
        printf("unable to close current directory\n");

    /* unset the frame busy cursor */

    xv_set(load_frame, FRAME_BUSY, FALSE, NULL);

    /* open the "select a file to load the database from" scrolling menu */

    xv_set(load_frame, XV_SHOW, TRUE, NULL);
    }
 
if (!strcmp((char *)xv_get(menu_item, MENU_STRING), "Save"))
    {
    /* 
       check that there is a non-empty current file name, if
       OK call the save database function, otherwise call "Save as..."
       to get the user to select a file name
    */ 

    if(strcmp(file_name,""))
        save_database();
    else
        goto LEAP_FROG;
    
    }
 
if (!strcmp((char *)xv_get(menu_item, MENU_STRING), "Save as..."))
    {
    LEAP_FROG:

    /* find out how many elements currently in the scrolling list */

    num_choices = (int) xv_get(save_list, PANEL_LIST_NROWS);

#ifdef DEBUG
    printf("deleting %d elements from save file scrolling list\n",num_choices);
#endif

    /* set the frame busy cursor */

    xv_set(save_frame, FRAME_BUSY, TRUE, NULL);

    /* clear the current contents of the save file scrolling list */

    if(num_choices > 0)
        {
        for( i = num_choices - 1 ; i >= 0 ; i-- )
            xv_set(save_list, PANEL_LIST_DELETE, i, NULL);
        }

    /* open the current directory */

    if((dp = opendir(".")) == NULL)
        {
        printf("unable to open current directory\n");
        return;
        }
    
    /* find the name of the current working directory and set the message */

    (void) getcwd(dummy_string,(size_t) MAX_STRING_LENGTH);
    xv_set(save_message, PANEL_LABEL_STRING, dummy_string, NULL);

    /* read its contents, one entry at a time */

    while((dirp = readdir(dp)) != NULL)
        {
        /* ignore . and .. - they will be added at the end */
        if((strcmp(dirp->d_name, ".") == 0) || (strcmp(dirp->d_name, "..") == 0))
            continue;

        /* get the info on the entry */
        if(lstat(dirp->d_name, &statbuf) < 0 )
            {
            printf("problem with lstat on file %s\n",dirp->d_name);
            continue;
            }

        /* 
           if the entry is a directory, add it to the scrolling list 
           with an extra slash at the end to indicate a directory
        */
        if(S_ISDIR(statbuf.st_mode))
            {
            sprintf(dummy_string,"%s/",dirp->d_name);
            xv_set(save_list, 
                      PANEL_LIST_INSERT,        0, 
                      PANEL_LIST_STRING,        0,    dummy_string,
                      PANEL_LIST_SELECT,        0,    FALSE,
                      NULL);
            }

        /* if entry is a normal file, consider adding it to the scrolling list */
        if((statbuf.st_mode & S_IFMT) == S_IFREG)
            {
#ifdef DEBUG
            printf("%s is a regular file\n",dirp->d_name);
#endif
            /*
               check we can open the file for reading
            */
            if((fp = fopen(dirp->d_name,"r")) != NULL)
                {
#ifdef DEBUG
                printf("file opened\n");
#endif
                /*
                   read in the first line and search for the string "cadellinsoft"
                */

                fgets(dummy_string, 255, fp);
                if(strstr(dummy_string,"cadellinsoft") != NULL)
                    {
#ifdef DEBUG
                    printf("file is a database file\n");
#endif
                    /* if found, add the file name to the scrolling list */
                    xv_set(save_list, 
                              PANEL_LIST_INSERT,        0, 
                              PANEL_LIST_STRING,        0,    dirp->d_name,
                              PANEL_LIST_SELECT,        0,    FALSE,
                              NULL);

                    }
                /* close the file after use */
                fclose(fp);
                }
            }
        }

    /* add . and .. - this ensures that they appear at the top of the list */

    xv_set(save_list, 
              PANEL_LIST_INSERT,        0, 
              PANEL_LIST_STRING,        0,    "../",
              PANEL_LIST_SELECT,        0,    FALSE,
              NULL);

    xv_set(save_list, 
              PANEL_LIST_INSERT,        0, 
              PANEL_LIST_STRING,        0,    "./",
              PANEL_LIST_SELECT,        0,    FALSE,
              NULL);


    /* close the current directory */

    if(closedir(dp) < 0)
        printf("unable to close current directory\n");

    /* unset the frame busy cursor */

    xv_set(save_frame, FRAME_BUSY, FALSE, NULL);

    /* open the "select a file to save the database to" scrolling menu */

    xv_set(save_frame, XV_SHOW, TRUE, NULL);
    }
 
if (!strcmp((char *)xv_get(menu_item, MENU_STRING), "Properties..."))
    {
    /* display the settings menu */

    xv_set(settings_frame, XV_SHOW, TRUE, NULL);
    }

if (!strcmp((char *)xv_get(menu_item, MENU_STRING), "Exit"))
    {
    /* exit! */

    xv_destroy_safe(fdb_frame);

    }
}
 
/* edit menu selection callback */

void edit_menu_selection(menu, menu_item)
Menu       menu;
Menu_item  menu_item;
{
int    i, j, new_index, mother_index, father_index, spouse_index, selection;

/* ensure the text entries are up to date before a possible change of current person */

update_text();

#ifdef DEBUG
    printf("%s selected from edit menu\n", (char *)xv_get(menu_item, MENU_STRING));
#endif

if (!strcmp((char *)xv_get(menu_item, MENU_STRING), "Add Mother Entry"))
    {
    /*
        create a new entry marked as the mother of the current person
        and make this new entry the current entry
    */

    new_index = create_new_entry();

    /* if there was a problem, create_new_entry returns the current index */

    if(new_index != current_index)
        {
        /* 
            if the current person already had a mother, remove the
            current person from the old mothers list of children
        */
        if(entry[current_index]->mother != -1)
            {
            selection = 0;
            for(j=0;j<entry[entry[current_index]->mother]->number_of_children;j++)
                if(entry[entry[current_index]->mother]->child[j] == current_index)
                    selection = j;
            /* 
               check that the current person actually was a child of 
               its old mother
            */
            if(selection == entry[entry[current_index]->mother]->number_of_children)
                {
                /* indicate the problem */
                sprintf(error_message,"Database Inconsistency - Current Person Was Not a Child of Their Mother!");
                show_error();
                }
            else
                {
                /* 
                   remove the current person from the list of the old mothers 
                   children shunting the remaining children down in the array as necessary
                */

                if(selection < entry[entry[current_index]->mother]->number_of_children - 1)
                    {
                    for(i=selection;i<entry[entry[current_index]->mother]->number_of_children - 1;i++)
                        entry[entry[current_index]->mother]->child[i] = 
                                       entry[entry[current_index]->mother]->child[i+1];
                    } 
                entry[entry[current_index]->mother]->number_of_children -= 1;
                }
            }
    
        /* set the mother index in the current (childs) entry */
        entry[current_index]->mother = new_index;

        /* set the necessary links in the mothers entry */
        entry[new_index]->child[entry[new_index]->number_of_children] = current_index; 
        entry[new_index]->number_of_children += 1;

        /* make the new entry female */
        entry[new_index]->gender = FEMALE;

        /* make the mothers entry the current entry */
        current_index = new_index;

        /* update the display */
        update_panel_items();
        }
    }
 
if (!strcmp((char *)xv_get(menu_item, MENU_STRING), "Add Father Entry"))
    {
    /*
        create a new entry marked as the father of the current person
        and make this new entry the current entry
    */

    new_index = create_new_entry();

    /* if there was a problem, create_new_entry returns the current index */

    if(new_index != current_index)
        {
        /* 
            if the current person already had a father, remove the
            current person from the old fathers list of children
        */
        if(entry[current_index]->father != -1)
            {
            /* 
               now remove the current person from the list of the old
               fathers children
            */
            selection = 0;
            for(j=0;j<entry[entry[current_index]->father]->number_of_children;j++)
                if(entry[entry[current_index]->father]->child[j] == current_index)
                    selection = j;
            /* 
               check that the current person actually was a child of 
               its old father
            */
            if(selection == entry[entry[current_index]->father]->number_of_children)
                {
                /* indicate the problem */
                sprintf(error_message,"Database Inconsistency - Current Person Was Not a Child of Their Father!");
                show_error();
                }
            else
                {
                /* 
                   remove the current person from the list of the old fathers 
                   children shunting the remaining children down in the array as necessary
                */

                if(selection < entry[entry[current_index]->father]->number_of_children - 1)
                    {
                    for(i=selection;i<entry[entry[current_index]->father]->number_of_children - 1;i++)
                        entry[entry[current_index]->father]->child[i] = 
                                       entry[entry[current_index]->father]->child[i+1];
                    } 
                entry[entry[current_index]->father]->number_of_children -= 1;
                }
            }

        /* set the father index in the current (childs) entry */
        entry[current_index]->father = new_index;

        /* set the necessary links in the fathers entry */
        entry[new_index]->child[entry[new_index]->number_of_children] = current_index; 
        entry[new_index]->number_of_children += 1;

        /* make the new entry female */
        entry[new_index]->gender = MALE;

        /* make the fathers entry the current entry */
        current_index = new_index;

        /* update the display */
        update_panel_items();
        }
    }
 
if (!strcmp((char *)xv_get(menu_item, MENU_STRING), "Add Spouse Entry"))
    {
    /* check the number of spouses does not exceed the maximum */
    if((entry[current_index]->number_of_spouses + 1) == MAX_SPOUSES)
        {
        /* indicate the failure */
        strcpy(error_message,"Too Many Spouses! - Please Increment MAX_SPOUSES in definitions.h and Recompile!");
        show_error();
        return;
        }

    /*
        create a new entry marked as the spouse of the current person
        and make this new entry the current entry
    */

    new_index = create_new_entry();

    /* if there was a problem, create_new_entry returns the current index */

    if(new_index != current_index)
        {
        /* 
           increment the number of spouses and enter the index in the 
           spouse array 
        */
        entry[current_index]->spouse[entry[current_index]->number_of_spouses] = new_index;
        entry[current_index]->number_of_spouses += 1;

        /* set the necessary links in the spouses entry */
        entry[new_index]->spouse[entry[new_index]->number_of_spouses] = current_index; 
        entry[new_index]->number_of_spouses += 1;

        /* give the new entry the appropriate gender */
        if(entry[current_index]->gender == MALE)
            entry[new_index]->gender = FEMALE;
        else
            entry[new_index]->gender = MALE;

        /* make the spouses entry the current entry */
        current_index = new_index;

        /* update the display */
        update_panel_items();
        }
    }
 
if (!strcmp((char *)xv_get(menu_item, MENU_STRING), "Add Child Entry"))
    {
    /* check the number of children does not exceed the maximum */
    if(entry[current_index]->number_of_children == MAX_CHILDREN)
        {
        /* indicate the failure */
        strcpy(error_message,"Too Many Children! - Please Increment MAX_CHILDREN in definitions.h and Recompile!");
        show_error();
        return;
        }

    /*
        create a new entry marked as a child of the current person
        and make this new entry the current entry
    */

    new_index = create_new_entry();

    /* if there was a problem, create_new_entry returns the current index */

    if(new_index != current_index)
        {
        /* 
           increment the number of children and enter the index in the 
           child array 
        */
        entry[current_index]->child[entry[current_index]->number_of_children] = new_index;
        entry[current_index]->number_of_children += 1;

        /* 
           set the necessary links in the childs entry depending on whether
           the parent is female or male 
        */
        if(entry[current_index]->gender == 1)
            entry[new_index]->mother = current_index; 
        else
            entry[new_index]->father = current_index; 

        /* make the spouses entry the current entry */
        current_index = new_index;

        /* update the display */
        update_panel_items();
        }
    }
 
if (!strcmp((char *)xv_get(menu_item, MENU_STRING), "Add New Entry"))
    {
    /*
       create a new entry with no initial relationships with
       any other member of the database
    */
    
    new_index = create_new_entry();

    /* 
       set the current index equal to the new index, note that
       the two will be equal if there was a problem with creating
       a new index
    */

    current_index = new_index;

    /* update the display */

    update_panel_items();
    }

if (!strcmp((char *)xv_get(menu_item, MENU_STRING), "Delete Current Entry"))
    {
    /*
       make one of the current entries relatives the new current
       entry, if no relatives select the first entry in the database, if none
       create a new empty entry and make that the current entry
    */

    if(entry[current_index]->mother != -1)
        new_index = entry[current_index]->mother;
    else if(entry[current_index]->father != -1)
        new_index = entry[current_index]->father;
    else if(entry[current_index]->number_of_spouses > 0)
        new_index = entry[current_index]->spouse[0];
    else if(entry[current_index]->number_of_children > 0)
        new_index = entry[current_index]->child[0];
    else
        {
        for(i=0;i<=max_index;i++)
            {
            if((entry[i] != NULL) && (i != current_index))
                {
                new_index = i;
                break;
                }
            }
        if(i > max_index)
            {
            new_index = create_new_entry();
            }
        }
    /* remove any links to the current person from other entries */

    if(entry[current_index]->mother != -1)
        {
        /* 
           loop through the mothers children to find the position of the
           current person
        */
        mother_index = entry[current_index]->mother;
        selection = 0;
        for(i=0;i<entry[mother_index]->number_of_children;i++)
            if(entry[mother_index]->child[i] == current_index)
                selection = i;
        /* check that the current person actually was a child of its mother! */
        if(selection == entry[mother_index]->number_of_children)
            {
            /* indicate the problem */
            sprintf(error_message,"Database Inconsistency - Current Person Was Not a Child of its Mother!");
            show_error();
            }
        else
            {
            /* 
               remove the current person from the list of its mothers children 
               shunting the remaining children down in the array as necessary
            */

            if(selection < entry[mother_index]->number_of_children - 1)
                {
                for(i=selection;i<entry[mother_index]->number_of_children - 1;i++)
                    entry[mother_index]->child[i] = entry[mother_index]->child[i+1];
                } 
            entry[mother_index]->number_of_children -= 1;
            }
        }

    if(entry[current_index]->father != -1)
        {
        /* 
           loop through the fathers children to find the position of the
           current person
        */
        father_index = entry[current_index]->father;
        selection = 0;
        for(i=0;i<entry[father_index]->number_of_children;i++)
            if(entry[father_index]->child[i] == current_index)
                selection = i;
        /* check that the current person actually was a child of its father! */
        if(selection == entry[father_index]->number_of_children)
            {
            /* indicate the problem */
            sprintf(error_message,"Database Inconsistency - Current Person Was Not a Child of its Father!");
            show_error();
            }
        else
            {
            /* 
               remove the current person from the list of its fathers children 
               shunting the remaining children down in the array as necessary
            */

            if(selection < entry[father_index]->number_of_children - 1)
                {
                for(i=selection;i<entry[father_index]->number_of_children - 1;i++)
                    entry[father_index]->child[i] = entry[father_index]->child[i+1];
                } 
            entry[father_index]->number_of_children -= 1;
            }
        }

    /* 
        if there are any spouses, remove the current person from their
        list of spouses
    */

    for(i=0;i<entry[current_index]->number_of_spouses;i++)
        {
        /* 
           loop through the spouses spouses (sic) to find the position of the
           current person
        */
        spouse_index = entry[current_index]->spouse[i];
        selection = 0;
        for(j=0;j<entry[spouse_index]->number_of_spouses;j++)
            if(entry[spouse_index]->spouse[j] == current_index)
                selection = j;
        /* check that the current person actually was a spouse of its spouse! */
        if(selection == entry[spouse_index]->number_of_spouses)
            {
            /* indicate the problem */
            sprintf(error_message,"Database Inconsistency - Current Person Was Not a Spouse of its Spouse!");
            show_error();
            }
        else
            {
            /* 
               remove the current person from the list of its spouses spouses 
               shunting the remaining spouses down in the array as necessary
            */
            if(selection < entry[spouse_index]->number_of_spouses - 1)
                {
                for(j=selection;j<entry[spouse_index]->number_of_spouses - 1;j++)
                    entry[spouse_index]->spouse[j] = entry[spouse_index]->spouse[j+1];
                } 
            entry[spouse_index]->number_of_spouses -= 1;
            }
        }
        
    /* 
        if there are any children, remove the current person from the
        appropriate parent entry (mother or father depending on the
        gender of the current person)
    */

    for(i=0;i<entry[current_index]->number_of_children;i++)
        {
        if(entry[current_index]->gender == MALE)  /* person is a father */
            {
            /* check that current entry is the father of his children! */
            if(entry[entry[current_index]->child[i]]->father != current_index)
                {
                /* indicate the problem */
                sprintf(error_message,"Database Inconsistency - Current Person was not the Father of his Children!");
                show_error();
                }
            else
                {
                /* remove the reference to the current person */
                entry[entry[current_index]->child[i]]->father = -1;
                }
            }
        else                                      /* person is a mother */
            {
            /* check that current entry is the mother of her children! */
            if(entry[entry[current_index]->child[i]]->mother != current_index)
                {
                /* indicate the problem */
                sprintf(error_message,"Database Inconsistency - Current Person was not the Mother of her Children!");
                show_error();
                }
            else
                {
                /* remove the reference to the current person */
                entry[entry[current_index]->child[i]]->mother = -1;
                }
            }
        }
    
    /* delete the specified entry */
    free(entry[current_index]);
    entry[current_index] = (struct db_record *) NULL;

    /* reduce the number of entries in the database counter */

    num_entries--;

    /* create the new status display string and set it */

    if(num_entries == 1)
        strcpy(status_string,"Currently 1 Entry in Database");
    else
        sprintf(status_string,"Currently %d Entries in Database",num_entries);

    xv_set(fdb_frame, FRAME_RIGHT_FOOTER, status_string, NULL);

    /* recalculate the max index in use (just in case) */
 
    for(i=0;i<MAX_INDICES;i++)
        if(entry[i] != NULL)
            max_index = i;

    /* set the new current entry */
    current_index = new_index;

    /* set the default file menu selection to save since something has changed */

    xv_set(file_menu, MENU_DEFAULT, 3, NULL);

    /* update the display */
    update_panel_items();
    }
}

/* select menu selection callback */

void select_menu_selection(menu, menu_item)
Menu       menu;
Menu_item  menu_item;
{
/* ensure the text entries are up to date before a possible change of current person */

update_text();

#ifdef DEBUG
    printf("%s selected from select menu\n", (char *)xv_get(menu_item, MENU_STRING));
#endif

if (!strcmp((char *)xv_get(menu_item, MENU_STRING), "Select Mother Entry"))
    {
    /*
        set a person as the mother of the current person
        amend the current person and mother appropriately
    */
    /* set the title of the entries list frame */

    xv_set(entries_frame, FRAME_LABEL, "Select Mother Entry", NULL);

    /* 
       set the selection type so that the list selection callback
       knows what we are selecting
    */ 

    selection_nature = SELECTION;
    selection_type   = MOTHER;

#ifdef DEBUG
    printf("selection_type set to %d for Mother Selection\n",selection_type);
#endif

    /* have the user select a mother */

    select_entry(FEMALE);

    return;
    }
 
if (!strcmp((char *)xv_get(menu_item, MENU_STRING), "Select Father Entry"))
    {
    /*
        set a person as the father of the current person
        amend the current person and father appropriately
    */
    /* set the title of the entries list frame */

    xv_set(entries_frame, FRAME_LABEL, "Select Father Entry", NULL);

    /* 
       set the selection type so that the list selection callback
       knows what we are selecting
    */ 

    selection_nature = SELECTION;
    selection_type   = FATHER;

#ifdef DEBUG
    printf("selection_type set to %d for Father Selection\n",selection_type);
#endif

    /* have the user select a father */

    select_entry(MALE);

    return;
    }
 
if (!strcmp((char *)xv_get(menu_item, MENU_STRING), "Select Spouse Entry"))
    {
    /*
        set a person as the spouse of the current person
        amend the current person and spouse appropriately
    */
    /* set the title of the entries list frame */

    xv_set(entries_frame, FRAME_LABEL, "Select Spouse Entry", NULL);

    /* 
       set the selection type so that the list selection callback
       knows what we are selecting
    */ 

    selection_nature = SELECTION;
    selection_type   = SPOUSE;

#ifdef DEBUG
    printf("selection_type set to %d for Spouse Selection\n",selection_type);
#endif

    /* have the user select a spouse of the appropriate gender */

    if(entry[current_index]->gender == MALE)
        select_entry(FEMALE);
    else
        select_entry(MALE);

    return;
    }
 
if (!strcmp((char *)xv_get(menu_item, MENU_STRING), "Select Child Entry"))
    {
    /*
        set a person as a child of the current person
        amend the current person and child appropriately
    */
    /* set the title of the entries list frame */

    xv_set(entries_frame, FRAME_LABEL, "Select Child Entry", NULL);

    /* 
       set the selection type so that the list selection callback
       knows what we are selecting
    */ 

    selection_nature = SELECTION;
    selection_type   = CHILD;

#ifdef DEBUG
    printf("selection_type set to %d for Child Selection\n",selection_type);
#endif

    /* have the user select a child */

    select_entry(ANY_GENDER);

    return;
    }

if (!strcmp((char *)xv_get(menu_item, MENU_STRING), "Select New Current Entry"))
    {
    /*
       display all persons not related to the current person to allow
       one to be selected to be the new current person
    */ 
    /* set the title of the entries list frame */

    xv_set(entries_frame, FRAME_LABEL, "Select New Current Entry", NULL);

    /* 
       set the selection type so that the list selection callback
       knows what we are selecting
    */ 

    selection_nature = SELECTION;
    selection_type   = ANYBODY;

    /* have the user select a person */

    select_entry(ANY_GENDER);

    return;
    }
}

/* deselect menu selection callback */

void deselect_menu_selection(menu, menu_item)
Menu       menu;
Menu_item  menu_item;
{
int    i, j, selection, mother_index, father_index, spouse_index;

/* ensure the text entries are up to date (just to be on the safe side) */

update_text();

#ifdef DEBUG
    printf("%s selected from deselect menu\n", (char *)xv_get(menu_item, MENU_STRING));
#endif

if (!strcmp((char *)xv_get(menu_item, MENU_STRING), "Deselect Mother Entry"))
    {
    /* if the current person has a mother, deselect her */
    if(entry[current_index]->mother != -1)
        {
        /* 
           loop through the mothers children to find the position of the
           current person
        */
        mother_index = entry[current_index]->mother;
        selection = 0;
        for(i=0;i<entry[mother_index]->number_of_children;i++)
            if(entry[mother_index]->child[i] == current_index)
                selection = i;
        /* check that the current person actually was a child of its mother! */
        if(selection == entry[mother_index]->number_of_children)
            {
            /* indicate the problem */
            sprintf(error_message,"Database Inconsistency - Current Person Was Not a Child of its Mother!");
            show_error();
            goto amend_m_records;
            }
        /* 
           remove the current person from the list of its mothers children 
           shunting the remaining children down in the array as necessary
        */
        if(selection < entry[mother_index]->number_of_children - 1)
            {
            for(i=selection;i<entry[mother_index]->number_of_children - 1;i++)
                entry[mother_index]->child[i] = entry[mother_index]->child[i+1];
            } 
        entry[mother_index]->number_of_children -= 1;
        }
    else
        {
        /* if no mother to deselect, just return */
        return;
        }

    amend_m_records:

    /* amend the current persons record */

    entry[current_index]->mother = -1;

    }
 
if (!strcmp((char *)xv_get(menu_item, MENU_STRING), "Deselect Father Entry"))
    {
    /*
        deselect the father of the current person
        amend the current person and father appropriately
    */
    /* if the current person has a father, deselect him */
    if(entry[current_index]->father != -1)
        {
        /* 
           loop through the fathers children to find the position of the
           current person
        */
        father_index = entry[current_index]->father;
        selection = 0;
        for(i=0;i<entry[father_index]->number_of_children;i++)
            if(entry[father_index]->child[i] == current_index)
                selection = i;
        /* check that the current person actually was a child of its father! */
        if(selection == entry[father_index]->number_of_children)
            {
            /* indicate the problem */
            sprintf(error_message,"Database Inconsistency - Current Person Was Not a Child of its Father!");
            show_error();
            goto amend_f_records;
            }
                 
        /* 
           remove the current person from the list of its fathers children 
           shunting the remaining children down in the array as necessary
        */
        if(selection < entry[father_index]->number_of_children - 1)
            {
            for(i=selection;i<entry[father_index]->number_of_children - 1;i++)
                entry[father_index]->child[i] = entry[father_index]->child[i+1];
            } 
        entry[father_index]->number_of_children -= 1;
        }
    else
        {
        /* if no father to deselect, just return */
        return;
        }

    amend_f_records:

    /* amend the current persons record */

    entry[current_index]->father = -1;

    }
 
if (!strcmp((char *)xv_get(menu_item, MENU_STRING), "Deselect Spouse Entry"))
    {
    /*
        only need to do anything if there is 1 or more spouse
    */
    if(entry[current_index]->number_of_spouses > 0)
        {
        if(entry[current_index]->number_of_spouses == 1)
            {
            /* 
               only one spouse so deselect them! 
            */
            /* 
               loop through the spouses spouses (sic) to find the position of the
               current person
            */
            spouse_index = entry[current_index]->spouse[0];
            selection = 0;
            for(j=0;j<entry[spouse_index]->number_of_spouses;j++)
                if(entry[spouse_index]->spouse[j] == current_index)
                    selection = j;
            /* check that the current person actually was a spouse of its spouse! */
            if(selection == entry[spouse_index]->number_of_spouses)
                {
                /* indicate the problem */
                sprintf(error_message,"Database Inconsistency - Current Person Was Not a Spouse of its Spouse!");
                show_error();
                }
            else
                {
                /* 
                   remove the current person from the list of its spouses spouses 
                   shunting the remaining spouses down in the array as necessary
                */

                if(selection < entry[spouse_index]->number_of_spouses - 1)
                    {
                    for(i=selection;i<entry[spouse_index]->number_of_spouses - 1;i++)
                        entry[spouse_index]->spouse[i] = entry[spouse_index]->spouse[i+1];
                    } 
                entry[spouse_index]->number_of_spouses -= 1;
                }
            entry[current_index]->number_of_spouses = 0;
            }
        else
            {
            /* 
               more than one spouse so need to display a list
            */
            /* set the title of the entries list frame */

            xv_set(entries_frame, FRAME_LABEL, "Deselect Spouse Entry", NULL);

            /* set the nature and type of the selection */

            selection_nature = DESELECTION;
            selection_type   = SPOUSE;

            deselect_entry(SPOUSE);
            }
        }
    else
        {
        /* if no spouses to deselect, just return */
        return;
        }
    }
 
if (!strcmp((char *)xv_get(menu_item, MENU_STRING), "Deselect Child Entry"))
    {
    /*
        only need to do anything if there is 1 or more child
    */
    if(entry[current_index]->number_of_children > 0)
        {
        if(entry[current_index]->number_of_children == 1)
            {
            /* 
               only one child so deselect them! 
            */
            /* check that the current person actually was the parent of its child! */

            if(((entry[current_index]->gender == MALE) && 
                  (entry[entry[current_index]->child[0]]->father != current_index)) || 
                ((entry[current_index]->gender == FEMALE) && 
                  (entry[entry[current_index]->child[0]]->mother != current_index)))
                {
                /* indicate the problem */
                sprintf(error_message,"Database Inconsistency - Current Person Was Not a Parent of its Child!");
                show_error();
                }
            else
                {
                /* 
                   remove the current person from the appropriate parent 
                   entry of the child
                */
                if(entry[current_index]->gender == MALE)
                    entry[entry[current_index]->child[0]]->father = -1;
                else
                    entry[entry[current_index]->child[0]]->mother = -1;
                }
            entry[current_index]->number_of_children = 0;
            }
        else
            {
            /* 
               more than one child so need to display a list
            */
            /* set the title of the entries list frame */

            xv_set(entries_frame, FRAME_LABEL, "Deselect Child Entry", NULL);

            /* set the nature and type of the selection */

            selection_nature = DESELECTION;
            selection_type   = CHILD;

            deselect_entry(CHILD);
            }
        }
    else
        {
        /* if no children to deselect, just return */
        return;
        }
    }

/* set the default file menu selection to save since something has changed */

xv_set(file_menu, MENU_DEFAULT, 3, NULL);

/* update the display */

update_panel_items();

}

/* output menu selection callback */

void output_menu_selection(menu, menu_item)
Menu       menu;
Menu_item  menu_item;
{
char    working[MAX_FORENAMES], substring[MAX_FORENAMES];
char    filename[MAX_SURNAME + 50], test_filename[MAX_SURNAME + 50], initial[2];
char   *ptr;
int     i, j, k, l, initialise, num_items;
FILE   *fp;

/* ensure the text entries are up to date before a possible output of data */

update_text();

#ifdef DEBUG
    printf("%s selected from output menu\n", (char *)xv_get(menu_item, MENU_STRING));
#endif

if (!strcmp((char *)xv_get(menu_item, MENU_STRING), "Output Current Person") &&
             (output_type_flag == LATEX))
    {
    /*
        output the current persons data to a file in LaTeX format
    */

    /* initialise the file name string */
    strcpy(filename,"");

    /* read the persons forename string into the work space, converting to upper case on the way */

    i = 0;
    while((working[i] = toupper(entry[current_index]->forenames[i])) != '\0')
        i++;

#ifdef DEBUG
    printf("forename string: %s\n",working);
#endif

    /* read in substrings and extract the first letters */

    j = 0;

    while( ((num_items = sscanf((working + j),"%s",substring)) != EOF) &&
           (j < (int) strlen(working)) )
        {
#ifdef DEBUG
        printf("working on substring: %s\n",substring);
#endif

        sprintf(initial,"%c",substring[0]);

#ifdef DEBUG
        printf("first letter: **%s**\n",initial);
#endif
        /* add the initial to the file name */

        strcat(filename,initial);

        /* add an underscore */

        strcat(filename,"_");
	
        /* increment the distance along the string counter */

        j += strlen(substring) + 1;
#ifdef DEBUG
        printf("j = %d\n",j);
#endif
        }

    /* add the surname on the end, ensuring the first letter is capitalised */

    sscanf(entry[current_index]->surname,"%s",working);
    working[0] = toupper(working[0]);

    /* append the surname to the file name */

    strcat(filename,working);

    /* 
       ensure that file does not already exist, add an integer to the
       file name to make it unique if necessary 
    */

    sprintf(test_filename,"%s.tex",filename);

    /* check if the file already exists */

    j = 1;
    while((fp = fopen(test_filename,"r")) != NULL)
        {
        sprintf(test_filename,"%s%d.tex",filename,j);
        j++;
        }

    strcpy(filename, test_filename);

#ifdef DEBUG
    printf("file name: %s\n",filename);
#endif

    /* open the file for writing and write the initial LaTeX header info */

    if((fp = fopen(filename,"w")) == NULL)
        {
        /* if problems create error message and return */
        sprintf(error_message,"unable to open file %s for writing",filename);
        show_error();
        return;
        }

    /* display the name of the file created */
    sprintf(error_message,"LaTeX plot created in file %s",filename);
    show_error();

    fprintf(fp,"%% generated by fdb V%s\n\n", FDB_VERSION);

    fprintf(fp,"\\documentstyle[11pt,a4,doublespace]{article}\n\n");

    fprintf(fp,"\\setlength{\\textwidth}{6.0in}\n");
    fprintf(fp,"\\setlength{\\oddsidemargin}{0.3in}\n");
    fprintf(fp,"\\setlength{\\evensidemargin}{-0.3in}\n");
    fprintf(fp,"\\setlength{\\topmargin}{-0.5in}\n");
    fprintf(fp,"\\setlength{\\textheight}{9.5in}\n");
    fprintf(fp,"\\setstretch{1.4}\n");
    fprintf(fp,"\\parindent 0 pt\n\n");

    fprintf(fp,"\\begin{document}\n");

    /* call the LaTeX output function for the current person */

    latex_output(fp, current_index);

    /* end the document and close the file */

    fprintf(fp,"\\end{document}\n");

    fclose(fp);
    return;
    }
 
if (!strcmp((char *)xv_get(menu_item, MENU_STRING), "Output Current Person") &&
            (output_type_flag == POSTSCRIPT))
    {
    /*
        output the current persons data to a file in PostScript format
    */

    /* initialise the file name string */
    strcpy(filename,"");

    /* read the persons forename string into the work space, converting to upper case on the way */

    i = 0;
    while((working[i] = toupper(entry[current_index]->forenames[i])) != '\0')
        i++;

#ifdef DEBUG
    printf("forename string: %s\n",working);
#endif

    /* read in substrings and extract the first letters */

    j = 0;

    while( ((num_items = sscanf((working + j),"%s",substring)) != EOF) &&
           (j < (int) strlen(working)) )
        {
#ifdef DEBUG
        printf("working on substring: %s\n",substring);
#endif

        sprintf(initial,"%c",substring[0]);

#ifdef DEBUG
        printf("first letter: **%s**\n",initial);
#endif
        /* add the initial to the file name */

        strcat(filename,initial);

        /* add an underscore */

        strcat(filename,"_");
	
        /* increment the distance along the string counter */

        j += strlen(substring) + 1;
#ifdef DEBUG
        printf("j = %d\n",j);
#endif
        }

    /* add the surname on the end, ensuring the first letter is capitalised */

    sscanf(entry[current_index]->surname,"%s",working);
    working[0] = toupper(working[0]);

    /* append the surname to the file name */

    strcat(filename,working);

    /* 
       ensure that file does not already exist, add an integer to the
       file name to make it unique if necessary 
    */

    sprintf(test_filename,"%s.ps",filename);

    /* check if the file already exists */

    j = 1;
    while((fp = fopen(test_filename,"r")) != NULL)
        {
        sprintf(test_filename,"%s%d.ps",filename,j);
        j++;
        }

    strcpy(filename, test_filename);

#ifdef DEBUG
    printf("file name: %s\n",filename);
#endif

    /* open the file for writing and write the initial Postscript header info */

    if((fp = fopen(filename,"w")) == NULL)
        {
        /* if problems create error message and return */
        sprintf(error_message,"unable to open file %s for writing",filename);
        show_error();
        return;
        }

    /* initialise the PostScript file with header info */

    postscript_init(fp);

    /* call the PostScript output function for the current person */

    postscript_output(fp, current_index);

    /* end the document and close the file */

    fprintf(fp,"\n\nshowpage\n\n");

    fclose(fp);

    /* display the name of the file created */

    sprintf(error_message,"PostScript plot created in file %s",filename);
    show_error();

    return;
    }
 
if (!strcmp((char *)xv_get(menu_item, MENU_STRING), "Output Entire Database") &&
            (output_type_flag == LATEX))
    {
    /*
        output the data on each person in the database, one to a page,
        in LaTeX format
    */

    /* initialise the file name string to be the current database file name */

    strcpy(filename,file_name);

    /* if there is a trailing .fdb, remove it */

    if((ptr = strstr(filename,".fdb")) != NULL)
        *ptr = '\0';

    /* 
       ensure that file does not already exist, add an integer to the
       file name to make it unique if necessary 
    */

    sprintf(test_filename,"%s.tex",filename);

    /* check if the file already exists */

    j = 1;
    while((fp = fopen(test_filename,"r")) != NULL)
        {
        sprintf(test_filename,"%s%d.tex",filename,j);
        j++;
        }

    strcpy(filename, test_filename);

#ifdef DEBUG
    printf("file name: %s\n",filename);
#endif

    /* open the file for writing and write the initial LaTeX header info */

    if((fp = fopen(filename,"w")) == NULL)
        {
        /* if problems create error message and return */
        sprintf(error_message,"unable to open file %s for writing",filename);
        show_error();
        return;
        }

    /* display the name of the file created */
    sprintf(error_message,"LaTeX plot created in file %s",filename);
    show_error();

    fprintf(fp,"\\documentstyle[11pt,a4,doublespace]{article}\n\n");

    fprintf(fp,"\\setlength{\\textwidth}{6.0in}\n");
    fprintf(fp,"\\setlength{\\oddsidemargin}{0.3in}\n");
    fprintf(fp,"\\setlength{\\evensidemargin}{-0.3in}\n");
    fprintf(fp,"\\setlength{\\topmargin}{-0.5in}\n");
    fprintf(fp,"\\setlength{\\textheight}{9.5in}\n");
    fprintf(fp,"\\setstretch{1.4}\n");
    fprintf(fp,"\\parindent 0 pt\n\n");

    fprintf(fp,"\\begin{document}\n");

    /* title page and table of contents */

    fprintf(fp,"\\vspace*{75mm}\n");
    fprintf(fp,"\\Huge\n");
    fprintf(fp,"\\begin{center}\n");
    fprintf(fp,"FDB Output\n");
    fprintf(fp,"\\end{center}\n");
    fprintf(fp,"\\normalsize\n");
    fprintf(fp,"\\newpage\n\n");
    fprintf(fp,"\\tableofcontents\n");

    /* call the LaTeX output function for each person in turn */

    for(i=0;i<MAX_INDICES;i++)
        {
        if(entry[i] != (struct db_record *) NULL)
            {
            fprintf(fp,"\\newpage\n\n");
            latex_output(fp, i);
            }
        }

    /* end the document and close the file */

    fprintf(fp,"\\end{document}\n");

    fclose(fp);
    return;

    }

if (!strcmp((char *)xv_get(menu_item, MENU_STRING), "Output Entire Database") &&
            (output_type_flag == POSTSCRIPT))
    {
    /*
        output the data on each person in the database, one to a page,
        in PostScript format
    */

    /* initialise the file name string to be the current database file name */

    strcpy(filename,file_name);

    /* if there is a trailing .fdb, remove it */

    if((ptr = strstr(filename,".fdb")) != NULL)
        *ptr = '\0';

    /* 
       ensure that file does not already exist, add an integer to the
       file name to make it unique if necessary 
    */

    sprintf(test_filename,"%s.ps",filename);

    /* check if the file already exists */

    j = 1;
    while((fp = fopen(test_filename,"r")) != NULL)
        {
        sprintf(test_filename,"%s%d.ps",filename,j);
        j++;
        }

    strcpy(filename, test_filename);

#ifdef DEBUG
    printf("file name: %s\n",filename);
#endif

    /* open the file for writing */

    if((fp = fopen(filename,"w")) == NULL)
        {
        /* if problems create error message and return */
        sprintf(error_message,"unable to open file %s for writing",filename);
        show_error();
        return;
        }

    /* initialise the PostScript file with header info */

    postscript_init(fp);

    /* call the PostScript output function for each person in turn */

    for(i=0;i<MAX_INDICES;i++)
        {
        if(entry[i] != (struct db_record *) NULL)
            {
            postscript_output(fp, i);
            fprintf(fp,"showpage\n");
            }
        }

    /* close the file */

    fclose(fp);

    /* display the name of the file created */

    sprintf(error_message,"PostScript plot created in file %s",filename);
    show_error();

    return;
    }

if (!strcmp((char *)xv_get(menu_item, MENU_STRING), "Output Current Descent Line"))
    {
    /*
       output the currently displayed descent line in PostScript format
    */
    xv_set(fdb_frame, FRAME_BUSY, TRUE, NULL);
    plot_gen_tree();
    xv_set(fdb_frame, FRAME_BUSY, FALSE, NULL);
    }

if (!strcmp((char *)xv_get(menu_item, MENU_STRING), "Output Current Descendants Tree"))
    {
    /*
       output the currently displayed family tree in PostScript format
    */
    xv_set(fdb_frame, FRAME_BUSY, TRUE, NULL);
    plot_des_tree();
    xv_set(fdb_frame, FRAME_BUSY, FALSE, NULL);
    }
}

/* view database menu selection callback */

void view_menu_selection(menu, menu_item)
Menu       menu;
Menu_item  menu_item;
{
Xv_Window   temp_window;
Xv_object   return_code;
int         i;

/* ensure the text entries are up to date before a possible display of data */

update_text();

#ifdef DEBUG
    printf("%s selected from view menu\n", (char *)xv_get(menu_item, MENU_STRING));
#endif

if (!strcmp((char *)xv_get(menu_item, MENU_STRING), "View Male Descent Line"))
    {
    /* set the frame cursor to busy */

    xv_set(fdb_frame, FRAME_BUSY, TRUE, NULL);

    /* build the descent line data structure */

    build_line(MALE);

    /*
       show the descent display window
    */

    if((return_code = xv_set(gen_frame, XV_SHOW, TRUE, NULL)) != XV_OK)
        {
        printf("unable to show the 'Descent Line Display' window\n");
        }

    /*
       call the redraw function directly for each view
    */

    i = 0;
    while(temp_window = (Xv_window) xv_get(gen_canvas, CANVAS_NTH_PAINT_WINDOW, i))
        {
        draw_line(gen_canvas, temp_window,
                  (Display *) xv_get(temp_window, XV_DISPLAY),
                  xv_get(temp_window, XV_XID), (Xv_xrectlist *) NULL);
        i++;
        }

    /* set the frame cursor to not busy */

    xv_set(fdb_frame, FRAME_BUSY, FALSE, NULL);

    }

if (!strcmp((char *)xv_get(menu_item, MENU_STRING), "View Female Descent Line"))
    {
    /* set the frame cursor to busy */

    xv_set(fdb_frame, FRAME_BUSY, TRUE, NULL);

    /* build the descent line data structure */

    build_line(FEMALE);

    /*
       show the descent display window
    */

    if((return_code = xv_set(gen_frame, XV_SHOW, TRUE, NULL)) != XV_OK)
        {
        printf("unable to show the 'Descent Display Line' window\n");
        }

    /*
       call the redraw function directly for each view
    */

    i = 0;
    while(temp_window = (Xv_window) xv_get(gen_canvas, CANVAS_NTH_PAINT_WINDOW, i))
        {
        draw_line(gen_canvas, temp_window,
                  (Display *) xv_get(temp_window, XV_DISPLAY),
                  xv_get(temp_window, XV_XID), (Xv_xrectlist *) NULL);
        i++;
        }

    /* set the frame cursor to not busy */

    xv_set(fdb_frame, FRAME_BUSY, FALSE, NULL);

    }

if (!strcmp((char *)xv_get(menu_item, MENU_STRING), "View Descendants Tree"))
    {

    /* set the frame cursor to busy */

    xv_set(fdb_frame, FRAME_BUSY, TRUE, NULL);

    /* recursively build the descendants tree structure */

    build_descendants();

    /*
       show the descendants display window
    */

    if((return_code = xv_set(des_frame, XV_SHOW, TRUE, NULL)) != XV_OK)
        {
        printf("unable to show the 'Descendants Display Tree' window\n");
        }

    /*
       call the redraw function directly for each view
    */

    i = 0;
    while(temp_window = (Xv_window) xv_get(des_canvas, CANVAS_NTH_PAINT_WINDOW, i))
        {
        draw_descendants(des_canvas, temp_window,
                  (Display *) xv_get(temp_window, XV_DISPLAY),
                  xv_get(temp_window, XV_XID), (Xv_xrectlist *) NULL);
        i++;
        }

    /* set the frame cursor to not busy */

    xv_set(fdb_frame, FRAME_BUSY, FALSE, NULL);
    }
}

