// QSHMEM.C - An OS/2 demonstration program for passing variable-sized messages
//   messages over queues using shared memory.  Uses a single server servicing
//   multiple, semaphore-attached queues. Clients pass variable-sized messages
//   using pointers into shared memory.
#define INCL_BASE
#define INCL_DOSPROCESS
#include <os2.h>
#include <stdio.h>
#include <string.h>
#include <time.h>
#include <stdlib.h>

#define MAXQUEUES 16
#define MAXFILENAME 80
#define MAXTIMEOUT 10000L

typedef struct {
    PID client_pid[MAXQUEUES];  // Client puts his pid here.
    char szMuxSem[MAXFILENAME]; // Name of the mux sem.
    char szQueue[MAXQUEUES][MAXFILENAME];   // Names of the client queues.
    char szSem[MAXQUEUES][MAXFILENAME]; // Names of queue semaphores.
    SEMRECORD Sem[MAXQUEUES];   // semaphore array to multiplex.
    char data[MAXQUEUES][4096];   // Buffer area messages are alloced from.
    } SHAREMEM;
#define SHAREMEM_SIZE sizeof(SHAREMEM)
REQUESTDATA rd; // buffer used by readqueue.
char *ptr;  // generic pointer.
ULONG length;
int i, j;
PSZ szShareMem = "\\SHAREMEM\\QSRV";    // Name of shared memory.
SHAREMEM *pShareMem;    // Pointer to base of shared memory.
// Base of queue name, queues become <base>.<index> where index
// is the clients index into the shared memory array.
PSZ szQueueBase = "\\QUEUES\\QSRV";
// Base of semaphore name, semaphores become <base>.<index> where index
// is the clients index into the shared memory array.
PSZ szSemBase = "\\SEM32\\QSRV";
PSZ szMuxSem = "\\SEM32\\QSRV.MUX"; // Name of multiplex semaphore.
HMUX hMux;      // Handle to multiplexed semaphore.
HQUEUE hQ[MAXQUEUES];   // Client queue handles live here, client
            // process uses hQ[0] for its single queue.
ULONG ulPostCount;
APIRET rc;
char *pReadData;
UCHAR ucPriority;
PTIB ptib;      // Thread info block
PPIB ppib;      // Process info block, where pid lives
PID srvPid, mypid;  // various process ids.
ULONG ulUser;

int DoClient( void );
int DoServer( void );

// main - Decide what type of process we are, call either client or server.
int main( int argc, char *argv[] )
{
DosGetInfoBlocks( &ptib, &ppib );
mypid = ppib->pib_ulpid;
if( strcmp( argv[1], "-c" ) == 0 )
    DoClient();
else
    DoServer();
return( 0 );
}
// DoClient - Attach to shared memory, claim a slot in the pid[] array,
//      open the queue for that slot, then write 100 messages into that queue.
int DoClient()
{
int MyIndex;
printf( "Started client process %lu.\n", mypid );
for( i = 0; i <= 30; i++ )
    {
    if( i == 30 )
        {
        printf( "Client attach error %d\n", rc );
        goto AttachError;
        }
    if(( rc = DosGetNamedSharedMem( (PVOID *)&pShareMem, szShareMem, 
                                                   PAG_READ|PAG_WRITE )) == 0 )
        break;
    else
        DosSleep( 1000L );
    }
printf( "Opened shared memory %p!\n", pShareMem );
for( i = 0; i <= MAXQUEUES; i++ )
    {
    if( i == MAXQUEUES )
        {
        printf( "Maximum clients (%d) reached!\n", i );
        goto GetQError;
        }
    if( pShareMem->client_pid[i] == 0 )
        {
        MyIndex = i;
        pShareMem->client_pid[i] = mypid;
        break;
        }
    }
if(( rc = DosOpenQueue( &srvPid, &hQ[0], pShareMem->szQueue[MyIndex])) != 0 )
    {
    printf( "Client get q error %d\n", rc );
    goto GetQError;
    }
printf( "Client opened queue %lu\n", hQ[0] );
// We need this Sleep in here so that multiple instances of the
// client seed the randomizer with different values.
DosSleep( (MyIndex+1)*1000L );
randomize();
ptr = pShareMem->data[MyIndex];
for( i = 0; i < 100; i++ )
    {
    sprintf( ptr, "Client %d msg %d", MyIndex, i );
    if(( rc = DosWriteQueue( hQ[0], 0L, (ULONG )strlen( ptr ), 
                                                     (PVOID )ptr, 0L )) != 0 )
        {
        printf( "Client Write error %d\n", rc );
        goto WriteError;
        }
    printf( "CLIENT[%d] delivered: %s\n", MyIndex, ptr );
    j = random( (INT )MAXTIMEOUT );
    DosSleep( (ULONG )j );
    ptr += strlen( ptr );
    }
DosCloseQueue( hQ[0] );
DosFreeMem( pShareMem );
return( 0 );
WriteError:
    DosCloseQueue( hQ[0] );
GetQError:
    DosFreeMem( pShareMem );
AttachError:
return( 1 );
}
// DoServer - Create the shared memory block, create all the client queues,
//   create a semaphore for each queue, attach the semaphores to the queues,
//   then create a multiplex semaphore from all the individual semaphores. Sit
//   blocking on multiplex sem, reading all queues whenever a semaphore clears.
int DoServer()
{
int MessageCount;
 printf( "Started server process %lu.\n", mypid );
    if(( rc = DosAllocSharedMem( (PVOID *)&pShareMem, szShareMem, 
                         SHAREMEM_SIZE, PAG_READ|PAG_WRITE|PAG_COMMIT )) != 0 )
        {
        printf( "Server alloc failed %d\n", rc );
        goto AllocError;
        }
    printf( "Server alloced share mem at %p\n", pShareMem );
    memset( pShareMem, '\0', SHAREMEM_SIZE );
    for( i = 0; i < MAXQUEUES; i++ )
        {
        sprintf( pShareMem->szQueue[i], "%s.%d", szQueueBase, i );
        if(( rc = DosCreateQueue( &hQ[i], 0L, 
                                               pShareMem->szQueue[i] )) != 0 )
            {
            printf( "Server queue create failed %d\n", rc );
            goto CreateError;
            }
        printf( "Server created Q (%s) handle (%lu)\n", 
                                                pShareMem->szQueue[i], hQ[i] );
        sprintf( pShareMem->szSem[i], "%s.%d", szSemBase, i );
        if(( rc = DosCreateEventSem( pShareMem->szSem[i], 
                          (HEV *)&pShareMem->Sem[i].hsemCur, 0, FALSE )) != 0 )
            {
            printf( "Server create semaphore error %d\n", rc );
            goto CreateError;
            }
        }
    strcpy( pShareMem->szMuxSem, szMuxSem );
    if(( rc = DosCreateMuxWaitSem( pShareMem->szMuxSem, &hMux, MAXQUEUES, 
                                        pShareMem->Sem, DCMW_WAIT_ANY )) != 0 )
        {
        printf( "Server create mux semaphore error %d\n", rc );
        goto CreateError;
        }
    // Attach the semaphores to the queues
    for( i = 0; i < MAXQUEUES; i++ )
        {
        if(( rc = DosReadQueue( hQ[i], &rd, &length, 
                           (void **)&pReadData, 0L, DCWW_NOWAIT, &ucPriority, 
                           (HEV )pShareMem->Sem[i].hsemCur )) != 0 )
            {
            if( rc != ERROR_QUE_EMPTY )
                {
                printf( "Server dosread error attaching 
                                                   sem to queue (%d)!\n", rc );
                goto ReadError;
                }
            }
        }
    // Now sit in a loop reading the queues
    for( j = 0; j < 100; j++ )
        {
        MessageCount = 0;
        // Wait on all the queue semaphores.
        if(( rc = DosWaitMuxWaitSem( hMux, MAXTIMEOUT,&ulUser )) != 0 )
            {
            if( rc == ERROR_INTERRUPT )
                {
                printf( "Mux wait interrupted!\n" );
                break;
                }
            else
                {
                printf( "Mux wait timeout\n" );
                if( rc == ERROR_TIMEOUT )
                    continue;
                }
            printf( "Server mux wait error %d\n", rc );
            break;
            }
        for( i = 0; i < MAXQUEUES; i++ )
            {
            // ReadQueue fills pReadData with a pointer into shared
                        // memory that contains our data.
            if(( rc = DosReadQueue( hQ[i], &rd, &length, 
                           (void **)&pReadData, 0L, DCWW_NOWAIT, &ucPriority, 
                           (HEV )pShareMem->Sem[i].hsemCur )) != 0 )
                continue;
            printf( "SERVER rcvd from CLIENT[%d]: %s\n", i, 
                                                                   pReadData );
            // MessageCount tracks how many messages we read for 
                        // triggering of muxsem so that we can trap any false 
                        // triggers of the muxsem.
            MessageCount++;
            // You MUST RESET semaphore or it will keep triggering.
            if(( rc = DosResetEventSem( (HEV )
                             pShareMem->Sem[i].hsemCur, &ulPostCount )) != 0 )
                {
                printf("Server Reset event sem error %d\n",rc);
                }
            }
        if( MessageCount == 0 )
            printf("No messages(%d) read: error!\n", MessageCount);
        }
ReadError:
    for( i = 0; i < MAXQUEUES; i++ )
        {
        DosCloseQueue( hQ[i] );
        DosCloseEventSem( (HEV )pShareMem->Sem[i].hsemCur );
        }
    DosCloseMuxWaitSem( hMux );
CreateError:
    DosFreeMem( pShareMem );
AllocError:
    return( rc );
}


