/*

TBTEXT.PRG (c) Hannes Ziegler ( CIS 100142,302 )
Version 1.0    9.Okt.93

********
Content:
  Proc TBText( cTextFile, nTop, nLeft, nBottom, nRight )
    cTextfile      - Name of a textfile including path & extension
    nTop...nBottom - Screen-coordinates for display

  Func TBMouse( oTBrowse ) 
    oTBrowse       - guess what?

************
Description:

  ********
  TBText() is a text-file-viewer for unformatted and formatted text-
  files. This version displays formatted files that have been created
  using Word 5 for DOS or Winword 2 as well as normal ASCII files.
  The program recognizes the text-processing-program a file has been
  formatted with.
  File size does not matter. The only limit is: one single paragraph
  in a formatted file must be less than 64k, because text is being read
  in one line- or paragraph after the next (a paragraph is delimited
  with CRLF but may contain a couple of lines of a given linelength.
  See mlcount() )
  Display of text is being done with TBrowse, which is fast but not really
  fast. However, in most cicumstances it's sufficient.

  *********
  TBMouse() implements a mouse-control for TBrowse. The mouse-functions
  I have used can be found in NANFOR.LIB in the CIS forum. 
  The mouse-hot-spot is given by TBrowse-coordinates. The TBrowse-cursor
  (highlighted cell) follows the mouse-cursor if the left button is
  pressed (and held). TBrowse reacts as follows:

  Mouse-cursor is in:
    TBColumn:Heading        -> fast scrolling up
    TBrowse :HeadSep        -> slow scrolling up
    TBrowse data-area       -> set TBrowse-cursor to indicated RowPos
    TBrowse :FootSep        -> slow scrolling down
    TBColumn:Footing        -> fast scrolling down

  If the right mouse-button is pressed, the program terminates.

Compile + link:
  clipper tbtext /n/m
  rtlink fi tbtext lib nanfor

Call:
  C:\>tbtext winword.doc  -> Display Winword file
  C:\>tbtext word5.txt    -> Display Word 5 file
  C:\>tbtext tbtext.prg   -> Display unformatted file

***********
Disclaimer:
  This program is freeware. I'm glad if you use it, but you do it
  on your own risk.
  Be aware of static function TBEvent(). It accesses TBrowse internals.
  If you don't know what's going on, see TBINFO.ZIP for details about
  TBrowse internals. If you still don't like it, delete this file from
  your harddisk and forget it.

*******
Remark:
  If you have questions or comments, please let me know

  I'd like to give TBText() a more global scope and for this I need
  formatted text-files. If you have a textprocessing system, that is 
  not being recognized by TBText(), please leave a message.
  (my CIS ID is in line #3)

*/

#include "fileio.ch"
#include "inkey.ch"


// *#define MOUSE // if not linked with NANFOR.LIB 
#define MOUSE  

#ifndef MOUSE

  #xtrans  FT_MHIDECRS() =>
  #xtrans  FT_MSHOWCRS() =>

#endif


#define BUFF_SIZE         2048   // read buffer size

#define WORD_FOR_DOS_50  -16847  // file is formatted using Word 5 
#define WIN_WORD_20      -23077  // file is formatted using WinWord 2

#define CRLF   chr(13)+chr(10)

static scText                    // current paragraph
static snLineCount               // lines in paragraph
static snCurLine                 // current line
static snLineLen  := 78          // line length for memoline()
static snHandle   := 0           // DOS handle for textfile
static snFileType := 0           // file type
static snFilePos  := 0           // current position in file
static snFirstPos := 0           // first position for text in file
static snLastPos  := 0           // last position for text in file 
static scReadBuff                // read buffer
static snByteRead                // number of bytes for paragraph read in
static slBof      :=.F.          // BoF flag
static slEof      :=.F.          // EoF flag
static scEofChar  :=" "          // char for End-of-text or End-of-file

******************************************************
Proc TBText( cTextFile, nTop, nLeft, nBottom, nRight )
local oTBrowse, oTBColumn


if cTextFile==nil
  cls
  ? "Usage: C:\TBTEXT>  <Text file & extension>" 
  quit
end

snHandle := TextUse(cTextFile)

if snHandle < 0
  return qout("File not found:",cTextFile)
end

// TBTEXT.PRG is a Stand-alone-utility
// If TBText() is linked into a program the next four lines can be omitted
if valtype( nTop    ) == "C" ; nTop   :=  val( nTop    ) ; end
if valtype( nLeft   ) == "C" ; nLeft  :=  val( nLeft   ) ; end
if valtype( nBottom ) == "C" ; nBottom:=  val( nBottom ) ; end
if valtype( nRight  ) == "C" ; nRight :=  val( nRight  ) ; end

if valtype( nTop    ) != "N" ; nTop   :=  2  ; end
if valtype( nLeft   ) != "N" ; nLeft  :=  1  ; end
if valtype( nBottom ) != "N" ; nBottom:=  22 ; end
if valtype( nRight  ) != "N" ; nRight :=  78 ; end

snLineLen:= nRight-nLeft+1

oTBrowse := TBrowseNew( nTop, nLeft, nBottom, nRight )
oTBrowse : SkipBlock := {|n|TextSkipper(n)} 
oTBrowse : GoTopBloc := {|| TextTop() } 
oTBrowse : GoBottomB := {|| TextBottom() } 
oTBrowse : ColorSpec := "N/BG,N/R"

oTBColumn:=TBColumnNew( "FILE: "+upper(cTextFile)   ,;
           {|| memoline(scText,snLineLen,snCurLine) } )

oTBColumn:Footing:="TBTEXT.PRG: Textfile-Viewer for ASCII, Word5 and WinWord2. (c) H.Ziegler"

oTbrowse:AddColumn(oTBColumn)

TBMouse( oTBrowse )

TextClose()

return

********************
* Open Textfile    *
********************
Static Func TextUse( cTextFile )

if (snHandle := fopen( cTextFile ))  >  0
  TextInit()  // initialize Static Vars 
end

return snHandle

*************************
* Close text file       *
* release Static Vars   *
*************************
Static Func TextClose()

fclose(snHandle)

scText     := ""
snCurLine  := 0
snLineCount:= 0

snHandle   := 0
snFileType := 0
snFilePos  := 0
snFirstPos := 0
snLastPos  := 0
scReadBuff := ""
snByteRead := 0
slBof      :=.F.
slEof      :=.F.
scEofChar  :=" "

return nil

***************************
* initialize static vars  *
***************************
Static Proc TextInit()

scReadBuff := space(BUFF_SIZE)
snFilePos  := 0
snLastPos  := BUFF_SIZE+1    
scText     := TextReadDown()
snFileType := bin2i(scText)   // byte 1+2 is ID

// Determine area in file where text is being stored
// snFirstpos - exclude fileheader 
// snLastPos  - exclude formatting bytes (filetail)

if snFileType == WORD_FOR_DOS_50
  snFirstPos := 128
  snLastPos  := bin2l(substr(scText,15))

elseif snFileType == WIN_WORD_20
  snFirstPos := bin2l(substr(scText,25))
  snLastPos  := snFirstPos+bin2l(substr(scText,53))

else
  snFileType := 0
  snFirstPos := 0
  snLastPos  := fseek(snHandle,0,FS_END)

end

// Read Eof-character
fseek(snHandle,snLastPos-1,FS_SET)
fread(snHandle,@scEofChar,1)
fseek(snHandle,snFilePos,FS_SET)

scText    := substr(scText,snFirstPos+1)
snByteRead:= len(scText)
TextFormat()

return

********************************
* Format paragraph for display *
********************************
Static Proc TextFormat()
local i, j, k, cStr1, cStr2, cChar

if snFileType == WIN_WORD_20

// You won't need to translate German Umlauts

    // Umlaute bersetzen
    //                                                          
//  cStr1:=chr(228)+chr(196)+chr(246)+chr(214)+chr(252)+chr(220)+chr(223)
//  cStr2:=chr(132)+chr(142)+chr(148)+chr(153)+chr(129)+chr(154)+chr(225)
//  for i:=1 to 7
//    if (cChar:=substr(cStr1,i,1)) $ scText
//      scText:=strtran(scText,cChar,substr(cStr2,i,1))
//    end
//  next

  // Remove hidden text and field-functions
  while (i:=at(chr(19),scText)) > 0
    j    := at(chr(21),scText)
    cStr1:= substr(scText,i,j-i+1)
    if (k := at(chr(20),cStr1)) > 0
      cStr2:= substr(cStr1,k+1,j-i-k) // save text of field-functions
    else
      cStr2:= "" // field-func has no text (eg. an index)
    end
    scText:=stuff(scText,i,len(cStr1),cStr2)
  end

end

// number of lines in paragraph
snLineCount:= max(1,mlcount( scText, snLineLen ))
snCurLine  := 1

// translate vertical tab to CRLF
while ( i:=at(chr(11),scText)) > 0
  scText   := stuff(scText,i,1,CRLF)
  snLineCount++
end

// translate formfeed to CRLF
while ( i:=at(chr(12),scText)) > 0
  scText   := stuff(scText,i,1,CRLF)
  snLineCount++
end

return

**********************************
* Skip-Function for TBrowse      *
* Maintains pointer to line and  *
* pointer to paragraph           *
**********************************
Static Func TextSkipper( nSkip )
local nDoSkip := 0

if nSkip > 0

  while !slEof .and. nDoSkip < nSkip // browse down
    nDoSkip++
    snCurLine++
        
    if snCurLine > snLineCount       // Next paragraph
      TextReadDown()
      TextFormat()
      snCurLine  := 1
      snDirection:= 1
    end
  end
   
  if slEof
    snCurLine := snLineCount
  end
   
elseif nSkip < 0
  while !slBof .and. nDoSkip > nSkip // Browse up
    nDoSkip-- 
    snCurLine--

    if snCurLine == 0                // Previous paragraph
      TextReadUp()
      TextFormat()
      snDirection:= -1
      snCurLine  := snLineCount
    end
  end
   
  if slBof
    nDoSkip++
    snCurLine := 1
  end

end

return nDoSkip

*********************************
* Read next paragraph from file *
*********************************
Static Proc TextReadDown()
local i, j := BUFF_SIZE, nBytes:=0

scText   := ""
while ( i:= at(CRLF,scText) ) == 0 .and. nBytes+snFilePos < snLastPos
  j      := fread( snHandle, @scReadBuff, BUFF_SIZE )
  nBytes += j
  scText += scReadBuff
end

slEof    := (i==0 .or. ( i+snFilePos > snLastPos ) )

if slEof 
  slBof     := (snFirstPos==snLastPos)
  snFilePos := fseek(snHandle,snLastPos, FS_SET )

  if asc(scEofChar) < 27       // Eof-Char exist
    scText:= "*** EOF ***"

  else                         // Eof-Char is missing
    slEof := (nBytes == 0)     // Eof() if nothing has been read
    if slEof
      scText := "*** EOF ***"
    else
      scText:= left(scText,nBytes)
    end
  end

else

  // Move pointer to beginning of next paragraph
  slBof     := .F.
  snFilePos := fseek(snHandle,i-nBytes+1, FS_RELATIVE )
  scText    := left(scText,i+1)

end

snByteRead := len(scText)      // Save it! scText gets formatted!

return

*************************************
* Read previous paragraph from file *
*************************************
Static Proc TextReadUp()
local i, j

// The file pointer always is at the end of the current paragraph,
// ie at the beginning of the next one.
// Go back to beginning of current paragraph and skip preceding CRLF
// Watch Eof !!

if slEof
  if scEofChar $ chr(12)+chr(26)  // Eof-char exist
    snFilePos := fseek(snHandle, -3, FS_RELATIVE )

  elseif scEofChar == chr(10)     // EoF-char is CRLF
    snFilePos := fseek(snHandle, -2, FS_RELATIVE )

  end

else 
  snFilePos := fseek(snHandle, -snByteRead-2, FS_RELATIVE )

end

// Hit Bof
if snFilePos < snFirstPos

  snFilePos := fseek( snHandle, snFirstPos, FS_SET )
  TextReadDown()
  slBof     := .T.
  
// Approaching Bof
// can't read in the total of BUFF_SIZE Bytes
elseif snFilePos < BUFF_SIZE

  // Read Bytes just from bof to FilePos 
  scText := space(snFilePos-snFirstPos)
  j      := len(scText)
  fseek( snHandle, snFirstPos, FS_SET )
  fread( snHandle, @scText, j ) 
  
  // It's not the first paragraph yet
  if (i := rat(CRLF,scText)) > 0
    scText := substr(scText,i+2)
  end

// File pointer is in the middle of nowhere, ie between
// snFirstPos+BUFF_SIZE and snLastPos
else

  scText    := ""
  slEof     := slBof := .F.
  j         := BUFF_SIZE

  while (i:=rat(CRLF,scText)) == 0 .and. snFilePos > snFirstPos
    snFilePos:= fseek(snHandle, -BUFF_SIZE, FS_RELATIVE )
    j        := fread(snHandle,@scReadBuff, BUFF_SIZE   )
    scText   := scReadBuff + scText
    fseek(snHandle, -BUFF_SIZE, FS_RELATIVE )
  end

  if i > 0
    scText:= substr(scText,i+2)   // CRLF is at the beginning of the paragraph
  end

  snFilePos := fseek(snHandle,i+len(scText)+1, FS_RELATIVE )

end

if substr(scText,-1)==chr(10)  .or.;  // CRLF+CRLF at the end
   (slEof .and. asc(scEofChar) > 26)  // last line without Eof-char
   // nix tun

elseif substr(scText,-1)==chr(13)     // Split CRLF 
  scText    += chr(10)                // Add LF and adjust FilePos
  snFilePos := fseek( snHandle, 1, FS_RELATIVE )

else
  scText    += CRLF                   // Normal situation. Add CRLF for memoline()
  snFilePos := fseek( snHandle, 2, FS_RELATIVE )

end

snByteRead  := len(scText)      // Save it! scText gets formatted!
slEof       := .F.

return

*************************
* GoTop() for text-file *
*************************
static proc TextTop()

snFilePos := fseek(snHandle,snFirstPos,FS_SET)
slEof     :=.F.
slBof     :=.F.
TextReadDown()
TextFormat()

return

****************************
* GoBottom() for text-file *
****************************
static proc TextBottom()

snFilePos := fseek(snHandle,snLastPos,FS_SET)
snByteRead:= 0
slEof     :=.T.
slBof     :=.F.
scText    := "*** EOF ***"
TextReadUp()
TextFormat()
snCurLine := snLineCount

return


*************************************************************************
*************************************************************************
/*
  This part contains mouse control for TBrowse
  NANFOR.LIB must be linked
  If there is no mouse, uncomment #define MOUSE at the beginning
*/

#define  NO_EVENT                -1
#define  KEYBOARD_EVENT           0
#define  LBUTTON_PRESSED_EVENT    1 // see FT_MGETCOORD()
#define  RBUTTON_PRESSED_EVENT    2 // see FT_MGETCOORD()

************************
Func TBMouse( oTBrowse ) 
local nEvent:=NO_EVENT, nCursor:=SetCursor(0)

if empty(oTBrowse:HeadSep)   // Need HeadSep and ColSep for mousecontrol
  oTBrowse:Headsep:=chr(196)
end

if empty(oTBrowse:FootSep)
  oTBrowse:FootSep:=chr(196)
end

TBEvent( oTBrowse , .T. )

FT_MSHOWCRS()                // show mouse cursor

while  nEvent != RBUTTON_PRESSED_EVENT 

  // replace stabilize() loop with event() 
  nEvent := TBEvent( oTBrowse )

  if oTBrowse:HitTop .or. oTBrowse:HitBottom
    tone(1000,0)
    oTBrowse:HitTop := oTBrowse:HitBottom:= .F.
  end

end

FT_MHIDECRS()                // hide mouse cursor 

SetCursor(nCursor)

return

*************************************
* Process mouse and keyboard events *
*************************************
Static Func TBEvent( oTBrowse, lInit )
static snMouseStatus := -1
static snFirstRow  , snLastRow
static scTBInternal,  snRequest, snPending
static snCellTop

local nMouseRow, nMouseCol, nEvent:=NO_EVENT, lWait:=.F., nKey

if lInit!=nil     // init static vars 

  if snMouseStatus < 0
    snMouseStatus := FT_MRESET()    // Reset mouse once only
  end

	// Force display and init atail(oTBrowse)
  oTbrowse:configure()
  while ! oTbrowse:stabilize() ; end

  // get TBrowse internals
  scTBInternal:= atail(oTBrowse)

  // row() of first datarow in browse area - 1
  snFirstRow := oTBrowse:nTop+bin2i(substr(scTBInternal,33))-1

  // row() of last datarow in browse area + 1
  snLastRow  := oTBrowse:nTop+bin2i(substr(scTBInternal,35))+1

  snCellTop  := row()

  return NO_EVENT

end

// Check for keyboard input first
if ( nKey := inkey() ) != 0

  FT_MHIDECRS()
  if (nEvent := TBKeyBoard( oTBrowse, nKey )) == KEYBOARD_EVENT
    while ! oTBrowse:stabilize() ; end
  elseif nKey > 31
    nEvent := nKey
  end
  FT_MSHOWCRS()

end

  #ifdef MOUSE


if nEvent == NO_EVENT              .and. ; // No significant key was pressed
  FT_MINREGION( oTBrowse:nTop,;            // mouse is in TBrowse area
                oTBrowse:nLeft,;
                oTBrowse:nBottom,;
                oTBrowse:nRight  ) .and. ;
(  nEvent:= ;                              // Left button is pressed
  FT_MGETCOORD( @nMouseRow, @nMouseCol ) ) == LBUTTON_PRESSED_EVENT

	// Move RowPos according to MouseRow
  if snCellTop != nMouseRow
    oTBrowse : DeHilite()

		// This is a wee bit dirty...
		// Calculate how many records are to be skipped in either
		// direction and stuff() it into TBrowse.
		// TBrowse knows pending skips, so add Requested to Pending 
		// (it's the FASTEST way to do it using Clipper code)

    snRequest := max(snFirstRow,min(snLastRow,nMouseRow)) - snCellTop
    snPending := bin2i(substr(scTBInternal,47))

    if snRequest != snPending
      snPending    += snRequest
      scTBInternal := stuff(scTBInternal,47,2,i2bin(snPending))
      oTBrowse[len(oTBrowse)]:=scTBInternal
      oTBrowse:stable:=.F.
    end

    // waitstate for slow scrolling
    lWait := (nMouseRow==snFirstRow .or. nMouseRow==snLastRow)

  end
  
  if lWait
    inkey(.1)
  end

end

  #endif /* #define MOUSE */

if ! oTBrowse:stable
  FT_MHIDECRS()             // Hide mouse

  if ! oTBrowse:Stabilize() // Update screen for scroll-request
    oTBrowse:Stabilize()    // Update screen for data diaply
  end

  snCellTop  := row()       // save row() of TBrowse-cursor (hilited cell)
  FT_MSHOWCRS()             // Show mouse
end


return nEvent

****************************************
Static Func TBKeyBoard( oTBrowse, nKey )

do case
  case nKey == K_UP         ; oTbrowse:up()      
  case nKey == K_DOWN       ; oTbrowse:Down()    
  case nKey == K_LEFT       ; oTbrowse:Left()    
  case nKey == K_RIGHT      ; oTbrowse:Right()   
  case nKey == K_PGDN       ; oTbrowse:PageDown()
  case nKey == K_PGUP       ; oTbrowse:PageUp()  
  case nKey == K_HOME       ; oTbrowse:Home()    
  case nKey == K_END        ; oTbrowse:End()     
  case nKey == K_CTRL_LEFT  ; oTbrowse:PanLeft() 
  case nKey == K_CTRL_RIGHT ; oTbrowse:PanRight()
  case nKey == K_CTRL_PGDN  ; oTbrowse:goBottom()
  case nKey == K_CTRL_PGUP  ; oTbrowse:goTop()   
  case nKey == K_CTRL_HOME  ; oTbrowse:PanHome() 
  case nKey == K_CTRL_END   ; oTbrowse:PanEnd()  
  case nKey == K_ESC        ; return RBUTTON_PRESSED_EVENT
  otherwise                  
    return NO_EVENT
end

return KEYBOARD_EVENT
