* FileName: GRADIOBN.PRG
*
* This file contains the functions required to allow a Radio Button object
* within a READ command. 
*
* (c) Copyright 1992, John D. Lay
* ALL RIGHTS RESERVED
*
***************************************************************************
* This is the beginning of a Get:RadioButton                              *
***************************************************************************
*
* Create Get Object for the following command:
* RADIO BUTTON <var> OPTIONS  @ <Row1>, <Col1> SAY <cButton1>  ;
*                           [,@ <Rown>, <Coln> SAY <cButtonn>] ;
*                           [COLOR <clr>]                      ;
*                           [VALID <vald>]                     ;
*                           [WHEN <when>]                      ;
*                           [SEND <msg>]
* => GetNewButton( <var>, ;
*                 {{<Row1>,<Col1>,<cButton1>}[,{<Rown>,<Coln>,<cButtonn>}]},;
*                 <clr>, <{vald}>, <{when}> )
*
#define FRESH

#include "Set.ch"
#include "Inkey.ch"
#include "Getexit.ch"
#include "SetCurs.ch"
#ifdef FRESH
#include "Fresh.ch"
#endif
*
* Defines for Get Cargo := Button object
#define GBUTTONS     3

#define K_UNDO  K_CTRL_U

**************************************************************************
*
* Function: GetNewButton( nVar, aButtons, cColor, bWhen )
*
*
STATIC bInsToggle := { || SETCURSOR( IF( READINSERT( !READINSERT() ), ;
                                         SC_NORMAL, SC_INSERT ) ) }

FUNCTION GetNewButton( bVar, cName, aButtons, cColor, bWhen )
   LOCAL oGet, x, nNumButtons
   LOCAL nTop    := MAXROW(),;
         nLeft   := MAXCOL(),;
         nBottom := 0,;
         nRight  := 0

   nNumButtons := LEN( aButtons )
   // Insure that each button has a valid clause
   // (allows selection of individual button)
   FOR x := 1 TO nNumButtons
      aButtons[x, 4] := IF( aButtons[x, 4]==NIL, {||.T.}, aButtons[x, 4] )
      // Get Button Coordinates
      nTop    := MIN( nTop,    aButtons[x, 1] )
      nLeft   := MIN( nLeft,   aButtons[x, 2] )
      nBottom := MAX( nBottom, aButtons[x, 1] )
      nRight  := MAX( nRight,  aButtons[x, 2] + LEN( aButtons[x,3] ) + 3 )
   NEXT x

   oGet := GetNew( nTop, nLeft )

   oGet:name      := cName
   oGet:block     := bVar
   oGet:colorSpec := cColor
   oGet:Reader := {|get, aGets, nGetPos| gRadioReader( get, aGets, @nGetPos)}

   oGet:cargo := { "RADIO", { nTop, nLeft, nBottom, nRight }, aButtons }

   gbDisplay( oGet )

   RETURN oGet
**************************************************************************
*
* Function: gbtnReader( oGet )
*
* This function is the Reader method for the getbutton object.
* It 1) activates the browse (get), 
*    2) PreValidates the get, 
*    3) calls gbApplyKey for each keyboard message,
* 
*
FUNCTION gRadioReader( oGet, aGetList, nGetPos )
   LOCAL nPos, nSaveCursor
   LOCAL nMsButton, nMsX, nMsY, nNumGets, n, nKey
   LOCAL nMsGetPos

   nSaveCursor := SETCURSOR( SC_NORMAL )

*  Check WHEN clause
	IF GetPreValidate( oGet )

*  Get Current Button
   nPos := gbNextValid( oGet, EVAL( oGet:block ) )
*  Activate Get (update display)
   gbDisplay( oGet, nPos )
   
*  Process keyboard messages
      WHILE oGet:exitState == GE_NOEXIT
#ifndef FRESH
         DO WHILE (nKey := inkey() ) == 0
         ENDDO
         gbApplyKey( oGet, @nPos, nKey, 0, 0, 0 )
#else
         nMsButton := 0
         WHILE ( ( nKey := inkey() ) == 0 .AND. ( nMsButton == 0 ) )
            MsStatus( @nMsButton, @nMsX, @nMsY )
         ENDDO
         IF ( nMsButton == MB_LEFT )
            nMsGetPos := nGetPos
            nNumGets  := LEN( aGetList )
            FOR n := 1 to nNumGets
               // See if mouse press is in a GET
               IF IsMouseInGet( aGetList[n], nMsX, nMsY )
                  nMsGetPos := n            // Assign it to mget position
                  EXIT
               ENDIF
            NEXT n
            IF ( nGetPos == nMsGetPos )
               // Click within Buttons
               gbApplyKey( oGet, @nPos, 0, nMsButton, nMsX, nMsY )
            ELSE
               oGet:exitState := GE_WRITE
               nGetPos := IF( nMsGetPos > 0, nMsGetPos, nGetPos )
            ENDIF
         ELSEIF ( nMsButton == MB_RIGHT )
            __keyboard( CHR( K_ESC ) )
         ELSE
            gbApplyKey( oGet, @nPos, nKey, 0, nMsX, nMsY )
         ENDIF
#endif
      ENDDO

*  Update button variable
      IF oGet:exitState != GE_ESCAPE
         EVAL( oGet:block, nPos )
      ENDIF
*  Update Display
      gbDisplay( oGet )
   ENDIF

   SETCURSOR( nSaveCursor )

*  Done...
   RETURN NIL
**************************************************************************
*
* Function: gbApplyKey( oGet, oBrowse, nKey )
*
* This function will process keystrokes for the getbrowse.
* It handles dispatching keyboard messages to the get and browse
* objects.
*
STATIC FUNCTION gbApplyKey( oGet, nPos, nKey, nMsPress, nMsX, nMsY )
   LOCAL lUpdated := .F., n, nNumButtons, aButtons, nSavePos

   DO CASE
      CASE nMsPress == MB_LEFT
         aButtons    := oGet:cargo[ GBUTTONS ]
         nNumButtons := LEN( aButtons )
         nSavePos    := nPos
         FOR n := 1 TO nNumButtons
            IF aButtons[ n, 1 ] == nMsX
               IF aButtons[n, 2] <= nMsY .AND. ;
                  ( aButtons[n, 2] + LEN( aButtons[n, 3] ) + 4 ) >= nMsY
                  nPos := n
                  EXIT
               ENDIF
            ENDIF
         NEXT n
         IF nPos != nSavePos
            IF (nPos := gbNextValid( oGet, nPos )) == 0
               nPos := nSavePos
            ELSE
               lUpdated := .T.
            ENDIF
          ELSE
            nPos := nSavePos
         ENDIF

      CASE nKey == K_UP .OR. nKey == K_LEFT
         IF nPos <= 1
            oGet:exitState := GE_UP
         ELSE
            IF (nPos := gbPriorValid( oGet, --nPos )) == 0
               oGet:exitState := GE_UP
            ELSE
               lUpdated := .T.
            ENDIF
         ENDIF

      CASE nKey == K_SH_TAB
         oGet:exitState := GE_UP

      CASE nKey == K_DOWN .OR. nKey == K_RIGHT
         IF nPos >= LEN( oGet:cargo[ GBUTTONS ] )
            oGet:exitState := GE_DOWN
         ELSE
            IF (nPos := gbNextValid( oGet, ++nPos )) == 0
               oGet:exitState := GE_DOWN
            ELSE
               lUpdated := .T.
            ENDIF
         ENDIF

      CASE nKey == K_TAB
         oGet:exitState := GE_DOWN

      CASE nKey == K_ENTER
         oGet:exitState := GE_ENTER

      CASE nKey == K_ESC
         oGet:exitState := GE_ESCAPE

      CASE nKey == K_PGUP
         oGet:exitState := GE_WRITE

      CASE nKey == K_CTRL_PGUP
         oGet:exitState := GE_WRITE

      CASE nKey == K_PGDN
         oGet:exitState := GE_WRITE

      CASE nKey == K_CTRL_PGDN
         oGet:exitState := GE_WRITE

      CASE nKey == K_CTRL_HOME
         oGet:exitState := GE_TOP

      CASE nKey == K_CTRL_W
         oGet:exitState := GE_WRITE

      CASE nKey == K_HOME
         nPos := 1
         lUpdated := .T.

      CASE nKey == K_END
         nPos := LEN( oGet:cargo[ GBUTTONS ] )
         lUpdated := .T.

      CASE nKey == K_DEL
         nPos := 0
         lUpdated := .T.

      CASE nKey == K_UNDO
         nPos := EVAL( oGet:block )
         lUpdated := .T.

   ENDCASE

   IF lUpdated
      gbDisplay( oGet, nPos )
      ReadUpdated( TRUE )
   ENDIF

   RETURN NIL
**************************************************************************
*
* Function: GetPreValidate( oGet )
*
* This function handles processing of the Get VALID clause.
*
STATIC FUNCTION GetPreValidate( oGet )
   LOCAL lWhen := .T.

   oGet:exitState := GE_NOEXIT

   IF oGet:preBlock <> NIL
      lWhen := EVAL( oGet:preBlock, oGet )
   ENDIF

   IF !(lWhen)
      oGet:exitState := GE_WHEN
   ENDIF

   RETURN (lWhen)
**************************************************************************
*
* Function: gbDisplay( nVar, aButtons )
*
* This function is the Display Method for the GetButton Object
* records.
*
STATIC FUNCTION gbDisplay( oGet, nVar )
   LOCAL aButtons, nNumButtons, n
   LOCAL nPos, cNormal, cHilite, cInvalid, cColor

   nVar     := IF( nVar == NIL, EVAL( oGet:block ), nVar )
   aButtons := oGet:cargo[ GBUTTONS ]
   cColor   := oGet:colorSpec

   MsHideAt( oGet:cargo[2, 1], oGet:cargo[2, 2],;
             oGet:cargo[2, 3], oGet:cargo[2, 4] )

   IF (nPos := AT( ",", cColor )) == 0
      cNormal  := cColor
      cHilite  := cColor
      cInvalid := cColor
   ELSE
      cNormal := SUBSTR( cColor, 1, nPos - 1 )
      cHilite := SUBSTR( cColor, nPos + 1 )
      IF (nPos := AT( ",", cHilite )) == 0
         cInvalid := cHilite
      ELSE
         cInvalid := SUBSTR( cHilite, nPos + 1 )
      ENDIF
   ENDIF

   nNumButtons := LEN( aButtons )

   FOR n := 1 TO nNumButtons
      cColor := IF( EVAL( aButtons[ n, 4 ] ), cNormal, cInvalid )
      SETPOS( aButtons[ n, 1 ], aButtons[ n, 2 ] )
      DEVOUT( IF( nVar == n , "("+CHR(7)+") ", "( ) " ), cColor )
      DEVOUT( aButtons[ n, 3 ], cColor )
   NEXT n

   n := IF( nVar < 1, 1, nVar )
   SETPOS( aButtons[ n, 1 ], aButtons[ n, 2  ] + 1 )

   MsShow()

   RETURN NIL
**************************************************************************
*
* Function: gbNextValid( oGet, nPos )
*
* This function will return the position passed if valid or the next 
* valid position.  If no position are valid then 0 is returned.
*
STATIC FUNCTION gbNextValid( oGet, nPos )
   LOCAL nNumButtons, nStart

   nNumButtons := LEN( oGet:cargo[ GBUTTONS ] )

   nPos := IF( EMPTY( nPos ), 1, ;
           IF( nPos > nNumButtons, nNumButtons, nPos ) )

   DO WHILE !EVAL( oGet:cargo[ GBUTTONS, nPos, 4 ] )
      IF ++nPos > nNumButtons
         nPos := 0
         EXIT
      ENDIF
   ENDDO

   RETURN (nPos)
**************************************************************************
*
* Function: gbNextValid( oGet, nPos )
*
* This function will return the position passed if valid or the next 
* valid position.  If no position are valid then 0 is returned.
*
STATIC FUNCTION gbPriorValid( oGet, nPos )
   LOCAL nNumButtons, nStart

   nNumButtons := LEN( oGet:cargo[ GBUTTONS ] )

   nPos := IF( EMPTY( nPos ), 0, ;
           IF( nPos > nNumButtons, nNumButtons, nPos ) )

   DO WHILE !EVAL( oGet:cargo[ GBUTTONS, nPos, 4 ] ) 
      IF --nPos < 1
         nPos := 0
         EXIT
      ENDIF
   ENDDO

   RETURN (nPos)
***************************************************************************
