#define VERSION "v1.32 2/9/94"

static char what[] = "@(#) helix.c " VERSION " by Alex Howansky";

/*
 *  helix.c
 *
 *  By Alex Howansky
 *     261 Boght Road
 *     Watervliet, NY 12189-1121
 *
 *  Please feel free to distribute and make changes to this code, as long
 *   as you include this header and maintain the version list.
 *
 *  Version 1.0 8/27/91 Alex Howansky
 *    initial version for DKBTrace v2.12
 *
 *  Version 1.1 8/30/91 Alex Howansky
 *    added input file support for user declared node objects
 *
 *  Version 1.2 8/23/93 Alex Howansky
 *    modified for use with POV-Ray v1
 *
 *  Version 1.3 1/27/94 Alex Howansky
 *    modified for use with POV-Ray v2
 *
 *  Version 1.31 2/7/94 Alex Howansky
 *    added option to indicate declared helix union label in output file
 *
 *  Version 1.32 2/9/94 Alex Howansky
 *    added option to control declared node object rotational orientation
 */

#include <stdio.h>
#include <math.h>

void exit(int);

void main (int argc, char *argv[])
 {
   FILE *ofp = stdout, *ifp = NULL;
   int ch;
   char *ifname = NULL, *ofname = NULL, *label = "Helix";
   double x, y, z, theta;
   double r = 1, R = 10, s = 0.1, S = 10, d = M_PI / 64, O = 1;

   if (argc == 1)
    {
      fprintf(stderr,"\nhelix " VERSION " by Alex Howansky\n");
      fprintf(stderr,"\nUsage: helix [-iFILE] [-oFILE] [-r#] [-R#] [-s#] [-S#] [-d#] [-O] [-lNAME] \n");
      fprintf(stderr,"\nOptions:\n");
      fprintf(stderr,"  -i    Names the node description input file.\n");
      fprintf(stderr,"  -o    Names the output file.\n");
      fprintf(stderr,"  -r#   Radius of node spheres. Applies only if -i is not used.\n");
      fprintf(stderr,"  -R#   Radius of helix.\n");
      fprintf(stderr,"  -s#   Y increment between nodes (\"springiness\").\n");
      fprintf(stderr,"  -S#   Size (length) of helix. The helix will stretch from -S/2 to S/2.\n");
      fprintf(stderr,"  -d#   Density of the helix (measure in degrees between adjacent nodes).\n");
      fprintf(stderr,"  -O    Doesn't rotate user declared node objects to normal position.\n");
      fprintf(stderr,"  -l    Names the label of the helix union.\n\n");
      exit(1);
    }

   while (--argc > 0)
    if (argv[argc][0] == '-')
     switch (argv[argc][1])
      {
        case 'o' :
         ofname = &argv[argc][2];
         if ((ofp = fopen(ofname,"w")) == NULL)
          {
            fprintf(stderr,"Error: unable to open output file %s\n",ofname);
            exit(1);
          }
         break;
        case 'i' :
         ifname = &argv[argc][2];
         if ((ifp = fopen(ifname,"r")) == NULL)
          {
            fprintf(stderr,"Error: unable to open input file %s\n",ifname);
            exit(1);
          }
         break;
        case 'r' : r = atof(&argv[argc][2]); break;
        case 'R' : R = atof(&argv[argc][2]); break;
        case 's' : s = atof(&argv[argc][2]); break;
        case 'S' : S = atof(&argv[argc][2]); break;
        case 'd' : d = atof(&argv[argc][2]) * M_PI / 180; break;
        case 'O' : O = 0; break;
        case 'l' : label = &argv[argc][2]; break;
        default  :
         fprintf(stderr,"Error: unknown option %s\n",argv[argc]);
         fprintf(stderr,"Usage: helix [-iFILE] [-oFILE] [-r#] [-R#] [-s#] [-S#] [-d#] [-O] [-lNAME]\n");
         exit(1);
      }

   fprintf(ofp,"/*\n   This file was generated by\n\n");
   fprintf(ofp,"   Helix " VERSION " by Alex Howansky\n\n");
   if (ifname != NULL)
    fprintf(ofp,"   i = %s\n",ifname);
   if (ofname != NULL)
    fprintf(ofp,"   o = %s\n",ofname);
   fprintf(ofp,"   r = %9.4f\n",r);
   fprintf(ofp,"   R = %9.4f\n",R);
   fprintf(ofp,"   s = %9.4f\n",s);
   fprintf(ofp,"   S = %9.4f\n",S);
   fprintf(ofp,"   d = %9.4f degrees\n",d * 180 / M_PI);
   if (O == 0)
    fprintf(ofp,"   O\n");
   fprintf(ofp,"   l = %s\n*/\n\n",label);

   fprintf(ofp,"#declare %s = union {\n",label);

   theta = 0;
   y = - S / 2;
   while (y <= S / 2)
    {
      x = sin(theta) * R;
      z = cos(theta) * R;
      if (ifp == NULL)
       fprintf(ofp,"  sphere { < %.4f, %.4f, %.4f >, %.4f }\n",x,y,z,r);
      else
       {
         fprintf(ofp,"  object {\n");
         rewind(ifp);
         while ((ch = getc(ifp)) != EOF)
          fputc(ch,ofp);
         if (O)
          {
            fprintf(ofp,"translate < %.4f, %.4f, 0 >\n",R,y);
            fprintf(ofp,"rotate < 0, %.4f, 0 >\n",(theta * 180 / M_PI) - 90);
          }
         else
          fprintf(ofp,"    translate < %.4f, %.4f, %.4f >\n",x,y,z);
         fprintf(ofp,"  }\n");
       }
      y += s;
      theta += d;
    }

   fprintf(ofp,"}\n");
   if (ifp != NULL)
    fclose(ifp);

   fclose(ofp);
 }
