// G_TEST.CPP
//
// Genesis Graphical Interface Application Library version 2
// Copyright (c) '93 by Kevin C.W. Hise - All rights reserved.
//
// sample and test program.

#include "g_test.hpp"


#if defined ( USE_META )
extern unsigned _stklen = 14336U;
#include <metaqery.hpp>
#endif



// ================
// code starts here
// ================

int main( int argc, char **argv )
{

    // allocate the GL2 kernel

    KERNEL * kernel  = new KERNEL( 16, 32 );

    // set the default system button size
    kernel->system_button_height =
        kernel->system_button_width = 24;

    kernel->color_table = colorTable;


    // initialize the unknown mouse device to null

    MOUSE_DEVICE * mouse = NULL;


#if defined ( USE_BGI )

    // setup the BGI display

    BGI_DISPLAY * display = new BGI_DISPLAY( "\\bc\\bgi", VGA, VGAHI );

// using super-vga drivers
//    BGI_DISPLAY * display = new BGI_DISPLAY( "\\BC\\BGI", DETECT, SVGA640x480, "SVGA256" );

    if ( !display || display->is_installed == false )
    {
        int err = display ? display->error_code : 0;
        cout << "BGI error: " << err << ' ' << grapherrormsg( err ) << "\n\n";
        exit( 0 );
    }

    san_font = display->AddFont( SANS_SERIF_FONT, 4 );
    tri_font = display->AddFont( TRIPLEX_FONT, 4 );
    scr_font = display->AddFont( TRIPLEX_SCR_FONT, 4 );

#elif defined ( USE_META )

    int card = -1, input;

    MetaQuery( card, input, argc, argv );

    METAWINDOW_DISPLAY * display = new METAWINDOW_DISPLAY( card );

    if ( !display || display->is_installed == false )
    {
        int err = display ? display->error_code : 0;
        cout << "METAGRAPHICS error: " << err << "\n\n";
        exit(0);
    }

    scr_font = display->AddFont( "SYSTEM96.FNT" );
    tri_font = display->AddFont( "SYSTEM64.FNT" );
    san_font = display->AddFont( "SYSTEM72.FNT" );

#endif

    display->explode_effect = 10;

    // parse the command line (if there is one) to determine the pointer device desired

    *kernel + display;                      // display device

    if ( argc == 1 )
#if defined ( USE_BGI )
        mouse = new MS_MOUSE;
#elif defined ( USE_META )
        mouse = new METAGRAPHICS_MOUSE(input);
#endif
    else
        for( int argn = 1; argn < argc; argn++ )
        {
            strupr( argv[ argn ] );

            if ( !strncmp( argv[ argn ], "MOUSE=", 6 ) && !mouse )
            {
                if ( !strncmp( argv[ argn ]+6, "ELO", 3 ) )
                    *kernel + new ELODEV( 0x65, 2 );
            }
        }

    // tell the display there's a mouse ( if mouse is NULL, there is no effect )

    display->mouse = mouse;


    // add some devices to the kernel

    *kernel + mouse                         // mouse ( a NULL is ignored )
            + new KEYBOARD( 0x011B )        // BIOS keyboard with Esc as hot-key
            + ( event_timer = new TIMER )   // programmable event timer
            + new MENU_MANAGER              // POS demo menu manager
            + new TICKET_MANAGER;           // ticket manager

    // add the screen saver timer and it's window

    *kernel + new SCREEN_SAVER( 60, new STARFIELD( 50, 25 ) );

    // add the desktop and main window

    // create a desktop ( must add after the display )

    desktop = new WINDOW;

    desktop->flags = WOF_DESKTOP | WOF_BACKGROUND | WOF_CAN_DELETE;
    desktop->color = kernel->GetScheme( SID_CANVAS );

    *kernel + desktop
            + new MAIN_WINDOW;

    // process events and return a code

    int code = kernel->Exec();

    // remove all windows, shut-down the system, and return to DOS

    delete kernel;

    // display a copyright / sign-off message

    cout << "\
The GXL Graphical Interface Library Demo (v2.1).\n\
Copyright (c) 1992-1995 Hise Development - All rights reserved.\n\n\
This demonstration, including drivers, may be copied or distributed freely.\n\n\
For inquiries contact:\n\
  GeNT Computer Services\n\
  114-C Gemstone\n\
  Ridgecrest, CA 93555\n\
  Voice: (619) 384-2847\n\
  Fax:   (619) 371-1054\n\n\
Complete GXL package includes class libraries and C++ source with examples\n\
(over 600KB) and 120+ page manual.  Supports Borland(tm) 3.1 or later.\n\
\n\n\
DISPLAY_DEVICE Graphics Primitives and fonts:\n";

#if defined ( USE_BGI )
    cout << "  > Borland Graphics Interface (BGI) version 3.1\n  > Copyright (c) Borland International.\n\nAlso supports Metagraphics MetaWINDOW 4.x";
#elif defined ( USE_META )
    cout << "  > Metagraphics MetaWINDOW version 4.3 (DOS real-mode)\n  > Copyright (c) Metagraphics Software Corporation.\n  > (MetaWINDOW must be purchased separately.)\n\nAlso supports Borland BGI 3.1";
#endif

    // return a child-process code

    return code;

}








// ==========================
// MAIN_WINDOW implementation


MAIN_WINDOW::MAIN_WINDOW( void ) : WINDOW()
{
    // set the windows coords on the display

    object_area = CenterRect( display->area, 240, 415 );

    // init the color scheme

    color = kernel->GetScheme( SID_MAIN_WINDOW );

    // add a border, title bar and help bar

    *this + new THREE_D_BORDER( SID_BORDER )
          + new TITLE_BAR( SID_TITLE_BAR, "GXL DEMO" )
          + ( help_bar = new HELP_BAR( SID_MAIN_WINDOW, help_context[0] ) );

    // now add the menu selection buttons

    *this + new LABEL( 115, 8, SID_TEXT, "Sample Objects", 0, 0, WOF_TEXT_CJ );

    gRECT r = { 5, 25, -70, -70 };
    *this + new OPTION( r, SID_BUTTON, "Exit", 0, ico_exit, C_EXIT, WOF_P_BUTTON | WOF_SEND_QUEUE );

    MoveRectX( r, 75 );
    *this + new OPTION( r, SID_BUTTON, "Windows", 0, ico_windows, C_WINDOW, WOF_P_BUTTON | WOF_SEND_QUEUE );

    MoveRectX( r, 75 );
    *this + new OPTION( r, SID_BUTTON, "Buttons", 0, ico_buttons, C_BUTTON_OBJECTS, WOF_P_BUTTON | WOF_SEND_QUEUE );

    MoveRectY( r, 75 );  r.left = 5;
    *this + new OPTION( r, SID_BUTTON, "Desktop\rColors", 0, ico_paint, C_SET_DESKTOP, WOF_P_BUTTON | WOF_SEND_QUEUE );

    MoveRectX( r, 75 );
    *this + new OPTION( r, SID_BUTTON, "Text", 0, ico_text, C_TEXT_OBJECTS, WOF_P_BUTTON | WOF_SEND_QUEUE );

    MoveRectX( r, 75 );
    *this + new OPTION( r, SID_BUTTON, "Clocks", 0, ico_clock, C_CLOCK, WOF_P_BUTTON | WOF_SEND_QUEUE );

    MoveRectY( r, 75  );  r.left = 5;
    *this + new OPTION( r, SID_BUTTON, "List\rBox", 0, NULL, C_LIST, WOF_P_BUTTON | WOF_SEND_QUEUE );

    MoveRectX( r, 75 );
    *this + new OPTION( r, SID_BUTTON, "Input", 0, NULL, C_INPUT, WOF_P_BUTTON | WOF_SEND_QUEUE );

    MoveRectX( r, 75 );
    *this + new OPTION( r, SID_BUTTON, "Pick\rFile", 0, NULL, C_PICK_FILE, WOF_P_BUTTON | WOF_SEND_QUEUE );

    *this + new LABEL( 115, 255, SID_TEXT, "Sample Applications", 0, 0, WOF_TEXT_CJ );

    InitRect( r, 5, 275, -70, -70 );
    *this + new OPTION( r, SID_BUTTON, "Point\rOf Sale", 0, NULL, C_POS, WOF_P_BUTTON | WOF_SEND_QUEUE );

}


// handle events for the main window

EVENT_STATUS MAIN_WINDOW::Event( EVENT & event )
{
    EVENT_STATUS ret = ES_OK;   // default return value

    switch( event.type )
    {
    case EV_OPTION:     // a menu option was selected
        if ( event.data.mouse_t.button )
            switch( event.code )    // find out which one was selected ( in 'code' )
            {
            case C_EXIT:            // set the flag in the kernel object to true
                kernel->halt = true;
                break;

            case C_WINDOW:          // add the sample window to the screen
                kernel->AddWindow( (new SAMPLE_WINDOW() ), event.area );
                break;

            case C_SET_DESKTOP:    // add the wallpaper color/pattern modifier
                kernel->AddWindow( (new SET_DESKTOP()) ,event.area );
                break;

            case C_TEXT_OBJECTS:    // add the sample text objects window
                kernel->AddWindow( (new TEXT_WINDOW()), event.area );
                break;

            case C_BUTTON_OBJECTS:  // add the sample button objects
                kernel->AddWindow( (new BUTTON_WINDOW()), event.area );
                break;

            case C_CLOCK:           // add the analog and digital clocks window
                kernel->AddWindow( (new CLOCK_WINDOW()), event.area );
                break;

            case C_LIST:
                kernel->AddWindow( (new LIST_WINDOW()), event.area );
                break;

            case C_INPUT:
                kernel->AddWindow( (new INPUT_WINDOW()), event.area );
                break;

            case C_PICK_FILE:
                kernel->AddWindow( (new PICK_WINDOW()), event.area );
                break;

            case C_POS:

                // tell the window manager to close all windows

                AddUIEvent( EV_CLOSE_ALL );

                // then add the main POS window - use an event so window
                // is added after all other windows are closed

                EVENT ev;

                ev.type         = EV_ADD_WINDOW;
                ev.target_id    = OI_KERNEL;
                ev.area         = event.area;
                ev.ptr.window_t = new POS_MENU_WINDOW();
                AddUIEvent( ev );

                ev.ptr.window_t = new POS_TICKET_WINDOW();
                AddUIEvent( ev );
                break;
            }
        break;

    default:            // otherwise, send the event to the base class 'window'
        ret = WINDOW::Event( event );

    }

    return ret;

}







// ====================
// a menu OPTION button


OPTION::OPTION
(
    gRECT &              r,
    SCHEME_ID           c,
    const char *        t,
    int                 fh,
    EVENT_CODE          opt,
    bit_flag_t          bf

) : COMBO_BUTTON( r, c, t, fh, bf, EV_OPTION, opt )
{
    option = opt;
}



OPTION::OPTION
(
    gRECT &              r,
    SCHEME_ID           c,
    const pixel_array * pa,
    EVENT_CODE          opt,
    bit_flag_t          bf

) : COMBO_BUTTON( r, c, pa, bf, EV_OPTION, opt )
{
    option = opt;
}


OPTION::OPTION
(
    gRECT &              r,
    SCHEME_ID           c,
    const char *        t,
    int                 fh,
    const pixel_array * pa,
    EVENT_CODE          opt,
    bit_flag_t          bf

) : COMBO_BUTTON( r, c, t, fh, pa, bf, EV_OPTION, opt )
{
    option = opt;
}



EVENT_STATUS OPTION::Event( EVENT & event )
{
    EVENT_STATUS ret = ES_OK;

    switch( event.type )
    {
    case EV_CURRENT:
        help_bar->ChangeText( help_context[ option ] );
        goto CONTINUE;

    case EV_NON_CURRENT:
        help_bar->ChangeText( help_context[ 0 ] );
        goto CONTINUE;

    CONTINUE:

    default:
        ret = COMBO_BUTTON::Event( event );
    }

    return ret;

}








// ====================================
// implementation for the sample window


SAMPLE_WINDOW::SAMPLE_WINDOW( void ) : WINDOW()
{
    object_area = CenterRect( display->area, 200, 150 );

    color = kernel->GetScheme( SID_MAIN_WINDOW );

    gRECT r = { 60, 70, -70, -30 };

    *this + new THREE_D_BORDER( SID_BORDER )
          + new TITLE_BAR( SID_TITLE_BAR, "Blank Window" )
          + new OPTION( r, SID_BUTTON, "Close", 0, ico_ok, C_CLOSE )
          + new LABEL( 95, 30, SID_TEXT, "Sample Dialog Text", 0, 0, WOF_TEXT_CJ );

// *** programmable TIMER bug-busting, 1/94
//    EVENT ev;
//    ev.type = EV_TIMEOUT;
//    thandle1 = event_timer->Add( TT_INTERVAL, this, ev, BIOS_TICKS_PER_SECOND * 25, 1, false );

//    ev.type = EV_TIMER;
//    thandle2 = event_timer->Add( TT_SECOND, this, ev, 0, 20, false );

}



EVENT_STATUS SAMPLE_WINDOW::Event( EVENT & event )
{
    EVENT_STATUS ret = ES_OK;

    switch( event.type )
    {
//    case EV_TIMER:
//        display->DrawLine( window, client_area.left, client_area.top, client_area.right, client_area.bottom, ((event.data.int_t.z%2) ? black : white) );
//        break;

    case EV_OPTION:
//        event_timer->Remove( thandle1 );
//
//    case EV_TIMEOUT:
//        event_timer->Remove( thandle2 );
        kernel->RemoveWindow( this );
//        break;

    default:
        ret = WINDOW::Event( event );
    }

    return ret;

}








// ===========================================
// implementation for the sample button window


BUTTON_WINDOW::BUTTON_WINDOW( void ) : WINDOW()
{
    object_area = CenterRect( display->area, 450, 300 );

    color = kernel->GetScheme( SID_BUTTON );

    strcpy( x_buf, "X = 0" );
    strcpy( y_buf, "Y = 0" );

    *this + new THREE_D_BORDER( SID_BORDER )
          + new TITLE_BAR( SID_TITLE_BAR, "Button object types" )
          + new SCROLL_BAR( SID_SCROLL, SID_BUTTON, 0, 20, 10, SB_V_EDGE | SB_USE_CORNER )
          + new SCROLL_BAR( SID_SCROLL, SID_BUTTON, 0, 20, 10, SB_H_EDGE )
          + ( x_text = new LABEL( 10, 170, SID_BLACK_TEXT, x_buf, 0 ) )
          + ( y_text = new LABEL( 10, 190, SID_BLACK_TEXT, y_buf, 0 ) )
    ;

    x_scroll =
        y_scroll = 0;

    gRECT r = { 5, 5, -70, -50 };
    *this + new COMBO_BUTTON( r, SID_BUTTON, "Text\rOnly", 0 );

    MoveRectY( r, 55 );
    *this + new COMBO_BUTTON( r, SID_BUTTON, "Icon\r& Text", 0, ico_ok, WOF_P_BUTTON );

    MoveRectY( r, 55 );
    *this + new COMBO_BUTTON( r, SID_BUTTON, ico_ok );


    r.left = 85;  r.top = 5;  r.right = -55;  r.bottom = -55;
    *this + new ROUND_BUTTON( r, SID_BUTTON, "Round", 0 );

    MoveRectY( r, 60 );
    *this + new ARROW_BUTTON( r, SID_BUTTON, ABD_LEFT, "Left", 0 );

    MoveRectY( r, 60 ); gRECT b = r;  b.left += 5;  b.right += 10;
    *this + new ROUND_BUTTON( b, SID_BUTTON, "Oval", 0 );

    MoveRectX( r, 60 );  r.top = 5;
    *this + new ARROW_BUTTON( r, SID_BUTTON, ABD_UP, "Up", 0 );

    MoveRectY( r, 60 );
    *this + new STOP_BUTTON( r, SID_W_RED, "Stop", 0 );

    MoveRectY( r, 60 );
    *this + new ARROW_BUTTON( r, SID_BUTTON, ABD_DOWN, "Down", 0 );

    MoveRectX( r, 60 );  r.top = 5;  b = r;  b.top += 5;  b.bottom += 10;
    *this + new ROUND_BUTTON( b, SID_BUTTON, "Oval", 0 );

    MoveRectY( r, 60 );
    *this + new ARROW_BUTTON( r, SID_BUTTON, ABD_RIGHT, "Right", 0 );

    MoveRectY( r, 60 );
    *this + new ROUND_BUTTON( r, SID_BUTTON, "Round", 0 );

    MoveRectX( r, 60 ); r.top = 5; b = r;  b.bottom += 10;
    *this + new TRIANGLE_BUTTON( b, SID_BUTTON, ABD_UP, "Up", 0 );

    MoveRectY( r, 60 ); b = r;  b.bottom += 10;
    *this + new TRIANGLE_BUTTON( b, SID_BUTTON, ABD_DOWN, "Down", 0 );

    MoveRectX( r, 60 ); r.top = 5; b = r;  b.right += 10;
    *this + new TRIANGLE_BUTTON( b, SID_BUTTON, ABD_LEFT, "Left", 0 );

    MoveRectY( r, 60 ); b = r;  b.right += 10;
    *this + new TRIANGLE_BUTTON( b, SID_BUTTON, ABD_RIGHT, "Right", 0 );


    InitRect( r, 110, 195, -70, -30 );
    *this + new OPTION( r, SID_BUTTON, "Close", 0, ico_ok, C_CLOSE );

}




EVENT_STATUS BUTTON_WINDOW::Event( EVENT & event )
{
    EVENT_STATUS ret = ES_OK;

    switch( event.type )
    {
    case EV_OPTION:
        kernel->RemoveWindow( this );
        break;

    case EV_VSCROLL:
        y_scroll += event.code;
        sprintf( y_buf, "Y = %d", y_scroll );
        y_text->ChangeData( y_buf );
        break;

    case EV_HSCROLL:
        x_scroll += event.code;
        sprintf( x_buf, "X = %d", x_scroll );
        x_text->ChangeData( x_buf );
        break;

    default:
        ret = WINDOW::Event( event );
    }

    return ret;

}








// =================================
// the desktop editor implementation


SET_DESKTOP::SET_DESKTOP( void ) : WINDOW()
{
    object_area = CenterRect( display->area, 415, 250 );

    color = kernel->GetScheme( SID_MAIN_WINDOW );

    *this + new THREE_D_BORDER( SID_BORDER )
          + new TITLE_BAR( SID_TITLE_BAR, "Change Desktop Colors" );

    gRECT r = { 10, 0, -175, -30 };
    *this + new LABEL( r, SID_BLACK_TEXT, "Colors" );

    r.left = 220;
    *this + new LABEL( r, SID_BLACK_TEXT, "Patterns" );

    int bsize = 32;

    InitRect( r, 10, 30, -bsize, -bsize );

    for (int n = 0; n < 16; n++ )
    {
        if ( n && !(n % (200/(bsize+2))) )
        {
            r.left = 10;
            MoveRectY( r, bsize + 2 );
        }
        *this + new COLOR_OPTION( r, n, FILL_SOLID, n );
        MoveRectX( r, bsize + 2 );
    }

    int pmin, pmax;

#if defined ( USE_BGI )
    bsize = 32;
    pmin  = SOLID_FILL,
    pmax  = CLOSE_DOT_FILL;

#elif defined ( USE_META )
    bsize = 24;
    pmin  = 1;
    pmax  = 31;

#endif

    InitRect( r, 220, 30, -bsize, -bsize );

    for (n = pmin; n < pmax; n++ )
    {
        if ( (n-pmin) && !((n-pmin) % (200/(bsize+2))) )
        {
            r.left = 220;
            MoveRectY( r, bsize + 2 );
        }

        *this + new COLOR_OPTION( r, white, n, n+1000 );
        MoveRectX( r, bsize + 2 );
    }

    InitRect( r, 120, 180, -70, -30 );
    *this + new OPTION( r, SID_BUTTON, "Ok", 0, ico_ok, C_OK );

    MoveRectX( r, 90 );
    *this + new OPTION( r, SID_BUTTON, "Cancel", 0, ico_cancel, C_CANCEL );

    old_color   = desktop->color->fill;
    old_pattern = desktop->color->pattern;

}




EVENT_STATUS SET_DESKTOP::Event( EVENT & event )
{
    EVENT_STATUS ret = ES_OK;

    switch( event.type )
    {
    case EV_COLOR:
        if ( event.code < 1000 )
        {
            desktop->color->fill = event.code;
            kernel->RedrawArea( display->area, NULL, true, 1 );
        }
        else
        {
            desktop->color->pattern = event.code - 1000;
            kernel->RedrawArea( display->area, NULL, true, 1 );
        }
        break;

    case EV_OPTION:
        switch( event.code )
        {
        case C_CANCEL:
            desktop->color->fill    = old_color;
            desktop->color->pattern = old_pattern;

            kernel->RedrawArea( display->area, NULL, true, 1 );

        case C_OK:
            kernel->RemoveWindow( this );
            break;

        }
        break;

    default:
        ret = WINDOW::Event( event );
    }

    return ret;

}









// ==============================================
// color and pattern swatch button implementation


COLOR_OPTION::COLOR_OPTION
(
    gRECT &     r,   // area of button
    int        c,   // color of swatch
    int        p,   // pattern of swatch
    EVENT_CODE opt  // option to return

) : BUTTON( r, SID_BUTTON )
{
    col    = c;
    pat    = p;
    option = opt;

}



EVENT_STATUS COLOR_OPTION::Event( EVENT & event )
{
    EVENT_STATUS ret = ES_OK;

    switch( event.type )
    {
    case EV_INITIALIZE:
        BUTTON::Event( event );
        swatch_area = data_area - 2;
        break;

    case EV_MOVE:
        MoveRect( swatch_area, event.data.int_t.x, event.data.int_t.y );

    default:
        ret = BUTTON::Event( event );
    }

    return ret;

}



void COLOR_OPTION::Draw( void )
{
    BUTTON::Draw();

    display->SetFillStyle( pat, col );
    display->DrawFilledRectangle( swatch_area );
    display->DrawRectangle( swatch_area, black );

}



void COLOR_OPTION::Selected( boolean status )
{
    BUTTON::Selected( status );

    if ( status == false )
    {
        EVENT ev;

        ev.type      = EV_COLOR;
        ev.code      = option;
        ev.target_id = OI_KERNEL;
        ev.source    = this;

        AddUIEvent( ev );
    }

}









// ==============================
// the sample text objects window


TEXT_WINDOW::TEXT_WINDOW( void ) : WINDOW()
{
    object_area = CenterRect( display->area, 400, 210 );

    color = kernel->GetScheme( SID_MAIN_WINDOW );

    *this + new THREE_D_BORDER( SID_BORDER )
          + new TITLE_BAR( SID_TITLE_BAR, "Text Objects" );

    *this + new LABEL( 195, 5,   SID_BLACK_TEXT,  "Left Justify",   tri_font, 0, WOF_TEXT_LJ  )
          + new LABEL( 195, 30,  SID_BLACK_TEXT,  "Right Justify",  san_font, 0, WOF_TEXT_RJ )
          + new LABEL( 195, 60,  SID_BLACK_TEXT,  "Center Justify", scr_font, 0, WOF_TEXT_CJ )
          + new LABEL( 195, 110, SID_TEXT, "Blinking Text",  0, 0, WOF_TEXT_CJ | WOF_BLINK );

    gRECT r = { 160, 130, -70, -30 };
    *this + new OPTION( r, SID_BUTTON, "Close", 0, ico_ok, C_CLOSE );

}




EVENT_STATUS TEXT_WINDOW::Event( EVENT & event )
{
    EVENT_STATUS ret = ES_OK;

    switch( event.type )
    {
    case EV_OPTION:
        if ( event.code == C_CLOSE )
            kernel->RemoveWindow( this );
        break;

    default:
        ret = WINDOW::Event( event );
    }


    return ret;

}
















// ============================
// clocks window implementation



CLOCK_WINDOW::CLOCK_WINDOW( void ) : WINDOW()
{
    object_area = CenterRect( display->area, 255, 300 );

    color = kernel->GetScheme( SID_MAIN_WINDOW );

    *this + new THREE_D_BORDER( SID_BORDER )
          + new TITLE_BAR( SID_TITLE_BAR, "Clocks" );


    gRECT r = { 5, 5, -80, -90 };
    *this + new CLOCK( r, SID_BUTTON, SID_CLOCK, 0 );

    MoveRectY( r, 95 ); r.right = -80;
    *this + new CLOCK( r, SID_BUTTON, SID_CLOCK, 0 , CF_ANALOG | CF_SHOW_TIME | CF_PORTRAIT );

    MoveRectY( r, 95 ); r.bottom = -60;
    *this + new CLOCK( r, SID_BUTTON, 0 );

    InitRect( r, 90, 5, -150, -60 );
    *this + new CLOCK( r, SID_BUTTON, SID_CLOCK, 0, CF_ANALOG | CF_LANDSCAPE | CF_SHOW_TIME | CF_SHOW_SECONDS | CF_SHOW_DATE );

    MoveRectY( r, 65 );  r.bottom = -40;
    *this + new CLOCK( r, SID_BUTTON, 0, CF_DIGITAL | CF_LANDSCAPE | CF_SHOW_DATE | CF_SHOW_TIME | CF_SHOW_SECONDS );

    MoveRectY( r, 45 );  r.right = -80; r.bottom = -30;
    *this + new CLOCK( r, SID_BUTTON, 0, CF_DIGITAL | CF_SHOW_TIME | CF_AMPM | CF_PORTRAIT );

    MoveRectY( r, 35 ); r.bottom = -40;
    *this + new CLOCK( r, SID_BUTTON, 0, CF_DIGITAL | CF_SHOW_DATE | CF_PORTRAIT );


    InitRect( r, 130, 220, -70, -30 );
    *this + new OPTION( r, SID_BUTTON, "Ok", 0, ico_ok, C_CLOSE );

}





EVENT_STATUS CLOCK_WINDOW::Event( EVENT & event )
{

    EVENT_STATUS ret = ES_OK;

    switch( event.type )
    {
    case EV_OPTION:
        kernel->RemoveWindow( this );
        break;

    default:
        ret = WINDOW::Event( event );
    }

    return ret;

}




// ==========================
// List window implementation


LIST_WINDOW::LIST_WINDOW( void ) : WINDOW()
{
    object_area = CenterRect( display->area, 200, 300 );

    color = kernel->GetScheme( SID_MAIN_WINDOW );

    *this + new THREE_D_BORDER( SID_BORDER )
          + new TITLE_BAR( SID_TITLE_BAR, "List Box" );

    gRECT r = { 10, 10, RectWidth(object_area) - 30, 198 };
    VLIST * vlist = new VLIST( r, SID_VLIST );
    *vlist + new SCROLL_BAR( SID_SCROLL, SID_BUTTON, 0, 0, 0, SB_V_EDGE );

    data = 0x1248;

    for( int n = 0; n < 16; n++ )
    {
        char buf[ 20 ];
        sprintf( buf, "Item #%d", n+1 );

        int f = 1 << n;

        *vlist << new FLAG_ITEM( SID_ITEM, buf, 0, &data, false, f, ~f, WOF_FLAG_ITEM | WOF_ALLOCATE_TEXT );
    }

    *this + vlist;

    InitRect( r, 60, 220, -70, -30 );
    *this + new OPTION( r, SID_BUTTON, "Ok", 0, ico_ok, C_CLOSE );
}



EVENT_STATUS LIST_WINDOW::Event( EVENT & event )
{

    EVENT_STATUS ret = ES_OK;

    switch( event.type )
    {
    case EV_OPTION:
        kernel->RemoveWindow( this );
        break;

    default:
        ret = WINDOW::Event( event );
    }

    return ret;

}









// ===================
// input fields window


INPUT_WINDOW::INPUT_WINDOW( void ) : WINDOW()
{
    object_area = CenterRect( display->area, 300, 295 );

    color = kernel->GetScheme( SID_MAIN_WINDOW );

    *this + new THREE_D_BORDER( SID_BORDER )
          + new TITLE_BAR( SID_TITLE_BAR, "Input Fields" );

    gRECT r;

    strcpy( string_field_1, "GXL v2.1 DEMO" );
    strcpy( string_field_2, "Touch Screen GUI Lib!" );

    int_data    = 0;
    long_data   = 0;
    double_data = 0.0;

    *this + new LABEL( 20, 10, SID_TEXT, "Program Name:", 0 )
          + new FIELD( 20, 25, SID_TEXT_FIELD, string_field_1, 0, 30, 30 )

          + new LABEL( 20, 45, SID_TEXT, "Cost:", 0 )
          + new FIELD( 20, 60, SID_TEXT_FIELD, string_field_2, 0, 30, 30 )

          + new LABEL( 20, 80, SID_TEXT, "Integer:", 0 )
          + new FIELD( 20, 95, SID_TEXT_FIELD, &int_data, "9999", 0 )

          + new LABEL( 20, 115, SID_TEXT, "Long:", 0 )
          + new FIELD( 20, 130, SID_TEXT_FIELD, &long_data, "99999999", 0 )

          + new LABEL( 20, 150, SID_TEXT, "Double (fixed decimal point):", 0 )
          + new FIELD( 20, 165, SID_TEXT_FIELD, &double_data, "9999999.9999", 0 )

          + new LABEL( 20, 185, SID_TEXT, "Double (floating decimal point):", 0 )
          + new FIELD( 20, 200, SID_TEXT_FIELD, &double_data, "999999999999", 0 );

    InitRect( r, 200, 220, -70, -30 );
    *this + new OPTION( r, SID_BUTTON, "Ok", 0, ico_ok, C_CLOSE );
}



EVENT_STATUS INPUT_WINDOW::Event( EVENT & event )
{

    EVENT_STATUS ret = ES_OK;

    switch( event.type )
    {
    case EV_OPTION:
        kernel->RemoveWindow( this );
        break;

    default:
        ret = WINDOW::Event( event );
    }

    return ret;

}







PICK_WINDOW::PICK_WINDOW( void )
{
    object_area = CenterRect( display->area, 300, 300 );

    color = kernel->GetScheme( SID_MAIN_WINDOW );

    *this + new THREE_D_BORDER( SID_BORDER )
          + new TITLE_BAR( SID_TITLE_BAR, "Pick File" );

    gRECT r = { 10, 10, 0, 0 };

    *this + new PICK_FILE( r, "", SID_MAIN_WINDOW, WOF_PICK_FILE | WOF_3D_BORDER,
                                  SID_TEXT,        WOF_TEXT_LJ,
                                  SID_TEXT_FIELD,  WOF_FIELD,
                                  SID_VLIST,       WOF_VLIST,
                                  SID_SCROLL,      WOF_SCROLL_BAR,
                                  SID_BUTTON,
                                  SID_ITEM,
                                  0, 16 );

    object_area = CenterRect( display->area, r.right + 20, r.bottom + 100 );

    InitRect( r, 10, r.bottom + 20, -70, -30 );
    *this + new OPTION( r, SID_BUTTON, "Ok", 0, ico_ok, C_CLOSE );
}


EVENT_STATUS PICK_WINDOW::Event( EVENT & event )
{
    EVENT_STATUS ret = ES_OK;

    switch( event.type )
    {
    case EV_FILE_SELECT:
        if ( event.code < 2 )
            break;

    case EV_OPTION:
        kernel->RemoveWindow( this );
        break;

    default:
        ret = WINDOW::Event( event );
    }

    return ret;

}





// ===========================
// Point-Of-Sale Demonstration
// ===========================


// define some clock settings and other bit-flag defaults

const bit_flag_t

    CSET_FLAGS = CF_ANALOG | CF_LANDSCAPE | CF_SHOW_TIME | CF_SHOW_SECONDS,
    COBJ_FLAGS = WOF_CAN_DELETE | WOF_IS_DRAWN,
    ITEM_FLAGS = WOF_L_BUTTON | WOF_SEND_QUEUE

    ;





// -------------------------------
// the menu manager implementation

MENU_MANAGER::MENU_MANAGER( void )
{
    object_id      = OI_MENU_MANAGER;
    current_menu_n = 0;
    current_menu_p = NULL;

}


MENU_MANAGER::~MENU_MANAGER( )
{

}


EVENT_STATUS MENU_MANAGER::Event( EVENT & event )
{
    EVENT_STATUS ret = ES_OK;

    switch( event.type )
    {
    case EV_SELECT_MENU:
        if ( event.code != current_menu_n )
        {
            POS_MENU * new_menu = new POS_MENU;

            gRECT r;

            if ( new_menu )
            {
                InitRect( new_menu->object_area, 250, 68, 640, 480 );

                switch( event.code )
                {
                case MI_MAIN_MENU:
                    new_menu->color = kernel->GetScheme( SID_POS_MAIN_MENU );

                    InitRect( r,   4, 4, -72, -64 );  *new_menu + new MENU_BUTTON( r, MI_DOUBLE );
                    MoveRectY( r, 68 );
                    MoveRectY( r, 68 );
                    MoveRectY( r, 68 );               *new_menu + new MENU_BUTTON( r, MI_LARGE );


                    InitRect( r,  81, 4, -72, -64 );  *new_menu + new MENU_BUTTON( r, MI_BURGER );
                    MoveRectY( r, 68 );               *new_menu + new MENU_BUTTON( r, MI_CLUCK_BURGER );
                    MoveRectY( r, 68 );               *new_menu + new MENU_BUTTON( r, MI_BLT );
                    MoveRectY( r, 68 );               *new_menu + new MENU_BUTTON( r, MI_HOT_DOG );
                    MoveRectY( r, 68 );               *new_menu + new MENU_BUTTON( r, MI_POLISH_DOG );
                    MoveRectY( r, 68 );               *new_menu + new MENU_BUTTON( r, MI_ROAST_BEEF );

                    InitRect( r, 158, 4, -72, -64 );

                    InitRect( r, 235, 4, -72, -64 );  *new_menu + new MENU_BUTTON( r, MI_CHEESE );
                    MoveRectY( r, 68 );               *new_menu + new MENU_BUTTON( r, MI_LETTUCE );
                    MoveRectY( r, 68 );               *new_menu + new MENU_BUTTON( r, MI_TOMATO );
                    MoveRectY( r, 68 );               *new_menu + new MENU_BUTTON( r, MI_ONION  );
                    MoveRectY( r, 68 );               *new_menu + new MENU_BUTTON( r, MI_PICKLE );

                    InitRect( r, 312, 4, -72, -64 );  *new_menu + new MENU_BUTTON( r, MI_MAYO );
                    MoveRectY( r, 68 );               *new_menu + new MENU_BUTTON( r, MI_MUSTARD );
                    MoveRectY( r, 68 );               *new_menu + new MENU_BUTTON( r, MI_KETCHUP );
                    MoveRectY( r, 68 );               *new_menu + new MENU_BUTTON( r, MI_RELISH );
                    MoveRectY( r, 68 );               *new_menu + new MENU_BUTTON( r, MI_1000_ISLAND );
                    MoveRectY( r, 68 );               *new_menu + new MENU_BUTTON( r, MI_BBQ_SAUCE );
                    break;

                case MI_SIDES_MENU:
                    new_menu->color = kernel->GetScheme( SID_POS_SIDES_MENU );

                    InitRect( r,   4, 4, -72, -64 );  *new_menu + new MENU_BUTTON( r, MI_FRIES );
                    MoveRectY( r, 68 );               *new_menu + new MENU_BUTTON( r, MI_ONION_RINGS );
                    MoveRectY( r, 68 );               *new_menu + new MENU_BUTTON( r, MI_ZUCCINI_STICKS );
                    MoveRectY( r, 68 );               *new_menu + new MENU_BUTTON( r, MI_BROCCOLI );
                    MoveRectY( r, 68 );               *new_menu + new MENU_BUTTON( r, MI_MUSHROOMS );
                    MoveRectY( r, 68 );               *new_menu + new MENU_BUTTON( r, MI_MOZZARELLA_STICKS );

                    InitRect( r,  81, 4, -72, -64 );  *new_menu + new MENU_BUTTON( r, MI_RANCH_DRESSING );
                    MoveRectY( r, 68 );               *new_menu + new MENU_BUTTON( r, MI_BBQ_SAUCE );
                    MoveRectY( r, 68 );               *new_menu + new MENU_BUTTON( r, MI_SWEET_N_SOUR );
                    MoveRectY( r, 68 );               *new_menu + new MENU_BUTTON( r, MI_TARTAR_SAUCE );
                    MoveRectY( r, 68 );               *new_menu + new MENU_BUTTON( r, MI_1000_ISLAND );

                    InitRect( r, 158, 4, -72, -64 );

                    InitRect( r, 235, 4, -72, -64 );  *new_menu + new MENU_BUTTON( r, MI_SMALL_CUP );
                    MoveRectY( r, 68 );               *new_menu + new MENU_BUTTON( r, MI_MEDIUM_CUP );
                    MoveRectY( r, 68 );               *new_menu + new MENU_BUTTON( r, MI_LARGE_CUP );
                    MoveRectY( r, 68 );
                    MoveRectY( r, 68 );               *new_menu + new MENU_BUTTON( r, MI_DIET );

                    InitRect( r, 312, 4, -72, -64 );  *new_menu + new MENU_BUTTON( r, MI_RC_COLA );
                    MoveRectY( r, 68 );               *new_menu + new MENU_BUTTON( r, MI_7_UP );
                    MoveRectY( r, 68 );               *new_menu + new MENU_BUTTON( r, MI_ROOT_BEER );
                    MoveRectY( r, 68 );               *new_menu + new MENU_BUTTON( r, MI_ICE_TEA );
                    MoveRectY( r, 68 );               *new_menu + new MENU_BUTTON( r, MI_LEMON_WEDGE );
                    break;

                case MI_CASH_MENU:
                    new_menu->color = kernel->GetScheme( SID_POS_CASH_MENU );

                    *new_menu + new LABEL( 150, 10,  SID_TEXT, "Sub-Total:", 0, 0, WOF_TEXT_RJ )
                              + ( new_menu->sub_total_text = new LABEL( 212,  10, SID_TEXT, "", 0, 0, WOF_TEXT_RJ ))

                              + new LABEL( 150, 30,  SID_TEXT, "Tax:", 0, 0, WOF_TEXT_RJ )
                              + ( new_menu->tax_text = new LABEL( 212,  30, SID_TEXT, "", 0, 0, WOF_TEXT_RJ ))

                              + ( new_menu->order_type_text = new LABEL( 150, 50,  SID_TEXT, "", 0, 0, WOF_TEXT_RJ ))
                              + ( new_menu->order_cost_text = new LABEL( 212,  50, SID_TEXT, "", 0, 0, WOF_TEXT_RJ ))

                              + new LABEL( 150, 70,  SID_TEXT, "Total:", 0, 0, WOF_TEXT_RJ )
                              + ( new_menu->grand_total_text = new LABEL( 212,  70, SID_TEXT, "", 0, 0, WOF_TEXT_RJ ))

                              + new LABEL( 150, 100, SID_TEXT, "Tendered:", 0, 0, WOF_TEXT_RJ )
                              + ( new_menu->tendered_field = new FIELD( 220, 100, SID_TEXT_FIELD, &new_menu->tendered, "9999.99", 0, ((WOF_FIELD & ~WOF_JUSTIFY_LEFT) | WOF_JUSTIFY_RIGHT) ) );

                    InitRect( r,   4, 140, -72, -64 );  *new_menu + new MENU_BUTTON( r, MI_DINING   );
                    MoveRectY( r, 68 );                 *new_menu + new MENU_BUTTON( r, MI_DELIVERY );
                    MoveRectY( r, 68 );                 *new_menu + new MENU_BUTTON( r, MI_TAKE_OUT );

                    InitRect( r,  81, 140, -72, -64 );  *new_menu + new MENU_BUTTON( r, MI_7 );
                    MoveRectY( r, 68 );                 *new_menu + new MENU_BUTTON( r, MI_4 );
                    MoveRectY( r, 68 );                 *new_menu + new MENU_BUTTON( r, MI_1 );
                    MoveRectY( r, 68 );                 *new_menu + new MENU_BUTTON( r, MI_0 );

                    InitRect( r, 158, 140, -72, -64 );  *new_menu + new MENU_BUTTON( r, MI_8 );
                    MoveRectY( r, 68 );                 *new_menu + new MENU_BUTTON( r, MI_5 );
                    MoveRectY( r, 68 );                 *new_menu + new MENU_BUTTON( r, MI_2 );
                    MoveRectY( r, 68 );                 *new_menu + new MENU_BUTTON( r, MI_BACK_SPACE );

                    InitRect( r, 235, 4, -72, -64 );    *new_menu + new MENU_BUTTON( r, MI_1_DOLLAR );
                    MoveRectY( r, 68 );                 *new_menu + new MENU_BUTTON( r, MI_10_DOLLAR );
                    MoveRectY( r, 68 );                 *new_menu + new MENU_BUTTON( r, MI_9 );
                    MoveRectY( r, 68 );                 *new_menu + new MENU_BUTTON( r, MI_6 );
                    MoveRectY( r, 68 );                 *new_menu + new MENU_BUTTON( r, MI_3 );
                    MoveRectY( r, 68 );                 *new_menu + new MENU_BUTTON( r, MI_CLEAR_ENTRY );

                    InitRect( r, 312, 4, -72, -64 );    *new_menu + new MENU_BUTTON( r, MI_5_DOLLAR );
                    MoveRectY( r, 68 );                 *new_menu + new MENU_BUTTON( r, MI_20_DOLLAR );
                    MoveRectY( r, 68 );                 *new_menu + new MENU_BUTTON( r, MI_CASH );
                    MoveRectY( r, 68 );                 *new_menu + new MENU_BUTTON( r, MI_CHECK );
                    MoveRectY( r, 68 );                 *new_menu + new MENU_BUTTON( r, MI_VISA );
                    MoveRectY( r, 68 );                 *new_menu + new MENU_BUTTON( r, MI_MASTER_CARD );

                    // get the latest sub total and sale type

                    event.type      = EV_GET_SUB_TOTAL;
                    event.target_id = OI_TICKET_MANAGER;
                    AddBGEvent( event );

                    break;

                default:
                    delete new_menu;
                    new_menu = NULL;
                }
            }

            if ( new_menu )
                kernel->ReplaceWindow( current_menu_p, new_menu, event.area );

            current_menu_n = event.code;
            current_menu_p = new_menu;
        }
        break;

    case EV_SUB_TOTAL:
    case EV_KEYBOARD:
        if ( current_menu_p )
            current_menu_p->Event( event );
        break;

    default:
        ret = ES_NOT_USED;

    }

    return ret;

}


EVENT_STATUS MENU_MANAGER::Poll( void )
{
    return ES_OK;

}



// -------------------
// ticket item manager

TICKET_LIST * TICKET_MANAGER::ticket_list = NULL;


TICKET_MANAGER::TICKET_MANAGER( void )
{
    object_id = OI_TICKET_MANAGER;

    sub_total         = 0;

    current_sale_type = MI_DINING;

}

TICKET_MANAGER::~TICKET_MANAGER()
{
    if ( itemList.Begin() )
    {
        do
        {
            delete (*itemList);

        }
        while( itemList.Next() );

        itemList.Flush();
    }

}

EVENT_STATUS TICKET_MANAGER::Event( EVENT & event )
{
    EVENT_STATUS ret = ES_OK;

    TICKET_ITEM * new_item,
                * item_p = NULL;

    int item_n;

    // syncronize the lists

    if ( ticket_list )
    {
        item_p = (TICKET_ITEM *) ticket_list->p_selected;
        item_n = ticket_list->n_selected;

        itemList.Goto( item_n );
    }

    long old_cost = item_p ? item_p->cost : 0;

    switch( event.type )
    {
    case EV_CLEAR_TICKET:
        if ( itemList.Begin() )
        {
            if ( ticket_list )
                ticket_list->Clear();

            do
            {
                delete (*itemList);
            }
            while( itemList.Next() );

            itemList.Flush();

            current_sale_type = MI_DINING;

            old_cost          = sub_total;  // set 'old_cost' so it will reset the sub-total amount
            item_p            = NULL;
        }
        break;

    case EV_EDIT_ITEM:
        switch( event.code )
        {
        case C_MORE:                                                    // increment the quantity
            if ( item_p )
            {
                item_p->Quantity( item_p->quantity + 1 );

                if ( ticket_list )
                    ticket_list->IsVisible( item_n, true );
            }
            break;

        case C_LESS:                                                    // decrement the quantity
            if ( item_p && item_p->quantity > 0 )
            {
                item_p->Quantity( item_p->quantity - 1 );

                if ( ticket_list )
                    ticket_list->IsVisible( item_n, true );
            }
            break;

        case C_NO:                                                      // toggle the "NO" text
            if ( item_p && item_p->type == IT_MODIFIER )
            {
                item_p->Quantity( ((item_p->quantity > 0) ? 0 : 1) );

                if ( ticket_list )
                    ticket_list->IsVisible( item_n, true );
            }
            break;

        case C_TRASH:                                                   // delete an modifier or food + modifier
            if ( item_p )
            {
                ITEM_TYPE type = item_p->type;      // get a copy of the item type

                itemList.Goto( item_n );            // move to the item index in the list

                if ( (*itemList) == item_p )        // do they match ??
                {
                    old_cost = item_p->cost;

                    if ( ticket_list )              // is there a ticket-list on the display ?
                        *ticket_list -= item_p;     // delete the item from the list

                    delete (*itemList);             // remove the item from the ticket-manager's list
                    itemList.Remove();

                    // if the item was a FOOD item, delete the trailing modifiers

                    if ( type == IT_FOOD && itemList.Index() <= (itemList.Count() - 1) )
                        while( (*itemList)->type != IT_FOOD && item_n < itemList.Count() )
                        {
                            old_cost += (*itemList)->cost;

                            if ( ticket_list )
                                *ticket_list -= (*itemList);

                            delete (*itemList);
                            itemList.Remove();
                        }
                }

                item_p = NULL;      // set to NULL so cost calculation works
            }
            break;
        }
        break;

    case EV_SELECT_ITEM:
        switch( menu_list[ event.code ].type )
        {
        case IT_FOOD:
            if ( item_p && item_p->type == IT_SIZE )
            {
                item_p->Item( (MENU_ITEMS) event.code );

                if ( ticket_list )
                    ticket_list->IsVisible( item_n, true );
            }
            else if ( item_p && item_p->item_index == (MENU_ITEMS) event.code )
            {
                item_p->Quantity( item_p->quantity + 1 );

                if ( ticket_list )
                    ticket_list->IsVisible( item_n, true );
            }
            else
            {
                new_item = new TICKET_ITEM( (MENU_ITEMS) event.code );

                new_item->Quantity( 1 );

                itemList.Add( new_item );

                if ( ticket_list )
                    ticket_list->Add( new_item );

                old_cost = 0;
                item_p   = new_item;
            }
            break;

        case IT_MODIFIER:
            if ( item_p && item_p->type == IT_FOOD || item_p->type == IT_MODIFIER )
            {
                if ( item_p->type == IT_MODIFIER && item_p->item_index == (MENU_ITEMS) event.code )
                {
                    item_p->Quantity( item_p->quantity + 1 );

                    if ( ticket_list )
                        ticket_list->IsVisible( item_n, true );
                }
                else
                {
                    new_item = new TICKET_ITEM( (MENU_ITEMS) event.code );
                    new_item->Quantity( 1 );

                    itemList.PostInsert( new_item );

                    if ( ticket_list )
                        ticket_list->Insert( new_item );

                    old_cost = 0;
                    item_p   = new_item;
                }
            }
            break;

        case IT_SIZE:
            if ( item_p == NULL || item_p->type != IT_SIZE )
            {
                new_item = new TICKET_ITEM( (MENU_ITEMS) event.code );
                new_item->Quantity( 1 );

                itemList.Add( new_item );

                if ( ticket_list )
                    ticket_list->Add( new_item );

                old_cost = 0;
                item_p   = new_item;
            }
            else
            {
                item_p->Size( (MENU_ITEMS) event.code );

                if ( ticket_list )
                    ticket_list->IsVisible( item_n, true );
            }
            break;

        }

        if ( ticket_list && ticket_list->IsVisible( ticket_list->n_selected ) )
            display->ExplodeRectangle( event.area, ticket_list->p_selected->object_area );

        break;

    case EV_GET_SUB_TOTAL:
        event.target_id     = OI_MENU_MANAGER;      // send the event back to the menu manager
        event.type          = EV_SUB_TOTAL;         // change the event to a sub total event
        event.code          = current_sale_type;    // get the current sale type
        event.data.long_t.a = sub_total;            // get the current sub total

        AddBGEvent( event );                        // send the event

        item_p   = NULL;    // defeat the sub-total recalc
        old_cost = 0;

        break;

    case EV_SALE_TYPE_KEY:

        if ( current_sale_type != (MENU_ITEMS) event.code )
        {
            current_sale_type   = (MENU_ITEMS) event.code;  // store the new sale type

            event.type          = EV_SUB_TOTAL;             // change the event to a sub-total event
            event.target_id     = OI_MENU_MANAGER;          // send it to the menu manager so it can send to the menu
            event.data.long_t.a = sub_total;                // copy the latest sub total value

            AddBGEvent( event );                            // send the event
        }

        item_p   = NULL;    // defeat the sub-total recalc
        old_cost = 0;

        break;
    }

    long new_cost = item_p ? item_p->cost : 0;

    if ( old_cost != new_cost )
    {
        sub_total += ( new_cost - old_cost );

        event.type          = EV_SUB_TOTAL;
        event.data.long_t.a = sub_total;

        if ( ticket_list )
            ticket_list->owner->Event( event );

        event.target_id = OI_MENU_MANAGER;
        AddBGEvent( event );
    }

    return ret;

}


EVENT_STATUS TICKET_MANAGER::Poll( void )
{
    return ES_OK;
}





// --------------------
// a ticket item object

TICKET_ITEM::TICKET_ITEM
(
    MENU_ITEMS index

) : WINDOW_OBJECT()
{
    flags &= ~WOF_CAN_DELETE;

    selected = false;

    size       = 0;
    t_size     = NULL;
    t_name     = NULL;
    type       = IT_UNKNOWN;
    item_index = MI_NONE;
    size_index = MI_NONE;

    if ( index > MI_NONE )
    {
        if ( (type = menu_list[ index ].type) == IT_SIZE )
            t_size = menu_list[ size_index = index ].name;
        else
            t_name = menu_list[ item_index = index ].name;
    }

    color = kernel->GetScheme( SID_ITEM );

}


EVENT_STATUS TICKET_ITEM::Event( EVENT & event )
{
    EVENT_STATUS ret = ES_OK;

    switch( event.type )
    {
    case EV_SET_COORDS:
        WINDOW_OBJECT::Event( event );
        y_quan = ( RectHeight(object_area) - display->GetFontHeight( 0 ) ) / 2;
        break;

    case EV_SELECT:
        selected = true;
        break;

    case EV_UNSELECT:
        selected = false;
        break;

    default:
        ret = WINDOW_OBJECT::Event( event );
    }

    return ret;
}


void TICKET_ITEM::Draw( void )
{
    display->SetDrawColor( selected == true ? color->fill : color->text );

    if ( t_name )
        if ( type == IT_FOOD || type == IT_MODIFIER )
            display->DrawString( object_area.left + x_item, object_area.top + y_quan, t_name, 0 );

    if ( type != IT_MODIFIER )
    {
        if ( t_size )
            display->DrawString( object_area.left + x_size, object_area.top + y_quan, t_size, 0 );

        display->DrawString( object_area.right - x_cost, object_area.top + y_quan, t_cost, 0 );
    }

    if ( type == IT_MODIFIER && quantity == 0 )
        display->SetDrawColor( lightred );

    if ( type != IT_MODIFIER || quantity > 1 || quantity == 0 )
        display->DrawString( object_area.left + x_quan, object_area.top + y_quan, t_quan, 0 );

}


void TICKET_ITEM::Item( MENU_ITEMS index )
{
    if ( index > MI_NONE )
    {
        if ( index == item_index )
            Quantity( quantity + 1 );

        else if ( type == IT_SIZE && menu_list[ index ].type == IT_FOOD )
        {
            item_index = index;
            t_name     = menu_list[ index ].name;
            type       = IT_FOOD;

            Size( size_index );
        }
    }

}


void TICKET_ITEM::Quantity( int q )
{
    quantity = q;

    if ( type == IT_FOOD )
        cost = quantity * menu_list[ item_index ].cost[ size ];
    else
        cost = 0;

    if ( type == IT_MODIFIER )
    {
        if ( quantity > 0 )
            sprintf( t_quan, "%d X", quantity );
        else if ( quantity == 0 )
            strcpy( t_quan, "NO " );
    }
    else
        sprintf( t_quan, "%d", quantity );

    sprintf( t_cost, "%2.2f", cost / 100.0 );

    display->SelectFont( 0 );


    x_quan = 3;

    if ( type == IT_MODIFIER )
        x_quan += display->font_width * 3;

    x_size = x_quan + display->GetStringWidth( t_quan ) + display->font_width;

    x_item = x_size;

    if ( type != IT_MODIFIER )
        x_item += display->GetStringWidth( t_size ) + display->font_width;

    x_cost = display->GetStringWidth( t_cost ) + 3;

}


void TICKET_ITEM::Size( MENU_ITEMS index )
{
    size_index = index;

    size   = 0;
    t_size = menu_list[ size_index ].name;

    if ( type == IT_FOOD )
    {
        int s = menu_list[ size_index ].size[ 0 ];          // grab the size value

        for( int n = 0; n < MAX_SIZES; n++ )                // locate it in the food item's table
            if ( menu_list[ item_index ].size[ n ] == s )
            {
                size   = n;
                Quantity( quantity );
                return;
            }

        t_size = NULL;
        Quantity( quantity );
    }

}





POS_MENU::POS_MENU( void )
{
    sub_total_text =
        order_type_text  =
        order_cost_text  =
        tax_text         =
        grand_total_text = NULL;

    sub_total_buf[ 0 ] =
        tax_buf[ 0 ]         =
        order_cost_buf[ 0 ]  =
        grand_total_buf[ 0 ] = '\0';

    tendered_field = NULL;

    tendered = 0.0;
}


EVENT_STATUS POS_MENU::Event( EVENT & event )
{
    EVENT_STATUS ret = ES_OK;

    switch( event.type )
    {
    case EV_SUB_TOTAL:
        if ( sub_total_text )
        {
            sprintf( sub_total_buf, "%3.2f", event.data.long_t.a / 100.0 );
            sub_total_text->ChangeData( sub_total_buf );

            if ( order_type_text )
                order_type_text->ChangeData( menu_list[ event.code ].name );

            int tc = menu_list[ event.code ].cost[ 0 ];

            sprintf( order_cost_buf, "%3.2f", tc / 100.0 );

            if ( order_cost_text )
                order_cost_text->ChangeData( order_cost_buf );

            long tx = event.data.long_t.a * .0725;

            sprintf( tax_buf, "%3.2f", tx / 100.0 );

            if ( tax_text )
                tax_text->ChangeData( tax_buf );

            long gt = event.data.long_t.a + tc + tx;

            sprintf( grand_total_buf, "%3.2f", gt / 100.0 );

            if ( grand_total_text )
                grand_total_text->ChangeData( grand_total_buf );

        }
        break;

    case EV_PRESET_DOLLAR_KEY:
        if ( event.code == 0 )
            tendered = 0.0;
        else
            tendered += (double) event.code / 100.0;

        if ( tendered_field )
            tendered_field->ChangeData();

        break;

    case EV_KEYBOARD:
        if ( tendered_field )
            ret = tendered_field->Event( event );
        break;

    default:
        ret = WINDOW::Event( event );
    }

    return ret;
}


// ------------------------
// the menu selector window


POS_MENU_WINDOW::POS_MENU_WINDOW( void ) : WINDOW()
{
    InitRect( object_area, 250, 0, 640, 68 );

    color = kernel->GetScheme( SID_POS_MAIN_WIN );

    SBH = kernel->system_button_height;
    SBW = kernel->system_button_width;

    kernel->system_button_height =
        kernel->system_button_width = 50;

    gRECT r = { 4, 4, -60, -60 };         *this + new MENU_BUTTON( r, MI_MAIN_MENU );
    MoveRectX( r, 64 );                  *this + new MENU_BUTTON( r, MI_SIDES_MENU );
    MoveRectX( r, 64 );                  *this + new MENU_BUTTON( r, MI_CASH_MENU );
    MoveRectX( r, 64 );                  *this + new COMBO_BUTTON( r, SID_BUTTON, "Exit", 0, WOF_L_BUTTON | WOF_SEND_QUEUE, EV_POS_EXIT );
    MoveRectX( r, 64 ); r.right = -126;  *this + new CLOCK( r, SID_BUTTON, SID_CLOCK, 0, CSET_FLAGS, COBJ_FLAGS );

    // tell the menu manager to initialize itself

    EVENT ev;

    ev.type      = EV_SELECT_MENU;
    ev.code      = MI_MAIN_MENU;
    ev.target_id = OI_MENU_MANAGER;
    ev.area      = kernel->explode_default;

    AddBGEvent( ev );

}


EVENT_STATUS POS_MENU_WINDOW::Event( EVENT & event )
{
    EVENT_STATUS ret = ES_OK;

    switch( event.type )
    {
    case EV_POS_EXIT:

        // restore the system button size

        kernel->system_button_height = SBH;
        kernel->system_button_width  = SBW;

        // select a non-menu

        EVENT ev;

        ev.type = EV_SELECT_MENU;
        ev.code = PM_NONE;
        ev.target_id = OI_MENU_MANAGER;

        AddBGEvent( ev );

        // close all windows

        AddUIEvent( EV_CLOSE_ALL );

        // add the main window

        ev.type         = EV_ADD_WINDOW;
        ev.area         = event.area;
        ev.target_id    = OI_KERNEL;
        ev.ptr.window_t = new MAIN_WINDOW;

        AddUIEvent( ev );

        // tell the TICKET_MANAGER to dump the items

        ev.type = EV_CLEAR_TICKET;
        ev.target_id = OI_TICKET_MANAGER;
        AddBGEvent( ev );
        break;


    default:
        ret = WINDOW::Event( event );
    }

    return ret;

}






// ----------------------
// a ticket item list box

TICKET_LIST::TICKET_LIST
(
    gRECT r,
    SCHEME_ID s

) : VLIST( r, s, WOF_VLIST | WOF_SEND_OWNER, NO_ITEM_SELECTED, 40 )
{
    TICKET_MANAGER::ticket_list = this;

    scroll_changes = true;
}


TICKET_LIST::~TICKET_LIST()
{
    TICKET_MANAGER::ticket_list = NULL;
}


void TICKET_LIST::Add( WINDOW_OBJECT * element )
{
    *this << element;
}

void TICKET_LIST::Insert( WINDOW_OBJECT * element )
{
    *this <= element;
}





// --------------------------------------------------------
// main POS window with ticket list, exit button, and clock

POS_TICKET_WINDOW::POS_TICKET_WINDOW( void )
{
    // set the size and color scheme

    InitRect( object_area, 0, 0, 250, 480 );
    color = kernel->GetScheme( SID_POS_TICKET_WIN );


    // add a ticket item viewer

    gRECT r;

    InitRect( r, 5, 5, -240, -380 );
    *this + ( ticket_list = new TICKET_LIST( r, SID_VLIST ) );

    *ticket_list + new SCROLL_BAR( SID_SCROLL, SID_BUTTON, 0, 0, 0, SB_V_EDGE, WOF_SCROLL_BAR );


    // add a ticket sub-total display

    strcpy( sub_total_buf, "0.00" );

    int x1 = 245 - ( kernel->system_button_width + 3 );
    display->SelectFont( 0 );
    int x2 = x1 - ( 8 * display->font_width );

    *this + new LABEL( x2, 395, SID_TEXT, "Sub-total:", 0, 0, WOF_TEXT_RJ )
          + ( sub_total_text = new LABEL( x1, 395, SID_TEXT, sub_total_buf, 0, 0, WOF_TEXT_RJ ) );


    // add some control buttons

    InitRect( r, 5, 415, -60, -60 );
    *this + new COMBO_BUTTON( r, SID_BUTTON, "More", 0, ico_plus, WOF_P_BUTTON | WOF_SEND_QUEUE, EV_EDIT_ITEM, C_MORE );

    MoveRectX( r, 60 );
    *this + new COMBO_BUTTON( r, SID_BUTTON, "Less", 0, ico_minus, WOF_P_BUTTON | WOF_SEND_QUEUE, EV_EDIT_ITEM, C_LESS );

    MoveRectX( r, 60 );
    *this + new COMBO_BUTTON( r, SID_BUTTON, "No", 0, ico_no, WOF_P_BUTTON | WOF_SEND_QUEUE, EV_EDIT_ITEM, C_NO );

    MoveRectX( r, 60 );
    *this + new COMBO_BUTTON( r, SID_BUTTON, "Cancel", 0, ico_trash, WOF_P_BUTTON | WOF_SEND_QUEUE, EV_EDIT_ITEM, C_TRASH );

}


EVENT_STATUS POS_TICKET_WINDOW::Event( EVENT & event )
{
    EVENT_STATUS ret = ES_OK;

    switch( event.type )
    {
    case EV_KEYBOARD:
        event.target_id = OI_MENU_MANAGER;
        AddUIEvent( event );
        break;

    case EV_VLIST_SELECT:
        event.target_id = OI_TICKET_MANAGER;
        AddBGEvent( event );
        break;

    case EV_EDIT_ITEM:
        if ( event.data.mouse_t.button > M_NO_BUTTONS )
        {
            event.target_id = OI_TICKET_MANAGER;
            AddBGEvent( event );
        }
        break;

    case EV_SUB_TOTAL:
        sprintf( sub_total_buf, "%3.2f", event.data.long_t.a / 100.0 );
        sub_total_text->ChangeData( sub_total_buf );
        break;

    default:
        ret = WINDOW::Event( event );
    }

    return ret;

}







// ------------------
// a menu item button

MENU_BUTTON::MENU_BUTTON
(
    gRECT &     r,
    MENU_ITEMS index

) : COMBO_BUTTON( r, menu_list[ index ].scheme_id,
                     menu_list[ index ].menu,
                     menu_list[ index ].font,
                     menu_list[ index ].icon,
                     WOF_P_BUTTON )
{
    item_index = index;
}



void MENU_BUTTON::GetIconSize( int * x, int * y )
{
    if ( menu_list[ item_index ].icon_type == PCX_TYPE )
        *x = *y = 0;
//        size_PCX( menu_list[ item_index ].icon, x, y );
    else
        COMBO_BUTTON::GetIconSize( x, y );
}


void MENU_BUTTON::DrawIcon( void )
{
    if ( menu_list[ item_index ].icon_type == PCX_TYPE )
    {
//        int error = 0;
//
//#if defined USE_BGI
//        error = draw_PCX( (char *) menu_list[ item_index ].icon, data_area.left + icon_x, data_area.top + icon_y, 0 );
//#elif definded USE_META
//        rect m = { data_area.left + icon_x,
//                   data_area.top  + icon_y,
//                   data_area.left + icon_x + 75,
//                   data_area.top  + icon_y + 75 };
//
//        error = ReadPCX( (char *) menu_list[ item_index ].icon, NULL, &m );
//#endif
//        if ( error )
//            error = error;
    }
    else
        COMBO_BUTTON::DrawIcon();
}

void MENU_BUTTON::Draw( void )
{
    COMBO_BUTTON::Draw();

    gPOINT p[3] = { { data_area.left,    data_area.top },
                   { data_area.left+15, data_area.top },
                   { data_area.left,    data_area.top+15 } };

    display->DrawFilledPolygon( p, 3, kernel->GetScheme( SID_W_BLUE ) );

}


void MENU_BUTTON::Selected( boolean status )
{
    COMBO_BUTTON::Selected( status );

    if ( status == true )
    {
        EVENT ev;

        ev.type = EV_UNDEFINED;
        ev.code = item_index;
        ev.area = object_area;

        switch( menu_list[ item_index ].type )
        {
        case IT_MENU:
            ev.type      = EV_SELECT_MENU;
            ev.target_id = OI_MENU_MANAGER;
            break;

        case IT_MODIFIER:
        case IT_FOOD:
        case IT_SIZE:
            ev.type      = EV_SELECT_ITEM;
            ev.target_id = OI_TICKET_MANAGER;
            break;

        case IT_SALE_KEY:
            ev.type      = EV_SALE_TYPE_KEY;
            ev.target_id = OI_TICKET_MANAGER;
            break;

        case IT_PAY_KEY:
            ev.type      = EV_SELECT_MENU;
            ev.code      = MI_MAIN_MENU;
            ev.target_id = OI_MENU_MANAGER;
            ev.area      = kernel->explode_default;

            AddBGEvent( ev );

            ev.type      = EV_CLEAR_TICKET;
            ev.target_id = OI_TICKET_MANAGER;
            break;

        case IT_NUMBER_KEY:
            ev.type      = EV_KEYBOARD;
            ev.target_id = OI_KERNEL;
            ev.data.key_t.ascii = menu_list[ item_index ].size[ 0 ];
            break;

        case IT_DOLLAR_KEY:
            ev.type      = EV_PRESET_DOLLAR_KEY;
            ev.target_id = OI_KERNEL;
            ev.code      = menu_list[ item_index ].size[ 0 ];
            break;
        }

        if ( ev.type != EV_UNDEFINED )
        {
            if ( ev.type == EV_SELECT_ITEM )
                AddBGEvent( ev );
            else
                AddUIEvent( ev );
        }
    }
}

// end of file: G_TEST.CPP
