/* Copyright (C) 1989, 1992, 1993, 1994 Aladdin Enterprises.  All rights reserved.
  
  This file is part of Aladdin Ghostscript.
  
  Aladdin Ghostscript is distributed with NO WARRANTY OF ANY KIND.  No author
  or distributor accepts any responsibility for the consequences of using it,
  or for whether it serves any particular purpose or works at all, unless he
  or she says so in writing.  Refer to the Aladdin Ghostscript Free Public
  License (the "License") for full details.
  
  Every copy of Aladdin Ghostscript must include a copy of the License,
  normally in a plain ASCII text file named PUBLIC.  The License grants you
  the right to copy, modify and redistribute Aladdin Ghostscript, but only
  under certain conditions described in the License.  Among other things, the
  License requires that the copyright notice and this notice be preserved on
  all copies.
*/

/* gxdevice.h */
/* Device description structure for Ghostscript library */

#ifndef gxdevice_INCLUDED
#  define gxdevice_INCLUDED

#include "gsmatrix.h"
#include "gsxfont.h"
#include "gxbitmap.h"
#include "gxdcolor.h"

/* See drivers.doc for documentation of the driver interface. */
#ifndef gx_device_DEFINED
#  define gx_device_DEFINED
typedef struct gx_device_s gx_device;
#endif

/* We need at least an abstract type for a graphics state, */
/* which is passed to the page device procedures. */
#ifndef gs_state_DEFINED
#  define gs_state_DEFINED
typedef struct gs_state_s gs_state;
#endif

/* Define the type for gray or RGB values at the driver interface. */
typedef unsigned short gx_color_value;
#define arch_sizeof_gx_color_value arch_sizeof_short
/* We might use less than the full range someday. */
/* ...bits must lie between 8 and 16. */
#define gx_color_value_bits (sizeof(gx_color_value) * 8)
#define gx_max_color_value ((gx_color_value)((1L << gx_color_value_bits) - 1))
#define gx_color_value_to_byte(cv)\
  ((cv) >> (gx_color_value_bits - 8))
#define gx_color_value_from_byte(cb)\
  (((cb) << (gx_color_value_bits - 8)) + ((cb) >> (16 - gx_color_value_bits)))

/* Define the structure for device color capabilities. */
typedef struct gx_device_color_info_s {
	int num_components;		/* 1 = gray only, 3 = RGB, */
					/* 4 = CMYK */
	int depth;			/* # of bits per pixel */
	gx_color_value max_gray;	/* # of distinct gray levels -1 */
	gx_color_value max_rgb;		/* # of distinct color levels -1 */
					/* (only relevant if num_comp. > 1) */
	gx_color_value dither_gray;	/* size of gray ramp for dithering */
	gx_color_value dither_rgb;	/* size of color cube ditto */
					/* (only relevant if num_comp. > 1) */
} gx_device_color_info;
#define dci_black_and_white { 1, 1, 1, 0, 2, 0 }
#define dci_color(depth,maxv,dither) { 3, depth, maxv, maxv, dither, dither }
#define gx_device_has_color(dev) ((dev)->color_info.num_components > 1)

/* Structure for device procedures */
typedef struct gx_device_procs_s gx_device_procs;

/* Structure for standard "page device" properties. */
typedef struct gx_pagedevice_s {

	float page_size[2];		/* PageSize */
	bool imaging_bbox_set;
	float imaging_bbox[4];		/* ImagingBBox */
#if 0
	/* We use x/y_pixels_per_inch instead of hw_resolution, */
	/* for backward compatibility. */
	float hw_resolution[2];		/* HWResolution */
#endif
	float margins[2];		/* Margins */
	bool orientation_set;
	int orientation;		/* Orientation */

	/* "Page device" procedures. */
	/* Note that these take the graphics state as a parameter. */

#define dev_page_proc_install(proc)\
  int proc(P2(gx_device *dev, gs_state *pgs))
	dev_page_proc_install((*install));

#define dev_page_proc_begin_page(proc)\
  int proc(P2(gx_device *dev, gs_state *pgs))
	dev_page_proc_begin_page((*begin_page));

#define dev_page_proc_end_page(proc)\
  int proc(P3(gx_device *dev, int reason, gs_state *pgs))
	dev_page_proc_end_page((*end_page));

} gx_pagedevice;
/* Default procedures */
dev_page_proc_install(gx_default_install);
dev_page_proc_begin_page(gx_default_begin_page);
dev_page_proc_end_page(gx_default_end_page);
/* Default pagedevice value */
#define gx_pagedevice_defaults\
 { {612, 792}, false, {0, 0, 612, 792}, /*{300, 300},*/ {0, 0}, false, 0,\
   gx_default_install, gx_default_begin_page, gx_default_end_page\
 }

/*
 * Structure for generic device description.  The device procedures can
 * have three different configurations:
 * 
 *	- Statically initialized devices predating release 2.8.1
 *	set the procs pointer to point to a separate procedure record,
 *	and do not initialize std_procs.
 *
 *	- Statically initialized devices starting with release 2.8.1
 *	set the procs pointer to 0, and initialize std_procs.
 *
 *	- Dynamically initialized devices set the procs pointer to
 *	point to the embedded std_procs record.
 *
 * The gx_device_set_procs procedure converts the first two of these to
 * the third, which is what all client code starting in 2.8.1 expects
 * (using the std_procs record, not the procs pointer, to call the
 * driver procedures).
 */
#define gx_device_common\
	int params_size;		/* OBSOLETE */\
					/* size of this structure */\
	gx_device_procs *procs;		/* OBSOLETE */\
					/* pointer to std_procs */\
	const char *dname;		/* the device name */\
	int width;			/* width in pixels */\
	int height;			/* height in pixels */\
	float x_pixels_per_inch;	/* x density */\
	float y_pixels_per_inch;	/* y density */\
	float l_margin, b_margin, r_margin, t_margin;	/* margins around */\
					/* imageable area, in inches */\
	gx_device_color_info color_info;	/* color information */\
		/* dev_init_misc initializes all values between */\
		/* this point and std_procs */\
	gs_memory_t *memory;		/* (0 if statically allocated instance) */\
	bool is_open;			/* true if device has been opened */\
	long page_count;		/* number of pages written */\
	long showpage_count;		/* number of calls on showpage */\
	struct {\
		gx_color_index white;	/* (cache) device white color */\
		gx_color_index black;	/* (cache) device black color */\
	} cached;\
	gx_pagedevice page;\
		/* end of dev_init_misc */\
	gx_device_procs std_procs	/* standard procedures */
#define no_margins 0, 0, 0, 0
/* Because some compilers don't allow casts in initialization expressions, */
/* the preopen argument must be an int, not a bool. */
#define dev_init_misc_is_open_(preopen)\
  NULL, preopen, 0, 0,\
  { (preopen ? 1 : gx_no_color_index), (preopen ? 0 : gx_no_color_index) },\
  gx_pagedevice_defaults
#define dev_init_misc dev_init_misc_is_open_(0 /*false*/)
#define dev_init_open_misc dev_init_misc_is_open_(1 /*true*/)
/* Accessors for device procedures */
#define dev_proc(dev, p) ((dev)->std_procs.p)
#define set_dev_proc(dev, p, proc) ((dev)->std_procs.p = (proc))
#define fill_dev_proc(dev, p, dproc)\
  if ( dev_proc(dev, p) == 0 ) set_dev_proc(dev, p, dproc)

/* Define an opaque type for parameter lists. */
#ifndef gs_param_list_DEFINED
#  define gs_param_list_DEFINED
typedef struct gs_param_list_s gs_param_list;
#endif

/* Definition of device procedures. */
/* Note that the gx_device * argument is not declared const, */
/* because many drivers maintain dynamic state in the device structure. */

struct gx_device_procs_s {

#define dev_proc_open_device(proc)\
  int proc(P1(gx_device *dev))
	dev_proc_open_device((*open_device));

#define dev_proc_get_initial_matrix(proc)\
  void proc(P2(gx_device *dev, gs_matrix *pmat))
	dev_proc_get_initial_matrix((*get_initial_matrix));

#define dev_proc_sync_output(proc)\
  int proc(P1(gx_device *dev))
	dev_proc_sync_output((*sync_output));

#define dev_proc_output_page(proc)\
  int proc(P3(gx_device *dev, int num_copies, int flush))
	dev_proc_output_page((*output_page));

#define dev_proc_close_device(proc)\
  int proc(P1(gx_device *dev))
	dev_proc_close_device((*close_device));

#define dev_proc_map_rgb_color(proc)\
  gx_color_index proc(P4(gx_device *dev,\
    gx_color_value red, gx_color_value green, gx_color_value blue))
	dev_proc_map_rgb_color((*map_rgb_color));

#define dev_proc_map_color_rgb(proc)\
  int proc(P3(gx_device *dev,\
    gx_color_index color, gx_color_value rgb[3]))
	dev_proc_map_color_rgb((*map_color_rgb));

#define dev_proc_fill_rectangle(proc)\
  int proc(P6(gx_device *dev,\
    int x, int y, int width, int height, gx_color_index color))
	dev_proc_fill_rectangle((*fill_rectangle));

#define dev_proc_tile_rectangle(proc)\
  int proc(P10(gx_device *dev,\
    const gx_tile_bitmap *tile, int x, int y, int width, int height,\
    gx_color_index color0, gx_color_index color1,\
    int phase_x, int phase_y))
	dev_proc_tile_rectangle((*tile_rectangle));

#define dev_proc_copy_mono(proc)\
  int proc(P11(gx_device *dev,\
    const unsigned char *data, int data_x, int raster, gx_bitmap_id id,\
    int x, int y, int width, int height,\
    gx_color_index color0, gx_color_index color1))
	dev_proc_copy_mono((*copy_mono));

#define dev_proc_copy_color(proc)\
  int proc(P9(gx_device *dev,\
    const unsigned char *data, int data_x, int raster, gx_bitmap_id id,\
    int x, int y, int width, int height))
	dev_proc_copy_color((*copy_color));

#define dev_proc_draw_line(proc)\
  int proc(P6(gx_device *dev,\
    int x0, int y0, int x1, int y1, gx_color_index color))
	dev_proc_draw_line((*draw_line));

		/* Added in release 2.4 */

#define dev_proc_get_bits(proc)\
  int proc(P4(gx_device *dev,\
    int y, unsigned char *data, unsigned char **actual_data))
	dev_proc_get_bits((*get_bits));

		/* Added in release 2.4, changed in 2.8, */
		/* renamed in 2.9.6 */

#define dev_proc_get_params(proc)\
  int proc(P2(gx_device *dev, gs_param_list *plist))
	dev_proc_get_params((*get_params));
    
#define dev_proc_put_params(proc)\
  int proc(P2(gx_device *dev, gs_param_list *plist))
	dev_proc_put_params((*put_params));

		/* Added in release 2.6 */

#define dev_proc_map_cmyk_color(proc)\
  gx_color_index proc(P5(gx_device *dev,\
    gx_color_value cyan, gx_color_value magenta, gx_color_value yellow,\
    gx_color_value black))
	dev_proc_map_cmyk_color((*map_cmyk_color));
    
#define dev_proc_get_xfont_procs(proc)\
  gx_xfont_procs *proc(P1(gx_device *dev))
	dev_proc_get_xfont_procs((*get_xfont_procs));

		/* Added in release 2.6.1 */

#define dev_proc_get_xfont_device(proc)\
  gx_device *proc(P1(gx_device *dev))
	dev_proc_get_xfont_device((*get_xfont_device));

		/* Added in release 2.7.1 */

#define dev_proc_map_rgb_alpha_color(proc)\
  gx_color_index proc(P5(gx_device *dev,\
    gx_color_value red, gx_color_value green, gx_color_value blue,\
    gx_color_value alpha))
	dev_proc_map_rgb_alpha_color((*map_rgb_alpha_color));

		/* Added in release 2.8.1 */

#define dev_proc_get_page_device(proc)\
  gx_device *proc(P1(gx_device *dev))
	dev_proc_get_page_device((*get_page_device));

};

/* A generic device */
struct gx_device_s {
	gx_device_common;
};
extern_st(st_device);
#define public_st_device()	/* in gsdevice.c */\
  gs_public_st_simple(st_device, gx_device, "gx_device")
#define st_device_max_ptrs 0

/* Enumerate or relocate a pointer to a device. */
/* These take the containing space into account properly. */
gx_device *gx_device_enum_ptr(P1(gx_device *));
gx_device *gx_device_reloc_ptr(P2(gx_device *, gc_state_t *));

/* Define typedefs for some of the device procedures, because */
/* ansi2knr can't handle dev_proc_xxx((*xxx)) in a formal argument list. */
typedef dev_proc_map_rgb_color((*dev_proc_map_rgb_color_t));
typedef dev_proc_map_color_rgb((*dev_proc_map_color_rgb_t));

/* A device that forwards non-display operations to another device */
/* called the "target".  This is used for clipping, banding, image, */
/* and null devices. */
#define gx_device_forward_common\
	gx_device_common;\
	gx_device *target
/* A generic forwarding device. */
typedef struct gx_device_forward_s {
	gx_device_forward_common;
} gx_device_forward;
extern_st(st_device_forward);
#define public_st_device_forward()	/* in gsdevice.c */\
  gs_public_st_composite(st_device_forward, gx_device_forward,\
    "gx_device_forward", device_forward_enum_ptrs, device_forward_reloc_ptrs)
#define st_device_forward_max_ptrs (st_device_max_ptrs + 1)

/* A null device.  This is used to temporarily disable output. */
typedef struct gx_device_null_s {
	gx_device_forward_common;
} gx_device_null;
extern_st(st_device_null);
#define public_st_device_null()	/* in gsdevice.c */\
  gs_public_st_suffix_add0(st_device_null, gx_device_null, "gx_device_null",\
    device_forward_enum_ptrs, device_forward_reloc_ptrs)
#define st_device_null_max_ptrs st_device_forward_max_ptrs
/* Make a null device. */
/* The gs_memory_t argument is 0 if the device is temporary and local, */
/* or the allocator that was used to allocate it if it is a real object. */
void	gs_make_null_device(P2(gx_device_null *, gs_memory_t *));

/* Calculate the raster (number of bytes in a scan line), */
/* with byte or word padding. */
uint	gx_device_raster(P2(const gx_device *dev, int pad_to_word));

/* Adjust the resolution for devices that only have a fixed set of */
/* geometries, so that the apparent size in inches remains constant. */
/* If fit=1, the resolution is adjusted so that the entire image fits; */
/* if fit=0, one dimension fits, but the other one is clipped. */
int	gx_device_adjust_resolution(P4(gx_device *dev, int actual_width, int actual_height, int fit));

/*
 * Macros to help the drawing procedures clip coordinates to
 * fit into the drawing region.  Note that these may modify
 * x, y, w, h, data, data_x, and id.
 */

/* Macro for fill_rectangle and tile_rectangle. */
#define fit_fill(dev, x, y, w, h)\
	if ( (x | y) < 0 )\
	{	if ( x < 0 ) w += x, x = 0;\
		if ( y < 0 ) h += y, y = 0;\
	}\
	if ( x > dev->width - w ) w = dev->width - x;\
	if ( y > dev->height - h ) h = dev->height - y;\
	if ( w <= 0 || h <= 0 ) return 0

/* Macro for copy_mono and copy_color. */
#define fit_copy(dev, data, data_x, raster, id, x, y, w, h)\
	if ( (x | y) < 0 )\
	{	if ( x < 0 ) w += x, data_x -= x, x = 0;\
		if ( y < 0 ) h += y, data -= y * raster, id = gx_no_bitmap_id, y = 0;\
	}\
	if ( x > dev->width - w ) w = dev->width - x;\
	if ( y > dev->height - h ) h = dev->height - y;\
	if ( w <= 0 || h <= 0 ) return 0

/* Default implementations of optional procedures. */
/* Note that the default map_xxx_color routines assume black-and-white. */
dev_proc_open_device(gx_default_open_device);
dev_proc_get_initial_matrix(gx_default_get_initial_matrix);
dev_proc_sync_output(gx_default_sync_output);
dev_proc_output_page(gx_default_output_page);
dev_proc_close_device(gx_default_close_device);
dev_proc_map_rgb_color(gx_default_map_rgb_color);
dev_proc_map_color_rgb(gx_default_map_color_rgb);
dev_proc_tile_rectangle(gx_default_tile_rectangle);
dev_proc_copy_color(gx_default_copy_color);
dev_proc_draw_line(gx_default_draw_line);
dev_proc_get_bits(gx_default_get_bits);
dev_proc_get_params(gx_default_get_params);
dev_proc_put_params(gx_default_put_params);
dev_proc_put_params(gx_default_put_params_only);	/* no close/reopen */
dev_proc_map_cmyk_color(gx_default_map_cmyk_color);
dev_proc_get_xfont_procs(gx_default_get_xfont_procs);
dev_proc_get_xfont_device(gx_default_get_xfont_device);
dev_proc_map_rgb_alpha_color(gx_default_map_rgb_alpha_color);
dev_proc_get_page_device(gx_default_get_page_device);	/* returns NULL */
dev_proc_get_page_device(gx_page_device_get_page_device);	/* returns dev */
/* Color mapping routines for gray scale, true RGB, and true CMYK color. */
dev_proc_map_rgb_color(gx_default_gray_map_rgb_color);
dev_proc_map_color_rgb(gx_default_gray_map_color_rgb);
dev_proc_map_rgb_color(gx_default_rgb_map_rgb_color);
dev_proc_map_color_rgb(gx_default_rgb_map_color_rgb);
dev_proc_map_cmyk_color(gx_default_cmyk_map_cmyk_color);

/* Default implementations for forwarding devices */
dev_proc_get_initial_matrix(gx_forward_get_initial_matrix);
dev_proc_map_rgb_color(gx_forward_map_rgb_color);
dev_proc_map_color_rgb(gx_forward_map_color_rgb);
dev_proc_get_params(gx_forward_get_params);
dev_proc_put_params(gx_forward_put_params);
dev_proc_map_cmyk_color(gx_forward_map_cmyk_color);
dev_proc_get_xfont_procs(gx_forward_get_xfont_procs);
dev_proc_get_xfont_device(gx_forward_get_xfont_device);
dev_proc_map_rgb_alpha_color(gx_forward_map_rgb_alpha_color);
dev_proc_get_page_device(gx_forward_get_page_device);

/* Convert the device procedures to the proper form (see above). */
void	gx_device_set_procs(P1(gx_device *));

/* Fill in defaulted procedures in a device procedure record. */
void	gx_device_fill_in_procs(P1(gx_device *));
void	gx_device_forward_fill_in_procs(P1(gx_device_forward *));

/* Temporarily install a null device, or a special device such as */
/* a clipping device. */
void gx_device_no_output(P1(gs_state *));
void gx_set_device_only(P2(gs_state *, gx_device *));

/* Close a device. */
int gs_closedevice(P1(gx_device *));

/* ------ Device types ------ */

#define dev_type_proc_initialize(proc)\
  int proc(P1(gx_device *))

typedef struct gx_device_type_s {
	gs_memory_type_ptr_t stype;
	dev_type_proc_initialize((*initialize));
} gx_device_type;

#define device_type(dtname, stype, initproc)\
private dev_type_proc_initialize(initproc);\
const gx_device_type dtname = { &stype, initproc }

dev_type_proc_initialize(gdev_initialize);

#endif					/* gxdevice_INCLUDED */
