/* Copyright (C) 1989, 1992, 1993, 1994 Aladdin Enterprises.  All rights reserved.
  
  This file is part of Aladdin Ghostscript.
  
  Aladdin Ghostscript is distributed with NO WARRANTY OF ANY KIND.  No author
  or distributor accepts any responsibility for the consequences of using it,
  or for whether it serves any particular purpose or works at all, unless he
  or she says so in writing.  Refer to the Aladdin Ghostscript Free Public
  License (the "License") for full details.
  
  Every copy of Aladdin Ghostscript must include a copy of the License,
  normally in a plain ASCII text file named PUBLIC.  The License grants you
  the right to copy, modify and redistribute Aladdin Ghostscript, but only
  under certain conditions described in the License.  Among other things, the
  License requires that the copyright notice and this notice be preserved on
  all copies.
*/

/* gdevmem2.c */
/* 8-and-more-bit-per-pixel "memory" (stored bitmap) devices */
/* for Ghostscript library. */
#include "memory_.h"
#include "gx.h"
#include "gxdevice.h"
#include "gxdevmem.h"			/* semi-public definitions */
#include "gdevmem.h"			/* private definitions */

/* ------ Generic procedures ------ */

/* Copy a rectangle of bytes from a source to a destination. */
#undef chunk
#define chunk byte
private int
copy_byte_rect(gx_device_memory *dev, const byte *source, int sraster,
  int offset, int y, int byte_count, int h)
{	uint draster = dev->raster;
	byte *dest = scan_line_base(dev, y) + offset;
	while ( h-- > 0 )
	   {	memcpy(dest, source, byte_count);
		source += sraster;
		dest += draster;
	   }
	return 0;
}

/* Map a r-g-b color to a color index. */
/* This requires searching the palette. */
gx_color_index
mem_mapped_map_rgb_color(gx_device *dev, gx_color_value r, gx_color_value g,
  gx_color_value b)
{	byte br = gx_color_value_to_byte(r);
	byte bg = gx_color_value_to_byte(g);
	byte bb = gx_color_value_to_byte(b);
	register const byte *pptr = mdev->palette.data;
	int cnt = mdev->palette.size;
	const byte *which;
	int best = 256*3;
	while ( (cnt -= 3) >= 0 )
	   {	register int diff = *pptr - br;
		if ( diff < 0 ) diff = -diff;
		if ( diff < best )	/* quick rejection */
		   {	int dg = pptr[1] - bg;
			if ( dg < 0 ) dg = -dg;
			if ( (diff += dg) < best )	/* quick rejection */
			   {	int db = pptr[2] - bb;
				if ( db < 0 ) db = -db;
				if ( (diff += db) < best )
					which = pptr, best = diff;
			   }
		   }
		pptr += 3;
	   }
	return (gx_color_index)((which - mdev->palette.data) / 3);
}

/* Map a color index to a r-g-b color. */
int
mem_mapped_map_color_rgb(gx_device *dev, gx_color_index color,
  gx_color_value prgb[3])
{	const byte *pptr = mdev->palette.data + (int)color * 3;
	prgb[0] = gx_color_value_from_byte(pptr[0]);
	prgb[1] = gx_color_value_from_byte(pptr[1]);
	prgb[2] = gx_color_value_from_byte(pptr[2]);
	return 0;
}

/* ------ Mapped 8-bit color ------ */

/* Procedures */
declare_mem_procs(mem_mapped8_copy_mono, mem_mapped8_copy_color, mem_mapped8_fill_rectangle);

/* The device descriptor. */
/* The instance is exported for gdevmem1.c. */
const gx_device_memory far_data mem_mapped8_color_device =
  mem_device("image(8)", 8, 0,
    mem_mapped_map_rgb_color, mem_mapped_map_color_rgb,
    mem_mapped8_copy_mono, mem_mapped8_copy_color, mem_mapped8_fill_rectangle);

/* Convert x coordinate to byte offset in scan line. */
#undef x_to_byte
#define x_to_byte(x) (x)

/* Fill a rectangle with a color. */
private int
mem_mapped8_fill_rectangle(gx_device *dev,
  int x, int y, int w, int h, gx_color_index color)
{	declare_scan_ptr(dest);
	fit_fill(dev, x, y, w, h);
	setup_rect(dest);
	while ( h-- > 0 )
	   {	memset(dest, (byte)color, w);
		inc_chunk_ptr(dest, draster);
	   }
	return 0;
}

/* Copy a monochrome bitmap. */
/* We split up this procedure because of limitations in the bcc32 compiler. */
private void mapped8_copy01(P9(chunk *, const byte *, int, int, uint,
  int, int, byte, byte));
private void mapped8_copyN1(P8(chunk *, const byte *, int, int, uint,
  int, int, byte));
private void mapped8_copy0N(P8(chunk *, const byte *, int, int, uint,
  int, int, byte));
private int
mem_mapped8_copy_mono(gx_device *dev,
  const byte *base, int sourcex, int sraster, gx_bitmap_id id,
  int x, int y, int w, int h, gx_color_index zero, gx_color_index one)
{	const byte *line;
	int first_bit;
	declare_scan_ptr(dest);
	fit_copy(dev, base, sourcex, sraster, id, x, y, w, h);
	setup_rect(dest);
	line = base + (sourcex >> 3);
	first_bit = 0x80 >> (sourcex & 7);
#define is_color(c) ((int)(c) != (int)gx_no_color_index)
	if ( is_color(one) )
	{	if ( is_color(zero) )
		  mapped8_copy01(dest, line, first_bit, sraster, draster,
				 w, h, (byte)zero, (byte)one);
		else
		  mapped8_copyN1(dest, line, first_bit, sraster, draster,
				 w, h, (byte)one);
	}
	else if ( is_color(zero) )
	  mapped8_copy0N(dest, line, first_bit, sraster, draster,
			 w, h, (byte)zero);
#undef is_color
	return 0;
}
/* Macros for copy loops */
#define COPY_BEGIN\
	while ( h-- > 0 )\
	{	register byte *pptr = dest;\
		const byte *sptr = line;\
		register int sbyte = *sptr;\
		register uint bit = first_bit;\
		int count = w;\
		do\
		{
#define COPY_END\
			if ( (bit >>= 1) == 0 )\
				bit = 0x80, sbyte = *++sptr;\
			pptr++;\
		}\
		while ( --count > 0 );\
		line += sraster;\
		inc_chunk_ptr(dest, draster);\
	}
/* Halftone coloring */
private void
mapped8_copy01(chunk *dest, const byte *line, int first_bit,
  int sraster, uint draster, int w, int h, byte b0, byte b1)
{	COPY_BEGIN
	*pptr = (sbyte & bit ? b1 : b0);
	COPY_END
}
/* Stenciling */
private void
mapped8_copyN1(chunk *dest, const byte *line, int first_bit,
  int sraster, uint draster, int w, int h, byte b1)
{	COPY_BEGIN
	if ( sbyte & bit )
	  *pptr = b1;
	COPY_END
}
/* Reverse stenciling (probably never used) */
private void
mapped8_copy0N(chunk *dest, const byte *line, int first_bit,
  int sraster, uint draster, int w, int h, byte b0)
{	COPY_BEGIN
	if ( !(sbyte & bit) )
	  *pptr = b0;
	COPY_END
}
#undef COPY_BEGIN
#undef COPY_END

/* Copy a color bitmap. */
private int
mem_mapped8_copy_color(gx_device *dev,
  const byte *base, int sourcex, int sraster, gx_bitmap_id id,
  int x, int y, int w, int h)
{	fit_copy(dev, base, sourcex, sraster, id, x, y, w, h);
	return copy_byte_rect(mdev, base + x_to_byte(sourcex), sraster,
		x_to_byte(x), y, x_to_byte(w), h);
}

/* ------ 16-bit true color ------ */
/* The 16 bits are divided 5 for red, 6 for green, and 5 for blue. */
/* Note that the bits must always be kept in big-endian order. */

/* Procedures */
declare_mem_map_procs(mem_true16_map_rgb_color, mem_true16_map_color_rgb);
declare_mem_procs(mem_true16_copy_mono, mem_true16_copy_color, mem_true16_fill_rectangle);

/* The device descriptor. */
/* The instance is exported for gdevmem1.c. */
const gx_device_memory far_data mem_true16_color_device =
  mem_device("image(16)", 16, 0,
    mem_true16_map_rgb_color, mem_true16_map_color_rgb,
    mem_true16_copy_mono, mem_true16_copy_color, mem_true16_fill_rectangle);

/* Map a r-g-b color to a color index. */
private gx_color_index
mem_true16_map_rgb_color(gx_device *dev, gx_color_value r, gx_color_value g,
  gx_color_value b)
{	return ((r >> (gx_color_value_bits - 5)) << 11) +
		((g >> (gx_color_value_bits - 6)) << 5) +
		(b >> (gx_color_value_bits - 5));
}

/* Map a color index to a r-g-b color. */
private int
mem_true16_map_color_rgb(gx_device *dev, gx_color_index color,
  gx_color_value prgb[3])
{	ushort value;
	value = color >> 11;
	prgb[0] = ((value << 11) + (value << 6) + (value << 1) + (value >> 4)) >> (16 - gx_color_value_bits);
	value = (color >> 6) & 0x7f;
	prgb[1] = ((value << 10) + (value << 4) + (value >> 2)) >> (16 - gx_color_value_bits);
	value = color & 0x3f;
	prgb[2] = ((value << 11) + (value << 6) + (value << 1) + (value >> 4)) >> (16 - gx_color_value_bits);
	return 0;
}

/* Convert x coordinate to byte offset in scan line. */
#undef x_to_byte
#define x_to_byte(x) ((x) << 1)

/* Fill a rectangle with a color. */
private int
mem_true16_fill_rectangle(gx_device *dev,
  int x, int y, int w, int h, gx_color_index color)
{
#if arch_is_big_endian
#  define color16 ((ushort)color)
#else
	ushort color16 = ((uint)(byte)color << 8) + ((ushort)color >> 8);
#endif
	declare_scan_ptr(dest);
	fit_fill(dev, x, y, w, h);
	setup_rect(dest);
	while ( h-- > 0 )
	   {	ushort *pptr = (ushort *)dest;
		int cnt = w;
		do { *pptr++ = color16; } while ( --cnt > 0 );
		inc_chunk_ptr(dest, draster);
	   }
	return 0;
#undef color16
}

/* Copy a monochrome bitmap. */
private int
mem_true16_copy_mono(gx_device *dev,
  const byte *base, int sourcex, int sraster, gx_bitmap_id id,
  int x, int y, int w, int h, gx_color_index zero, gx_color_index one)
{
#if arch_is_big_endian
#  define zero16 ((ushort)zero)
#  define one16 ((ushort)one)
#else
	ushort zero16 = ((uint)(byte)zero << 8) + ((ushort)zero >> 8);
	ushort one16 = ((uint)(byte)one << 8) + ((ushort)one >> 8);
#endif
	const byte *line;
	int first_bit;
	declare_scan_ptr(dest);
	fit_copy(dev, base, sourcex, sraster, id, x, y, w, h);
	setup_rect(dest);
	line = base + (sourcex >> 3);
	first_bit = 0x80 >> (sourcex & 7);
	while ( h-- > 0 )
	   {	register ushort *pptr = (ushort *)dest;
		const byte *sptr = line;
		register int sbyte = *sptr++;
		register int bit = first_bit;
		int count = w;
		do
		   {	if ( sbyte & bit )
			   {	if ( one != gx_no_color_index )
				  *pptr = one16;
			   }
			else
			   {	if ( zero != gx_no_color_index )
				  *pptr = zero16;
			   }
			if ( (bit >>= 1) == 0 )
				bit = 0x80, sbyte = *sptr++;
			pptr++;
		   }
		while ( --count > 0 );
		line += sraster;
		inc_chunk_ptr(dest, draster);
	   }
	return 0;
#undef zero16
#undef one16
}

/* Copy a color bitmap. */
private int
mem_true16_copy_color(gx_device *dev,
  const byte *base, int sourcex, int sraster, gx_bitmap_id id,
  int x, int y, int w, int h)
{	fit_copy(dev, base, sourcex, sraster, id, x, y, w, h);
	return copy_byte_rect(mdev, base + x_to_byte(sourcex), sraster,
		x_to_byte(x), y, x_to_byte(w), h);
}

/* ------ 24-bit (RGB) color ------ */

/* Procedures */
declare_mem_procs(mem_true24_copy_mono, mem_true24_copy_color, mem_true24_fill_rectangle);

/* The device descriptor. */
const gx_device_memory far_data mem_true24_color_device =
  mem_device("image(24)", 24, 0,
    gx_default_rgb_map_rgb_color, gx_default_rgb_map_color_rgb,
    mem_true24_copy_mono, mem_true24_copy_color, mem_true24_fill_rectangle);

/* Convert x coordinate to byte offset in scan line. */
#undef x_to_byte
#define x_to_byte(x) ((x) * 3)

/* Unpack a color into its bytes. */
#define declare_unpack_color(r, g, b, color)\
	byte r = (byte)(color >> 16);\
	byte g = (byte)((uint)color >> 8);\
	byte b = (byte)color
#if arch_is_big_endian
#  define declare_pack_color(cword, rgb, r, g, b)\
	bits32 cword = (bits32)(rgb) << 8
#else
#  define declare_pack_color(cword, rgb, r, g, b)\
	bits32 cword =\
	  ((bits32)(b) << 16) | ((bits16)(g) << 8) | (r)
#endif
/* Put a 24-bit color into the bitmap. */
#define put3(ptr, r, g, b)\
	(ptr)[0] = r, (ptr)[1] = g, (ptr)[2] = b
/* Put 4 bytes of color into the bitmap. */
#define putw(ptr, wxyz)\
	*(bits32 *)(ptr) = (wxyz)
/* Load the 3-word 24-color cache. */
/* Free variables: [m]dev, rgbr, gbrg, brgb. */
#if arch_is_big_endian
#  define set_color24_cache(crgb, r, g, b)\
	mdev->color24.rgbr = rgbr = ((bits32)(crgb) << 8) | (r),\
	mdev->color24.gbrg = gbrg = (rgbr << 8) | (g),\
	mdev->color24.brgb = brgb = (gbrg << 8) | (b),\
	mdev->color24.rgb = (crgb)
#else
#  define set_color24_cache(crgb, r, g, b)\
	mdev->color24.rgbr = rgbr =\
		((bits32)(r) << 24) | ((bits32)(b) << 16) |\
		((bits16)(g) << 8) | (r),\
	mdev->color24.brgb = brgb = (rgbr << 8) | (b),\
	mdev->color24.gbrg = gbrg = (brgb << 8) | (g),\
	mdev->color24.rgb = (crgb)
#endif

/* Fill a rectangle with a color. */
private int
mem_true24_fill_rectangle(gx_device *dev,
  int x, int y, int w, int h, gx_color_index color)
{	declare_unpack_color(r, g, b, color);
	declare_scan_ptr(dest);
	fit_fill(dev, x, y, w, h);
	setup_rect(dest);
	if ( w >= 5 )
	  { if ( r == g && r == b)
	      {
#if 1
		/* We think we can do better than the library's memset.... */
		int bcntm7 = w * 3 - 7;
		register bits32 cword = color | (color << 24);
		while ( h-- > 0 )
		{	register byte *pptr = dest;
			byte *limit = pptr + bcntm7;
			/* We want to store full words, but we have to */
			/* guarantee that they are word-aligned. */
			switch ( x & 3 )
			  {
			  case 3: *pptr++ = (byte)cword;
			  case 2: *pptr++ = (byte)cword;
			  case 1: *pptr++ = (byte)cword;
			  case 0: ;
			  }
			/* Even with w = 5, we always store at least */
			/* 3 full words, regardless of the starting x. */
			*(bits32 *)pptr =
			  ((bits32 *)pptr)[1] =
			  ((bits32 *)pptr)[2] = cword;
			pptr += 12;
			while ( pptr < limit )
			  { *(bits32 *)pptr =
			      ((bits32 *)pptr)[1] = cword;
			    pptr += 8;
			  }
			switch ( pptr - limit )
			  {
			  case 0: pptr[6] = (byte)cword;
			  case 1: pptr[5] = (byte)cword;
			  case 2: pptr[4] = (byte)cword;
			  case 3: *(bits32 *)pptr = cword;
			    break;
			  case 4: pptr[2] = (byte)cword;
			  case 5: pptr[1] = (byte)cword;
			  case 6: pptr[0] = (byte)cword;
			  case 7: ;
			  }
			inc_chunk_ptr(dest, draster);
		}
#else
		int bcnt = w * 3;
		while ( h-- > 0 )
		{	memset(dest, r, bcnt);
			inc_chunk_ptr(dest, draster);
		}
#endif
	      }
	    else
	      {	int x3 = -x & 3, ww = w - x3;
		bits32 rgbr, gbrg, brgb;
		if ( mdev->color24.rgb == color )
		  rgbr = mdev->color24.rgbr,
		  gbrg = mdev->color24.gbrg,
		  brgb = mdev->color24.brgb;
		else
		  set_color24_cache(color, r, g, b);
		while ( h-- > 0 )
		  {	register byte *pptr = dest;
			int w1 = ww;
			switch ( x3 )
			  {
			  case 1:
				put3(pptr, r, g, b);
				pptr += 3; break;
			  case 2:
				pptr[0] = r; pptr[1] = g;
				putw(pptr + 2, brgb);
				pptr += 6; break;
			  case 3:
				pptr[0] = r;
				putw(pptr + 1, gbrg);
				putw(pptr + 5, brgb);
				pptr += 9; break;
			  case 0:
				;
			  }
			while ( w1 >= 4 )
			  {	putw(pptr, rgbr);
				putw(pptr + 4, gbrg);
				putw(pptr + 8, brgb);
				pptr += 12;
				w1 -= 4;
			  }
			switch ( w1 )
			  {
			  case 1:
				put3(pptr, r, g, b);
				break;
			  case 2:
				putw(pptr, rgbr);
				pptr[4] = g; pptr[5] = b;
				break;
			  case 3:
				putw(pptr, rgbr);
				putw(pptr + 4, gbrg);
				pptr[8] = b;
				break;
			  case 0:
				;
			  }
			inc_chunk_ptr(dest, draster);
		  }
	      }
	  }
	else			/* w < 5 */
	{	while ( h-- > 0 )
		  {	switch ( w )
			  {
			  case 4: put3(dest + 9, r, g, b);
			  case 3: put3(dest + 6, r, g, b);
			  case 2: put3(dest + 3, r, g, b);
			  case 1: put3(dest, r, g, b);
			  case 0: ;
			  }
			inc_chunk_ptr(dest, draster);
		  }
	}
	return 0;
}

/* Copy a monochrome bitmap. */
private int
mem_true24_copy_mono(gx_device *dev,
  const byte *base, int sourcex, int sraster, gx_bitmap_id id,
  int x, int y, int w, int h, gx_color_index zero, gx_color_index one)
{	const byte *line;
	int sbit;
	int first_bit;
	declare_scan_ptr(dest);
	fit_copy(dev, base, sourcex, sraster, id, x, y, w, h);
	setup_rect(dest);
	line = base + (sourcex >> 3);
	sbit = sourcex & 7;
	first_bit = 0x80 >> sbit;
	if ( zero != gx_no_color_index )
	  {	/* Loop for halftones or inverted masks */
		/* (never used). */
		declare_unpack_color(r0, g0, b0, zero);
		declare_unpack_color(r1, g1, b1, one);
		while ( h-- > 0 )
		   {	register byte *pptr = dest;
			const byte *sptr = line;
			register int sbyte = *sptr++;
			register int bit = first_bit;
			int count = w;
			do
			  {	if ( sbyte & bit )
				  { if ( one != gx_no_color_index )
				      put3(pptr, r1, g1, b1);
				  }
				else
				  put3(pptr, r0, g0, b0);
				pptr += 3;
				if ( (bit >>= 1) == 0 )
				  bit = 0x80, sbyte = *sptr++;
			  }
			while ( --count > 0 );
			line += sraster;
			inc_chunk_ptr(dest, draster);
		   }
	  }
	else if ( one != gx_no_color_index )
	  {	/* Loop for character and pattern masks. */
		/* This is used heavily. */
		declare_unpack_color(r1, g1, b1, one);
		int first_mask = first_bit << 1;
		int first_count, first_skip;
		if ( sbit + w > 8 )
		  first_mask -= 1,
		  first_count = 8 - sbit;
		else
		  first_mask -= first_mask >> w,
		  first_count = w;
		first_skip = first_count * 3;
		while ( h-- > 0 )
		   {	register byte *pptr = dest;
			const byte *sptr = line;
			register int sbyte = *sptr++ & first_mask;
			int count = w - first_count;
			if ( sbyte )
			  {	register int bit = first_bit;
				do
				  {	if ( sbyte & bit )
					  put3(pptr, r1, g1, b1);
					pptr += 3;
				  }
				while ( (bit >>= 1) & first_mask );
			  }
			else
			  pptr += first_skip;
			while ( count >= 8 )
			  {	sbyte = *sptr++;
				if ( sbyte & 0xf0 )
				  { if ( sbyte & 0x80 )
				      put3(pptr, r1, g1, b1);
				    if ( sbyte & 0x40 )
				      put3(pptr + 3, r1, g1, b1);
				    if ( sbyte & 0x20 )
				      put3(pptr + 6, r1, g1, b1);
				    if ( sbyte & 0x10 )
				      put3(pptr + 9, r1, g1, b1);
				  }
				if ( sbyte & 0xf )
				  { if ( sbyte & 8 )
				      put3(pptr + 12, r1, g1, b1);
				    if ( sbyte & 4 )
				      put3(pptr + 15, r1, g1, b1);
				    if ( sbyte & 2 )
				      put3(pptr + 18, r1, g1, b1);
				    if ( sbyte & 1 )
				      put3(pptr + 21, r1, g1, b1);
				  }
				pptr += 24;
				count -= 8;
			  }
			if ( count > 0 )
			  {	register int bit = 0x80;
				sbyte = *sptr++;
				do
				  {	if ( sbyte & bit )
					  put3(pptr, r1, g1, b1);
					pptr += 3;
					bit >>= 1;
				  }
				while ( --count > 0 );
			  }
			line += sraster;
			inc_chunk_ptr(dest, draster);
		   }
	  }
	return 0;
}

/* Copy a color bitmap. */
private int
mem_true24_copy_color(gx_device *dev,
  const byte *base, int sourcex, int sraster, gx_bitmap_id id,
  int x, int y, int w, int h)
{	fit_copy(dev, base, sourcex, sraster, id, x, y, w, h);
	return copy_byte_rect(mdev, base + x_to_byte(sourcex), sraster,
		x_to_byte(x), y, x_to_byte(w), h);
}

/* ------ 32-bit (CMYK) color ------ */

/* Procedures */
declare_mem_procs(mem_true32_copy_mono, mem_true32_copy_color, mem_true32_fill_rectangle);

/* The device descriptor. */
const gx_device_memory far_data mem_true32_color_device =
  mem_full_device("image(32)", 24, 8, mem_open,
    gx_default_map_rgb_color, gx_default_map_color_rgb,
    mem_true32_copy_mono, mem_true32_copy_color, mem_true32_fill_rectangle,
    mem_get_bits, gx_default_cmyk_map_cmyk_color);

/* Convert x coordinate to byte offset in scan line. */
#undef x_to_byte
#define x_to_byte(x) ((x) << 2)

/* Swap the bytes of a color if needed. */
#if arch_is_big_endian
#  define arrange_bytes(color) (color)
#else
#  define arrange_bytes(color)\
    (((color) >> 24) + (((color) >> 8) & 0xff00) +\
     (((color) & 0xff00) << 8) + ((color) << 24))
#endif

/* Fill a rectangle with a color. */
private int
mem_true32_fill_rectangle(gx_device *dev,
  int x, int y, int w, int h, gx_color_index color)
{	bits32 a_color = arrange_bytes(color);
	declare_scan_ptr(dest);
	fit_fill(dev, x, y, w, h);
	setup_rect(dest);
	while ( h-- > 0 )
	{	bits32 *pptr = (bits32 *)dest;
		int cnt = w;
		do { *pptr++ = a_color; } while ( --cnt > 0 );
		inc_chunk_ptr(dest, draster);
	}
	return 0;
}

/* Copy a monochrome bitmap. */
private int
mem_true32_copy_mono(gx_device *dev,
  const byte *base, int sourcex, int sraster, gx_bitmap_id id,
  int x, int y, int w, int h, gx_color_index zero, gx_color_index one)
{	bits32 a_zero = arrange_bytes(zero);
	bits32 a_one = arrange_bytes(one);
	const byte *line;
	int first_bit;
	declare_scan_ptr(dest);
	fit_copy(dev, base, sourcex, sraster, id, x, y, w, h);
	setup_rect(dest);
	line = base + (sourcex >> 3);
	first_bit = 0x80 >> (sourcex & 7);
	while ( h-- > 0 )
	{	register bits32 *pptr = (bits32 *)dest;
		const byte *sptr = line;
		register int sbyte = *sptr++;
		register int bit = first_bit;
		int count = w;
		do
		{	if ( sbyte & bit )
			{	if ( one != gx_no_color_index )
				  *pptr = a_one;
			}
			else
			{	if ( zero != gx_no_color_index )
				  *pptr = a_zero;
			}
			if ( (bit >>= 1) == 0 )
				bit = 0x80, sbyte = *sptr++;
			pptr++;
		}
		while ( --count > 0 );
		line += sraster;
		inc_chunk_ptr(dest, draster);
	}
	return 0;
}

/* Copy a color bitmap. */
private int
mem_true32_copy_color(gx_device *dev,
  const byte *base, int sourcex, int sraster, gx_bitmap_id id,
  int x, int y, int w, int h)
{	fit_copy(dev, base, sourcex, sraster, id, x, y, w, h);
	return copy_byte_rect(mdev, base + x_to_byte(sourcex), sraster,
		x_to_byte(x), y, x_to_byte(w), h);
}
