unit Table;

interface

uses
    WinTypes,
    WinProcs,
    Strings,
    Objects,
    MLBTypes;

const
    Reserve            = 100;
    MaxColNum          = 50;            { Maximum columns in a table }
    ItemsPerBlock      = 100;           { Number of items stored in one global memory object }
                                        { This constant should be tuned in depece of size of each item
                                          !!! Block size should not exceed 64K limit }
    BlocksLimit        = 50;            { Number of block handles to allocate initialy }
    BlocksDelta        = 10;            { Number of blocks handles to allocate when block
                                          table is growing }
type
    pItemHandle = ^tItemHandle;
    tItemHandle = longint;              { Item handle - HiWord contain block index in block table
                                                        LoWord contain item index in the block }

    pItemsTable = ^tItemsTable;         { The items table }
    tItemsTable = array [0..0] of tItemHandle;

    pBlockHandle = ^tBlockHandle;
    tBlockHandle = longint;             { Block handle - HiWord contain global memory object handle
                                                         LoWord contain number of items stored in block }

    pBlocksTable = ^tBlocksTable;       { The blocks table }
    tBlocksTable = array [0..0] of tBlockHandle;

    pOffsetTable = ^tOffsetTable;       { Memory block offset table }
    tOffsetTable = array [0..ItemsPerBlock] of Word;

    pBlockHeader = ^tBlockHeader;
    tBlockHeader = record
      OffsetTable:  tOffsetTable;       { Items offset table }
    end;

    pTabsOffsTable = ^tTabsOffsTable;   { Offset table inside item }
    tTabsOffsTable = array [1..MaxColNum] of word;

    pListItemHeader = ^tListItemHeader;
    tListItemHeader = record
      RecNo:        Longint;            { Record number }
      HasOwnColor:  Boolean;            { Does item has own color }
      Color:        tColorRef;          { If yes = Color }
    end;
 
const
    { Error codes }
    tSuccess           = 0;
    tAlreadyBuilded    = 3001;
    tInvalidRecNo      = 3002;
    tInvalidRecSize    = 3003;
    tNotBuilded        = 3004;
    tInvalidItem       = 3005;
    tMemory            = 3006;
 
type
    pListTable = ^tListTable;
    tListTable = object(tObject)
      RowNumber:  Longint;              { Number of rows in table }
      ItemsList:  pItemsList;           { Description of table }
      Builded:    Boolean;              { Builded flag }
      constructor Init(AnItemsList: pItemsList);
      destructor  Done; virtual;
      function    BuildTable: Integer; virtual;
      function    CreateNewItem(Buffer: PChar; ARecNo: LongInt): tItemHandle;
      function    DeleteItem(ItemHandle: tItemHandle): Integer; virtual;
      function    DestroyTable: Integer;
      function    GetColAlign(ColNum: Integer): Word;
      function    GetColNumber: Integer;
      function    GetColWrap(ColNum: Integer): Word;
      procedure   GetFieldString(ItemHandle: tItemHandle; Index: Word; Str: PChar);
      function    GetItemColor(ItemHandle: tItemHandle; var ItemColor: Longint): Boolean;
      function    GetItemHandle(Row: Longint): tItemHandle;
      function    GetItemRecno(ItemHandle: tItemHandle): Longint;
      function    GetItemType(ColNum: Integer): TItemType;
      function    GetRows: Longint;
      function    ID2Index(ID: Integer): Integer;
      function    Index2ID(Index: Integer): Integer;
      procedure   SetSortOrder(ColID: Integer);
      procedure   SetItemColor(ItemHandle: tItemHandle; AColor: LongInt; OwnColor: Boolean);
      function    UpdateItem(ItemHandle: tItemHandle; RecNo: Longint; NewItem: Boolean): tItemHandle; virtual;
      {---  Abstract functions and procedures ---}
      procedure   CloseStrip; virtual;
      procedure   CreateStrip(ATitle: PChar; ATotal: Longint); virtual;
      function    GetBmpField(RecNo: LongInt; Index: Word): HBitmap; virtual;
      function    GetRecordCount: Longint; virtual;
      function    GetRecordField(RecNo: LongInt; Index: Word): PChar; virtual;
      function    GetRecordNo: Longint; virtual;
      function    GetRecordWidth: Word; virtual;
      function    GetStrField(RecNo: LongInt; Index: Word): PChar; virtual;
      procedure   NewStrip(Current: LongInt); virtual;
      function    NextRecord: Boolean; virtual;
      function    SkipRecord(dwRecno: Longint): Boolean; virtual;
      { Internally using variables and methods }
      private
      hTable         : tHandle;              { Items table memory handle }
      hBlocks        : tHandle;              { Blocks table memory handle }
      BlocksNum      : Word;
      BlocksTableLen : Word;
      function MemPtr(Address: pointer; Idx: longint): pointer;
    end;

(*----------------------------------------------------------------------------*)
implementation

Const
   AHi: word = 8;
   AHs: byte = 3;

(*----------------------------------------------------------------------------*)
procedure AHIncr;  FAR; External 'KERNEL' Index 114; {magic function}
procedure AHShift; FAR; External 'KERNEL' Index 113; {dito}

(*----------------------------------------------------------------------------*)
function MaxFit (Size: Word; anOffset: Word): Word;
{-return maximum number of bytes that fit in a Segment}
  inline(
    $5B/                   {  Pop Bx       ; anOffset}
    $58/                   {  Pop Ax       ; Size}
    $01/$C3/               {  Add Bx,Ax}
    $73/$02/               {  Jnc @@1}
    $29/$D8);              {  Sub Ax,Bx    ; $1.0000-anOffset}
                           {@@1:           ; Ax= Min(Size,$1.0000-anOffset)}

(*----------------------------------------------------------------------------*)
function MaxChunk (Size: LongInt): Word;
{-return maximum number of bytes that can be transferred
  in one block using conventional functions}
  inline(
    $5B/                   {  Pop Bx       ; Word(Size)}
    $5A/                   {  Pop Dx       ; Word(Size+2)}
    $B8/$FF/$FF/           {  Mov Ax,$FFFF}
    $09/$D2/               {  Or Dx,Dx     ; Dx=0 ?}
    $75/$02/               {  Jne @@1}
    $89/$D8);              {  Mov Ax,Bx}
                           {@@1:           ; Ax=Min($FFFF,Size)}

(*----------------------------------------------------------------------------*)
function LIncPtr (aPtr: Pointer; anOffset: LongInt): Pointer; assembler;
  asm
    Mov Dx,Word(anOffset+2)
    Mov Ax,Word(anOffset)
    Mov Cx,OFFSET AHShift
    Shl Dx,Cl
    Add Dx,Word(aPtr+2)
    Add Ax,Word(aPtr)
    Jnc @@1
    Add Dx,Offset AHincr
  @@1:
  end;

(*----------------------------------------------------------------------------*)
function LDecPtr (aPtr: Pointer; anOffset: LongInt): Pointer; assembler;
  asm
    Mov Bx,Word(anOffset+2)
    Mov Cx,Offset AHShift
    Shl Bx,Cl
    Mov Dx,Word(aPtr+2)
    Mov Ax,Word(aPtr)
    Sub Dx,Bx
    Sub Ax,Word(anOffset)
    Jnc @@1
    Sub Dx,Offset AHincr
  @@1:
  end;

(*----------------------------------------------------------------------------*)
function IncPtrMac (aPtr: pointer; anOffset: word): pointer;
  inline(
    $5B/                   {  Pop Bx    ; anOffset}
    $58/                   {  Pop Ax    ; Word(aPtr)}
    $5A/                   {  Pop Dx    ; Word(aPtr+2)}
    $01/$D8/               {  Add Ax,Bx}
    $73/$04/               {  Jnc @@1}
    $03/$16/>AHI);         {  Add Dx,[>AHi]}
                           {@@1:}

(*----------------------------------------------------------------------------*)
function DecPtrMac (aPtr: pointer; anOffset: word): pointer;
  inline(
    $5B/                   {  Pop Bx    ; anOffset}
    $58/                   {  Pop Ax    ; Word(aPtr)}
    $5A/                   {  Pop Dx    ; Word(aPtr+2)}
    $29/$D8/               {  Sub Ax,Bx}
    $73/$04/               {  Jnc @@1}
    $2B/$16/>AHI);         {  Sub Dx,[>AHi]}
                           {@@1:}

(*----------------------------------------------------------------------------*)
function PtrDiff (Ptr1,Ptr2: Pointer): LongInt; assembler;
  asm
    Mov Dx,Word(Ptr1+2)
    Mov Bx,Word(Ptr2+2)
    Mov Cx,Offset AHshift
    Shr Dx,Cl
    Shr Bx,Cl
    Mov Ax,Word(Ptr1)
    Sub Ax,Word(Ptr2)
    Sbb Dx,Bx
    Jnc @@1
    Neg Ax
    Adc Dx,0
    Neg Dx
  @@1:
  end;

(*----------------------------------------------------------------------------*)
procedure hMove(srcPtr, dstPtr: pointer; Size: longint);

  var
     Count: word;
     S : LONGINT;

     { +++++++++++++++++++++++++++++++++++++++++++++++++++ }

     function Min (a,b: Word): Word;
       begin
          if a > b then
             Min:= b
          else
             Min:= a
       end;

     { +++++++++++++++++++++++++++++++++++++++++++++++++++ }

  begin
     S := Size;
     if longint(dstPtr) > longint(srcPtr) then
       begin {Shift up}
         srcPtr:= LIncPtr(srcPtr,Size-1);
         dstPtr:= LIncPtr(dstPtr,Size-1);
         while Size > 0 do
           begin
             Count:= Min(Min(MaxChunk(Size),MaxChunk(LONGINT(Word(srcPtr))+1)),
                                            MaxChunk(LONGINT(Word(dstPtr))+1));
             srcPtr:= DecPtrMac(srcPtr, Count-1);
             dstPtr:= DecPtrMac(dstPtr, Count-1);
             Move(srcPtr^, dstPtr^, Count);
             srcPtr:= DecPtrMac(srcPtr, 1);
             dstPtr:= DecPtrMac(dstPtr, 1);
             Dec(Size, Count);
           end
       end
     else {shift down}
       begin
         while Size > 0 do
           begin
             Count:= MaxFit(
                     MaxFit(MaxChunk(Size),Word(srcPtr))
                                          ,Word(dstPtr));
             Move(srcPtr^, dstPtr^, Count);
             srcPtr:= IncPtrMac(srcPtr,Count);
             dstPtr:= IncPtrMac(dstPtr,Count);
             Dec(Size,Count)
           end
       end
  end;

(*----------------------------------------------------------------------------*)
function ItemBlockIndex(ItemHandle: tItemHandle): Word;
  begin
    ItemBlockIndex := HiWord(ItemHandle);
  end;

(*----------------------------------------------------------------------------*)
function ItemIndex(ItemHandle: tItemHandle): Word;
  begin
    ItemIndex := LoWord(ItemHandle);
  end;

(*----------------------------------------------------------------------------*)
function BlockHandle(ABlockHandle: tBlockHandle): tHandle;
  begin
    BlockHandle := HiWord(ABlockHandle);
  end;

(*----------------------------------------------------------------------------*)
function ItemsInBlock(BlockHandle: tBlockHandle): Word;
  begin
    ItemsInBlock := LoWord(BlockHandle);
  end;

(*----------------------------------------------------------------------------*)
constructor tListTable.Init;
  begin
    inherited Init;
      if AnItemsList^.ColNumber > MaxColNum then
      Fail;
    RowNumber := 0;
    hTable := 0;
    ItemsList := AnItemsList;
    Builded := False;
    BlocksNum := 0;
    BlocksTableLen := 0;
  end;

(*----------------------------------------------------------------------------*)
destructor tListTable.Done;
  begin
    DestroyTable;
    inherited Done;
  end;

(*----------------------------------------------------------------------------*)
function tListTable.BuildTable: Integer;

  var
    BuffLen      : longint; { Maximum size reqired for item }
    RecSize      : word; { Maximum record size }
    hWork        : tHandle; { Working buffer handle }
    hItem        : tItemHandle; { Item }
    WorkBuf      : pchar; { Working buffer }
    TotalRecs    : longint; { Number of total records in database }
    Table        : pItemsTable; { The table itself }
    i, RecNo, k  : longint; { Iterations }
    ContLoop     : Boolean; { Loop flag }
    ItemAddress  : pointer;

  begin
    BuildTable := tSuccess;
      if Builded then
      begin
      { Table was already builded }
        BuildTable := tAlreadyBuilded;
        Exit;
      end;
    RecNo := GetRecordNo;
      if RecNo < 1 then
      begin
      { Invalid record number given }
        BuildTable := tInvalidRecNo;
        Exit;
      end;
    RecSize := GetRecordWidth;
      if RecSize < 1 then
      begin
      { Invalid record size given }
        BuildTable := tInvalidRecSize;
        Exit;
      end;
      { Memory allocation for block table }
    BlocksTableLen := BlocksLimit;
    hBlocks := GlobalAlloc(GMEM_MOVEABLE or GMEM_ZEROINIT,
               BlocksTableLen * SizeOf(tBlockHandle) +
               { Number to allocate initialy }
    SizeOf(tBlockHandle)); { Plus one extra }
    { Calculation of maximum size of item }
    BuffLen := SizeOf(tListItemHeader) + { Item header }
    MaxColNum * SizeOf(Word) + { Tab offset table }
    MaxColNum * SizeOf(Char) + { Tab characters (tab delimiters) }
    RecSize + { Maximum possible record size }
    Reserve; { Plus some additional space }
    { Memory allocation for working buffer }
    hWork := GlobalAlloc(GMEM_MOVEABLE or GMEM_ZEROINIT, BuffLen);
    WorkBuf := GlobalLock(hWork);
    { Get record count in database }
    TotalRecs := GetRecordCount;
    { Memory allocation for primary table }
    hTable := GlobalAlloc(GMEM_MOVEABLE or GMEM_ZEROINIT,
              (TotalRecs * SizeOf(tItemHandle) +
              { Number of records in database }
    SizeOf(tItemHandle))); { Plus one extra }
    { Exit if table is empty }
      if TotalRecs = 0 then
      begin
        GlobalUnlock(hWork);
        GlobalFree(hWork);
        RowNumber := 0;
        Builded := True;
        Exit;
      end;
      { Get table pointer }
    Table := GlobalLock(hTable);
    { Create strip dialog }
    CreateStrip('Reading table...', TotalRecs);
    i := 0;
    k := 0;
    ContLoop := True;
    while ContLoop do
      begin
        Inc(i);
        { Skip filtering records }
          if not SkipRecord(i) then
          begin
          { Create new item }
            hItem := CreateNewItem(WorkBuf, GetRecordNo);
              if hItem < 0 then
              begin
                BuildTable := tMemory;
                ContLoop := False;
              end;
              { Put new item in table }
              if ContLoop then
              begin
                ItemAddress := MemPtr(Table, k);
                plongint(ItemAddress)^ := hItem;
{                Table^[k] := hItem;}
                Inc(k);
              end;
          end;
          { Update strip dialog }
          NewStrip(i);
          { Get next record from database }
          if i < TotalRecs then
            ContLoop := NextRecord;
          if i = TotalRecs then
            ContLoop := False;
      end;
      { Calculate actual number of items in table }
    RowNumber := k;
    { Close strip dialog }
    CloseStrip;
    { release & free allocated memory }
    GlobalUnlock(hWork);
    GlobalFree(hWork);
    { reallocate table memory according to RowNumber }
    GlobalUnlock(hTable);
    hTable := GlobalReAlloc(hTable,
              (RowNumber * SizeOf(tItemHandle) +
              { Actual number of items in the table }
    SizeOf(tItemHandle)), { Plus one extra }
    GMEM_MOVEABLE);
    Builded := True;
  end;

(*----------------------------------------------------------------------------*)
function tListTable.CreateNewItem(Buffer: PChar; ARecNo: LongInt): tItemHandle;
  
  var
    lpBlocks: pBlocksTable;
    Block: tBlockHandle;
    hBlock: tHandle;
    lpBlock: pChar;
    Hndl: tHandle;
    RoomFound: Boolean;
    i, j, index, IndexItem: Integer;
    MaxOffs: Longint;
    CurAddr: PChar;
    CurLen: Word;
    ItemLen, StartOffs: Word;
    FieldsOffs: tTabsOffsTable;
    hItem: tItemHandle;
    lpItem: pChar;
    ItemsNum: Word;
    ItemSize: Word;
    BlockSize: Longint;

  begin
    CreateNewItem := - 1;
    { Loop to place data into memory }
    CurAddr := Buffer;
    ItemLen := 0;
    StartOffs := SizeOf(Word) * ItemsList^.ColNumber;
    for i := 1 to ItemsList^.ColNumber do
      begin
        FieldsOffs[i] := (CurAddr - Buffer) + StartOffs + SizeOf(
                         TListItemHeader);
          case ItemsList^.Items^[i].ItemType of
            ct_String:
            begin
              strcopy(CurAddr, GetRecordField(ARecno, i));
              CurLen := strlen(CurAddr);
            end;
            ct_Bitmap:
            begin
              move(GetRecordField(ARecno, i)^, CurAddr^, SizeOf(HBitmap));
              CurLen := SizeOf(HBitmap);
            end;
          end;
        CurAddr[CurLen] := #8;
        Inc(ItemLen, CurLen + 1);
        CurAddr := Buffer + ItemLen;
      end;
    Inc(ItemLen);
    CurAddr[ItemLen] := #0;
    { Calculate actual item size }
    ItemSize := ItemLen + ItemsList^.ColNumber * SizeOf(Word) + SizeOf(
                tListItemHeader);
    { Lock blocks table }
    lpBlocks := GlobalLock(hBlocks);
    i := - 1;
    RoomFound := False;
    while not RoomFound do
      begin
        Inc(i);
        Block := lpBlocks^[i];
          if i = BlocksNum then
          begin
          { We have no blocks with room for our item }
          { First check if we have space in the block table }
              if i = BlocksTableLen - 1 then
              begin
              { Block table filled out - reallocate BlockTable memory }
                GlobalUnlock(hBlocks);
                Inc(BlocksTableLen, BlocksDelta);
                Hndl := GlobalRealloc(hBlocks,
                        BlocksTableLen * SizeOf(tBlockHandle) + SizeOf(
                        tBlockHandle), GMEM_MOVEABLE);
                  if Hndl = 0 then
                  begin
                    GlobalUnlock(hBlocks);
                    Exit;
                  end;
                hBlocks := Hndl;
                lpBlocks := GlobalLock(hBlocks);
              end;
              { Make memory allocation for new block }
            hBlock := GlobalAlloc(GMEM_MOVEABLE or GMEM_ZEROINIT,
                      SizeOf(tBlockHeader) + Reserve); { Block header size }
              if hBlock = 0 then
              begin
                GlobalUnlock(hBlocks);
                Exit;
              end;
              { Initialize new allocated block header }
            lpBlock := GlobalLock(hBlock);
            pBlockHeader(lpBlock)^.OffsetTable[0] := SizeOf(tBlockHeader);
            GlobalUnlock(hBlock);
            { Put initial data into the block handle }
            Block := MakeLong(0, hBlock);
            lpBlocks^[BlocksNum] := Block;
                                    { Store block handle in the block table }
            Inc(BlocksNum); { We had allocated new block }
            RoomFound := True;
          end
          else
            if ItemsInBlock(Block) < ItemsPerBlock then
            begin
            { This block contain a room for new item }
              RoomFound := True;
            end;
      end;
{ Now "i" contains index of block to store item in a block table and "Block"
    contains this block handle }
    BlockSize := GlobalSize(BlockHandle(Block));
    Hndl := GlobalReAlloc(BlockHandle(Block), BlockSize + ItemSize,
            GMEM_MOVEABLE);
      if Hndl = 0 then
      begin
        GlobalUnlock(hBlocks);
        Exit;
      end;
    hBlock := Hndl;
    lpBlock := GlobalLock(hBlock);
    { Find maximum offset in a block }
    MaxOffs := 0;
    IndexItem := 0;
    for j := 0 to ItemsPerBlock do

      with pBlockHeader(lpBlock)^ do
        if OffsetTable[j] > MaxOffs then
        begin
          MaxOffs := OffsetTable[j];
          IndexItem := j;
        end;
    lpItem := lpBlock + pBlockHeader(lpBlock)^.OffsetTable[IndexItem];
    { fill current item }

    with pListItemHeader(lpItem)^ do
    begin
      RecNo := ARecno;
      HasOwnColor := False;
    end;
    move(FieldsOffs, lpItem[SizeOf(TListItemHeader)], StartOffs);
    move(Buffer^, lpItem[SizeOf(TListItemHeader) + StartOffs], ItemLen);
    { update block header }
    j := - 1;
    index := 0;

    with pBlockHeader(lpBlock)^ do
    repeat
      Inc(j);
        if OffsetTable[j] = 0 then
        index := j;
    until index <> 0;
    pBlockHeader(lpBlock)^.OffsetTable[index] := MaxOffs + ItemSize;
    ItemsNum := ItemsInBlock(Block) + 1;
    GlobalUnlock(hBlock);
    { update blocks table }
    Block := MakeLong(ItemsNum, { New number of items in block }
    hBlock);
{ New block handle - it's important, because this value
                                  could be changed after memory reallocation }
    lpBlocks^[i] := Block;
    GlobalUnlock(hBlocks);
    { Now we can make newly created item handle }
    hItem := MakeLong(IndexItem, i);
    CreateNewItem := hItem;
  end;

(*----------------------------------------------------------------------------*)
function tListTable.DeleteItem(ItemHandle: tItemHandle): Integer;

  var
    Table    : pItemsTable;
    Blocks   : pBlocksTable;
    Block    : tBlockHandle;
    hBlock   : tHandle;
    i, k     : longint;
    Reached  : Boolean;
    hItem    : tItemHandle;
    s        : pchar;
    OffsItem,
    OffsNext,
    Offs     : longint;
    NewSize,
    BlockSize: longint;
    ItemsNum : word;
    ItIndex,
    BlIndex  : word;
    TPtr,
    SPtr     : pointer;

  begin
    DeleteItem := tInvalidItem;
      if not Builded then
      begin
        DeleteItem := tNotBuilded;
        Exit;
      end;
    Table := GlobalLock(hTable);
    Reached := False;
    ItIndex := ItemIndex(ItemHandle);
    BlIndex := ItemBlockIndex(ItemHandle);
    i := 0;
    { Find item in ItemsTable }
    while (i < RowNumber) do
      begin
        hItem := plongint(MemPtr(Table, i))^;
        if hItem = ItemHandle then
          begin
            Reached := True;
            break;
          end;
{        if Reached then
          Table^[i] := Table^[i + 1];
}        Inc(i);
      end;
      if Reached then
       { Shift items table }
        if i <> (RowNumber - 1) then
         begin
           TPtr := MemPtr(Table, i);
           SPtr := MemPtr(Table, i + 1);
           hMove(SPtr, TPtr, (RowNumber - 1 - i) * SizeOf(tItemHandle));
        end;
      if Reached then
      begin
        Dec(RowNumber);
        GlobalUnlock(hTable);
        { Reallocate ItemsTable }
        hTable := GlobalReAlloc(hTable,
                  RowNumber * SizeOf(tItemHandle) + SizeOf(tItemHandle),
                  GMEM_MOVEABLE);
        Blocks := GlobalLock(hBlocks);
        Block := Blocks^[BlIndex];
        hBlock := BlockHandle(Block);
        ItemsNum := ItemsInBlock(Block);
        BlockSize := GlobalSize(hBlock);
        s := GlobalLock(hBlock);
        OffsItem := pBlockHeader(s)^.OffsetTable[ItIndex];
        OffsNext := MaxLongint;
        i := 0;
        while i <= ItemsPerBlock do
          begin
            Offs := pBlockHeader(s)^.OffsetTable[i];
              if (Offs > OffsItem) and (Offs < OffsNext) then
              OffsNext := Offs;
            Inc(i);
          end;
          if OffsNext = MaxLongint then
          OffsNext := OffsItem;
          { Shift items inside the block }
          if OffsItem <> OffsNext then
          { if this is not last item }
          Move((s + OffsNext)^, (s + OffsItem)^, BlockSize - OffsNext);
        NewSize := BlockSize - (OffsNext - OffsItem);
        { Update block offset table }
          if OffsNext <> OffsItem then
          pBlockHeader(s)^.OffsetTable[ItIndex] := 0;
        i := 0;
        while i <= ItemsPerBlock do
          begin
              if pBlockHeader(s)^.OffsetTable[i] > OffsItem then
              Dec(pBlockHeader(s)^.OffsetTable[i], (OffsNext - OffsItem));
            Inc(i);
          end;
        GlobalUnlock(hBlock);
        { Update blocks table }
        Dec(ItemsNum);
        hBlock := GlobalRealloc(hBlock, NewSize, GMEM_MOVEABLE);
        Block := MakeLong(ItemsNum, hBlock);
        Blocks^[ItemBlockIndex(ItemHandle)] := Block;
        GlobalUnlock(hBlocks);
        DeleteItem := tSuccess;
      end
      else
      GlobalUnlock(hTable);
  end;

(*----------------------------------------------------------------------------*)
function tListTable.DestroyTable: Integer;

  var
    Blocks: pBlocksTable;
    i: LongInt;
    Block: tBlockHandle;
    hBlock: tHandle;

  begin
      if not Builded then
      begin
        DestroyTable := tNotBuilded;
        Exit;
      end;
    Blocks := GlobalLock(hBlocks);
    i := 0;
    while i < BlocksNum do
      begin
        Block := Blocks^[i];
        hBlock := BlockHandle(Block);
        GlobalFree(hBlock);
        Inc(i);
      end;
    GlobalUnLock(hBlocks);
    GlobalFree(hBlocks);
    GlobalFree(hTable);
    RowNumber := 0;
    BlocksNum := 0;
    BlocksTableLen := 0;
    Builded := False;
    DestroyTable := tSuccess;
  end;

(*----------------------------------------------------------------------------*)
function tListTable.GetColAlign(ColNum: Integer): Word;
  begin
    GetColAlign := ItemsList^.Items^[ColNum].Align;
  end;

(*----------------------------------------------------------------------------*)
function tListTable.GetColNumber: Integer;
  begin
    GetColNumber := ItemsList^.ColNumber;
  end;

(*----------------------------------------------------------------------------*)
function tListTable.GetColWrap(ColNum: Integer): Word;
  begin
    GetColWrap := ItemsList^.Items^[ColNum].Wrap;
  end;

(*----------------------------------------------------------------------------*)
procedure tListTable.GetFieldString(ItemHandle: tItemHandle; index: Word;
                                    Str: PChar);

  var
    i: Integer;
    s: PChar;
    Offs: Longint;
    hBlock: tHandle;
    Blocks: pBlocksTable;
    Block: tBlockHandle;

  begin
      if not Builded then
      Exit;
    Blocks := GlobalLock(hBlocks);
    Block := Blocks^[ItemBlockIndex(ItemHandle)];
    hBlock := BlockHandle(Block);
    s := GlobalLock(hBlock);
    { Calculation offset inside block }
    Offs := pBlockHeader(s)^.OffsetTable[ItemIndex(ItemHandle)];
    { Now make s a pointer to item }
    s := s + Offs;
    { Calculation offset inside item }
    offs := Word(s[SizeOf(TListItemHeader) + index * SizeOf(Word)]);
    { Make s a pointer to tab string }
    s := s + Offs;
    i := 0;
    while s[i] <> #8 do
      begin
        Str[i] := s[i];
        Inc(i);
      end;
    Str[i] := #0;
    GlobalUnlock(hBlock);
    GlobalUnlock(hBlocks);
  end;

(*----------------------------------------------------------------------------*)
function tListTable.GetItemColor(ItemHandle: tItemHandle;
                                 var ItemColor: Longint): Boolean;

  var
    hBlock: tHandle;
    Blocks: pBlocksTable;
    Block: tBlockHandle;
    P: pListItemHeader;
    s: pChar;
    Offs: Longint;

  begin
      if not Builded then
      Exit;
    Blocks := GlobalLock(hBlocks);
    Block := Blocks^[ItemBlockIndex(ItemHandle)];
    hBlock := BlockHandle(Block);
    s := GlobalLock(hBlock);
    { Calculation offset inside block }
    Offs := pBlockHeader(s)^.OffsetTable[ItemIndex(ItemHandle)];
    { Now make s a pointer to item }
    s := s + Offs;
    P := pListItemHeader(s);
      if P^.HasOwnColor then
      begin
        ItemColor := P^.Color;
        GetItemColor := True;
      end
      else
      GetItemColor := False;
    GlobalUnlock(hBlock);
    GlobalUnlock(hBlocks);
  end;

(*----------------------------------------------------------------------------*)
function tListTable.GetItemHandle(Row: Longint): tItemHandle;

  var
    Table: pItemsTable;
    hItem: tItemHandle;

  begin
      if not Builded then
      Exit;
    Table := GlobalLock(hTable);
    hItem := plongint(MemPtr(Table, Row))^;
    GlobalUnlock(hTable);
    GetItemHandle := hItem;
  end;

(*----------------------------------------------------------------------------*)
function tListTable.GetItemRecno(ItemHandle: tItemHandle): Longint;

  var
    hBlock: tHandle;
    Blocks: pBlocksTable;
    Block: tBlockHandle;
    P: pListItemHeader;
    s: pChar;
    Offs: Longint;
  
  begin
      if not Builded then
      Exit;
    Blocks := GlobalLock(hBlocks);
    Block := Blocks^[ItemBlockIndex(ItemHandle)];
    hBlock := BlockHandle(Block);
    s := GlobalLock(hBlock);
    { Calculation offset inside block }
    Offs := pBlockHeader(s)^.OffsetTable[ItemIndex(ItemHandle)];
    { Now make s a pointer to item }
    s := s + Offs;
    P := pListItemHeader(s);
    GetItemRecno := P^.RecNo;
    GlobalUnlock(hBlock);
    GlobalUnlock(hBlocks);
  end;

(*----------------------------------------------------------------------------*)
function tListTable.GetItemType(ColNum: Integer): TItemType;
  begin
    GetItemType := ItemsList^.Items^[ColNum].ItemType;
  end;

(*----------------------------------------------------------------------------*)
function tListTable.GetRows: Longint;
  begin
    GetRows := RowNumber;
  end;

(*----------------------------------------------------------------------------*)
function tListTable.ID2Index(ID: Integer): Integer;
  
  var
    i: Integer;
  
  begin
    for i := 1 to ItemsList^.ColNumber do
        if ItemsList^.Items^[i].ItemID = ID then
        begin
          ID2Index := i;
          Exit;
        end;
  end;

(*----------------------------------------------------------------------------*)
function tListTable.Index2ID(index: Integer): Integer;
  begin
    Index2ID := ItemsList^.Items^[index].ItemID;
  end;

(*----------------------------------------------------------------------------*)
procedure tListTable.SetSortOrder(ColID: Integer);

  var
    i: Integer;

  begin
    for i := 1 to ItemsList^.ColNumber do
      begin
          if ItemsList^.Items^[i].ItemID <> ColID then
          ItemsList^.Items^[i].Sort := False
          else
          ItemsList^.Items^[i].Sort := True;
      end;
  end;

(*----------------------------------------------------------------------------*)
procedure tListTable.SetItemColor(ItemHandle: tItemHandle; AColor: Longint;
                                  OwnColor: Boolean);
  
  var
    hBlock: tHandle;
    Blocks: pBlocksTable;
    Block: tBlockHandle;
    P: pListItemHeader;
    s: pChar;
    Offs: Longint;

  begin
      if not Builded then
      Exit;
    Blocks := GlobalLock(hBlocks);
    Block := Blocks^[ItemBlockIndex(ItemHandle)];
    hBlock := BlockHandle(Block);
    s := GlobalLock(hBlock);
    { Calculation offset inside block }
    Offs := pBlockHeader(s)^.OffsetTable[ItemIndex(ItemHandle)];
    { Now make s a pointer to item }
    s := s + Offs;
    P := pListItemHeader(s);
    { fill item color }

    with P^ do
    begin
      HasOwnColor := OwnColor;
      Color := AColor;
    end;
    GlobalUnlock(hBlock);
    GlobalUnlock(hBlocks);
  end;

(*----------------------------------------------------------------------------*)
function tListTable.UpdateItem(ItemHandle: tItemHandle; RecNo: Longint;
                               NewItem: Boolean): tItemHandle;
  
  var
    BuffLen,
    RecSize     : word;
    hWork       : tHandle;
    WorkBuf     : pchar;
    Table       : pItemsTable;
    Hndl        : tHandle;
    k, index    : longint;
    i           : longint;
    hItem       : tItemHandle;
    ItemAddress : plongint;

  begin
      if not Builded then
      Exit;
    RecSize := GetRecordWidth;
      if RecSize < 1 then
      begin
      { Invalid record size given }
        UpdateItem := 0;
        Exit;
      end;
      if not NewItem then
      begin
        Table := GlobalLock(hTable);
        i := 0;
        index := - 1;
        { Find item in ItemsTable }
        while i < RowNumber do
          begin
            hItem := plongint(MemPtr(Table, i))^;
              if hItem = ItemHandle then
              index := i;
            Inc(i);
          end;
        GlobalUnlock(hTable);
          if index < 0 then
          begin
            UpdateItem := 0;
            Exit;
          end;
      end;
      { Calculation of maximum size of item }
    BuffLen := SizeOf(tListItemHeader) + { Item header }
    MaxColNum * SizeOf(Word) + { Tab offset table }
    MaxColNum * SizeOf(Char) + { Tab characters (tab delimiters) }
    RecSize + { Maximum possible record size }
    Reserve; { Plus some additional space }
    { Memory allocation for working buffer }
    hWork := GlobalAlloc(GMEM_MOVEABLE or GMEM_ZEROINIT, BuffLen);
      if hWork = 0 then
      begin
        UpdateItem := 0;
        Exit;
      end;
    WorkBuf := GlobalLock(hWork);
      if NewItem then
      begin
        Inc(RowNumber);
        { Add new item to table }
        Hndl := GlobalReAlloc(hTable,
                RowNumber * SizeOf(tItemHandle) + SizeOf(tItemHandle),
                GMEM_NODISCARD);
          if Hndl = 0 then
          begin
            GlobalUnlock(hWork);
            GlobalFree(hWork);
            UpdateItem := 0;
            Exit;
          end;
        hTable := Hndl;
        index := RowNumber - 1;
      end;
    hItem := CreateNewItem(WorkBuf, RecNo);
      if hItem < 0 then
      begin
        GlobalUnlock(hWork);
        GlobalFree(hWork);
        UpdateItem := 0;
        Exit;
      end;
    Table := GlobalLock(hTable);
    ItemAddress := MemPtr(Table, index);
    ItemAddress^ := hItem;
{    Table^[index] := hItem;}
    UpdateItem := hItem;
    GlobalUnlock(hTable);
    { release & free allocated memory }
    GlobalUnlock(hWork);
    GlobalFree(hWork);
  end;
  { !!! Below functions must be overriden by decenants !!! }

(*----------------------------------------------------------------------------*)
procedure tListTable.CloseStrip;
  begin
  { This abstract function doesn't know how to close strip dialog box }
  end;

(*----------------------------------------------------------------------------*)
procedure tListTable.CreateStrip(ATitle: PChar; ATotal: Longint);
  begin
  { This abstract function doesn't know how to create strip dialog box }
  end;

(*----------------------------------------------------------------------------*)
function tListTable.GetBmpField(RecNo: LongInt; index: Word): HBitmap;
  begin
  { This abstract function doesnt know how to get bitmap handle }
  end;

(*----------------------------------------------------------------------------*)
function tListTable.GetRecordCount;
  begin
  { This abstract function doesn't know how to get record count }
    GetRecordCount := 0;
  end;

(*----------------------------------------------------------------------------*)
function tListTable.GetRecordField;
  begin
  { This abstract function doesn't know how to get record field }
    GetRecordField := #0;
  end;

(*----------------------------------------------------------------------------*)
function tListTable.GetRecordNo: Longint;
  begin
  { This abstract function doesn't know how to get current record number }
    GetRecordNo := 0;
  end;

(*----------------------------------------------------------------------------*)
function tListTable.GetRecordWidth;
  begin
  { This abstract function doesn't know how to get record width }
    GetRecordWidth := 0;
  end;

(*----------------------------------------------------------------------------*)
function tListTable.GetStrField(RecNo: LongInt; index: Word): PChar;
  begin
  { This abstract function doesnt know how to get string }
  end;

(*----------------------------------------------------------------------------*)
procedure tListTable.NewStrip(Current: LongInt);
  begin
  { This abstract function doesn't know how to update strip dialog box }
  end;

(*----------------------------------------------------------------------------*)
function tListTable.NextRecord: Boolean;
  begin
  { This abstract function doesn't know how to get next record }
  end;

(*----------------------------------------------------------------------------*)
function tListTable.SkipRecord;
  begin
  { This abstract function doesn't know wether to skip or not to skip record }
    SkipRecord := True;
  end;

(*----------------------------------------------------------------------------*)
function tListTable.MemPtr(Address: pointer; Idx: longint) : pointer;
  begin
     MemPtr := LIncPtr(Address, Idx * SizeOf(pointer))
  end;

end.
