{$A+,B-,F-,G+,I-,P+,Q-,R-,S-,T-,V-,X+}

Unit MEM32bit;

{ Encapsulated access to the linear memory through WINMEM32.DLL       }
{ (c) Jochen Magnus (CIS 100333,2377), Lahnstein/Germany 10/94        }
{ All rights reserved. You are free to use, but not to sell the Unit. }

INTERFACE
uses
   winTypes,winProcs,objects,winmem32;

type
   PMem32   =  ^TMem32;
   TMem32   =  Object(TObject)               { Ancestor TObject is not a must;
                                               implemented here for the option
                                               of inserting PMem32 in TCollection }
                  { protected ;-}
                  msize,                     { actually allocated memory }
                  size     :  longint;       { maximum for reallocation  }
                  selec       :  word;       { selector }

                  { public }
                  constructor Init(initSize,maxSize:longint; var res:word);
                  constructor Copy(const sourceObj:PMem32; var res:word);
                  destructor  Done; VIRTUAL;
                  function    ReAlloc(newSize:longint):word;
                  function    MemCopy(source,dest,len:longint):word;
                  function    MemFill(dest,len:longint; pattern:byte):word;
                  function    SetData(var source; dest:longint; len:word):word;
                  function    GetData(source:longint; var dest; var len:word):word;
                  function    ReadFromFile(fname:PChar; fofs,dest,len:longint):word;
                  function    WriteToFile(source,len:longint; fname:PChar;
                                          fofs:longint):word;
                  function    TellSize:longint;
               end;


IMPLEMENTATION
const
   maxMemSize     =     $00FF0000;


function _hread(hf:integer; hpvBuffer:pointer; cbBuffer:longint):longint; FAR;
         EXTERNAL 'KERNEL' index 349;
function _hwrite(hf:integer; hpvBuffer:pointer; cbBuffer:longint):longint; FAR;
         EXTERNAL 'KERNEL' index 350;
procedure AHIncr; FAR;
         EXTERNAL 'KERNEL' index 114;

{$L mem32bit}
procedure Mem32Copy(ssel:word; sofs:longint;
                    tsel:word; tofs, len:longint); NEAR; EXTERNAL;
procedure Mem32Fill(sel:word; ofs,len:longint; pattern:byte); NEAR; EXTERNAL;


function _LFileSize(handle:integer; fname:PChar):longint;
{ uses either handle or - if handle=0 - filename }
var
   curPos : longint;
   sopen    :  boolean;
begin
   if handle=0 then
   begin
      sopen:=true;
      handle:=_LOpen(fname,OF_READ or OF_SHARE_DENY_NONE);
   end else
      sopen:=false;

   if handle<=0 then
   begin
      if handle=0 then handle:=-1;
      _LFileSize:=handle; EXIT
   end;

   curPos:=_llseek(handle,0,1);
   _LFileSize:=_llseek(handle,0,2);
   _Llseek(handle,curPos,0);

   if sopen then _LClose(handle);
end;


constructor TMem32.Init(initSize,maxSize:longint; var res:word);
begin
   if not inherited Init then
   begin
      res:=$FFFF; FAIL;
   end;

   if (maxSize>maxMemSize) or (maxSize=0) then maxSize:=maxMemSize;
   size:=initSize; msize:=maxSize;
   res:=Global32Alloc(size,@selec,mSize,0);
   if res<>0 then FAIL;
end;


constructor TMem32.Copy(const sourceObj:PMem32; var res:word);
begin
   if sourceObj=NIL then
   begin
      res:=$FFFE; FAIL;
   end;

   if not inherited Init then
   begin
      res:=$FFFF; FAIL;
   end;

   size:=sourceObj^.size;
   msize:=sourceObj^.msize;

   res:=Global32Alloc(size,@selec,mSize,0);
   if res<>0 then FAIL;

   Mem32Copy(sourceObj^.selec,0,selec,0,size);
end;


destructor TMem32.Done;
begin
   Global32Free(selec,0);
   inherited done;
end;


function TMem32.ReAlloc(newSize:longint):word;
var
   res   :  word;
begin
    if newSize>msize then
      res:=wm32_invalid_arg
    else
      res:=Global32ReAlloc(selec,newSize,0);

    if res=0 then size:=newSize;
    ReAlloc:=res;
end;


function TMem32.MemCopy(source,dest,len:longint):word;
var
   res   :  word;
begin
   if (source<0) or (dest<0) then
   begin
      MemCopy:=wm32_invalid_arg; EXIT;
   end;

   if len<0 then
   begin
      if dest<source then
         len:=size-source
      else
         len:=size-dest;
   end;

   MemCopy:=0;
   if (source=dest) or (len=0) then EXIT;

   if dest+len>size then
   begin
      res:=ReAlloc(dest+len);
      if res<>0 then
      begin
         MemCopy:=res; EXIT;
      end;
   end;

   Mem32Copy(selec,source,selec,dest,len);
end;


function TMem32.MemFill(dest,len:longint; pattern:byte):word;
var
   res   :  word;
begin
   if len<0 then len:=size-dest;

   if dest<0 then
   begin
      MemFill:=wm32_invalid_arg; EXIT;
   end;

   if dest+len>size then
   begin
      res:=ReAlloc(dest+len);
      if res<>0 then
      begin
         MemFill:=res; EXIT;
      end;
   end;

   Mem32Fill(selec,dest,len,pattern);
   MemFill:=0;
end;


function TMem32.SetData(var source; dest:longint; len:word):word;
var
   res   :  word;
begin
   if (dest<0) or (len<0) or (@source=NIL) then
   begin
      SetData:=wm32_invalid_arg; EXIT;
   end;

   if dest+len>size then
   begin
      res:=ReAlloc(dest+len);
      if res<>0 then
      begin
         SetData:=res; EXIT;
      end;
   end;

   Mem32Copy(seg(source),ofs(source),selec,dest,len);
   SetData:=0;
end;


function TMem32.GetData(source:longint; var dest; var len:word):word;
begin
   if (source<0) or (len<0) or  (@dest=NIL) then
   begin
      GetData:=wm32_invalid_arg; EXIT;
   end;

   if source+len>size then len:=size-source;

   Mem32Copy(selec,source,seg(dest),ofs(dest),len);
   GetData:=0;
end;


function TMem32.ReadFromFile(fname:PChar; fofs,dest,len:longint):word;
var
   res   :  longint;
   destSeg,
   destOfs  :  word;
   handle   :  integer;
begin
   if (dest<0) or (fname=NIL) or (fname[0]=#0) then
   begin
      ReadFromFile:=wm32_invalid_arg; EXIT;
   end;

   if len<0 then
   begin
      len:=_LFileSize(0,fname);
      if len<0 then
      begin
         ReadFromFile:=-len; EXIT;
      end;
   end;

   if dest+len>size then
   begin
      res:=ReAlloc(dest+len);
      if res<>0 then
      begin
         ReadFromFile:=res; EXIT;
      end;
   end;

   destSeg:=selec+(dest div $10000)*ofs(AHincr);
   destOfs:=dest mod $10000;

   handle:=_lopen(fname,OF_READ or OF_SHARE_DENY_NONE);
   if handle>0 then
   begin
      res:=_llseek(handle,fofs,0);
      if res=fofs then
      begin
         res:=_hread(handle,ptr(destSeg,destOfs),len);
         _lclose(handle);
         if res=len then
            ReadFromFile:=0
         else
            ReadFromFile:=100;
      end else
         ReadFromFile:=res;
   end else
      ReadFromFile:=103;
end;


function TMem32.WriteToFile(source,len:longint; fname:PChar; fofs:longint):word;
var
   res         :  longint;
   sourceSeg,
   sourceOfs   :  word;
   handle         :  integer;
begin
   if (source<0) or (source+len>size) or (fname=NIL) or (fname[0]=#0) then
   begin
      WriteToFile:=wm32_invalid_arg; EXIT;
   end;
   if len<0 then len:=size-source;

   sourceSeg:=selec+(source div $10000)*ofs(AHincr);
   sourceOfs:=source mod $10000;

   handle:=_lopen(fname,OF_WRITE or OF_SHARE_DENY_WRITE);
   if handle<=0 then handle:=_lcreat(fname,0);
   if handle>0 then
   begin
      res:=_llseek(handle,fofs,0);
      if res=fofs then
      begin
         res:=_hwrite(handle,ptr(sourceSeg,sourceOfs),len);
         _lclose(handle);
         if res=len then
            WriteToFile:=0
         else
            WriteToFile:=101;
      end else
         WriteToFile:=res;
   end else
      WriteToFile:=103;
end;


function TMem32.TellSize:longint;
begin
   TellSize:=size;
end;


end.
